/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.text;

import java.awt.*;

import javax.swing.JComponent;
import javax.swing.text.*;

import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.painter.text.SubstanceTextPainter;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.*;

/**
 * Text utilities for painting views of text components with
 * {@link SubstanceTextPainter}-based approach.
 * 
 * @author Kirill Grouchnikov
 */
public class TextUtilities {
	/**
	 * Gets the view component.
	 * 
	 * @param view
	 *            Text view.
	 * @return View component.
	 */
	private static JComponent getJComponent(View view) {
		if (view != null) {
			Component component = view.getContainer();
			if (component instanceof JComponent) {
				return (JComponent) component;
			}
		}
		return null;
	}

	/**
	 * Draws the specified text segment.
	 * 
	 * @param view
	 *            Text view.
	 * @param s
	 *            Text segment.
	 * @param x
	 *            X coordinate for the text painting.
	 * @param y
	 *            Y coordinate for the text painting.
	 * @param g
	 *            Graphic context.
	 * @param e
	 *            Tab expander.
	 * @param startOffset
	 *            The starting offset.
	 * @return X coordinate for drawing the next segment.
	 */
	public static int drawTabbedText(View view, Segment s, int x, int y,
			Graphics g, TabExpander e, int startOffset) {
		JComponent component = getJComponent(view);
		FontMetrics metrics = g.getFontMetrics();
		int nextX = x;
		char[] txt = s.array;
		int txtOffset = s.offset;
		int flushLen = 0;
		int flushIndex = s.offset;
		int spaceAddon = 0;
		int spaceAddonLeftoverEnd = -1;
		int startJustifiableContent = 0;
		int endJustifiableContent = 0;
		int n = s.offset + s.count;

		for (int i = txtOffset; i < n; i++) {
			if (txt[i] == '\t'
					|| ((spaceAddon != 0 || i <= spaceAddonLeftoverEnd)
							&& (txt[i] == ' ') && startJustifiableContent <= i && i <= endJustifiableContent)) {
				if (flushLen > 0) {
					nextX = drawChars(component, g, txt, flushIndex, flushLen,
							x, y);
					flushLen = 0;
				}
				flushIndex = i + 1;
				if (txt[i] == '\t') {
					if (e != null) {
						nextX = (int) e.nextTabStop(nextX, startOffset + i
								- txtOffset);
					} else {
						nextX += metrics.charWidth(' ');
					}
				} else if (txt[i] == ' ') {
					nextX += metrics.charWidth(' ') + spaceAddon;
					if (i <= spaceAddonLeftoverEnd) {
						nextX++;
					}
				}
				x = nextX;
			} else if ((txt[i] == '\n') || (txt[i] == '\r')) {
				if (flushLen > 0) {
					nextX = drawChars(component, g, txt, flushIndex, flushLen,
							x, y);
					flushLen = 0;
				}
				flushIndex = i + 1;
				x = nextX;
			} else {
				flushLen += 1;
			}
		}
		if (flushLen > 0) {
			nextX = drawChars(component, g, txt, flushIndex, flushLen, x, y);
		}

		// SubstanceTextPainter textPainter = SubstanceLookAndFeel
		// .getCurrentTextPainter();
		// if (textPainter.needsBackgroundImage()) {
		// textPainter.setBackgroundImage(component, null, component
		// .getBackground(), false, 0, 0, null);
		// }
		// Graphics2D g2d = (Graphics2D) g.create();
		// textPainter.paintText(g, component, textLines);
		// g2d.dispose();
		return nextX;
	}

	/**
	 * Draws the specified text characters.
	 * 
	 * @param c
	 *            Component.
	 * @param g
	 *            Graphic context.
	 * @param data
	 *            Characters to draw.
	 * @param offset
	 *            Offset into the characters array.
	 * @param length
	 *            Number of characters to draw.
	 * @param x
	 *            X coordinate for the text painting.
	 * @param y
	 *            Y coordinate for the text painting.
	 * @return X coordinate for drawing the next segment.
	 */
	public static int drawChars(JComponent c, Graphics g, char[] data,
			int offset, int length, int x, int y) {
		if (length <= 0) { // no need to paint empty strings
			return x;
		}
		FontMetrics fm = g.getFontMetrics();
		int nextX = x + g.getFontMetrics().charsWidth(data, offset, length);
		SubstanceTextPainter textPainter = SubstanceLookAndFeel
				.getCurrentTextPainter();
		textPainter.attachText(c, new Rectangle(x, y - fm.getAscent(), nextX
				- x, fm.getHeight()), new String(data, offset, length), -1, g
				.getFont(), g.getColor(), null);
		return nextX;
	}

	public static int drawSelectedText(Graphics g, View view,
			TabExpander tabExpander, int x, int y, int p0, int p1)
			throws BadLocationException {
		JTextComponent host = (JTextComponent) view.getContainer();
		Highlighter h = host.getHighlighter();
		g.setFont(host.getFont());
		Caret c = host.getCaret();
		ComponentState hostState = host.isEnabled() ? ComponentState.DEFAULT
				: ComponentState.DISABLED_UNSELECTED;
		SubstanceTheme theme = SubstanceThemeUtilities
				.getTheme(host, hostState);
		Color color = theme.getForegroundColor();
		if (c.isSelectionVisible() && h != null) {
			color = theme.getSelectionForegroundColor();
		}
		g.setColor(color);
		Document doc = view.getDocument();
		Segment s = new Segment();
		doc.getText(p0, p1 - p0, s);
		int ret = TextUtilities.drawTabbedText(view, s, x, y, g, tabExpander,
				p0);
		// System.out.println("\tSelected " + p0 + "-" + p1);
		return ret;
	}

	public static int drawUnselectedText(Graphics g, View view,
			TabExpander tabExpander, int x, int y, int p0, int p1)
			throws BadLocationException {
		JTextComponent host = (JTextComponent) view.getContainer();
		ComponentState hostState = host.isEnabled() ? ComponentState.DEFAULT
				: ComponentState.DISABLED_UNSELECTED;
		Color unselected = SubstanceCoreUtilities.getForegroundColor(host,
				hostState, hostState);
		g.setColor(unselected);
		Document doc = view.getDocument();
		Segment s = new Segment();
		doc.getText(p0, p1 - p0, s);
		int ret = TextUtilities.drawTabbedText(view, s, x, y, g, tabExpander,
				p0);
		// System.out.println("\tUnselected " + p0 + "-" + p1);
		return ret;
	}

}
