/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;

import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Base class for gradient painters that overlays wave gradient on top of some
 * other gradient painter.This class is part of officially supported API. The
 * implementation borrows heavily from <a
 * href="http://weblogs.java.net/blog/campbell/archive/2006/07/java_2d_tricker.html">this</a>.
 * 
 * @author Kirill Grouchnikov.
 * @author Chris Campbell.
 */
public abstract class WaveDelegateGradientPainter extends BaseGradientPainter {
	/**
	 * Transformation on a color scheme.
	 * 
	 * @author Kirill Grouchnikov.
	 */
	public static interface ColorSchemeTransformation {
		/**
		 * Transforms the specified color scheme.
		 * 
		 * @param scheme
		 *            Color scheme.
		 * @return Transformed color scheme.
		 */
		public ColorScheme transform(ColorScheme scheme);
	}

	/**
	 * Display name of <code>this</code> painter.
	 */
	protected String painterName;

	/**
	 * Optional scheme transformation - may be <code>null</code>.
	 */
	protected ColorSchemeTransformation transformation;

	/**
	 * Mandatory delegate painter.
	 */
	protected SubstanceGradientPainter delegate;

	/**
	 * Creates a new wave-overlaying painter.
	 * 
	 * @param painterName
	 *            Painter display name.
	 * @param transformation
	 *            Optional scheme transformation. If <code>null</code>, the
	 *            result will be identical to using the delegate painter.
	 * @param delegate
	 *            Delegate painter.
	 */
	public WaveDelegateGradientPainter(String painterName,
			ColorSchemeTransformation transformation,
			SubstanceGradientPainter delegate) {
		this.painterName = painterName;
		this.transformation = transformation;
		this.delegate = delegate;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return this.painterName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getContourBackground(int,
	 *      int, java.awt.Shape, boolean, org.jvnet.substance.color.ColorScheme,
	 *      org.jvnet.substance.color.ColorScheme, float, boolean, boolean)
	 */
	public BufferedImage getContourBackground(int width, int height,
			Shape contour, boolean isFocused, ColorScheme colorScheme1,
			ColorScheme colorScheme2, float cyclePos, boolean hasShine,
			boolean useCyclePosAsInterpolation) {

		BufferedImage mixResult = this.getMixContourBackground(width, height,
				contour, isFocused, colorScheme1, colorScheme2, cyclePos,
				hasShine, useCyclePosAsInterpolation);
		if (mixResult != null)
			return mixResult;

		// ColorScheme interpolationScheme1 = colorScheme1;
		// ColorScheme interpolationScheme2 = useCyclePosAsInterpolation ? colorScheme2
		// : colorScheme1;
		//
		// double cycleCoef = 1.0 - cyclePos / 10.0;
		//
		// Color topBorderColor = SubstanceColorUtilities.getTopBorderColor(
		// interpolationScheme1, interpolationScheme2, cycleCoef);
		// Color midBorderColor = SubstanceColorUtilities.getMidBorderColor(
		// interpolationScheme1, interpolationScheme2, cycleCoef);
		// Color bottomBorderColor = SubstanceColorUtilities.getBottomBorderColor(
		// interpolationScheme1, interpolationScheme2, cycleCoef);

		BufferedImage topFullImage = this.delegate.getContourBackground(width,
				height, contour, isFocused, colorScheme1, colorScheme2,
				cyclePos, hasShine, useCyclePosAsInterpolation);

		GeneralPath clipBottom = new GeneralPath();
		clipBottom.moveTo(0, height);
		clipBottom.lineTo(width, height);
		clipBottom.lineTo(width, 0);
		clipBottom.curveTo(5 * width / 6, height / 3, 3 * width / 4,
				height / 2, width / 2, height / 2);
		clipBottom.curveTo(width / 3, height / 2, width / 4, height, 0,
				7 * height / 8);
		clipBottom.lineTo(0, height);

		BufferedImage clipShapeBottom = SubstanceCoreUtilities.createClipImage(
				clipBottom, width, height);

		BufferedImage bottomImage = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D bottomGraphics = (Graphics2D) bottomImage.getGraphics();
		bottomGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		ColorScheme bottomColorScheme1 = (this.transformation == null) ? colorScheme1
				: this.transformation.transform(colorScheme1);
		ColorScheme bottomColorScheme2 = (this.transformation == null) ? colorScheme1
				: this.transformation.transform(colorScheme2);
		BufferedImage bottomFullImage = this.delegate.getContourBackground(
				width, height, contour, isFocused, bottomColorScheme1,
				bottomColorScheme2, cyclePos, hasShine,
				useCyclePosAsInterpolation);

		// Render our clip shape into the image. Note that we enable
		// antialiasing to achieve the soft clipping effect. Try
		// commenting out the line that enables antialiasing, and
		// you will see that you end up with the usual hard clipping.
		bottomGraphics.setComposite(AlphaComposite.Src);
		bottomGraphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);
		bottomGraphics.drawImage(clipShapeBottom, 0, 0, null);

		// Here's the trick... We use SrcAtop, which effectively uses the
		// alpha value as a coverage value for each pixel stored in the
		// destination. For the areas outside our clip shape, the destination
		// alpha will be zero, so nothing is rendered in those areas. For
		// the areas inside our clip shape, the destination alpha will be fully
		// opaque, so the full color is rendered. At the edges, the original
		// antialiasing is carried over to give us the desired soft clipping
		// effect.
		bottomGraphics.setComposite(AlphaComposite.SrcAtop);
		bottomGraphics.drawImage(bottomFullImage, 0, 0, null);

		BufferedImage image = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		graphics.setClip(contour);
		graphics.drawImage(topFullImage, 0, 0, null);
		graphics.drawImage(bottomImage, 0, 0, null);

		graphics.setClip(null);

		// // Draw border
		// GradientPaint gradientBorderTop = new GradientPaint(0, 0,
		// topBorderColor, 0, height / 2, midBorderColor);
		// graphics.setPaint(gradientBorderTop);
		// graphics.setClip(0, 0, width, height / 2);
		// graphics.draw(contour);
		//
		// GradientPaint gradientBorderBottom = new GradientPaint(0, height / 2,
		// midBorderColor, 0, height - 2, bottomBorderColor);
		// graphics.setPaint(gradientBorderBottom);
		// graphics.setClip(0, height / 2, width, 1 + height / 2);
		// graphics.draw(contour);

		return image;
	}
}
