#include "pdflib.h"
#include "treedraw.h"
#include <time.h>
#include "FL/Fl_Native_File_Chooser.H"
#include "FL/Fl_Paged_Device.H"
#include <ctype.h>

static PDF *pdf = NULL;

extern char *prepare_ps_or_pdf_font(int font_num);

int pdf_printout(SEA_VIEW *view, const char *filename, 
	int fontsize, int block_size, paperformat pageformat, int vary_only, int ref0, int pdfkindvalue);
static void color_pdf_display(PDF *pdf, SEA_VIEW *view, int (*calc_color_function)( int ), char *oneline, 
	int widnames, double x, double y, int fontsize, double char_width, double descender,int num, int current);
static int calc_vary_lines(int *vary_pos, int widpos);
static void out_vary_pos(int *vary_pos, int widnames, int widpos, int nl, PDF *pdf, FILE *textfile);


int pdf_printout(SEA_VIEW *view, const char *filename, 
	int fontsize, int block_size, paperformat pageformat, int vary_only, int ref0, int pdfkindvalue)
{
int num, i, j, k, current, max_seq_length, fin, curr_lines, widnames, 
	res_per_line, nl, firstpage, lines_per_page, use_pdf;
FILE *textfile = NULL;
time_t heure;
static char unnamed[] = "<unnamed>";
static char num_line[200];
int lettre, char_per_line;
short *vary_need = NULL;
int *vary_pos; /* rang ds alignement de la colonne imprime */
char oneline[500];
int (*calc_color_function)(int);
double	char_width, width, height, descender, margin = 25;
const char	*fontname, *encoding;
int		font;

    fontname	= "Courier";
#ifdef __APPLE__
    encoding	= "macroman";
#else
    encoding	= "iso8859-1";
#endif
if(view->tot_seqs == 0) return 0;
if(view->protein) calc_color_function = get_color_for_aa;
else  calc_color_function = get_color_for_base;
if(pageformat == A4) { width = a4_width; height = a4_height; }
else /* LETTER */ { width = letter_width; height = letter_height; }
lines_per_page = (int)((height - 2*margin) / fontsize + 0.5);

use_pdf = (pdfkindvalue != TEXT_ONLY);
if(use_pdf) {
	pdf = PDF_new();
	if(pdf == NULL) return TRUE;
	if( PDF_begin_document(pdf, filename, 0, "compatibility=1.3") == -1) {
		PDF_delete(pdf);
		fl_alert("Error opening %s for writing\n", filename);
		return TRUE;
		}
	}
else {
	textfile = fopen(filename, "w");
	if(textfile == NULL) return TRUE;
	}

//PDF_TRY(pdf) {
if( (!use_pdf) || (pdf && (setjmp(pdf_jbuf(pdf)->jbuf) == 0))) {
if(use_pdf) {
	PDF_set_info(pdf, "Title", PREPARE_LABEL(view->masename) );
	PDF_set_info(pdf, "Creator", "seaview");
	font = PDF_load_font(pdf, fontname, 0, encoding, "");
	char_width = PDF_stringwidth(pdf, "X", font, (double)fontsize);
	char_per_line = (int)((width - 2*margin) / char_width + 0.5);
	descender = PDF_get_value(pdf, "descender", font) * fontsize;

	PDF_begin_page_ext(pdf, width, height, "");
	PDF_setfont(pdf, font, (double)fontsize);
	PDF_set_text_pos(pdf, margin, height - margin);
	}
else char_per_line = 90;
firstpage = TRUE;

if(ref0 < 0) vary_only = FALSE;
time(&heure);
sprintf(oneline,"Alignment: %s", view->masename == NULL ? unnamed : PREPARE_LABEL(view->masename) );
if(use_pdf) PDF_continue_text(pdf, oneline);
else {fputs(oneline, textfile); fputs("\n", textfile);}
curr_lines = 1;
if(vary_only) {
	const char fixed[] = "Displaying variable sites only.";
	if(use_pdf) PDF_continue_text(pdf, fixed);
	else {fputs(fixed, textfile); fputs("\n", textfile);}
	++curr_lines;
	}
sprintf(oneline,"Seaview [blocks=%d fontsize=%d %s] on %s",
		block_size, fontsize, pageformat == A4 ? "A4" : "LETTER", ctime(&heure));
if(use_pdf) {
	PDF_continue_text(pdf, oneline);
	PDF_continue_text(pdf, "");
	}
else {
	fputs("Seaview text-only output\n", textfile);
	}
curr_lines += 2;
max_seq_length = 0; widnames = 0;
for(i=0; i < view->tot_seqs; i++) {
	if(view->each_length[i] > max_seq_length) max_seq_length = view->each_length[i];
	if( ( fin=strlen(view->seqname[i]) ) > widnames) widnames = fin;
	}
widnames += 2;
if(vary_only) {
	vary_need = (short *)calloc(max_seq_length, sizeof(short));
	if(vary_need == NULL) return TRUE;
	vary_pos = (int *)calloc(char_per_line, sizeof(int));
	if(vary_pos == NULL) return TRUE;
	for(i = 0; i < max_seq_length; i++) {
		for(num = 0; num < view->tot_seqs; num++) {
			if( toupper(view->sequence[num][i]) != toupper(view->sequence[ref0][i]) ) { 
				vary_need[i] = TRUE;
				break;
				}
			}
		}
	}
/* nombre max de blocks qui tiennent sur une ligne de cpl chars */
fin = (char_per_line - widnames + 1) / (block_size + 1);
if(fin < 1) { /* garde fou */
	fin = 1; block_size = char_per_line - widnames;
	}
res_per_line = fin * block_size;
current = 0; 
while( current < max_seq_length ) {
	nl = 1;
	if(vary_only) { 
		memset(vary_pos, 0, res_per_line * sizeof(int) );
		i = -1; j = 0; k = 0;
		while( j < res_per_line) {
			if(current + i >= max_seq_length) break;
			if( !vary_need[current + ++i] ) continue;
			j++;
			vary_pos[k++] = current + i + 1;
			if( j % block_size == 0) k++;
			}
		nl = calc_vary_lines(vary_pos,  k);
		}
	if( use_pdf && (!firstpage) && (curr_lines + view->tot_seqs + nl > lines_per_page)) {
		PDF_end_page_ext(pdf, "");
		PDF_begin_page_ext(pdf, width, height, "");
		PDF_setfont(pdf, font, (double)fontsize);
		PDF_set_text_pos(pdf, margin, height - margin);
		curr_lines = 0;
		}
	if(vary_only) {
		out_vary_pos(vary_pos, widnames, k, nl, pdf, textfile);
		curr_lines += nl;
		}
	else	{
		sprintf(num_line, "%d", current + 1);
		fin = strlen(num_line);
		memmove(num_line + widnames - fin + 1, num_line, fin+1);
		if(fin <= widnames) memset(num_line, ' ', widnames - fin + 1);
		if( use_pdf) PDF_continue_text(pdf, num_line);
		else {fputs(num_line, textfile);fputs("\n",textfile);}
		++curr_lines;
		}
	for(num=0; num < view->tot_seqs; num++) {
		k = 0;
		for(j = 0; j < widnames; j++) {
			if(view->seqname[num][j] == 0) break;
			oneline[k++] = view->seqname[num][j];
			}
		while( j < widnames) {
			j++;
			oneline[k++] = ' ';
			}
		if(vary_only) {
			i = -1; j = 0;
			while( j < res_per_line) {
				if(current + i >= max_seq_length) break;
				if( !vary_need[current + ++i] ) continue;
				j++;
				if(current + i < view->each_length[num]) {
					if(num != ref0) lettre = ( toupper(view->sequence[num][current+i]) == 
						toupper(view->sequence[ref0][current+i]) ? '.' : view->sequence[num][current+i] );
					else lettre = view->sequence[ref0][current+i];
					oneline[k++] = lettre;
					}
				if( j % block_size == 0) oneline[k++] = ' ';
				}
			if(num == view->tot_seqs - 1) current = current + i + 1;
			}

		else	{
			fin = res_per_line;
			if(current+fin > view->each_length[num]) 
				fin = view->each_length[num] - current;
			if(ref0 != -1 && num != ref0) {
				/* ecriture par reference a seq ref0 */
				for(i=0; i<fin; i++) {
					lettre = ( toupper(view->sequence[num][current+i]) == 
						toupper(view->sequence[ref0][current+i]) ? '.' : view->sequence[num][current+i] );
					oneline[k++] = lettre;
					if( i < fin-1 && (i+1)%block_size == 0) 
						oneline[k++] = ' ';
					}
				}
			else	{ /* ecriture normale de seq */
				for(i=0; i<fin; i++) {
					oneline[k++] = view->sequence[num][current+i];
					if( i < fin-1 && (i+1)%block_size == 0) 
						oneline[k++] = ' ';
					}
				}
			}
		oneline[k] = 0;
		if(!view->allow_lower) majuscules(oneline + widnames);
		if(use_pdf && (curr_lines >= lines_per_page)) {
			PDF_end_page_ext(pdf, "");
			PDF_begin_page_ext(pdf, width, height, "");
			PDF_setfont(pdf, font, (double)fontsize);
			PDF_set_text_pos(pdf, margin, height - margin);
			curr_lines = 0;
			}
		if(!use_pdf) {
			fputs(oneline, textfile); fputs("\n", textfile);
			}
		else if(pdfkindvalue == PDF_BW) {
			PDF_show_xy(pdf, oneline, margin, height - margin - (curr_lines+1) * fontsize);
			}
		else 
			color_pdf_display(pdf, view, calc_color_function, oneline, widnames, margin, 
				height - margin - (curr_lines+1) * fontsize, fontsize, char_width, descender,
				num, current);
		++curr_lines;
		firstpage = FALSE;
		}
	if(!use_pdf) {
		fputs("\n", textfile);
		}
	else if(curr_lines + 1 <= lines_per_page) {
		PDF_continue_text(pdf, "");
		++curr_lines;
		}
	if( ! vary_only ) current += res_per_line;
	}
if(use_pdf) {
	PDF_end_page_ext(pdf, "");
	PDF_end_document(pdf, "");
	PDF_delete(pdf);
	}
else fclose(textfile);
} /* end of PDF_TRY */
//PDF_CATCH(pdf) {
if(use_pdf && pdf_catch(pdf)) {
		fl_alert("Error while writing pdf file:\n"
		 "[%d] %s: %s\n",
		  PDF_get_errnum(pdf), PDF_get_apiname(pdf), PDF_get_errmsg(pdf) );
		PDF_delete(pdf);
		return TRUE; 
		}
return FALSE;
}


static void color_pdf_display(PDF *pdf, SEA_VIEW *view, int (*calc_color_function)( int ), char *oneline, 
	int widnames, double x, double y, int fontsize, double char_width, double descender,
	int num, int current)
{
uchar red, green, blue;
double r, g, b, xx;
int c, l, count = 0;
char *p, **clines;

clines = (char **)malloc(sizeof(char *) * view->numb_gc); if(clines==NULL) return;
l = strlen(oneline);
for(c = 1; c < view->numb_gc; c++) {
	clines[c] = (char *)malloc(l + 1); if(clines[c] == NULL) return;
	memset(clines[c], ' ', l); clines[c][l] = 0;
	}
for(p = oneline + widnames; *p != 0; p++) {
	if(*p == ' ') continue;
	if(view->curr_colors != view->codoncolors) c = calc_color_function(*p);
	else c = view->col_rank[num][current + count++];
	if(c > 0) clines[c][p - oneline] = 'X';
	}
for(c = 1; c < view->numb_gc; c++) {
	if(strchr(clines[c], 'X') == NULL) continue;
	Fl::get_color((Fl_Color)view->curr_colors[c], red, green, blue);
	r = red/255.; g = green/255.; b = blue/255.;
	PDF_setcolor(pdf, "fillstroke", "rgb", r, g, b, 0);
	for(xx = x + widnames*char_width, p = clines[c] + widnames; *p != 0; p++, xx += char_width) {
		if(*p == ' ') continue;
		PDF_rect(pdf, xx, y + descender, char_width, (double)fontsize);
		}
	PDF_fill_stroke(pdf);
	}
PDF_setcolor(pdf, "fillstroke", "rgb", 0, 0, 0, 0);
PDF_show_xy(pdf, oneline, x, y);
for(c = 1; c < view->numb_gc; c++) free(clines[c]);
free(clines);
}


static int calc_vary_lines(int *vary_pos, int widpos)
{
int maxi = 0, num, nl;

for(num = 0; num < widpos; num++) 
	if(vary_pos[num] > maxi) maxi = vary_pos[num];
if(maxi >= 100000)
	 nl = 6;
else if(maxi >= 10000)
	 nl = 5;
else if(maxi >= 1000)
	 nl = 4;
else if(maxi >= 100)
	 nl = 3;
else if(maxi >= 10)
	 nl = 2;
else 	
	 nl = 1;
return nl;
}


static void out_vary_pos(int *vary_pos, int widnames, int widpos, int nl, PDF *pdf, FILE *textfile)
{
int num, l, k, echelle, digit, val;
static char chiffre[] = "0123456789";
char oneline[300];

echelle = 1; k = 0;
for(l = 2; l <= nl; l++) echelle *= 10;
for(l = nl; l > 0; l--) {
	for(num = 0; num < widnames; num++) oneline[k++] = ' ';
	for(num = 0; num < widpos; num++) {
		val = vary_pos[num];
		if(val < echelle)
			oneline[k++] = ' ';
		else	{
			digit = (val / echelle) % 10 ;
			oneline[k++] = *(chiffre + digit);
			}
		}
	oneline[k] = 0;
	if(textfile == NULL) PDF_continue_text(pdf, oneline);
	else {fputs(oneline, textfile); fputs("\n",textfile); }
	k = 0;
	echelle /= 10;
	}
}


class Fl_PDF_Graphics_Driver : public Fl_Graphics_Driver {
private:
  PDF *pdf;
  int pdf_font;
  const char *encoding;
  bool in_page;
public:
  Fl_PDF_Graphics_Driver();
protected:
  void rect(int x, int y, int w, int h);
  void rectf(int x, int y, int w, int h);
  void line_style(int style, int width, char *dashes=0);
  void line(int x1, int y1, int x2, int y2);
  void font(int f, int s);
  void draw(const char *str, int n, int x, int y);
  void draw(const char*, int, float, float) ;
  void draw(int, const char*, int, int, int) ;
  void rtl_draw(const char*, int, int, int) ;
  void color(uchar r, uchar g, uchar b);
  void color(Fl_Color c);
  void push_clip(int x, int y, int w, int h) ;
  void pop_clip();
  void draw_image(const uchar*, int, int, int, int, int, int) ;
  void draw_image_mono(const uchar*, int, int, int, int, int, int) ;
  void draw_image(void (*)(void*, int, int, int, uchar*), void*, int, int, int, int, int) ;
  void draw_image_mono(void (*)(void*, int, int, int, uchar*), void*, int, int, int, int, int) ;
  void draw(Fl_RGB_Image*, int, int, int, int, int, int) ;
  void draw(Fl_Pixmap*, int, int, int, int, int, int) ;
  void draw(Fl_Bitmap*, int, int, int, int, int, int) ;
  double width(const char*, int) ;
  int height() ;
  int descent() ;
friend class Fl_PDF_File_Device;
};


Fl_PDF_File_Device::Fl_PDF_File_Device()
{
  driver(new Fl_PDF_Graphics_Driver);
  filename = NULL;
}

Fl_PDF_File_Device::~Fl_PDF_File_Device()
{
  delete driver();
}

int Fl_PDF_File_Device::start_job(const char *name, enum Fl_Paged_Device::Page_Format format, 
	      enum Fl_Paged_Device::Page_Layout layout)
{
  Fl_Native_File_Chooser *chooser = new Fl_Native_File_Chooser();
  chooser->type(Fl_Native_File_Chooser::BROWSE_SAVE_FILE);
  chooser->title("Set PDF filename");       
  chooser->filter("PDF Files\t*.pdf");
  int l = name ? strlen(name) : 0;
  char *preset = new char[l + 5];
  strcpy(preset, name ? name : "" );
  char *p = strchr(preset, '.');
  if (p) *p = 0;
  strcat(preset, ".pdf");
  chooser->preset_file(preset);
  delete[] preset;
  chooser->options(Fl_Native_File_Chooser::SAVEAS_CONFIRM);
  char *plotfilename = run_and_close_native_file_chooser(chooser);
  if(plotfilename == NULL) return 1;
#ifdef WIN32
  FILE *f;
  if( (f = fopen(plotfilename, "r")) != NULL) {
    fclose(f);
    if(fl_choice("File %s already exists.", "Cancel", "Overwrite", NULL, plotfilename) == 0) return 1;
  }
#endif
  width = page_formats[format].width;
  height = page_formats[format].height;
  PDF *pdf = PDF_new();
  if(pdf == NULL) return 1;
  Fl_PDF_Graphics_Driver *d = (Fl_PDF_Graphics_Driver*)driver();
  d->pdf = pdf;
  if( PDF_begin_document(pdf, plotfilename, 0, "compatibility=1.3") == -1) {
    PDF_delete(pdf);
    fl_alert("Error opening %s for writing\n", plotfilename);
    return 1;
  }
  PDF_set_info(pdf, "Title", plotfilename );
  PDF_set_info(pdf, "Creator", "seaview");
  previous_surface = Fl_Surface_Device::surface();
  if (format == Fl_Paged_Device::A4) {
    left_margin = 18;
    top_margin = 18;
  }
  else {
    left_margin = 12;
    top_margin = 12;
  }
  filename = strdup(plotfilename);
  set_current();
  return 0;
}

PDF *Fl_PDF_File_Device::pdf()
{
  return ((Fl_PDF_Graphics_Driver*)driver())->pdf;
}

int Fl_PDF_File_Device::start_page()
{
  Fl_PDF_Graphics_Driver *d = (Fl_PDF_Graphics_Driver*)driver();
  PDF *p = d->pdf;
  PDF_begin_page_ext(p, width, height, "");
  PDF_translate(p, left_margin, height - top_margin);
  PDF_scale(p, 1, -1);
  PDF_setlinecap(p, 1);
  PDF_setlinewidth(p, 1);
  d->in_page = true;
  return 0;
}

int Fl_PDF_File_Device::printable_rect(int *w, int *h)
//returns 0 iff OK
{
  if(w) *w = (int)((width - 2 * left_margin) + .5);
  if(h) *h = (int)((height - 2 * top_margin) + .5);
  return 0;
}

int Fl_PDF_File_Device::end_page()
{
  Fl_PDF_Graphics_Driver *d = (Fl_PDF_Graphics_Driver*)driver();
  PDF_end_page_ext(d->pdf, "");
  d->in_page = false;
  return 0;
}

void Fl_PDF_File_Device::end_job()
{
  PDF *p = ((Fl_PDF_Graphics_Driver*)driver())->pdf;
  PDF_end_document(p, "");
  PDF_delete(p);
  if (filename) free(filename);
  filename = NULL;
  previous_surface->set_current();
}

void Fl_PDF_File_Device::error_catch()
{
  PDF *p = ((Fl_PDF_Graphics_Driver*)driver())->pdf;
  Fl_Display_Device::display_device()->set_current();
  fl_alert("Error while writing to pdf file:\n%s", filename);
  PDF_delete(p);
  if (filename) free(filename);
  filename = NULL;
}

Fl_PDF_Graphics_Driver::Fl_PDF_Graphics_Driver()
{
#ifdef __APPLE__
  encoding	= "macroman";
#else
  encoding	= "iso8859-1";
#endif
  pdf = NULL;
  pdf_font = -1;
  in_page = false;
}

void Fl_PDF_Graphics_Driver::line(int x1, int y1, int x2, int y2)
{
  PDF_moveto(pdf, x1, y1);
  PDF_lineto(pdf, x2, y2);
  PDF_stroke(pdf);
}

void Fl_PDF_Graphics_Driver::rect(int x, int y, int w, int h)
{
  PDF_rect(pdf, x, y, w, h);
  PDF_stroke(pdf);
}

void Fl_PDF_Graphics_Driver::rectf(int x, int y, int w, int h)
{
  PDF_rect(pdf, x, y, w, h);
  PDF_fill(pdf);
}

void Fl_PDF_Graphics_Driver::draw(const char *str, int n, int x, int y)
{
  PDF_save(pdf);
  PDF_translate(pdf, x, y);
  PDF_scale(pdf, 1, -1);
  PDF_show_xy2(pdf, str, n, 0, 0);
  PDF_restore(pdf);
}

void Fl_PDF_Graphics_Driver::draw(int angle, const char* str, int n, int x, int y)
{
  PDF_save(pdf);
  PDF_translate(pdf, x, y);
  PDF_rotate(pdf, -angle);
  draw(str, n, 0, 0);
  PDF_restore(pdf);
}

void Fl_PDF_Graphics_Driver::font(int f, int s)
{
  char *current_ps_font = prepare_ps_or_pdf_font(f);
  pdf_font = PDF_load_font(pdf, current_ps_font, 0, encoding, "");
  Fl_Graphics_Driver::font(f, s);
  if (in_page) PDF_setfont(pdf, pdf_font, s);
}

void Fl_PDF_Graphics_Driver::color(Fl_Color c)
{
  uchar red, green, blue;
  Fl::get_color(c, red, green, blue);
  color(red, green, blue);
}

void Fl_PDF_Graphics_Driver::color(uchar red, uchar green, uchar blue)
{
  float r, g, b;
  r = red/255.; g = green/255.; b = blue/255.;
  PDF_setcolor(pdf, "fillstroke", "rgb", r, g, b, 0);
}

double Fl_PDF_Graphics_Driver::width(const char* str, int l)
{
  return PDF_stringwidth2(pdf, str, l, pdf_font, size());
}

int Fl_PDF_Graphics_Driver::height() {
  return size();
}

int Fl_PDF_Graphics_Driver::descent() {
  return (int)(-PDF_get_value(pdf, "descender", pdf_font) * size() + 0.5) + 1;
}

void Fl_PDF_Graphics_Driver::draw(const char* str, int n, float fx, float fy) {
  draw(str, n, (int)fx, (int)fy);
}

void Fl_PDF_Graphics_Driver::push_clip(int x, int y, int w, int h) 
{
  PDF_save(pdf);
  PDF_moveto(pdf, x, y); PDF_lineto(pdf, x + w, y); PDF_lineto(pdf, x + w, y + h);
  PDF_lineto(pdf, x, y + h); 
  PDF_closepath(pdf); 
  PDF_clip(pdf);
}

void Fl_PDF_Graphics_Driver::pop_clip()
{
  PDF_restore(pdf);
}

void Fl_PDF_Graphics_Driver::draw_image(const uchar*, int, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::draw_image_mono(const uchar*, int, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::draw_image(void (*)(void*, int, int, int, uchar*), void*, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::draw_image_mono(void (*)(void*, int, int, int, uchar*), void*, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::draw(Fl_RGB_Image*, int, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::draw(Fl_Pixmap*, int, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::draw(Fl_Bitmap*, int, int, int, int, int, int) {}
void Fl_PDF_Graphics_Driver::line_style(int style, int width, char *dashes) {}
void Fl_PDF_Graphics_Driver::rtl_draw(const char*, int, int, int) {}


