/*
 *   dspinit.c -- Routines to provide DSP initialization
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "dspmgr.h"                    // Include function prototypes
#include "dspabil.h"                   // Include function prototypes
#include "dspinit.h"                   // Include function prototypes
#include "dspload.h"                   // Include function prototypes
#include "dspactiv.h"                  // Include function prototypes
#include "dspispos.h"                  // Include function prototypes
#include "dspbios.h"                   // Include function prototypes
#include "dspfree.h"                   // Include function prototypes
#include "dspstruc.h"                  // Include function prototypes
#include "dspglist.h"                  // Include function prototypes
#include "dspgpc.h"                    // Include function prototypes
#include "dspparse.h"                  // Include function prototypes
#include "dsppcmem.h"                  // Include function prototypes
#include "dspmem.h"                    // Include function prototypes
#include "dspfxutl.h"                  // Include function prototypes

RC NameToTaskHandle(HMOD hMod,PSZ pszVirtName,HTASK *phTask);

/**********************************************************************/
/* Define DspINIT_Notify as a dummy macro for OS/2 until a later time */
/* to allow it to compile.  At some future time a real function will  */
/* have to be added.                                                  */
/**********************************************************************/
#define DspINIT_Notify( hdsp, usBeginEndFlag, usInitDataAddress) DSP_NOERROR

#define  NOTIFY_BEGIN   0
#define  NOTIFY_END     1

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: MatchManagerType                                        */
/*                                                                          */
/* FUNCTION: This routine will get information about a DSP and save that    */
/*           information in the appropriate RDSP structure. This routine    */
/*           will probably call ABIOS to get the info about the DSP         */
/*                                                                          */
/* INPUT:                                                                   */
/*        prmgrTail - Ptr to list of managers                               */
/*        PRDSP  prdsp - Ptr to the DSP structure to fill in.               */
/*        USHORT usId  - ID number of the DSP.                              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/* 00) DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

BOOL MatchManagerType(PRDSP prdsp,PRMGR *pprmgr)

{
   prdsp = prdsp;
   //
   if (pg->prmgrTail == NULL)          // No managers yet
      {
      *pprmgr = NULL;
      return (FALSE);
   }
   else {
      *pprmgr = pg->prmgrTail;
      return (TRUE);
   }
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: CompareDSP                                              */
/*                                                                          */
/* FUNCTION: This routine will check if any task on this module is in the   */
/*           DSP. To determine if a module exists on a DSP, we must check   */
/*           each task of the module to see if at least one of them is      */
/*           on the DSP. If one is then this module is considered on this   */
/*           DSP.                                                           */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP   prdsp - Are any task of the module on this DSP ?          */
/*        PRMOD   prmod - module ptr.                                       */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        BOOL    fOnDsp -   TRUE =  At least one task is on PRDSP.         */
/*                           FALSE =  No task from PRMOD are on PRDSP.      */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/* 00) DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

BOOL CompareDSP(PRDSP prdsp,PRMOD prmod)

{
   PRTSK      prtsk;                   // current task of module
   PRTSK      prtskHead;               // Head of task list
   BOOL       fOnDsp;                  // boolean flag

   /* Traverse the module (each task), start at the head of the list        */
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspinit::CompareDSP entry prdsp %x prmod %x\n",(int)prdsp,(int)prmod);

   fOnDsp = FALSE;                     // so far no task of PRMOD on PRDSP
   if (prmod->MOD_prtskTail != NULL) {
      prtskHead = prtsk = prmod->MOD_prtskTail->TSK_prtskNext; // set head ptr
      do {
         if (prtsk->TSK_prdsp == prdsp) {
            fOnDsp = TRUE;
         }
         prtsk = prtsk->TSK_prtskNext; // go to next in list
      }  while (prtsk != prtskHead && !fOnDsp); // Back at head yet ?
   }
   else {                              // Module with no Task
      fOnDsp = TRUE;
   }                                   /* endif                             */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspinit::CompareDSP exit fOnDsp %x\n",fOnDsp);
   return (fOnDsp);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: InitDSP                                                 */
/*                                                                          */
/* FUNCTION: This routine will initialize the DSP and load ISPOS (the ISP   */
/*           operating system. It will start the os in the requested        */
/*           activation state.                                              */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP   prdsp - Load the operating system in this DSP.            */
/*        PSZ     pszFileName - Load the operating system from this file.   */
/*        USHORT  usFlags - Flag indicates mode to active module to         */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*        - Resets the DSP                                                  */
/*        - Unloads any modules that have at least one task on the DSP.     */
/*        - Trashes all current data structures for this DSP.               */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC InitDSP(PRDSP prdsp,PSZ pszFileName,USHORT usState)
{
   PRMOD      prmod;                   // Current module node pointer
   PRMOD      prmod_Next;              // Next Module in MGR's List
   PRMOD      prmod_Tail;              // End Of MGR's Module List
   RC         ulRC, ulRC0 = 0;         // Error return code            CH06
   ULONG      ulDoneInit;              // Mwave/OS init complete flag  CH02
   char *     pszIFName = NULL;        // Mwave/OS init filename       CH02
   int        tries = 0;               //                              CH02
   USHORT     usValue = 0;             //                              CH02
   PSZ        pszlpcount = 0;          // Sleep mode variable          CH05
   ULONG      ulLPCount;               // Sleep mode variable          CH05
   int        ilpcount;                // Sleep mode variable          CH05
   ULONG      ulInitData = 0;          // Mwave/OS INITDATA addr       CH06
   HTASK      hTask;


   if (pszFileName == NULL)           // Null DSP File Name
      return (DSP_FILE_NOT_FOUND);

   ulRC = CheckDSPStatus(prdsp);       /* CH01 */
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   /* Reset the DSP                                                         */
   MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspinit::InitDSP entry prdsp %x pszFileName %s usStater %x\n",
	   (int)prdsp,pszFileName,usState);

   
   if ((ulRC = DspBIOS_Reset(prdsp)) == DSP_NOERROR) {

      /**********************************************************************/
      /* Free all current data structures in the DSP tree.                  */
      /* Since modules can not span DSP's, therefore, the DSP               */
      /* pointer we get on entry to this routine points to an               */
      /* owner manager. The manager contains the linked-list of             */
      /* modules. We must now check each module in this list to             */
      /* see if it is in the DSP we want to INIT. We can only               */
      /* free the modules on that DSP.                                      */
      /**********************************************************************/

      prmod_Tail = prdsp->DSP_prmodOS; /* MWAVEOS module is last freed      */
      fDSPINIT_FLAG = TRUE;            /* Set global flag indicating        */
                                       /* initiaization                     */
      if (prmod_Tail != NULL) {        // Skip if NULL/Empty
         prmod_Next = prmod_Tail->MOD_prmodNext; // Start of Module List
         do {                          // Walk to end of Mod List
            prmod = prmod_Next;        // Next Module
            prmod_Next = prmod->MOD_prmodNext; // Save Next PTR
            if (CompareDSP(prdsp, prmod)) {
               ulRC = FreeModule(prmod, DSP_INIT_INACTIVE);
               if (ulRC != DSP_NOERROR) { // Error if FreeMod Fails
		 MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspinit::DSPInit error ulRC %lx DSPInit FreeModule failure\n",ulRC);
                  return (ulRC);
               }
            }
         }  while (prmod != prmod_Tail); // End of List yet?
      }

/****************************************************************************/
/* The IPCTable can not be removed because the PDD has established a        */
/* global address for this table in IOC1_SetupGlobal which is called        */
/* by the daemon during initialization only.  If it is removed and          */
/* reallocated, the PDD's global address will no longer point to the        */
/* correct location (interrupts will be lost).                              */
/****************************************************************************/

      /**********************************************************************/
      /* Add the IPC table to the DSP control block on the first pass and   */
      /* reinitialize the table on subsequent calls.                        */
      /**********************************************************************/

      ulRC = AddRIPCTableTail(&prdsp->DSP_prIPCTable);

      prdsp->DSP_ulCPSFree = prdsp->DSP_ulCPS; // Assure Free CPS reset
      pg->prmgrTail->MGR_ulRemBusCapacity =  // Assure Remaining Cap. reset
         pg->prmgrTail->MGR_ulBusCapacity;

      if (prdsp->DSP_ulCPSFree != prdsp->DSP_ulCPS || prdsp->DSP_prfmTail !=
         NULL || prdsp->DSP_prgpcTail != NULL ||
         prdsp->DSP_prmemDFreeTail->MEM_ulMemSize != prdsp->DSP_ulDMemTotal ||
         prdsp->DSP_prmemIFreeTail->MEM_ulMemSize != prdsp->DSP_ulIMemTotal)
         return (DSP_INTERNAL_CORRUPT);

      prdsp->DSP_prmodOS = NULL;

      /**********************************************************************/
      /* Load the DSP operating system. Set flags for Load:                 */
      /* Load only accepts Inactive or standby activation mode              */
      /* and set the priority to critical so other modules can              */
      /* not take any resources from os.                                    */
      /**********************************************************************/

      ulRC = LoadModule(prdsp, pszFileName, prdsp->DSP_ISPOS, usState, &prmod);

      fDSPINIT_FLAG = FALSE;              /* Reset global flag indicating      */

      if (DSP_NOERROR == ulRC)
         do {   // CH03

         //CH02 Begin ----------------------------------------------------------
         // If the DONEINIT variable is found in MWAVE/OS then look for the
         // Mwave/OS init file name in the [Mwave,Board] section of the Mwave.ini
         // If the init file is found load it and wait for it to complete the
         // initialization.
         //CH04 LBM - Change returns to break; remove intermediate returns()
         //           so that a single diagnostic call can follow all inits.

         if (DSP_NOERROR == IsposLabelToAddress(prdsp, "LPCOUNT", &ulLPCount))   /* CH05 */
         {                                                                       /* CH05 */
            pszlpcount = GetDSPStringAttribute(DSPATTR_LPCOUNT, prdsp);          /* CH05 */
            ilpcount = atoi(pszlpcount);
            if (0 != ilpcount)                                                   /* CH05 */
               ulRC = DspBIOS_D_Write(prdsp, ulLPCount, (ULONG)1, &ilpcount);    /* CH05 */
         }                                                                       /* CH05 */

         if (DSP_NOERROR == IsposLabelToAddress(prdsp, "DONEINIT", &ulDoneInit) )
         {

            pszIFName = GetDSPStringAttribute(DSPATTR_OSINITNAME, prdsp);

            if ( pszIFName != NULL ) {                                  
		                     /* CH06 */
		ulRC = LoadModule(prdsp, pszIFName, NULL, DSP_INIT_INACTIVE, &prmod);         /* CH06 */
	
               if (DSP_NOERROR == ulRC) {                                                    /* CH06 */
                  ulRC0 = NameToTaskHandle(prmod, "MWOSINIT", &hTask);  /* CH06 */
	
                  if (DSP_NOERROR == ulRC0) {                         /* CH06 */
                     ulRC0 = LabelToAddress(hTask, "INITDATA", &ulInitData);                 /* CH06 */
		  }

                  if (DSP_NOERROR == ulRC0) {                     /* CH06 */
		      (void) DspINIT_Notify( prdsp, NOTIFY_BEGIN, (USHORT)ulInitData ); /* CH06 */
		  }

                  if (DSP_NOERROR == ulRC) {  
		      ulRC = ChangeModuleState( prmod, usState );                              /* CH06 */

                     tries = 0;
                     do {
                        usleep(32);

                        if ((ulRC = DspBIOS_D_Read(prdsp, ulDoneInit, (ULONG)1, &usValue))
                           != DSP_NOERROR)
                           break;

                        if (++tries > 156) {       // wait approx 5 secs for
                           ulRC=DSP_ISPOS_ERROR;   // DONEINIT != 0 else error
                           break;
                        }
 
                     } while ( 0 == usValue );

		     if (ulInitData != 0)                                                       /* CH06 */
			 (void) DspINIT_Notify( prdsp, NOTIFY_END, (USHORT)ulInitData );   /* CH06 */

		     ulRC = FreeModule(prmod, DSP_INIT_INACTIVE);
		  }
               }
               else                                                                          /* CH06 */
                  {                                                                          /* CH06 */
                  ulRC = DSP_ISPOS_ERROR;                                                    /* CH06 */
                  break;                                                                     /* CH06 */
                  }                                                                          /* CH06 */
            }
            else {                                                                           /* CH06 */
               ulRC = DSP_ISPOS_ERROR;                                                       /* CH06 */
               break;                                                                        /* CH06 */
            }                                                                                /* CH06 */
         }
      } while(0);  /* CH04 Mark end of block for breaks */
   }

   //CH02 End --------------------------------------------------------------

   if (ulRC==DSP_NOERROR) {
      ulRC = Wait_DONERT(prdsp);  /* CH04 Verify DSP is running */
   }

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspinit::InitDSP exit ulRC %lx pszFileName %s\n",ulRC,pszFileName);

   return (ulRC);
}
