/****************************************************************************
 **
 ** $Id: Update.cpp,v 1.16.2.1 2005/01/18 01:17:13 frank Exp $
 **
 ** Copyright (C) 2001-2004 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "Update.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "LinCVSLog.h"
#include "UpdateOtherRevisionDialogImpl.h"
#include "UpdateOtherRevisionStdDialogImpl.h"

Update * Update::queryUpdateFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, QUERYFILE);
}

Update * Update::updateFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, FILE);
}

Update * Update::updateReplaceFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, REPLACEFILE);
}

Update * Update::updateToDirBranchFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, BRANCHFILE);
}

Update * Update::updateOtherRevisionFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, QStringList * currentTagList)
{
   return setup(whatsThisIconSet, parent, workBench, false, OTHERFILE, currentTagList);
}

Update * Update::removeLocalOptionsFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, OPTIONSFILE);
}

Update * Update::queryUpdateDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, QUERYDIR);
}

Update * Update::queryUpdateAllDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, QUERYALLDIR);
}

Update * Update::updateDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, DIR);
}

Update * Update::updateAllDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, ALLDIR);
}

Update * Update::updateReplaceDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, REPLACEDIR);
}

Update * Update::updateReplaceAllDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, REPLACEALLDIR);
}

Update * Update::updateToDirBranchDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, BRANCHDIR);
}

Update * Update::updateOtherRevisionDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, QStringList * tagList)
{
   return setup(whatsThisIconSet, parent, workBench, true, OTHERDIR, tagList);
}

Update * Update::updateVirtualDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, VIRTUALDIR);
}

Update * Update::removeLocalOptionsDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, OPTIONSDIR);
}

Update * Update::queryUpdateAuto(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, QObject * receiver)
{
   return setup(whatsThisIconSet, parent, workBench, dir, true, QUERYDIR, NULL, receiver);
}

Update * Update::queryUpdateAllAuto(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, QObject * receiver)
{
   return setup(whatsThisIconSet, parent, workBench, dir, true, QUERYALLDIR, NULL, receiver);
}

Update * Update::updateAuto(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, QObject * receiver)
{
   return setup(whatsThisIconSet, parent, workBench, dir, true, DIR, NULL, receiver);
}

Update * Update::updateAllAuto(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, QObject * receiver)
{
   return setup(whatsThisIconSet, parent, workBench, dir, true, ALLDIR, NULL, receiver);
}

Update * Update::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, int mode, QStringList * tagList) {
   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;
   else return setup(whatsThisIconSet, parent, workBench, dir, isDir, mode, tagList);
}

Update * Update::setup(const QIconSet &whatsThisIconSet,
      QWidget* parent,
      CvsDirListView * workBench,
      DirBase * dir,
      bool isDir,
      int mode,
      QStringList * tagList,
      QObject * receiver) {

   if (dir->getType() == DirBase::Cvs) {

      assert(dir);

      QDialog * dlg = NULL;
      switch(mode) {
	 case VIRTUALDIR: {
	    if (!dir->isVirtual()) isDir = false;
	    break;
	 }
	 case OTHERFILE: {
	    QString moduleName = dir->topControlledDir()->relativeName();
	    bool moreThanOneFile = (dir->getSelectedFiles().count() > 1);
	    if (LookAndFeel::g_stdDlg) {
	       dlg = new UpdateOtherRevisionStdDialogImpl( moreThanOneFile,
		     tagList,
		     moduleName,
		     tr("Update Other Revision (DIR)"),
		     whatsThisIconSet,
		     parent, "Update Other Revision",
		     true,  //modal
		     LookAndFeel::g_modalF);
	    } else {
	       dlg = new UpdateOtherRevisionDialogImpl( moreThanOneFile,
		     tagList,
		     moduleName,
		     tr("Update Other Revision (FILES)"),
		     whatsThisIconSet,
		     parent, "Update Other Revision",
		     true,  //modal
		     LookAndFeel::g_modalF);
    	       connect( dlg, SIGNAL(getSelectedTags()), parent, SLOT(readCurrentTagList()));
	       connect( dlg, SIGNAL(getProjectTags()), parent, SLOT(updateProjectTagList()));
	       connect( parent, SIGNAL(tagListFetched()), dlg, SLOT(tagListUpdated()));

	    }
	    break;
	 }
	 case OTHERDIR: {
	    QString moduleName = dir->topControlledDir()->relativeName();
	    if (LookAndFeel::g_stdDlg) {
	       dlg = new UpdateOtherRevisionStdDialogImpl( true,
		     tagList,
		     moduleName,
		     tr("Update Other Revision (DIR)"),
		     whatsThisIconSet,
		     parent, "Update Other Revision",
		     true,  //modal
		     LookAndFeel::g_modalF);
	    } else {
	       dlg = new UpdateOtherRevisionDialogImpl( true,
		     tagList,
		     moduleName,
		     tr("Update Other Revision (FILES)"),
		     whatsThisIconSet,
		     parent, "Update Other Revision",
		     true,  //modal
		     LookAndFeel::g_modalF);
    	       connect( dlg, SIGNAL(getSelectedTags()), parent, SLOT(readCurrentTagList()));
	       connect( dlg, SIGNAL(getProjectTags()), parent, SLOT(updateProjectTagList()));
	       connect( parent, SIGNAL(tagListFetched()), dlg, SLOT(tagListUpdated()));

	    }
	    break;
	    
	    break;
	 }
	 default:
	    break;
      }

      if ( dlg && !dlg->exec() ) {
	 delete dlg;
	 dlg = NULL;
      } else {
	 Update * p = new Update(whatsThisIconSet, parent, workBench, dir, isDir, mode, tagList);
	 if (dlg) p->m_dlg = dlg;
	 if (receiver) connect(p, SIGNAL(done(bool)), receiver, SLOT(callDone(bool)));
	 p->acceptCvs();
	 return p;
      }
   } else {
      qDebug("Update: Method not implemented");
   }
   return NULL;
}

Update::Update(const QIconSet &whatsThisIconSet, QWidget* parent,
      CvsDirListView * workBench, DirBase * dir,
      bool isDir, int mode, QStringList * currentTagList)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_mode(mode),
      m_currentTagList(currentTagList)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
   connect(this,SIGNAL(checkTree(DirBase *)),parent,SLOT(slot_checkStatusOfTree(DirBase *)));
   connect(this,SIGNAL(checkInProgress(bool)),parent,SLOT(checkInProgress(bool)));
   connect(this,SIGNAL(showWarning(const QString&,const QString&)),parent,SLOT(showWarning(const QString&,const QString&)));
   connect(this,SIGNAL(openTreeAndActivate(DirBase *,const QString)),parent,SLOT(slot_openTreeAndActivate(DirBase *,const QString)));
}

Update::~Update() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Update::acceptCvs() {

   assert(m_dir);

   QString cvsParameter = "";
   bool sticky = false;
   if (m_dlg) {
      switch(m_mode) {
	 case OTHERFILE:
	 case OTHERDIR: {
	    if (LookAndFeel::g_stdDlg) {
	       UpdateOtherRevisionStdDialogImpl * dlg = static_cast<UpdateOtherRevisionStdDialogImpl*>(m_dlg);
	       assert(dlg);
	       cvsParameter = dlg->cvsParameter();
	       sticky = dlg->useStickyTag();
	    } else {
	       UpdateOtherRevisionDialogImpl * dlg = static_cast<UpdateOtherRevisionDialogImpl*>(m_dlg);
	       assert(dlg);
	       cvsParameter = dlg->cvsParameter();
	       sticky = dlg->useStickyTag();
	    }
	    if (cvsParameter.isNull()) cvsParameter = "";
	    break;
	 }
	 default:
	    assert(false);
      }
      delete m_dlg;
      m_dlg = 0;
   }

   if (!m_isDir) {
      m_selectedFilesList = m_dir->getSelectedFiles();
      if (m_selectedFilesList.isEmpty()) {
	 reject(false);
	 return;
      }
      m_selectedFiles = masqWs(m_selectedFilesList.join("\" \""));
   } else {
      m_selectedFiles = "";
   }

   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr();
   int cmd = -1;
   QString files = "";
   switch(m_mode) {
      case QUERYFILE: {
	 command += " -n update " + m_selectedFiles;
	 cmd = CVS_QUERY_UPDATE_CMD;
	 break;
      }
      case FILE: {
	 command += " update " + m_selectedFiles;
	 cmd = CVS_UPDATE_FILES_CMD;
	 break;
      }
      case REPLACEFILE: {
	 command += " update -C " + m_selectedFiles;
	 cmd = CVS_UPDATE_FILES_CMD;
	 break;
      }
      case BRANCHFILE: {
	 QString dirBranch = m_dir->getDirTag();
	 QString dirType = "";
	 if (!dirBranch.isEmpty()) {
	    dirType = dirBranch.left(1);
	    dirBranch = dirBranch.mid(1);
	 }
	 if (dirBranch.isEmpty()) {
	    emit showWarning( tr("update to branch of dir"), tr("The selected dir has\nno branch tag set") );
	    reject(false);
	    return;
	 }
	 command += " update ";
	 if (!dirBranch.isEmpty()) {
	    if (dirType=="T" || dirType=="N") {
	       command += "-r " + dirBranch + " ";
	    } else if (dirType=="D") {
	       command += "-D " + dirBranch + " ";
	    } else {
	       Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	    }
	 }
	 command += m_selectedFiles;
	 cmd = CVS_UPDATE_FILES_CMD;
	 break;
      }
      case OTHERFILE: {
	 if (sticky) {
	    command += " update ";
	    if (!cvsParameter.isEmpty()) {
	       command += cvsParameter + " ";
	    }
	    command += m_selectedFiles;
	    cmd = CVS_UPDATE_FILES_CMD;
	 } else {
	    cvsRevertToRevFile(cvsParameter);
	    return;
	 }
	 break;
      }
      case OPTIONSFILE: {
	 QString dirBranch = m_dir->getDirTag();
	 QString dirType = "";
	 if (!dirBranch.isEmpty()) {
	    dirType = dirBranch.left(1);
	    dirBranch = dirBranch.mid(1);
	 }
	 if (!cvsRemoveOptionsOk(dirBranch,true)) {
	    reject(false);
	    return;
	 }
	 command += " update -A ";
	 if (!dirBranch.isEmpty()) {
	    if (dirType=="T" || dirType=="N") {
	       command += "-r " + dirBranch + " ";
	    } else if (dirType=="D") {
	       command += "-D " + dirBranch + " ";
	    } else {
	       Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	    }
	 }
	 command += m_selectedFiles;
	 cmd = CVS_UPDATE_FILES_CMD;
	 break;
      }
      case QUERYDIR: {
	 command += " -n update ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 cmd = CVS_QUERY_UPDATE_CMD;
	 break;
      }
      case QUERYALLDIR: {
	 command += " -n update ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 command += "-d ";
	 cmd = CVS_QUERY_UPDATE_ALL_CMD;
	 break;
      }
      case DIR: {
	 command += " update ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 cmd = CVS_UPDATE_DIR_CMD;
	 break;
      }
      case ALLDIR: {
	 command += " update ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 command += "-d ";
	 cmd = CVS_UPDATE_DIR_CMD;
	 break;
      }
      case REPLACEDIR: {
	 command += " update -C ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 cmd = CVS_UPDATE_DIR_CMD;
	 break;
      }
      case REPLACEALLDIR: {
	 command += " update -C ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 command += "-d ";
	 cmd = CVS_UPDATE_DIR_CMD;
	 break;
      }
      case BRANCHDIR: {
	 QString dirBranch = m_dir->getDirTag();
	 QString dirType = "";
	 if (!dirBranch.isEmpty()) {
	    dirType = dirBranch.left(1);
	    dirBranch = dirBranch.mid(1);
	 }
	 if (dirBranch.isEmpty()) {
	    emit showWarning( tr("update to branch of dir"), tr("The selected dir has\nno branch tag set") );
	    reject(false);
	    return;
	 }
	 command += " update ";
	 if (!dirBranch.isEmpty()) {
	    if (dirType=="T" || dirType=="N") {
	       command += "-r " + dirBranch + " ";
	    } else if (dirType=="D") {
	       command += "-D " + dirBranch + " ";
	    } else {
	       Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	    }
	 }
	 cmd = CVS_UPDATE_FILES_CMD;
	 break;
      }
      case OTHERDIR: {
	 if (sticky) {
	    command += " update ";
	    command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	    command += CvsOptions::g_bBringOverNewDirs ? "-d " : "";
	    command += cvsParameter;
	    cmd = CVS_UPDATE_DIR_CMD;
	 } else {
	    cvsRevertToRevDir(cvsParameter);
	    return;
	 }
	 break;
      }
      case VIRTUALDIR: {
	 command += " update -d ";
	 command += (CvsOptions::g_bPruneDirs ? "-P " : "");
	 m_newFileNameList.clear();
	 if (m_dir->isVirtual()) {
	    files = m_dir->fullName();
	    m_dir = m_dir->topDir()->searchLastValidDirOfPath(files);
	    assert(m_dir);
	    m_newFileNameList.append(files);
	    files = files.mid(m_dir->fullName().length()+1);
	    int pos = files.find("/");
	    if (pos > -1) {
	       files.truncate(pos);//update first virtual dir in hierarchy
	    }
	 } else {
	    files = m_selectedFiles;
	    m_newFileNameList = m_selectedFilesList;
	 }
	 cmd = CVS_UPDATE_VIRTUAL_DIR_CMD;
	 break;
      }
      case OPTIONSDIR: {
	 QString dirBranch = m_dir->getDirTag();
	 QString dirType = "";
	 if (!dirBranch.isEmpty()) {
	    dirType = dirBranch.left(1);
	    dirBranch = dirBranch.mid(1);
	 }
	 if (!cvsRemoveOptionsOk(dirBranch,false)) {
	    reject(false);
	    return;
	 }
	 command += " update -A ";
	 if (!dirBranch.isEmpty()) {
	    if (dirType=="T" || dirType=="N") {
	       command += "-r " + dirBranch + " ";
	    } else if (dirType=="D") {
	       command += "-D " + dirBranch + " ";
	    } else {
	       Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	    }
	 }
	 cmd = CVS_UPDATE_DIR_CMD;
	 break;
      }
      default:
	 assert(false);
   }

   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Update::cvsRevertToRevFile(QString param /*=QString::null*/) {

   static QString topModule;
   static QString dir;
   static QString module;
   static QString tag;
   static QString updateParam;
   static QString cvsRoot;
   static int state;

   QString command;
   QString files = "";
   int cmd = CVS_REVERT_FILES;
   assert(m_dir);
   if (!param.isNull()) {
      dir = m_dir->fullName();
      module = m_dir->repository();
      topModule = m_dir->topControlledDir()->relativeName();
      cvsRoot = m_dir->getRoot();
      updateParam = param;
      if (updateParam.startsWith(":")) state = 0;
      else state = 1;
      tag = "TMP-TAG-"+m_dir->userName()+"-"+QDateTime::currentDateTime().toString("yyyyMMddhhmmzzz")+"-";
      command = "tag "+tag+"CURRENT";
      files = m_selectedFiles;
   } else {
      switch( state) {
	 case 1: {
	    command = "-d \""+cvsRoot+"\" rtag -D \""+updateParam.mid(1)+"\" "+tag+"TO \""+module+"\"";
	    break;
	 }
	 case 2: {
	    command = "";
	    if (!bRWPermission) command += "-r ";
	    command += CvsOptions::cmprStr() + " update ";
	    if (updateParam.startsWith(":")) command += "-j \""+tag+"CURRENT\" -j \"" + tag + "TO\"";
	    else {
	       command += "-j \""+tag+"CURRENT\" -j \"" + updateParam + "\"";
	       ++state;
	    }
	    files = m_selectedFiles;
	    cmd = CVS_REVERT_FILES_CMD;
	    break;
	 }
	 case 3: {
	    command = "-d \""+cvsRoot+"\" rtag -d \""+tag+"TO\" \""+module+"\"";
	    break;
	 }
	 case 4: {
	    command = "tag -d "+tag+"CURRENT";
	    files = m_selectedFiles;
	    break;
	 }
	 case 5: {
	    DirBase * tmp = m_workBench->findBestMatch(dir);
	    if (tmp) {
	       tmp->recRemoveTmpEntries(FALSE);
	       if (m_dir->isSelected()) {
		  m_workBench->setSelected(tmp,true);
		  tmp->activateItem(TRUE);
	       }
	    }
	    reject(true);
	    return;
	 }
	 default: {
	    assert(false);
	 }
      }
   }
   ++state;
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,
	 NOROOT);
}

void Update::cvsRevertToRevDir(QString param /*=QString::null*/) {

   static QString topModule;
   static QString dir;
   static QString files;
   static QString module;
   static QString tag;
   static QString updateParam;
   static QString cvsRoot;
   static int state;

   QString command;
   int cmd = CVS_REVERT_DIR;
   assert(m_dir);
   if (!param.isNull()) {
      dir = m_dir->fullName();
      files = "";
      module = m_dir->repository();
      topModule = m_dir->topControlledDir()->relativeName();
      cvsRoot = m_dir->getRoot();
      updateParam = param;
      if (updateParam.startsWith(":")) state = 0;
      else state = 1;
      tag = "TMP-TAG-"+m_dir->userName()+"-"+QDateTime::currentDateTime().toString("yyyyMMddhhmmzzz")+"-";
      command = "tag "+tag+"CURRENT";
   } else {
      switch( state) {
	 case 1: {
	    command = "-d \""+cvsRoot+"\" rtag -D \""+updateParam.mid(1)+"\" "+tag+"TO \""+module+"\"";
	    break;
	 }
	 case 2: {
	    command = "";
	    if (!bRWPermission) command += "-r ";
	    command += CvsOptions::cmprStr() + " update " +
	       (CvsOptions::g_bPruneDirs ? "-P " : "");
	    command += CvsOptions::g_bBringOverNewDirs ? "-d " : "";
	    if (updateParam.startsWith(":")) command += "-j \""+tag+"CURRENT\" -j \"" + tag + "TO\"";
	    else {
	       command += "-j \""+tag+"CURRENT\" -j \"" + updateParam + "\"";
	       ++state;
	    }
	    cmd = CVS_REVERT_DIR_CMD;
	    break;
	 }
	 case 3: {
	    command = "-d \""+cvsRoot+"\" rtag -d \""+tag+"TO\" \""+module+"\"";
	    break;
	 }
	 case 4: {
	    command = "tag -d "+tag+"CURRENT";
	    break;
	 }
	 case 5: {
	    DirBase * tmp = m_workBench->findBestMatch(dir);
	    if (tmp) {
	       tmp->recRemoveTmpEntries(FALSE);
	       if (m_dir->isSelected()) {
		  m_workBench->setSelected(tmp,true);
		  tmp->activateItem(TRUE);
	       }
	    }
	    reject(true);
	    return;
	 }
	 default: {
	    assert(false);
	 }
      }
   }
   ++state;
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,
	 NOROOT);
}

bool Update::cvsRemoveOptionsOk(QString dirBranch,bool fileCollection) {
   bool test = true;
   if (dirBranch.isEmpty()) {//check that no file has any tag -> would be removed otherwise
      QListViewItem *myChild = m_dir->firstControlledFile();
      while(myChild) {
	 if (fileCollection) {
	    if( (myChild->isSelected()) && ( !(myChild->text(2).isEmpty())) ) {
	       test = false;
	       break;
	    }
	 } else if ( !(myChild->text(2).isEmpty())) {
	    test = false;
	    break;
	 }
	 myChild = myChild->nextSibling();
      }
      if (!test) {
	 showWarning( tr("Warning"), tr("There are tags set that would be removed too.\n"
		     "If this is what you want, run:\n"
		     "'update other revision'\n"
		     "with option 'reset sticky tag' selected") );
      }
   } else {//check that no file has a tag different to dirBranch ->would be removed otherwise
      QListViewItem *myChild = m_dir->firstControlledFile();
      while(myChild) {
	 if (fileCollection) {
	    if( (myChild->isSelected()) && ( myChild->text(2)!=dirBranch) ) {
	       test = false;
	       break;
	    }
	 } else if( myChild->text(2)!=dirBranch) {
	    test = false;
	    break;
	 }
	 myChild = myChild->nextSibling();
      }
      if (!test) {
	 showWarning( tr("Warning"), tr("There are tags set that differ from the branch\n"
		     "set for this dir. They would be removed too\n"
		     "If this is what you want, run:\n"
		     "'update other revision'\n"
		     "with option 'reset sticky tag' selected") );
      }
   }
   return test;
}

void Update::reject(bool result) {
   emit done(result);
   emit deleteObject(this);
}

void Update::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Update::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Update::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   if (failed) {
      reject(false);
      return;
   }
   assert(m_dir);
   emit checkInProgress(true);
   switch( cmd) {
      case NOOP: {
	 break;
      }
      case CVS_UPDATE_FILES_CMD: {
	 m_dir->removeTmpEntries();
	 if (!m_dir->parseCallResult( buf, cmd)) {
	    emit showWarning(tr("CVS Info"),m_dir->getLastError());
	 }
	 m_dir->postCallCheck(DirBase::Controled);
	 break;
      }
      case CVS_UPDATE_VIRTUAL_DIR_CMD: {
	 bool fromVirtualDir = false;
	 QString name = "";
	 DirBase * pOldDir = m_dir;

	 QStringList::Iterator iter;
	 for (iter = m_newFileNameList.begin(); iter != m_newFileNameList.end(); iter++) {
	    name = (*iter);
	    if (name.contains("/")) {//called from virtual dir in workbench
	       QString tmpName = name.mid(pOldDir->fullName().length()+1);
	       m_dir->removeTmpEntries(tmpName.left(tmpName.find("/")));
	       fromVirtualDir = true;
	       break;
	    } else {
	       m_dir->removeTmpEntries(name);
	    }
	 }
	 m_newFileNameList.clear();
	 if (!m_dir->parseCallResult( buf, CVS_UPDATE_DIR_CMD)) {
	    emit checkInProgress(false);
	    emit showWarning(tr("CVS Info"),m_dir->getLastError());
	    emit checkInProgress(true);
	 }
	 if (m_dir->isSelected()) {
	    if (fromVirtualDir) {
	       emit openTreeAndActivate(m_dir,name);
	    } else {
	       m_dir->postCallCheck(DirBase::Controled);
	    }
	 }
	 break;
      }
      case CVS_UPDATE_DIR_CMD: {
	 QString name = m_dir->fullName();
	 if (m_mode == ALLDIR || m_mode == REPLACEALLDIR || m_mode == OTHERDIR) {
	    m_dir->recRemoveTmpEntries(false);
	 }
	 if (!m_dir->parseCallResult( buf, cmd)) {
	    emit checkInProgress(false);
	    showWarning(tr("CVS Info"),m_dir->getLastError());
	    emit checkInProgress(true);
	 }
	 if (m_dir->isSelected()) {
	    DirBase * tmp = m_workBench->findBestMatch(name);
	    if (tmp) {
	       m_workBench->setSelected(tmp,true);
	       m_dir->postCallCheck(DirBase::Controled|DirBase::Tree);
	    }
	 }
	 break;
      }
      case CVS_QUERY_UPDATE_CMD: {
	 m_dir->recRemoveTmpEntries(false);
	 m_dir->parseCallResult( buf, cmd);
	 m_dir->postCallCheck(DirBase::Controled|DirBase::Force|(m_isDir ? DirBase::Tree : 0));
	 break;
      }
      case CVS_QUERY_UPDATE_ALL_CMD: {
	 m_dir->parseCallResult( buf, cmd);
	 m_dir->postCallCheck(DirBase::Controled|DirBase::Force|(m_isDir ? DirBase::Tree : 0));
	 break;
      }
      case CVS_REVERT_FILES: {
	 cvsRevertToRevFile();
	 return;
      }
      case CVS_REVERT_DIR: {
	 cvsRevertToRevDir();
	 return;
      }
      case CVS_REVERT_FILES_CMD: {
	 m_dir->parseCallResult( buf, CVS_UPDATE_FILES_CMD);
	 cvsRevertToRevFile();
	 return;
      }
      case CVS_REVERT_DIR_CMD: {
	 m_dir->parseCallResult( buf, CVS_UPDATE_DIR_CMD);
	 cvsRevertToRevDir();
	 return;
      }
      default: {
	 qDebug("Update::afterCall: Unknown cmd");
      }
   }
   emit checkInProgress(false);
   reject(true);
}
