/****************************************************************************
**
** Copyright (C) 2004-2006 Frank Hemer <frank@hemer.org>,
**                         Tilo Riemer <riemer@crossvc.com>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** CrossVC is available under two different licenses:
**
** If CrossVC is linked against the GPLed version of Qt 
** CrossVC is released under the terms of GPL also.
**
** If CrossVC is linked against a nonGPLed version of Qt 
** CrossVC is released under the terms of the 
** CrossVC License for non-Unix platforms (CLNU)
**
**
** CrossVC License for non-Unix platforms (CLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than CrossVC.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** CrossVC License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation, version 2 of the License.
** This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License version 2 for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "Remove.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"

Remove * Remove::removeFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, FILE);
}

Remove * Remove::removeRemoteFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false, REMOTE);
}

Remove * Remove::removeDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true, DIR);
}

Remove * Remove::removeSelectionFile(const QIconSet &whatsThisIconSet,
      QWidget* parent,
      CvsDirListView * workBench,
      DirBase * dir,
      QStringList list,
      QObject * receiver)
{
   return setup(whatsThisIconSet, parent, workBench, dir, false, FILE, list, receiver);
}

Remove * Remove::removeSelectionDir(const QIconSet &whatsThisIconSet,
      QWidget* parent,
      CvsDirListView * workBench,
      DirBase * dir,
      QObject * receiver)
{
   QStringList list;
   return setup(whatsThisIconSet, parent, workBench, dir, true, DIR, list, receiver);
}

Remove * Remove::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir, int mode) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   QStringList list;
   if (!isDir) {
      list = dir->getSelectedFiles();
      if (list.isEmpty()) return NULL;
   }

   return setup(whatsThisIconSet, parent, workBench, dir, isDir, mode, list);
}

Remove * Remove::setup(const QIconSet &whatsThisIconSet,
      QWidget* parent,
      CvsDirListView * workBench,
      DirBase * dir,
      bool isDir,
      int mode,
      QStringList list,
      QObject * receiver) {

   if (dir->getType() == DirBase::Cvs) {

      Remove * p = new Remove(whatsThisIconSet, parent, workBench, dir, isDir, mode);
      if (receiver) {
	 connect(p, SIGNAL(done(bool)), receiver, SLOT(callDone(bool)));
	 p->m_activate = false;
      }
      p->m_filesList = list;
      p->acceptCvs();
      return p;
   } else {
      qDebug("Remove: Method not implemented");
   }
   return NULL;
}

Remove::Remove(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir, int mode)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_mode(mode),
      m_activate(true)
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
   connect(this,SIGNAL(checkTree(DirBase *)),parent,SLOT(slot_checkStatusOfTree(DirBase *)));
   connect(this,SIGNAL(checkInProgress(bool)),parent,SLOT(checkInProgress(bool)));
}

Remove::~Remove() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void Remove::acceptCvs() {

   assert(m_dir);

   int cmd = -1;
   QString files = "";
   QString command = CvsOptions::cmprStr();
   command += " remove";

   switch(m_mode) {
      case FILE: {
         command += " -f";
         cmd = CVS_REMOVE_CMD;
         QStringList list = m_filesList;
         masqQuoteMarks(&list);
         files = masqWs(list.join("\" \""));
         break;
      }
      case REMOTE: {
         if (!m_dir->backupFiles(m_filesList)) {
            reject(false);
            return;
         }
         command += " -f";
         cmd = REMOVE_REMOTE;
         QStringList list = m_filesList;
         masqQuoteMarks(&list);
         files = masqWs(list.join("\" \""));
         break;
      }
      case DIR: {
         command += " -f";
         cmd = CVS_REMOVEDIR_CMD;
         break;
      }
      default: {
         assert(false);
         break;
      }
   }

   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 files, cmd,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void Remove::reject(bool state) {
   emit done(state);
   emit deleteObject(this);
}

void Remove::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void Remove::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void Remove::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   assert(m_dir);
   switch( cmd) {
      case NOOP: {
         break;
      }
      case REMOVE_REMOTE: {
         if (!failed) {
            m_dir->parseCallResult( buf, CVS_REMOVE_CMD);
         }
         if (!m_dir->restoreFiles(m_filesList)) {
            qDebug("restore failed");
         }
         if (failed) {
            reject(false);
            return;
         }
         if (m_activate) {
            emit checkInProgress(true);
            m_dir->postCallCheck(DirBase::Controled);
            emit checkInProgress(false);
         }
         m_filesList.clear();
         break;
      }
      case CVS_REMOVE_CMD: {
         if (failed) {
            reject(false);
            return;
         }
         m_dir->parseCallResult( buf, cmd);
         if (m_activate) {
            emit checkInProgress(true);
            m_dir->postCallCheck(DirBase::Controled);
            emit checkInProgress(false);
         }
         m_filesList.clear();
         break;
      }
      case CVS_REMOVEDIR_CMD: {
         if (failed) {
            reject(false);
            return;
         }
         m_dir->parseCallResult( buf, cmd);
         if (m_activate) {
            emit checkInProgress(true);
            m_dir->postCallCheck(DirBase::Controled);
            emit checkInProgress(false);
         }
         break;
      }
      default: {
         qDebug("Remove::afterCall: Unknown cmd");
      }
   }
   reject(true);
}

