package Zim::GUI::PropertiesDialog;

use strict;
use Gtk2;
use Zim::Utils;
use Zim::GUI::Component;

our $VERSION = '0.25';
our @ISA = qw/Zim::GUI::Component/;

=head1 NAME

Zim::GUI::PropertiesDialog - dialog for zim

=head1 DESCRIPTION

This module contains the preferences dialog for zim.
It is autoloaded when needed.

=head1 METHODS

=over 4

=item C<show()>

Present the dialog.

=cut

sub show {
	my $self = shift;
	my $notebook = $$self{app}{notebook};
	
	my $dialog = Gtk2::Dialog->new(
		__("Properties"), $$self{app}{window}, #. dialog title
	       	[qw/destroy-with-parent no-separator/],
		'gtk-help'   => 'help',
		'gtk-cancel' => 'cancel',
		'gtk-ok'     => 'ok',
	);
	$dialog->set_resizable(0);
	
	my $tabs = Gtk2::Notebook->new;
	$dialog->vbox->add($tabs);
	
	## Notebook tab
	my $nb_config = $$notebook{config};
	my $dir = $$notebook{dir};
	my $config_rw = $$notebook{config_file}->writable;
	my ($nb_vbox, $nb_data) =
		$self->get_notebook_tab($nb_config, $dir, 0, $config_rw);
	$tabs->append_page($nb_vbox, __('Notebook')); #. tab label
	
	## Show it all
	$dialog->show_all;
	while ($_ = $dialog->run) {
		if ($_ eq 'help') {
			$$self{app}->ShowHelp(':usage:properties');
			next;
		}
		last unless $_ eq 'ok';
		
		$self->save_notebook_tab($nb_data, $nb_config);
		$notebook->save;
		$notebook->signal_emit('config_changed');
		last;
	}
	$dialog->destroy;
}

=item C<get_notebook_tab(CONFIG, DIR, DIR_EDITABLE)>

Returns a widget and a hash. The widget is the notebook tab for the
properties dialog. The hash is needed to save values later.
CONFIG is the config hash of a notebook.

DIR needs to be given separatly since it is not in the config.
DIR_EDITABLE determines if DIR will be a label or an entry.

=item C<save_notebook_tab(DATA, CONFIG, NOTEBOOK)>

Save values set by the user to the notebook config.
DATA is the hash returned byt C<get_notebook_tab()>.
CONFIG is the config hash of a notebook.

=cut


sub get_notebook_tab {
	# Don't trust $$self{app} to exist, we can be called from
	# NotebookDialog without the whole app running
	my ($self, $config, $dir, $dir_rw, $config_rw) = @_;
	my $vbox = Gtk2::VBox->new(0,5);
	$vbox->set_border_width(5);
	
	my @fields = qw/name dir document_root icon home/;
	my %data = (
		name          => [__('Name'),          'string' ], #. input
		dir           => [__('Directory'),     'dir'    ], #. input
		document_root => [__('Document Root'), 'dir'    ], #. input
		icon          => [__('Icon'),          'file'   ], #. input
		home          => [__('Home Page'),     'page'   ], #. input
	);
	push @{$data{$_}}, $$config{$_} for @fields; # set values
	$data{dir}[-1] = $dir;

	my ($table, $entries) = $self->new_form(\@fields, \%data);
	$_->set_sensitive($config_rw) for @$entries;
	$$entries[1]->set_sensitive($dir_rw && $config_rw);

	my $checkbox = Gtk2::CheckButton->new(
		__('Slow file system') ); #. check box
	$checkbox->set_active($$config{slow_fs});
	$checkbox->set_sensitive($config_rw);
	$table->attach_defaults($checkbox, 0,2, 5,6);
	
	$vbox->pack_start($table,0,1,0);

	my %widgets = (map {($fields[$_] => $$entries[$_])} 0 .. $#fields);
	$widgets{slow_fs} = $checkbox;

	return $vbox, \%widgets;
}

sub save_notebook_tab {
	# Don't trust $$self{app} to exist, we can be called from
	# NotebookDialog without the whole app running
	my ($self, $widgets, $config) = @_;

	my $checkbox = delete $$widgets{slow_fs};
	$$config{'slow_fs'} = $checkbox->get_active || 0;
	my $dir = $$widgets{dir}->get_text;
	delete $$widgets{dir};

	for my $k (keys %$widgets) {
		my $string = $$widgets{$k}->get_text;
		$string = Zim::FS->rel_path($string, $dir) || $string
			if $k eq 'icon';
		$$config{$k} = $string;
	}
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2006 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

