/* computerview.cc 
 * (c) 2002 Christian Tpp <c.toepp@gmx.de>
 */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <qmessagebox.h>
#include "computerview.hh"
#include "icons/play.xpm"
#include "icons/stop.xpm"
#include "icons/pause.xpm"
#include "icons/menu.xpm"
#include "icons/ok.xpm"
#include "icons/griff.xpm"

computerview::computerview (QWidget * parent, QString title):QWidget (parent)
{
  initView ();
  initMenu ();
  viewTitle = title;
  gbox->setTitle (title);
  gbox->setFixedSize (150, 90);
  active = false;
  setAcceptDrops (true);
}

computerview::~computerview ()
{

}

QString
computerview::tarif ()
{
  return cmbTarif->currentText ();
}

QString
computerview::title ()
{
  return viewTitle;
}

void
computerview::initView ()
{
  icoStart = QPixmap (play_xpm);
  icoStop = QPixmap (stop_xpm);
  icoPause = QPixmap (pause_xpm);
  icoMenu = QPixmap (menu_xpm);
  icoOk = QPixmap (ok_xpm);
  griff = QPixmap (griff_xpm);
  gbox = new QGroupBox (this);
  priv = new QWidget (this);
  priv->setGeometry (5, 15, 140, 70);
  vbox = new QVBoxLayout (priv);
  // griff
  lbGriff = new ZLabel (priv);
  lbGriff->setPixmap (griff);
  connect (lbGriff, SIGNAL (pressed ()), this, SLOT (startDrag ()));
  vbox->addWidget (lbGriff);
  line1box = new QHBoxLayout (vbox);
  // line 1
  btnStart = new zbbutton (priv);
  btnStart->setPixmap (icoStart);
  btnStart->setFixedSize (20, 20);
  connect (btnStart, SIGNAL (clicked ()), this, SLOT (slotBtnStart ()));
  line1box->addWidget (btnStart);

  btnStop = new zbbutton (priv);
  btnStop->setPixmap (icoStop);
  btnStop->setFixedSize (20, 20);
  connect (btnStop, SIGNAL (clicked ()), this, SLOT (slotBtnStop ()));
  btnStop->setEnabled (false);
  line1box->addWidget (btnStop);

  cmbTarif = new QComboBox (priv);
  connect (cmbTarif, SIGNAL (activated (int)), this,
	   SLOT (slotTarifChanged (int)));
  cmbTarif->setFixedSize (80, 20);
  line1box->addWidget (cmbTarif);

  btnMenu = new zbbutton (priv);
  btnMenu->setPixmap (icoMenu);
  btnMenu->setFixedSize (20, 20);
  connect (btnMenu, SIGNAL (clicked ()), this, SLOT (slotBtnMenu ()));
  line1box->addWidget (btnMenu);

  // line 2
  line3box = new QHBoxLayout (vbox);

  lbTime = new QLabel (priv);
  lbTime->setText ("00:00");
  lbTime->setFixedSize (45, 20);
  lbTime->setAlignment (AlignHCenter | AlignVCenter);
  line3box->addWidget (lbTime);

  lbPrice = new QLabel (priv);
  lbPrice->setFixedSize (45, 20);
  lbPrice->setText ("0.00");
  lbPrice->setAlignment (AlignRight | AlignVCenter);
  line3box->addWidget (lbPrice);

  lbCurency = new QLabel (priv);
  lbCurency->setText ("EUR");
  lbCurency->setFixedSize (30, 20);
  lbCurency->setAlignment (AlignHCenter | AlignVCenter);
  line3box->addWidget (lbCurency);

  // line 3
  line4box = new QHBoxLayout (vbox);

  cmbItem = new QComboBox (priv);
  cmbItem->setFixedSize (120, 20);
  connect (cmbItem, SIGNAL (activated (int)), this,
	   SLOT (slotItemChanged (int)));
  line4box->addWidget (cmbItem);

  btnItem = new zbbutton (priv);
  btnItem->setPixmap (icoOk);
  btnItem->setFixedSize (20, 20);
  btnItem->setEnabled (false);
  connect (btnItem, SIGNAL (clicked ()), this, SLOT (slotBtnItem ()));
  line4box->addWidget (btnItem);

  // free
  free = new QWidget (this);
  free->setGeometry (5, 15, 140, 70);
  freeBox = new QVBoxLayout (free);
  lbInfo = new QLabel (free);
  lbInfo->setText (tr ("Drop users here"));
  lbInfo->setAlignment (AlignHCenter | AlignVCenter);
  freeBox->addWidget (lbInfo);
  newUserBox = new QHBoxLayout(0);
  newUser = new zbbutton (free);
  newUser->setText (tr ("New User"));
  connect (newUser, SIGNAL (clicked ()), this, SLOT (slotNewUser ()));
  newUserBox->addWidget(newUser);
  lock_unlock = new zbbutton(free);
  lock_unlock->setText(tr("Unlock"));
  newUserBox->addWidget(lock_unlock);
  connect(lock_unlock,SIGNAL(clicked()),SLOT(slotBtnLock()));
  freeBox->addLayout (newUserBox);
  priv->hide ();
}

void
computerview::initMenu ()
{
  mainMenu = new QPopupMenu (this);
  viewHistory =
    new QAction (tr ("View history"), tr ("View history"),
		 QAccel::stringToKey ("Ctrl+h"), this);
  connect (viewHistory, SIGNAL (activated ()), this,
	   SLOT (slotViewHistory ()));
  viewHistory->addTo (mainMenu);
  //viewHistory->setEnabled(false);
  printCalc =
    new QAction (tr ("Print Calc"), tr ("PrintCalc"),
		 QAccel::stringToKey ("Ctrl+p"), this);
  connect (printCalc, SIGNAL (activated ()), this, SLOT (slotPrintCalc ()));
  printCalc->addTo (mainMenu);
  printCalc->setEnabled(false);
}

void
computerview::setTarif (tarifList * tarife)
{
  tfList = tarife;
  cmbTarif->clear ();
  tarifList::iterator it;
  for (it = tfList->begin (); it != tfList->end (); ++it)
    cmbTarif->insertItem ((*it).name);
}

void
computerview::setProducts (productList * products, printInfo * prints)
{
  pInfo = prints;
  pList = products;
  cmbItem->clear ();
  // standards
  cmbItem->insertItem (tr ("Add Item"));
  cmbItem->insertItem (tr ("Printout"));
  // items
  productList::iterator it;
  for (it = pList->begin (); it != pList->end (); ++it)
    cmbItem->insertItem ((*it).name);
}

void
computerview::updateView (QString title)
{
  gbox->setTitle (title);
}

void
computerview::setClientIp (QString s)
{
  clientIp = s;
  srv = new server (this, clientIp);
  connect (this, SIGNAL (started ()), SLOT (slotStarted ()));
  connect (this, SIGNAL (stopped ()), SLOT (slotStopped ()));
  connect (this, SIGNAL (paused ()), SLOT (slotPaused ()));
  connect (this, SIGNAL (statusChanged (QString, QString)),
					 SLOT (slotStatus (QString, QString)));
  srv->stopComputer ();
}

void
computerview::slotBtnStart ()
{
  btnStart->disconnect ();
  connect (btnStart, SIGNAL (clicked ()), this, SLOT (slotBtnPause ()));
  btnStart->setPixmap (icoPause);
  btnStop->setEnabled (true);
  zbUser->timeStart ();
  zbUser->setTarif (&(*tfList)[cmbTarif->currentItem ()]);
  emit started ();
}

void
computerview::slotBtnPause ()
{
  btnStart->disconnect ();
  connect (btnStart, SIGNAL (clicked ()), this, SLOT (slotBtnStart ()));
  btnStart->setPixmap (icoStart);
  zbUser->timePause ();
  emit paused ();
}

void
computerview::slotBtnStop ()
{
  if(QMessageBox::query(tr("Delete user"),tr("do they want to really delete this user?"),tr("Yes"),tr("No"),this))
    {
      btnStart->disconnect ();
      connect (btnStart, SIGNAL (clicked ()), this, SLOT (slotBtnStart ()));
      btnStart->setPixmap (icoStart);
      btnStop->setEnabled (false);
      zbUser->timeStop ();
      priv->hide ();
      free->show ();
      zbUser->disconnect ();
      zbUser->~zbuser ();
      gbox->setTitle (viewTitle);
      emit stopped ();
      setAcceptDrops(true);
      active = false;
      emit updateUser();
    }
}

void
computerview::slotBtnMenu ()
{
  mainMenu->exec (btnMenu->mapToGlobal (QPoint (0, 0)));
}

void
computerview::slotBtnItem ()
{
  int i;
  if ((i = cmbItem->currentItem ()) > 0)
    {
      switch (i)
	{
	case 1:
	  {			// printout
	    printout *prn = new printout (this, "printout", true);
	    if (prn->exec () == QDialog::Accepted)
	      {
		zbUser->addPrint (prn->rdColor->isChecked (),
				  prn->count->value ());
	      }
	    prn->~printout ();
	  }
	  break;
	default:
	  zbUser->addItem (i - 2, 1);
	}
    }
  cmbItem->setCurrentItem (0);
  btnItem->setEnabled (false);
}

void
computerview::slotNewUser ()
{
  newuser *dlg = new newuser (this, "newuser", true);
  if (dlg->exec () == QDialog::Accepted)
    {
      zbUser =
	new zbuser (dlg->edName->text (), pList, pInfo,
		    dlg->edCountdown->isChecked (), dlg->edTime->time ());
      connect (zbUser, SIGNAL (nameChanged (QString)), this,
	       SLOT (slotNameChanged (QString)));
      connect (zbUser, SIGNAL (timeChanged (QTime)), this,
	       SLOT (slotTimeChanged (QTime)));
      connect (zbUser, SIGNAL (priceChanged (float)), this,
	       SLOT (slotPriceChanged (float)));
      connect(zbUser,SIGNAL(updateUser()),SLOT(slotUpdateUser()));
      slotNameChanged (dlg->edName->text ());
      zbUser->updateTarif (&(*tfList)[cmbTarif->currentItem ()]);
      free->hide ();
      priv->show ();
    }
  dlg->~newuser ();
  active = true;
  setAcceptDrops(false);
  emit updateUser();
}

void
computerview::slotViewHistory ()
{
  history *hst =
    new history (this, zbUser->name (), true, &(zbUser->usProdList), pList, zbUser->colorPrint, zbUser->grayPrint,zbUser->pInfo, zbUser->getPrice(), lbTime->text());
  hst->exec ();
}

void
computerview::slotPrintCalc ()
{

}

void
computerview::slotTarifChanged (int i)
{
  zbUser->updateTarif (&(*tfList)[i]);
}

void
computerview::slotTimeChanged (QTime tme)
{
  lbTime->setText (tme.toString ("hh:mm"));
  emit statusChanged (lbTime->text (),
		      lbPrice->text () + " " + lbCurency->text ());
  // 2002-08-23 c.toepp
  // if the countdown is out,
  // the screen will be locked
  if(tme.toString("hh:mm") == "00:00" && zbUser->countdown)
    {
      slotBtnPause();
      QMessageBox::information(NULL,"Users timeout",
			       QString("The time for the user %1 on computer %2 is out")
			       .arg(zbUser->name())
			       .arg(viewTitle));
    }
  // end changes
}

void
computerview::slotPriceChanged (float p)
{
  QString buf;
  buf.setNum (p,'f',2);
  lbPrice->setText (buf);
  emit statusChanged (lbTime->text (),
		      lbPrice->text () + " " + lbCurency->text ());
}

void
computerview::slotNameChanged (QString s)
{
  gbox->setTitle (gbox->title () + " - " + s);
}

void
computerview::slotItemChanged (int i)
{
  if (i > 0)
    btnItem->setEnabled (true);
  else
    btnItem->setEnabled (false);
}

bool
computerview::isActive ()
{
  return active;
  //return priv->isVisible();
}

void
computerview::setCurency (QString s)
{
  lbCurency->setText (s);
}

void
computerview::startDrag ()
{
  QPixmap p;
  p = QPixmap::grabWidget (priv);
  QDragObject *d = new QTextDrag (viewTitle, this);
  d->setPixmap (p, QPoint (70, 0));
  d->dragCopy ();
}

void
computerview::dragEnterEvent (QDragEnterEvent * event)
{
  event->accept (QTextDrag::canDecode (event));
  gbox->setFrameShadow (QFrame::Raised);
  gbox->setMidLineWidth (2);
}

void
computerview::dragLeaveEvent (QDragLeaveEvent * event)
{
  if (!event->spontaneous ())
    {
      gbox->setFrameShadow (QFrame::Sunken);
      gbox->setMidLineWidth (0);
    }
}

void
computerview::dropEvent (QDropEvent * event)
{
  QString s;
  gbox->setFrameShadow (QFrame::Sunken);
  gbox->setMidLineWidth (0);
  if (QTextDrag::decode (event, s))
    {
      QString type = event->source ()->className ();
      emit userDropped (s, viewTitle, type);
      setAcceptDrops(false);
    }
}

void
computerview::setOn (bool running)
{
  connect (zbUser, SIGNAL (nameChanged (QString)), this,
	   SLOT (slotNameChanged (QString)));
  connect (zbUser, SIGNAL (timeChanged (QTime)), this,
	   SLOT (slotTimeChanged (QTime)));
  connect (zbUser, SIGNAL (priceChanged (float)), this,
	   SLOT (slotPriceChanged (float)));
  connect(zbUser,SIGNAL(updateUser()),SLOT(slotUpdateUser()));
  zbUser->updateTarif (&(*tfList)[cmbTarif->currentItem ()]);
  free->hide ();
  priv->show ();
  if (running == true)
    {
      btnStart->disconnect ();
      connect (btnStart, SIGNAL (clicked ()), this, SLOT (slotBtnPause ()));
      btnStart->setPixmap (icoPause);
      btnStop->setEnabled (true);
      zbUser->timeStart ();
      zbUser->setTarif (&(*tfList)[cmbTarif->currentItem ()]);
      emit started ();
    }
  active = true;
  emit updateUser();
  setAcceptDrops(false);
}

void
computerview::setOff ()
{
  priv->hide ();
  free->show ();
  gbox->setTitle (viewTitle);
  btnStart->disconnect ();
  connect (btnStart, SIGNAL (clicked ()), this, SLOT (slotBtnStart ()));
  btnStart->setPixmap (icoStart);
  btnStop->setEnabled (false);
  zbUser->disconnect ();
  setAcceptDrops(true);
  active = false;
  emit updateUser();
}

bool
computerview::isOn ()
{
  return (btnStop->isEnabled ());
}

void
computerview::setCurrentTarif (QString s)
{
  cmbTarif->setCurrentText (s);
}

void
computerview::slotStarted ()
{
  srv->startComputer ();
}

void
computerview::slotStopped ()
{
  srv->stopComputer ();
}

void
computerview::slotPaused ()
{
  srv->stopComputer ();
}

void
computerview::slotStatus (QString time, QString price)
{
  srv->sendStatus (time, price);
}

void
computerview::slotBtnLock()
{
  if(lock_unlock->text() == tr("Unlock"))
    {
      lock_unlock->setText(tr("Lock"));
      srv->startComputer();
    }
  else
    {
      lock_unlock->setText(tr("Unlock"));
      srv->stopComputer();
    }
}
