/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static double SD_TTF_NAN=65536.0;
 
#include "swindow/SFontTTF.h"
static void debugChars (const char* msg, 
  const SS_GlyphIndex* gchars, unsigned int len);

#include "stoolkit/SStringVector.h"
#include "stoolkit/SExcept.h"
#include "stoolkit/SUniMap.h"
#include "stoolkit/SUtil.h"
#include "stoolkit/SCluster.h"
/*
 * ntohl 
 */
#ifndef USE_WINAPI
#include <netinet/in.h>
#else
#include <winsock.h>
#endif

#include <stdio.h>
#include <ctype.h>

/*
 * These glyphs should not exists khmm.. I said these glyphs should not exist.
 * Unicode magic :)
 */
#define SD_G_INDIC_ZWNJ 0xfffe
#define SD_G_INDIC_ZWJ 0xffff

/* 
 * TT_PLAT and TT_ENC from ttf.h from VFlib Library.
 * by Hirotsugu Kakugawa. Copyright: GNU.
 */

#define TT_ENC_ID_ANY                     -1
#define TT_ENC_ID_ISO_ASCII                0
#define TT_ENC_ID_ISO_10646                1
#define TT_ENC_ID_ISO_8859_1               2

#define TT_ENC_ID_MS_SYMBOL                0
#define TT_ENC_ID_MS_UNICODE               1
#define TT_ENC_ID_MS_SHIFT_JIS             2
#define TT_ENC_ID_MS_BIG5                  3
#define TT_ENC_ID_MS_RPC                   4
#define TT_ENC_ID_MS_WANSUNG               5
#define TT_ENC_ID_MS_JOHAB                 6

#define TT_ENC_ID_APPLE_DEFAULT            0
#define TT_ENC_ID_APPLE_UNICODE_1_1        1
#define TT_ENC_ID_APPLE_ISO_10646          2
#define TT_ENC_ID_APPLE_UNICODE_2_0        3

#define TT_ENC_ID_MAC_ROMAN                0
#define TT_ENC_ID_MAC_JAPANESE             1
#define TT_ENC_ID_MAC_TRADITIONAL_CHINESE  2
#define TT_ENC_ID_MAC_KOREAN               3
#define TT_ENC_ID_MAC_ARABIC               4
#define TT_ENC_ID_MAC_HEBREW               5
#define TT_ENC_ID_MAC_GREEK                6
#define TT_ENC_ID_MAC_RUSSIAN              7
#define TT_ENC_ID_MAC_RSYMBOL              8
#define TT_ENC_ID_MAC_DEVANAGARI           9
#define TT_ENC_ID_MAC_GURMUKHI             10
#define TT_ENC_ID_MAC_GUJARATI             11
#define TT_ENC_ID_MAC_ORIYA                12
#define TT_ENC_ID_MAC_BENGALI              13
#define TT_ENC_ID_MAC_TAMIL                14
#define TT_ENC_ID_MAC_TELUGU               15
#define TT_ENC_ID_MAC_KANNADA              16
#define TT_ENC_ID_MAC_MALAYALAM            17
#define TT_ENC_ID_MAC_SINHALESE            18
#define TT_ENC_ID_MAC_BURMESE              19
#define TT_ENC_ID_MAC_KHMER                20
#define TT_ENC_ID_MAC_THAI                 21
#define TT_ENC_ID_MAC_LAOTIAN              22
#define TT_ENC_ID_MAC_GEORGIAN             23
#define TT_ENC_ID_MAC_ARMENIAN             24
#define TT_ENC_ID_MAC_MALDIVIAN            25
#define TT_ENC_ID_MAC_SIMPLIFIED_CHINESE   25
#define TT_ENC_ID_MAC_TIBETAN              26
#define TT_ENC_ID_MAC_MONGOLIAN            27
#define TT_ENC_ID_MAC_GEEZ                 28
#define TT_ENC_ID_MAC_SLAVIC               29
#define TT_ENC_ID_MAC_VIETNAMESE           30
#define TT_ENC_ID_MAC_SINDHI               31
#define TT_ENC_ID_MAC_UNINTERP             32

#define TT_PLAT_ID_ANY         -1
#define TT_PLAT_ID_APPLE       0
#define TT_PLAT_ID_MACINTOSH   1
#define TT_PLAT_ID_ISO         2
#define TT_PLAT_ID_MICROSOFT   3

/*
'arial.ttf' '' format=4 platform=0 encoding_id=3.
'arial.ttf' '' format=0 platform=1 encoding_id=0.
'arial.ttf' '' format=4 platform=3 encoding_id=1.
'msgothic.ttf' '' format=0 platform=1 encoding_id=0.
'msgothic.ttf' '' format=4 platform=3 encoding_id=1.
'cyberbit.ttf' '' format=0 platform=1 encoding_id=0.
'cyberbit.ttf' '' format=4 platform=3 encoding_id=1.
'times.ttf' '' format=4 platform=0 encoding_id=3.
'times.ttf' '' format=0 platform=1 encoding_id=0.
'times.ttf' '' format=4 platform=3 encoding_id=1.
'msmincho.ttf' '' format=0 platform=1 encoding_id=0.
'msmincho.ttf' '' format=4 platform=3 encoding_id=1.
'mangal.ttf' '' format=0 platform=1 encoding_id=0.
'mangal.ttf' '' format=4 platform=3 encoding_id=1.
'freearial.ttf' 'cp-1250' format=4 platform=0 encoding_id=0.
'freearial.ttf' 'cp-1250' format=0 platform=1 encoding_id=0.
'freearial.ttf' 'cp-1250' format=4 platform=3 encoding_id=1.
*/

/**
 * This is a hash width key platform '~' encoding 
 * and value SUnimap.
 */
static const SString SS_TB_NAME("name");
static const SString SS_TB_HEAD("head");
static const SString SS_TB_HHEA("hhea");
static const SString SS_TB_POST("post");
static const SString SS_TB_GLYF("glyf");
static const SString SS_TB_CMAP("cmap");
static const SString SS_TB_KERN("kern");
static const SString SS_TB_MAXP("maxp");
static const SString SS_TB_HTMX("hmtx");
static const SString SS_TB_LOCA("loca");

static const long SS_TN_NOTICE=0;
static const long SS_TN_FAMILY=1;
static const long SS_TN_WEIGHT=2;
static const long SS_TN_X3=3;
static const long SS_TN_FULLNAME=4;
static const long SS_TN_VERSION=5;
static const long SS_TN_FONTNAME=6;
static const long SS_TN_X7=7;
static const long SS_TN_MAX=8;


/*
 * The following definitions were derived from
 * Andrew Weeks's excellent font converter program.
 * Many thanks!
 */
#define BYTE unsigned char
#define CHAR signed char
#define USHORT unsigned short
#define SHORT signed short
#define ULONG unsigned int
#define LONG signed int
#define FWORD SHORT
#define UFWORD USHORT

#define ONOROFF 0x01
#define XSHORT  0x02
#define YSHORT  0x04
#define REPEAT  0x08
#define XSAME   0x10
#define YSAME   0x20
#define ARG_1_AND_2_ARE_WORDS           0x0001
#define ARGS_ARE_XY_VALUES                      0x0002
#define XY_BOUND_TO_GRID                        0x0004
#define WE_HAVE_A_SCALE                         0x0008
#define MORE_COMPONENTS                         0x0020
#define WE_HAVE_AN_X_AND_Y_SCALE        0x0040
#define WE_HAVE_A_TWO_BY_TWO            0x0080
#define WE_HAVE_INSTRUCTIONS            0x0100
#define USE_MY_METRICS                          0x0200

typedef struct short_2 {
        SHORT   upper;
        USHORT  lower;
} SSFIX ;

typedef struct longhormetric {
        UFWORD  advanceWidth;
        FWORD   lsb;
} LONGHORMETRIC;

typedef struct ttf_hhea {
        BYTE    version[4];
        SHORT   ascender, descender, lineGap;
        USHORT  advnaceWidthMax;
        SHORT   minLSB, minRSB, xMaxExtent;
        SHORT   caretSlopeRise, caretSlopeRun;
        SHORT   reserved[5];
        SHORT   metricDataFormat;
        USHORT  numberOfHMetrics;
} TTF_HHEA;

typedef struct ttf_dir_entry {
        char    tag[4];
        ULONG   checksum;
        ULONG   offset;
        ULONG   length;
} TTF_DIR_ENTRY ;

typedef struct ttf_directory {
        ULONG                   sfntVersion;
        USHORT                  numTables;
        USHORT                  searchRange;
        USHORT                  entrySelector;
        USHORT                  rangeShift;
        TTF_DIR_ENTRY   list;
} TTF_DIRECTORY ;

typedef struct ttf_name_rec {
        USHORT  platformID;
        USHORT  encodingID;
        USHORT  languageID;
        USHORT  nameID;
        USHORT  stringLength;
        USHORT  stringOffset;
} TTF_NAME_REC;

typedef struct ttf_name {
        USHORT                  format;
        USHORT                  numberOfNameRecords;
        USHORT                  offset;
        TTF_NAME_REC    nameRecords;
} TTF_NAME ;

/*
 Flags is HEAD table - grossly ignored by yudit :)
Bit 0: Baseline for font at y=0;
Bit 1: Left sidebearing point at x=0;
Bit 2: Instructions may depend on point size; 
Bit 3: Force ppem to integer values for all internal scaler math;
may use fractional ppem sizes if this bit is clear; 
Bit 4: Instructions may alter advance width (the advance widths
might not scale linearly); 
Bits 5-10: These should be set according to Apple's specification
. However, they are not implemented in OpenType. 
Bit 11: Font data is 'lossless,' as a result of having been
compressed and decompressed with the Agfa MicroType
Express engine.
Bit 12: Font converted (produce compatible metrics)
Bit 13: Font optimised for ClearType
Bit 14: Reserved, set to 0
Bit 15: Reserved, set to 0 
*/
typedef struct ttf_head {
        ULONG   version;
        ULONG   fontRevision;
        ULONG   checksumAdjust;
        ULONG   magicNo;
        USHORT  flags;
        USHORT  unitsPerEm;
        BYTE    created[8];
        BYTE    modified[8];
        FWORD   xMin, yMin, xMax, yMax;
        USHORT  macStyle, lowestRecPPEM;
        SHORT   fontDirection, indexToLocFormat, glyphDataFormat;
} TTF_HEAD ;


typedef struct ttf_kern {
        USHORT  version, nTables;
} TTF_KERN ;

typedef struct ttf_kern_sub {
        USHORT version, length, coverage;
        USHORT nPairs, searchRange, entrySelector, rangeShift;
} TTF_KERN_SUB;

typedef struct ttf_kern_entry {
        USHORT  left, right;
        FWORD   value;
} TTF_KERN_ENTRY;

typedef struct ttf_cmap_fmt0 {
        USHORT  format;
        USHORT  length;
        USHORT  version;
        BYTE    glyphIdArray[256];
} TTF_CMAP_FMT0;

typedef struct ttf_cmap_fmt4 {
        USHORT  format;
        USHORT  length;
        USHORT  version;
        USHORT  segCountX2;
        USHORT  searchRange;
        USHORT  entrySelector;
        USHORT  rangeShift;
} TTF_CMAP_FMT4;

typedef struct ttf_cmap_entry {
        USHORT  platformID;
        USHORT  encodingID;
        ULONG   offset;
} TTF_CMAP_ENTRY;

typedef struct ttf_cmap {
        USHORT                  version;
        USHORT                  numberOfEncodingTables;
        TTF_CMAP_ENTRY  encodingTable[1];
} TTF_CMAP ;

typedef struct ttf_glyf {
        SHORT   numberOfContours;
        FWORD   xMin, yMin, xMax, yMax;
} TTF_GLYF ;

typedef struct ttf_maxp {
        ULONG   version;
        USHORT  numGlyphs, maxPoints, maxContours;
        USHORT  maxCompositePoints, maxCompositeContours;
        USHORT  maxZones, maxTwilightPoints, maxStorage;
        USHORT  maxFunctionDefs, maxInstructionsDefs;
        USHORT  maxSizeOfInstructions, maxComponentElements;
        USHORT  maxComponentDepth;
} TTF_MAXP ;

typedef struct ttf_post_head {
        ULONG   formatType;
        SSFIX   italicAngle;
        FWORD   underlinePosition;
        FWORD   underlineThickness;
        ULONG   isFixedPitch;
        ULONG   minMemType42;
        ULONG   maxMemType42;
        ULONG   minMemType1;
        ULONG   maxMemType1;
        USHORT  numGlyphs;
        USHORT  glyphNameIndex;
} TTF_POST_HEAD ;

static SS_GlyphIndex findGlyph0 (TTF_CMAP_FMT0* encoding0, SS_UCS2 ucs2);
static SS_GlyphIndex findGlyph4 (TTF_CMAP_FMT4* encoding4, SS_UCS2 ucs2);

static double f2dot14 (short x);
static void moveto (SCanvas* canvas, const SS_Matrix2D &m,
  SHORT _x, SHORT _y);
static void lineto (SCanvas* canvas, const SS_Matrix2D &m,
  SHORT _x, SHORT _y);
static void cureveto (SCanvas* canvas, const SS_Matrix2D &m,
  SHORT _x0, SHORT _y0,
  SHORT _x1, SHORT _y1,
  SHORT _x2, SHORT _y2);

/**
 * @author: Gaspar Sinai <gsinai@yudit.org>
 * @version: 2000-04-23
 * Many parts of this file are originally written by Andrew Weeks.
 */


/**
 * Initialize a TTF
 * @param name is the font file
 */
SFontTTF::SFontTTF (const SFile& _file,  const SString& _fontencoding) 
  : file (_file)
{
  ok = true;
  fontencoding = _fontencoding;
}

SFontTTF::~SFontTTF ()
{
}

bool
SFontTTF::isOK()
{
  if (!ok) return false;

  if (name.size()==0)
  {
    name = file.getName();
//fprintf (stderr, "Setting name to %*.*s\n", SSARGS(name));
    if (file.size() < 0)
    {
      ok = false;
    }
    else
    {
      image = file.getFileImage();
      ok = image.size()>0 && init();
    }
  }
  return ok;
}

/**
 * initialize all numbers.
 * return false if something is wrong with this font.
 */
bool
SFontTTF::init ()
{
  TTF_DIRECTORY* directory = (TTF_DIRECTORY *) image.array();
  if (ntohl (directory->sfntVersion) != 0x00010000)
  {
    fprintf (stderr, "SFontTTF: BAD TTF file [%*.*s].\n", SSARGS(name));
    return false;
  }
  TTF_DIR_ENTRY* dir_entry = &(directory->list);
  
  char tag[5];
  unsigned int i;
  for (i=0; i < (unsigned short)ntohs(directory->numTables); i++)
  {
    for (unsigned int j=0; j<4; j++)
    {
      tag[j] = dir_entry->tag[j];
    }
    tag[4] = 0;
    tables.put (tag, image.array() + ntohl (dir_entry->offset));
    if (memcmp(tag, "EBDT", 4)==0 || memcmp(tag, "EBLC", 4)==0
        || memcmp(tag, "EBSC", 4)==0)
    {
      //fprintf (stderr, "SFontTTF info: TTF file [%*.*s] contains bitmaps.\n",
       //  SSARGS(name));
    }
    dir_entry++;
  }
  if (!processName())
  {
     return false;
  }
  if (!checkTables())
  {
     return false;
  }
  broken = false;

  TTF_CMAP* cmap_table = (TTF_CMAP*) tables[SS_TB_CMAP];
  int num_tables = ntohs(cmap_table->numberOfEncodingTables);

  /* We go through all the tables and choose the bes table */
  int bestType = 0;
  if (fontencoding.size())
  {
    charEncoder = SUniMap(fontencoding);
    if (!charEncoder.isOK())
    {
      fprintf (stderr, "SFontTTF: umap '%*.*s' not found for '%*.*s'.\n",
        SSARGS(fontencoding), SSARGS(name));
    }
  }
  charEncoderTable = (unsigned int) num_tables; 
  for (i=0; i < (unsigned int) num_tables; i++)
  {
    TTF_CMAP_ENTRY* table_entry = &(cmap_table->encodingTable[i]);
    int offset = ntohl(table_entry->offset);
    TTF_CMAP_FMT4* encoding4 = (TTF_CMAP_FMT4 *) ((BYTE *)cmap_table + offset);
    int format = ntohs(encoding4->format);
    int platform = ntohs(table_entry->platformID);
    int encoding_id = ntohs(table_entry->encodingID);
    /**
     * All 
     *  platform == TT_PLAT_ID_MICROSOFT(3)
     *  encoding_id == TT_ENC_ID_ISO_10646 (1)
     * will have TTF_CMAP_FMT4 (format==4
     * character map. Others *might* have. 
     *
     * TODO: support more cmap formats like 32 bit unicode.
     *  currently 32 bit unicode is done through external map.
     */
    if (format != 4) continue;

    /* could not find any good table */
    if (charEncoderTable ==(unsigned int) num_tables)
    {
      charEncoderTable = i;
    }
/*
    fprintf (stderr, "%*.*s platform=%d encoding=%d at 0x%X\n", SSARGS(name),
       platform, encoding_id, (char*)&table_entry->encodingID - (char*)directory);
*/
    switch (platform)
    {
    case TT_PLAT_ID_MICROSOFT:
      switch (encoding_id)
      { 
      case TT_ENC_ID_MS_SYMBOL:
        break;
      case TT_ENC_ID_MS_UNICODE:
//fprintf (stderr, "Microsoft=%u %*.*s\n", i, SSARGS(name));
        bestType = 10; /* these mistic numbers are my scores */
        charEncoderTable = i;
        break;
      case TT_ENC_ID_MS_SHIFT_JIS:
      case TT_ENC_ID_MS_BIG5:
      case TT_ENC_ID_MS_RPC:
      case TT_ENC_ID_MS_WANSUNG:
      case TT_ENC_ID_MS_JOHAB:
      default:
         break;
     }
    case TT_PLAT_ID_ISO:
      switch (encoding_id)
      { 
      case TT_ENC_ID_ANY:
        break;
      case TT_ENC_ID_ISO_ASCII:
        if (bestType < 2) 
        {
           bestType = 2;
           charEncoderTable = i;
        }
        break;
      case TT_ENC_ID_ISO_10646:
        if (bestType < 8) 
        {
           bestType = 8;
           charEncoderTable = i;
        }
        break;
      case TT_ENC_ID_ISO_8859_1:
        if (bestType < 4) 
        {
           bestType = 4;
           charEncoderTable = i;
        }
        break;
      default:
        break;
     }
    case TT_PLAT_ID_APPLE:
      switch (encoding_id)
      { 
      case TT_ENC_ID_APPLE_DEFAULT:
        break;
      case TT_ENC_ID_APPLE_UNICODE_1_1:
      case TT_ENC_ID_APPLE_ISO_10646:
      case TT_ENC_ID_APPLE_UNICODE_2_0:
        if (bestType < 7) 
        {
           bestType = 8;
           charEncoderTable = i;
        }
      default:
         break;
     }
    case TT_PLAT_ID_MACINTOSH:
      switch (encoding_id)
      { 
      case TT_ENC_ID_MAC_ROMAN:
      /* al lot of other encodings missing */
      default:
       break;
     }
    default:
      break;
    }
  }
  /* look at all tables SGC */
  if (fontencoding.size()!=0) charEncoderTable =(unsigned int) num_tables;

  SString chk = name;
  chk.lower();
  return true;
}


/**
 * Process the name table
 */
bool
SFontTTF::processName ()
{
  TTF_NAME*  name_table = (TTF_NAME*) tables[SS_TB_NAME];
  if (name_table==0)
  {
    getName (SS_TN_FONTNAME, name.array(), name.size());
    fprintf (stderr, "SFontTTF: No name fields in %*.*s\n", SSARGS(name));
    fprintf (stderr, "SFontTTF: records:[");
    for (unsigned int i=0; i<tables.size(); i++)
    {
      for (unsigned int j=0; j<tables.size(i); j++)
      {
        SString key = tables.key(i, j);
        fprintf (stderr, " %*.*s", SSARGS(key));
        
      }
    }
    fprintf (stderr, " ]\n");
    return false;
  }

  TTF_NAME_REC* name_record = &(name_table->nameRecords);
  char* string_area = (char *)name_table + ntohs(name_table->offset);
    
  int found=0;
  int i;
  for (i=0; i < ntohs (name_table->numberOfNameRecords); i++) 
  {
    short platform = ntohs(name_record->platformID);
    if (platform == 3)
    {
      found = 1;
      short len = ntohs(name_record->stringLength);
      short strOffset = ntohs(name_record->stringOffset);
      long nameId = ntohs(name_record->nameID);
      if (nameId < SS_TN_MAX)
      {
        getName (nameId, &string_area[strOffset], len);
      }
    }
    name_record++;
  }

  name_record = &(name_table->nameRecords);
  if (!found) for (i=0; i < ntohs(name_table->numberOfNameRecords); i++) 
  {
   short platform = ntohs(name_record->platformID); if (platform ==1)
   {
      found = 1;
      short len = ntohs(name_record->stringLength);
      short strOffset = ntohs(name_record->stringOffset);
      long nameId = ntohs(name_record->nameID);
      if (nameId < SS_TN_MAX)
      {
        getName (nameId, &string_area[strOffset], len);
      }
    }
    name_record++;
  }
  if (!found)
  {
    fprintf (stderr, "SFontTTF: BAD Name fields in %*.*s\n", SSARGS(name));
    return false;
  }

  if (names.get (SS_TN_FONTNAME) == 0 || names[SS_TN_FONTNAME].size() == 0)
  {
    getName (SS_TN_FONTNAME, names[SS_TN_FULLNAME].array(),
         names[SS_TN_FULLNAME].size());
  }
  return true;
}

/**
 * put the string from str into names. 
 * @param id is SS_TN_ something.
 * @param str is the input string
 * @param len is the size of the stirng
 */
void
SFontTTF::getName (long id, const char* str, int len)
{
  SString s;
  for (int i=0; i<len; i++)
  {
    if (str[i] == 0) continue;
    if (id==SS_TN_FONTNAME)
    {
      if (isalnum(str[i]))
      {
        s.append (str[i]);
      }
      else
      {
        s.append ((char)'_');
      }
    }
    /* This is to make postscript files clean */
    else switch (str[i])
    {
    case '(':
      s.append ((char)'[');
      break;
    case ')':
      s.append ((char)']');
      break;
    default:
      s.append (str[i]);
    }
  }
  names.put (id, s);
}

/**
 * Do a sanity check.all tables
 */
bool
SFontTTF::checkTables ()
{

  if (tables[SS_TB_HEAD] == 0)
  {
    fprintf (stderr, "SFontTTF: BAD head table in %*.*s\n", SSARGS(name));
    return false;
  }
  TTF_HEAD* head_table = (TTF_HEAD*) tables[SS_TB_HEAD];
  longOffsets = ntohs (head_table->indexToLocFormat);

  if (longOffsets != 0 && longOffsets != 1)
  {
    fprintf (stderr, "SFontTTF: BAD TTF file [%*.*s] - indexToLocFormat.\n",
       SSARGS(name));
    return false;
  }

  if (tables[SS_TB_HHEA] == 0)
  {
    fprintf (stderr, "SFontTTF: BAD hhea table in %*.*s\n", SSARGS(name));
    return false;
  }
  if (tables[SS_TB_GLYF] == 0)
  {
    fprintf (stderr, "SFontTTF: BAD glyf table in %*.*s\n", SSARGS(name));
    return false;
  }
  if (tables[SS_TB_CMAP] == 0)
  {
    fprintf (stderr, "SFontTTF: BAD cmap table in %*.*s\n", SSARGS(name));
    return false;
  }
  if (tables[SS_TB_HTMX] == 0)
  {
    fprintf (stderr, "SFontTTF: BAD htmx table in %*.*s\n", SSARGS(name));
    return false;
  }
  if (tables[SS_TB_LOCA] == 0)
  {
    fprintf (stderr, "SFontTTF: BAD loca table in %*.*s\n", SSARGS(name));
    return false;
  }
  TTF_POST_HEAD*  post_table = (TTF_POST_HEAD*) tables[SS_TB_POST];
  if (post_table == 0)
  {
    fprintf (stderr, "SFontTTF: missing post table in %*.*s. Using defaults\n",
         SSARGS(name));
    italicAngle = 0.0;
    underlineThickness = 100;
    underlinePosition = 0.0;
    isFixedPitch = 0.0;
  }
  else
  {
    italicAngle = (double) (ntohs(post_table->italicAngle.upper)) +
      (ntohs(post_table->italicAngle.lower) / 65536.0);
    underlineThickness = (double)ntohs(post_table->underlineThickness);
    underlinePosition = (double)ntohs(post_table->underlinePosition);
    isFixedPitch = (ntohl(post_table->isFixedPitch))? true : false;
  }
  TTF_HHEA* hhea_table = (TTF_HHEA*) tables[SS_TB_HHEA];

  lineGap = (double) ((short)htons (hhea_table->lineGap));
  charWidth = (double) ((short)(ntohs(head_table->xMax)+ ntohs(head_table->xMin)));
  charAscent = (double) ((short)htons (hhea_table->ascender));
  /* Be aware charDescent is negative! */
  charDescent = (double) ((short)htons (hhea_table->descender));

  /* kairali-S: dscale=1000 charAscent=298 charDescent=-202 
        dscale_factor dscale/(charAscent-charDescent) */

  /* charDescent is negative - take charAscent aonly to determine size */
  double charheight = charAscent;
  if (charheight < 1) charheight = 1;
  scaleFactor = 1.0 / charheight;
/*
  short unitsPerEM  = ntohs (head_table->unitsPerEm);
  fprintf (stderr, "font=%*.*s charheight=%g unitsPerEM=%d\n", 
      SSARGS(name), charheight, unitsPerEM);
*/
  return true;
}

/**
 * Draw a single unicode character on canvas using the pen.
 * before calling this, you should call a newpath and
 * after calling this you may want to call fill.
 * @param canvas is the canvas to draw to
 * @param m is the transformation matrix.
 * @param uch is the unicode character
 * @param len is the length of unicode array
 * @return true if drawn
 */
bool
SFontTTF::draw (SCanvas* canvas, const SS_Matrix2D& matrix, SS_UCS4 _uch)
{
  if (!isOK()) return false;
  SV_GlyphIndex gi;
  if (!findGlyphs (_uch, &gi)) return false;
  /* This makes things a bit faster */
  if (gi.size()==1)
  {
    SS_GlyphIndex g = gi[0];
    if (g == SD_G_INDIC_ZWJ || g== SD_G_INDIC_ZWNJ) return true;
    drawGlyph (canvas, matrix, g);
    return true;
  }

  SS_Matrix2D mo = matrix;
  for (unsigned int i=0; i<gi.size(); i++)
  {
    /* do not draw them */
    SS_GlyphIndex g = gi[i];
    if (g != SD_G_INDIC_ZWJ &&  g != SD_G_INDIC_ZWNJ) drawGlyph (canvas, mo, g);
    double nwidth = widthGlyph (mo, g);
    if (nwidth < 0) nwidth = -nwidth;
    mo.translate (nwidth, 0.0);
  }
  return true;
}

/**
 * Return the advacnce width of the glyphs
 * The value is the value that is multipied with matrix.
 * @param m is the transformation matrix.
 * @param uch is the unicode character
 * @param len is the length of unicode array
 * @param used will show how many characters were used in uch
 * @return the calibrated advace width, if with_ is passed and 
 *  true if it exists.
 */
bool
SFontTTF::width (const SS_Matrix2D& m, SS_UCS4 _uch, double* width_)
{
  if (width_) *width_ = 0.0;
  if (!isOK()) return false;

  SV_GlyphIndex gi;
  if (!findGlyphs (_uch, &gi)) return false;
  if (!width_) return true;

  /**
   * Multiple glyphs draw on top of each other. 
   */
  double max = 0;
  /* We draw one after the other.*/
  for (unsigned int i=0; i<gi.size(); i++)
  {
    /* do not draw them */
    double nwidth = widthGlyph (m, gi[i]);
    if (nwidth < 0) nwidth = -nwidth;
    max += nwidth;
  }
  *width_ = max;
  return true;
}

/**
 * Find out the width, knowing the local glyph number
 * @param m is the transformation matrix
 * @param glyphno is the local glyph index in the glyph table.
 */
double
SFontTTF::widthGlyph (const SS_Matrix2D& m, SS_GlyphIndex glyphno)
{
  if (!isOK()) return 0.0;
  if (glyphno == SD_G_INDIC_ZWJ) return 0.0;
  if (glyphno == SD_G_INDIC_ZWNJ) return 0.0;

  SString key ((char*)&glyphno, sizeof (SS_GlyphIndex));
  double cw = char2Width.get (key);
  if (cw == SD_TTF_NAN) return 0.0;
  if (cw != 0.0) return cw * m.x0;

  TTF_HHEA* hhea_table = (TTF_HHEA*) tables[SS_TB_HHEA];
  LONGHORMETRIC* hmtx_entry = (LONGHORMETRIC*) tables[SS_TB_HTMX];

  int n_hmetrics = ntohs(hhea_table->numberOfHMetrics);
  FWORD* lsblist = (FWORD *) &hmtx_entry[n_hmetrics];
  
  short w;
  short lsb;

  if (glyphno >= n_hmetrics)
  {
    w = ntohs (hmtx_entry[n_hmetrics-1].advanceWidth);
    lsb = ntohs (lsblist[glyphno-n_hmetrics]);
  }
  else
  {
    w = ntohs(hmtx_entry[glyphno].advanceWidth);
    lsb = ntohs(hmtx_entry[glyphno].lsb);
  }
  /* replace 0.0 with SD_NAN */
  if (w==0.0)
  {
    char2Width.put (key, (double) SD_TTF_NAN);
  }
  else 
  {
    char2Width.put (key, (double)w);
  }
  return double (w) * m.x0;
}

/**
 * Return the calibrated ascent
 * @param m is the transformation matrix.
 */
double
SFontTTF::ascent (const SS_Matrix2D& m)
{
  if (!isOK()) return 0.0;
  double rvle =  charAscent * m.y1;
  if (rvle < 0)
  {
    fprintf (stderr, "FIXME negative ascent: SFontTTF.cpp\n");
    return 1;
  }
  return rvle;
}

/**
 * Return the calibrated descent
 * @param m is the transformation matrix.
 */
double
SFontTTF::descent (const SS_Matrix2D& m)
{
  if (!isOK()) return 0.0;
  double rvle =  - charDescent * m.y1;
  if (rvle < 0)
  {
    fprintf (stderr, "FIXME negative descent: SFontTTF.cpp\n");
    return 1;
  }
  return rvle;
}

/**
 * Return the calibrated avarage width
 * @param m is the transformation matrix.
 */
double
SFontTTF::width (const SS_Matrix2D& m)
{
  if (!isOK()) return 0.0;
  double rvle =  charWidth * m.x0;
  if (rvle < 0) return -rvle;
  return rvle;
}

/**
 * Return the calibrated gap
 * @param m is the transformation matrix.
 */
double
SFontTTF::gap (const SS_Matrix2D& m) 
{
  if (!isOK()) return 0.0;
  double rvle =  lineGap * m.y1;
  if (rvle < 0) return -rvle;
  return rvle;
}

/**
 * Return the scale factor. You multiply this with point size you want.
 * matrix diagonals for a 10 point font is scale, scale
 */
double
SFontTTF::scale ()
{
  if (!isOK()) return 0.0;
  return scaleFactor;
}


/**
 * This routine tries to find the glyph indeces of a unicode input stream
 * @param in is the input stream
 * @param len is the length if in
 * @param out is the output 
 * @return the nuber of characters processed in 'in'
 */
bool
SFontTTF::findGlyphs (SS_UCS4 in, SV_GlyphIndex* out)
{
  if (!isOK()) return false;
  SString key ((char*)&in, sizeof (SS_UCS4));
  const SString* cached = char2Glyphs.get (key);
  if (cached)
  {
    if (cached->size()<sizeof (SS_GlyphIndex)) return false;
    unsigned int usize = cached->size();
    for (unsigned int i=0; i<usize; i+= sizeof (SS_GlyphIndex))
    {
       out->append (*((SS_GlyphIndex*)&cached->array()[i]));
    }
    return true;
  }
  

  TTF_CMAP* cmap_table = (TTF_CMAP*) tables[SS_TB_CMAP];
  int num_tables = ntohs(cmap_table->numberOfEncodingTables);
  if (num_tables ==0)
  {
    SString chc; char2Glyphs.put (key, chc);
    return false;
  }

  /* Try to get the ligature index from OTF */
  SS_UCS4 lig = in;
  unsigned int liglen = 0;
  if (isLigature (lig) && hasOTFLigatures() 
       && (liglen=getLigatureUnicode(lig, 0)) > 0)
  {
     unsigned int scriptcode = getLigatureScriptCode (lig);
     bool fixedcluster = true;
     switch (scriptcode)
     {
     case SD_TAMIL:
       break;
     default: 
       fixedcluster = false;
       break;
     }
     /* we allocate one more to allow for LEFT_RIGHT vowel expansion */
     SS_UCS4* chars =  new SS_UCS4[liglen+1];
     CHECK_NEW (chars);
     SS_GlyphIndex* gchars = new SS_GlyphIndex[liglen+1];
     CHECK_NEW (gchars);
     getLigatureUnicode (lig, chars);
     char * script = getLigatureScript (lig);
     if (script == 0) script = "default";

     SUniMap umap = charEncoder;
     if (!umap.isOK())
     {
        delete chars;
        delete gchars;
        SString chc;
        char2Glyphs.put (key, chc);
        return false;
     }

     /* get the encoder for this table. */
     bool decoded = true; 
     /* we need this hocus-pocus because getLigature works on
        glyph indeces */
     /* for indic modifiers */
     unsigned int mstart = 0;
     unsigned int mend = 0;
     for (unsigned int i=0; i<liglen; i++)
     {
       SS_UCS2 ucs2 = umap.encode (chars[i]);
       if (ucs2==0)
       {
         if (chars[i] > 0xffff)
         {
            decoded = false;
            break;
         }
         /* BE AWARE HACK! Try straight unicode  */
         ucs2 = chars[i];
       }
       gchars[i] =  findGlyph(ucs2);
       if (gchars[i]==0)
       {
         decoded = false;
         break;
       }
       int endtype = getCharType (chars[i]);
       if (i>0 && endtype == SD_INDIC_MODIFIER && mstart == 0)
       {
          mstart = i; mend = liglen;
       }
     }
     /* adjust liglen to where modifiers start */
     if (mstart != 0) liglen = mstart;

     /* this is unicode encoded... */
     SS_GlyphIndex halant = findGlyph (getHalant (scriptcode));
     SS_GlyphIndex reorder = 0;
     SS_GlyphIndex addVirama = 0;
     unsigned int inlen = liglen;
     bool *gbase = NULL;

     // post-consonant Malayalam ra has to be reordered to syllable start
     if (scriptcode == SD_MALAYALAM)
        reorder = findGlyph (0x0d30);

     // special rules for clusters ending in virama
     if (decoded && liglen == 2 && chars[1] == getHalant(scriptcode))
     {
       decoded = false;
       unsigned int olen = getOTFLigatures (gchars, inlen, 
            script, "haln", halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS haln=", gchars, olen);
          decoded = true;
          inlen--;
       }
     }
     else if (decoded && chars[liglen-1] == getHalant(scriptcode))
     {
        // todo - RA+H RA+H
        addVirama = gchars[liglen-1];
        inlen--;
     }

     if (fixedcluster && decoded)
     {
       SS_GlyphIndex gi;
       unsigned int nind =  getOTFLigature (script, 0, gchars, liglen, &gi);
       if (nind == liglen)
       {
         out->append (gi);
       }
       else
       {
         decoded = false;
       }
     }
     else if (decoded)
     {
       /* ----> DEBUG Information */
       debugChars ("GCHARS=", gchars, liglen);
       /* ----< DEBUG Information */
       unsigned int olen = getOTFLigatures (gchars, inlen, 
            script, "akhn", halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS akhn=", gchars, olen);
          inlen = olen;
       }
       /* can be at beginning only */
       SS_GlyphIndex rphfGlyph = 0;
       SS_GlyphIndex rphfNone = 0;
       if (inlen>2 && gchars[2] != findGlyph(0x200d))
       {
         debugChars ("BEFORE RPH =", gchars, inlen);
         SS_GlyphIndex g[2]; g[0] = gchars[0]; g[1] = gchars[1];
         olen = getOTFLigatures (g, 2 , script, "rphf",
              halant, reorder, gbase);
         if (olen == 2)
           olen = getOTFLigatures (g, 2, script, "abvs",
                halant, reorder, gbase);
         if (olen == 1 && liglen > 2)
         {
           int ct = getCharType (chars[2]);
           // if chars[2] == 0x200d will be handled automagically here 
           if (ct == SD_INDIC_CONSONANT)
           {
             debugChars ("GCHARS rphf=", g, olen);
             rphfGlyph = g[0];
           }
           else
           {
             //fprintf (stderr, "GCHARS rphfNone\n");
             rphfNone = gchars[0];
           }
           /* remove */
           for (unsigned int i=2; i<inlen; i++) gchars[i-2] = gchars[i];
           inlen -= 2;
         }
         debugChars ("AFTER RPH =", gchars, inlen);
       }

       // Vowel placement in Malayalam is somewhat peculiar, as compared
       // to other Indic scripts; also Telugu and Kannada need special treatment
       if ((scriptcode == SD_MALAYALAM &&
           (getCharType (chars[liglen-1]) == SD_INDIC_LEFT_VOWEL ||
            getCharType (chars[liglen-1]) == SD_INDIC_LEFT_RIGHT_VOWEL)) ||
           ((scriptcode == SD_TELUGU || scriptcode == SD_KANNADA) &&
           liglen > 2))
       {
          gbase = new bool [inlen-1];
          for (unsigned int i=0; i<inlen-1; i++)
          {
             if (gchars[i] == halant)
                gbase[i] = false;
             else gbase[i] = true;
          }
       }

       olen = getOTFLigatures (gchars, inlen,
            script, "blwf", halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS blwf=", gchars, olen);
          inlen = olen;
       }
       olen = getOTFLigatures (gchars, inlen,
            script, "vatu", halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS vatu=", gchars, olen);
          inlen = olen;
       }
       olen = getOTFLigatures (gchars, inlen,
            script, "pstf", halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS pstf=", gchars, olen);
          inlen = olen;
       }
       olen = getOTFLigatures (gchars, inlen, script, "blws",
            halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS blws=", gchars, olen);
          inlen = olen;
       }
       olen = getOTFLigatures (gchars, inlen, script, "psts",
            halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS psts=", gchars, olen);
          inlen = olen;
       }
       /* if we still have U+0931 at this point, let's try U+0930 "half" */
       if (gchars[0] == findGlyph(0x0931) && inlen > 2)
       {
          gchars[0] = findGlyph(0x0930);
          olen = getOTFLigatures (gchars, inlen, script, "half",
               halant, reorder, gbase);
          if (olen != inlen)
          {
             debugChars ("GCHARS eyelash=", gchars, olen);
             inlen = olen;
          }
          else // otherwise we change it back to U+0931
          {
             gchars[0] = findGlyph(0x0931);
          }
       }
       /* Half-forms */
       olen = getOTFLigatures (gchars, inlen, script, "half",
            halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS half=", gchars, olen);
          inlen = olen;
       }

       olen = getOTFLigatures (gchars, inlen, script,
           "!pstf,blwf,vatu,blws,rphf,psts,haln", halant, reorder, gbase);
       while (olen != inlen)
       {
          debugChars ("GCHARS any=", gchars, olen);
          inlen = olen;
          olen = getOTFLigatures (gchars, inlen, script,
            "!pstf,blwf,vatu,blws,rphf,psts,haln", halant, reorder, gbase);
       }

       /* in fact, this alone should do all the junk job (above) */
       /*
        * From: http://www.microsoft.com/typography/otspec/indicot/reg.htm
        *
        * In scripts like Malayalam, the halant form of certain consonants
        * is represented by 'chillaksharams'. These can appear at any
        *  non-initial or final consonant location in a syllable. 
        *
        * - unfortunatelly it is very vague: 'scripts like Malayalam'
        *    gaspar
        */
       if (inlen > 1)
       {
         /* does it start with consonant + halant + ZWJ ? */
         bool firstHalanOK = scriptcode!=SD_MALAYALAM  /* bit vague */
           || (inlen > 2 && gchars[1] == halant && gchars[2] == SD_G_INDIC_ZWJ);

         if (firstHalanOK) /* a bit vague */
         {
            olen = getOTFLigatures (gchars, inlen, script,
              "haln", halant, reorder, gbase);
         }
         else
         {
            olen = getOTFLigatures (&gchars[1], inlen-1, script,
              "haln", halant, reorder, gbase?&gbase[1]:0);
            olen++;
         }
         inlen = olen;
       }

       /* insert back virama and search for feature "haln" */
       if (addVirama)
       {
         gchars[olen] = addVirama;
         inlen++;
         olen = getOTFLigatures (gchars, inlen, script, "haln",
              halant, reorder, gbase);
       }
       /* This is "haln" not applied in while loop because of a specific 
          check condition for SD_G_INDIC_ZWNJ in getOTFLigatures */
       else if (inlen > 1 && gchars[inlen-1] == SD_G_INDIC_ZWNJ)
       {
         if (scriptcode != SD_MALAYALAM) /* a bit vague */
         {
           olen = getOTFLigatures (gchars, inlen-1, script, "haln",
                halant, reorder, gbase);
           if (olen != inlen-1)
           {
              gchars[olen] = gchars[inlen-1];
              olen++;
           }
         }
       }
       /* insert back non repha after getOTFLigatures */
       if (rphfNone)
       {
         for (unsigned int i=olen-1; i>1; i--)
         {
            gchars[i] = gchars[i-2];
         }
         gchars[0] = rphfNone;
         gchars[1] = halant;
         olen += 2;
       }
          
       int endtype = getCharType (chars[liglen-1]);
       switch (endtype)
       {
       case SD_INDIC_LEFT_VOWEL:
         if (olen > 0)
         {
            SS_GlyphIndex g = gchars[olen-1];
            if (gbase)
            {
               unsigned int i;
               for (i=olen-2; i && !gbase[i]; i--);
               for (unsigned int j=olen-1; j > i; j--)
                  gchars[j]=gchars[j-1];
               gchars[i] = g;
            }
            else
            {
               for (unsigned int i=olen; i; i--)
                  gchars[i]=gchars[i-1];
               gchars[0] = g;
            }
         }
         break;
       case SD_INDIC_RIGHT_VOWEL:
       case SD_INDIC_TOP_VOWEL:
       case SD_INDIC_BOTTOM_VOWEL:
         if (olen > 0)
         {
            SS_GlyphIndex g = gchars[olen-1];
            if (gbase)
            {
               unsigned int i;
               for (i=olen-2; i && !gbase[i]; i--);
               for (unsigned int j=olen-1; j > i+1; j--)
                  gchars[j]=gchars[j-1];
               gchars[i+1] = g;
            }
         }
         break;
       case SD_INDIC_LEFT_RIGHT_VOWEL:
         if (olen > 0)
         {
           //FIRST CHARACTER OF THE DECOMPOSED VOWEL
           SS_GlyphIndex g1 = findGlyph (getLRVowelLeft(chars[liglen-1]));
           SS_GlyphIndex g2 = findGlyph (getLRVowelRight(chars[liglen-1]));
           if (g1 && g2)
           {
             if (gbase)
             {
               unsigned int i;
               for (i=olen-2; i && !gbase[i]; i--);
               for (unsigned int j=olen-1; j > i; j--)
                  gchars[j]=gchars[j-1];
               gchars[i] = g1;
             }
             else
             {
               for (unsigned int i=olen; i; i--)
                  gchars[i]=gchars[i-1];
               gchars[0] = g1;
             }
             gchars[olen] = g2;
             olen++;
           }
         }
       }
       if (rphfGlyph)
       {
          gchars[olen] = rphfGlyph;
          olen++;
       }
       /* add modifiers back */
       for (unsigned int i=mstart; i<mend; i++)
       {
          gchars[olen] = gchars[i];
          olen++;
          liglen++;
       }
       inlen = olen;
       olen = getOTFLigatures (gchars, inlen, script, "blws",
            halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS blws=", gchars, olen);
          inlen = olen;
       }
       olen = getOTFLigatures (gchars, inlen, script, "abvs",
            halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS abvs=", gchars, olen);
          inlen = olen;
       }
       olen = getOTFLigatures (gchars, inlen, script, "psts",
            halant, reorder, gbase);
       if (olen != inlen)
       {
          debugChars ("GCHARS psts=", gchars, olen);
          inlen = olen;
       }

       if (olen < liglen && olen > 0)
       {
          for (unsigned int i=0; i<olen; i++)
          {
            out->append (gchars[i]);
          }
       }
       else
       {
          decoded = false;
       }
     }
     if (decoded)
     {
       SString chc ((char*)out->array(), out->size() * sizeof (SS_GlyphIndex));
       char2Glyphs.put (key, chc);
#ifdef DEBUG_LIGATURE
       fprintf (stderr, "SFontTTF.cpp: Found OTF ligature:%s[%04X] %u -> %u: ", 
          script, (lig & 0xffff), liglen, out->size());
       for (unsigned int i=0; i<liglen; i++)
       {
         fprintf (stderr, " %X", chars[i]);
       }
       fprintf (stderr, "\n");
#endif
       delete chars;
       delete gchars;
       if (gbase) delete gbase;
       return true;
     }
     out->clear();
     delete chars;
     delete gchars;
     if (gbase) delete gbase;
  }

  /**
   * When using external maps we are using the same map for all
   * tables.
   */
  if (fontencoding.size()!=0 &&  charEncoder.isOK() && !charEncoder.isUMap())
  {
     /* max 3 */
     SV_UCS4 ucs4; ucs4.append (in); SV_UCS4 decd;
     SUniMap umap = charEncoder;
     unsigned int lifted = umap.lift (ucs4, 0, false, &decd);
     if (lifted == 0)
     {
        /* try straight - font has to have ascii mapping */
        SS_GlyphIndex gi = (in>=0x80) ? 0 :  findGlyph (in);
        if (gi)
        {
          out->append (gi);
          SString chc ((char*) &gi,  sizeof (SS_GlyphIndex));
          char2Glyphs.put (key, chc);
          return true;
        }
        SString chc; char2Glyphs.put (key, chc);
        return false;
     }
     for (unsigned int i=0; i<decd.size(); i++)
     {
        SS_GlyphIndex gi = findGlyph (decd[i]);
        if (gi == 0)
        {
          out->clear ();
          SString chc; char2Glyphs.put (key, chc);
          return false;
        }
        out->append (gi);
     }
     SString chc ((char*) out->array(), out->size() * sizeof (SS_GlyphIndex));
     char2Glyphs.put (key, chc);
     return true;
  }

  /* as I see there is no way to define multiple tables
   * for now so we just hardcode first one in reality we should
   * go through  0..num_tables
   */
  SUniMap umap = charEncoder;
  if (!umap.isOK())
  {
    SString chc; char2Glyphs.put (key, chc);
    return false;
  }
  /* get the encoder for this table. */
  SS_UCS2 ucs2 = umap.encode (in);
  if (ucs2==0)
  {
    SString chc; char2Glyphs.put (key, chc);
    return false;
  }
  SS_GlyphIndex o = findGlyph ((SS_UCS4) ucs2);
  if (o ==0)
  {
    SString chc; char2Glyphs.put (key, chc);
    return false;
  }
  out->append (o);
  SString chc ((char*)&o, sizeof (SS_GlyphIndex));
  char2Glyphs.put (key, chc);
  return true;
}

/**
 * Get pango ligatures.
 * @param ino is the input-output buffer
 * @param len is the input length
 * @param gbase contains true at base ligature.
 * @return output length
 */
unsigned int
SFontTTF::getOTFLigatures (SS_GlyphIndex* ino, unsigned int len,
  const char* script, const char* feature, SS_GlyphIndex halant, 
  SS_GlyphIndex reord, bool* base) 
{
  unsigned int i;
  for (i=0; i<len; i++)
  {
    if (ino[i] == 0) return len;
  }
  if (len == 1) return 1;
  /* collect all ligatures in one loop, starting from big ones. */
  unsigned int curin=0;
  unsigned int curout=0;
  /* moved this outside of the loop */
  bool needreorder = feature!=0 
    && (strcmp (feature, "vatu") ==0
        || strcmp (feature, "blwf")==0 || strcmp (feature, "pstf")==0
        || strcmp (feature, "blws")==0 || strcmp (feature, "psts")==0);
  while (curin < len)
  {
    SS_GlyphIndex lig;
    unsigned int nind = 0;
    bool fullglyph = true;
    bool reorder = false;
    /* with these features we always need to reorder stuff */
    if (needreorder) { 
      /* we have at least 3 characters to reorder */
      if (len-curin >= 3 && ino[curin+1] == halant)
      {
        SS_GlyphIndex tmp[3];
        tmp[0] = ino[curin];
        tmp[1] = ino[curin+2];
        tmp[2] = ino[curin+1];
        nind = getOTFLigature (script, feature, tmp, 3, &lig);
      }
      /* we have at least 2 characters to reorder */
      else if (len-curin >= 2 && ino[curin] == halant)
      {
        SS_GlyphIndex tmp[2];
        tmp[0] = ino[curin+1];
        tmp[1] = ino[curin];
        nind = getOTFLigature (script, feature, tmp, 2, &lig);
        fullglyph = false;
        /* set reorder if resulting/origianl glyphs is a reorder glyph */
        if (reord && tmp[0] == reord) reorder = true;
        /* reorder shows that this should go to zeor position */
      }
      else
      {
        /* dont reorder - just apply feature. nothing will happen. */
        nind = getOTFLigature (script, feature, &ino[curin], len-curin, &lig);
      }
    }
    else /* ! needreorder */
    {
      /* just apply feature.  */
      nind = getOTFLigature (script, feature, &ino[curin], len-curin, &lig);
    }
    /* nind is the index of the ligature found */
    bool isok = (nind != 0);

    /* i point to next character index. ino is not rewritten yet and lig may 
       contain a ligature that was found. */
    i=nind+curin;
    if (isok)
    {
      /* dont worry i can not be zero nind!=0 checks it*/
      /* SD_G_INDIC_ZWNJ prevents half form when halant comes. */
      /* halant + ZWNJ */
      if (i+1==len && ino[i-1] == halant && ino[i] == SD_G_INDIC_ZWNJ)
      {
        isok = false;
      }
    }
    /* substitution can go ahead */
    if (isok)
    {
      ino[curout] = lig;
      curout++;
      curin = i;
      /* we need to update base output parameter */
      if (base) {
         unsigned int j, k;
         for (j=curin, k=curout; j<len; j++, k++) base[k]=base[j];
         if (!fullglyph)
         {
            base[curout-1] = false;
         }
      }
      /* we need to update base output parameter */
      if (reorder)
      {
         /* shift stuff up, and insert to zero pos, instead of curout */
         for (unsigned int j=curout; j; j--) ino[j]=ino[j-1];
         ino[0] = lig;
      }
    }
    else  /* not ok, continue loop to find ligatures from next position */
    {
      ino[curout] = ino[curin];
      curout++;
      curin++;
    }
  }
  return curout;
}

/**
 * Find one single glyph. 
 * @return the index or 0.
 * TODO: SFontTTF currently handles TTF_CMAP_FMT4 only. 
 *  This can encode 16 bit unicode only. Make it handle
 *   format 8  Mixed 16 bit and 32 bit coverage using Surrogates(U+D800-U+DFFF)
 *   format 10 Trimmed Array using Surrogates (U+D800-U+DFFF)
 *   format 12 Segmented Coverage using Surrogates (U+D800-U+DFFF)
 * I am not aware of any other format that does not used 
 * this MS surrogate hack.
 * (Guess why are these ugly surrogates in Unicode ? :)
 */
SS_GlyphIndex
SFontTTF::findGlyph (SS_UCS4 in)
{
  /* Support only BMP for now */
  if (in > 0xffff) return 0;
  if (in == 0x200D) return SD_G_INDIC_ZWJ;
  if (in == 0x200C) return SD_G_INDIC_ZWNJ;

  SS_UCS2 ucs2 = (SS_UCS2) in;
  TTF_CMAP* cmap_table = (TTF_CMAP*) tables[SS_TB_CMAP];
  int num_tables = ntohs(cmap_table->numberOfEncodingTables);
  /* Go for it directly */
  if (charEncoderTable != (unsigned int) num_tables)
  {
    TTF_CMAP_ENTRY* table_entry = 
             &(cmap_table->encodingTable[charEncoderTable]);

    int offset = ntohl(table_entry->offset);
    int format = ntohs(*((USHORT*)((BYTE*)cmap_table+offset)));
    SS_GlyphIndex gi=0;
    switch (format)
    {
    case 0:
      gi =  findGlyph0 ((TTF_CMAP_FMT0 *)((BYTE *)cmap_table + offset), ucs2);
      break;
    case 4:
      gi =  findGlyph4 ((TTF_CMAP_FMT4 *)((BYTE *)cmap_table + offset), ucs2);
      break;
    case 8: /* TODO */
    case 10:
    case 12:
    default:
      break;
    }
    return gi;
  }
  if (fontencoding.size()==0) return 0;
  /* go through all tables */
  for (int i=0; i < num_tables; i++)
  {
    TTF_CMAP_ENTRY* table_entry = &(cmap_table->encodingTable[i]);
    int offset = ntohl(table_entry->offset);
    int format = ntohs(*((USHORT*)((BYTE*)cmap_table+offset)));
    SS_GlyphIndex gi=0;
    switch (format)
    {
    case 0:/* TODO - SGC 8 bit*/
      gi =  findGlyph0 ((TTF_CMAP_FMT0 *)((BYTE *)cmap_table + offset), ucs2);
      break; 
    case 4:
      gi =  findGlyph4 ((TTF_CMAP_FMT4 *)((BYTE *)cmap_table + offset), ucs2);
      break;
    case 8: /* TODO */
    case 10:
    case 12:
    default:
      break;
    }
    if (gi) return gi;
  }
  return 0;
}

/**
 * Try to find Glyph in an encoding format 4 table 
 * @param encoding4 is the encoding4 tables 
 * @param ucs2 is the character to find.
 * @return the glyph index.
 */
static SS_GlyphIndex
findGlyph0 (TTF_CMAP_FMT0* encoding0, SS_UCS2 ucs2)
{
  if (ucs2==0||ucs2>255) return 0;
  /* 1 byte does not need byteorder */
  return (SS_GlyphIndex) encoding0->glyphIdArray[ucs2];
}

/**
 * Try to find Glyph in an encoding format 4 table 
 * @param encoding4 is the encoding4 tables 
 * @param ucs2 is the character to find.
 * @return the glyph index.
 */
static SS_GlyphIndex
findGlyph4 (TTF_CMAP_FMT4* encoding4, SS_UCS2 ucs2)
{
  /* Finally we found it. Maybe */
  int seg_c2 = ntohs(encoding4->segCountX2);
  SHORT cmap_n_segs = seg_c2 >> 1;
  BYTE* ptr = (BYTE *)encoding4 + 14;
  USHORT* cmap_seg_end = (USHORT *) ptr;
  USHORT* cmap_seg_start = (USHORT *) (ptr + seg_c2 + 2);
  SHORT* cmap_idDelta = (SHORT *) (ptr + (seg_c2 * 2 )+ 2);
  SHORT* cmap_idRangeOffset = (SHORT *) (ptr + (seg_c2 * 3) + 2);

  /* No choice. Go through the segments */
  for (int j=0; j < cmap_n_segs-1; j++)
  {
    USHORT start = ntohs(cmap_seg_start[j]);
    USHORT end   = ntohs(cmap_seg_end[j]);
    SHORT delta = ntohs(cmap_idDelta[j]);
    USHORT ro    = ntohs(cmap_idRangeOffset[j]);

    if (ucs2> end || ucs2 < start) continue;

    unsigned int ind = (ro/2) + (ucs2 - start);
    USHORT n = (delta != 0) ? ucs2 + delta 
     : (USHORT) ntohs(*(&cmap_idRangeOffset [j] + ind));

    /* We really found it.. */
    if (n==0) return 0;
    return (SS_GlyphIndex) n;
  }
  return 0;
}

/**
 * Draw a single glyph.
 * @pama canvas is the canvas to draw to
 * @param pen is the pen to draw with
 * @param m is the transformation matrix.
 * @param glyphindex is the local glyph index
 * @return nothing 
 */
void
SFontTTF::drawGlyph (SCanvas* canvas, const SS_Matrix2D& matrix, 
  SS_GlyphIndex glyphno)
{
  if (!isOK()) return ;
  if (glyphno == SD_G_INDIC_ZWJ) return;
  if (glyphno == SD_G_INDIC_ZWNJ) return;

  BYTE* gstart = (BYTE *) tables[SS_TB_GLYF];

  TTF_GLYF* gtable;
  int len =0;
  if (longOffsets)
  {
     ULONG* lloca = (ULONG *) tables[SS_TB_LOCA];
     unsigned int offs1 = ntohl (lloca[glyphno]);
     unsigned int offs2 = ntohl (lloca[glyphno+1]);
     gtable = (TTF_GLYF *) ((char*)gstart + offs1);
     len = offs2-offs1;
/* SGC
fprintf (stderr, "name=%*.*s glyphno=%u numglyphs=%d\n", 
   SSARGS(name), glyphno, 0);
fprintf (stderr, "lloca=%lx offs1=%u offs2=%u gtable=%lx\n",
      (unsigned long)lloca, offs1, offs2, (unsigned long) gtable);
*/
  }
  else
  {
     USHORT* sloca = (USHORT *) tables[SS_TB_LOCA];
     gtable = (TTF_GLYF *) (gstart + (ntohs (sloca[glyphno]) << 1));
     len = (ntohs (sloca[glyphno+1]) - ntohs (sloca[glyphno])) << 1;
  }
  if (len <= 0)
  {
    if (!broken && len < 0)
    {
      fprintf (stderr, "SFontTTF-2: non-existent glyph %u in %*.*s %d\n",
          (unsigned int) glyphno,
          SSARGS (name), (int) len) ;
      broken = true;
    }
    return;
  }
  TTF_GLYF* kludge = 0;
  if ((((unsigned long) gtable) & 1) != 0)
  {
    if (!broken)
    {
      fprintf  (stderr, "SFontTTF: fixing unaligned %*.*s.\n", SSARGS(name));
      broken = true;
    }
    kludge = new TTF_GLYF[len]; 
    CHECK_NEW (kludge);
    memcpy (kludge, gtable, len * sizeof (TTF_GLYF));
    gtable = kludge;
  }

  int ncontours = (int) ((short)ntohs (gtable->numberOfContours));
  if (ncontours <= 0)
  {
    BYTE *ptr = ((BYTE *) gtable + sizeof(TTF_GLYF));
    SHORT *sptr = (SHORT *) ptr;
    /* Go through the contours and modyfy the transformation matrix */
    USHORT flagbyte;
    do
    {
      SS_Matrix2D m;
      flagbyte = ntohs(*sptr); sptr ++;
      SS_GlyphIndex glyphindex = ntohs(*sptr); sptr ++;

      if (flagbyte & ARG_1_AND_2_ARE_WORDS)
      {
         /* we need to make it short as it can be negative */
         m.t0 = (double) ((SHORT)ntohs(*sptr)); sptr++;
         m.t1 = (double) ((SHORT)ntohs(*sptr)); sptr++;
      }
      else
      {
         char* bptr = (char *)sptr;
         m.t0 = (signed char)bptr[0];
         m.t1 = (signed char)bptr[1];
         sptr ++;
      }
      if (flagbyte & WE_HAVE_A_SCALE)
      {
         m.x0 = m.y1 = f2dot14(*sptr);
         sptr ++;
      }
      else if (flagbyte & WE_HAVE_AN_X_AND_Y_SCALE)
      {
        m.x0 = f2dot14(*sptr); sptr ++;
        m.y1 = f2dot14(*sptr); sptr ++;
      }
      else if (flagbyte & WE_HAVE_A_TWO_BY_TWO)
      {
        m.x0 = f2dot14(*sptr); sptr ++;
        m.y0 = f2dot14(*sptr); sptr ++;
        m.x1 = f2dot14(*sptr); sptr ++;
        m.y1 = f2dot14(*sptr); sptr ++;
      }
      SS_Matrix2D mm = matrix * m;
      /* recursively call itself */
      drawGlyph (canvas, mm, glyphindex);
    } while (flagbyte & MORE_COMPONENTS);
    if (kludge) delete kludge;
    return;
  }

  USHORT* contour_end_pt = (USHORT *) ((char *)gtable + sizeof(TTF_GLYF));

  int last_point = (int) ntohs (contour_end_pt[ncontours-1]);

  int n_inst = (int) ntohs (contour_end_pt[ncontours]);

  BYTE* ptr = ((BYTE *)contour_end_pt) + (ncontours << 1) + n_inst + 2;

  int j = 0; int k = 0;

  SBinVector<BYTE> flags;
  while (k <= last_point)
  {
    flags.append (ptr[j]);
    if (ptr[j] & REPEAT)
    {
       for (int k1=0; k1 < ptr[j+1]; k1++)
       {
            k++;
            flags.append (ptr[j]);
       }
       j++;
    }
    j++; k++;
  }
  
  SBinVector<SHORT> xrel;
  SBinVector<SHORT> xcoord;

  for (k=0; k <= last_point; k++)
  {
    /* Process xrel */
    if (flags[k] & XSHORT)
    {
      if (flags[k] & XSAME)
      {
        xrel.append (ptr[j]);
      }
      else
      {
        xrel.append (-ptr[j]);
      }
      j++;
    }
    else if (flags[k] & XSAME)
    {
      xrel.append (0);
    }
    else
    {
      xrel.append (ptr[j] * 256 + ptr[j+1]);
      j += 2;
    }
    /* Process x coordinate */
    if (k==0)
    {
      xcoord.append (xrel[k]);
    }
    else
    {
      xcoord.append (xrel[k] + xcoord[k-1]);
    }
  }

  SBinVector<SHORT> yrel;
  SBinVector<SHORT> ycoord;

  /* one more run fore yrel and ycoord */
  for (k=0; k <= last_point; k++)
  {
    if (flags[k] & YSHORT)
    {
      if (flags[k] & YSAME)
      {
        yrel.append (ptr[j]);
      }
      else
      {
         yrel.append (- ptr[j]);
      }
      j++;
    }
    else if (flags[k] & YSAME)
    {
      yrel.append (0);
    }
    else
    {
      yrel.append (ptr[j] * 256 + ptr[j+1]);
      j += 2;
    }
    if (k==0)
    {
       ycoord.append (yrel[k]);
    }
    else
    {
       ycoord.append (yrel[k] + ycoord[k-1]);
    }
  }

  bool first = true;

  j = 0;
  int i = 0;
  int contour_start = 0;
  while (i <= ntohs(contour_end_pt[ncontours-1]))
  {
    int contour_end = ntohs(contour_end_pt[j]);

    if (first)
    {
      moveto (canvas, matrix, xcoord[i], ycoord[i]);
      contour_start = i;
      first = false;
    }
    else if (flags[i] & ONOROFF)
    {
      lineto (canvas, matrix, xcoord[i], ycoord[i]);
    }
    else /* curvy */
    {
      int cs = i-1;
      int ce = 0;
      bool finished = false;
      int nguide = 0;
      while (!finished)
      {
        if (i == contour_end+1)
        {
          ce = contour_start;
          finished = true;
        }
        else if (flags[i] & ONOROFF)
        {
          ce = i;
          finished = true;
        }
        else
        {
           i++;
           nguide++;
        }
      }

      switch (nguide)
      {
      case 0:
        lineto (canvas, matrix, xcoord[ce], ycoord[ce]);
        break;

      case 1:
        cureveto (canvas, matrix, 
           (xcoord[cs]+2*xcoord[cs+1])/3, (ycoord[cs]+2*ycoord[cs+1])/3,
           (2*xcoord[cs+1]+xcoord[ce])/3, (2*ycoord[cs+1]+ycoord[ce])/3,
            xcoord[ce], ycoord[ce]);
        break;

      case 2: 
        cureveto (canvas, matrix, 
            (-xcoord[cs]+4*xcoord[cs+1])/3, (-ycoord[cs]+4*ycoord[cs+1])/3,
            (4*xcoord[cs+2]-xcoord[ce])/3, (4*ycoord[cs+2]-ycoord[ce])/3,
            xcoord[ce], ycoord[ce]);
        break;

      case 3: 
        cureveto (canvas, matrix, 
            (xcoord[cs]+2*xcoord[cs+1])/3, (ycoord[cs]+2*ycoord[cs+1])/3,
            (5*xcoord[cs+1]+xcoord[cs+2])/6, (5*ycoord[cs+1]+ycoord[cs+2])/6,
            (xcoord[cs+1]+xcoord[cs+2])/2, (ycoord[cs+1]+ycoord[cs+2])/2);
        cureveto (canvas, matrix,
            (xcoord[cs+1]+5*xcoord[cs+2])/6, (ycoord[cs+1]+5*ycoord[cs+2])/6,
            (5*xcoord[cs+2]+xcoord[cs+3])/6, (5*ycoord[cs+2]+ycoord[cs+3])/6,
            (xcoord[cs+3]+xcoord[cs+2])/2, (ycoord[cs+3]+ycoord[cs+2])/2);
        cureveto (canvas, matrix,
            (xcoord[cs+2]+5*xcoord[cs+3])/6, (ycoord[cs+2]+5*ycoord[cs+3])/6,
            (2*xcoord[cs+3]+xcoord[ce])/3, (2*ycoord[cs+3]+ycoord[ce])/3,
            xcoord[ce], ycoord[ce]);
        break;

      default:
        int k1 = cs + nguide;
        cureveto (canvas, matrix, 
            (xcoord[cs]+2*xcoord[cs+1])/3, (ycoord[cs]+2*ycoord[cs+1])/3,
            (5*xcoord[cs+1]+xcoord[cs+2])/6, (5*ycoord[cs+1]+ycoord[cs+2])/6,
            (xcoord[cs+1]+xcoord[cs+2])/2, (ycoord[cs+1]+ycoord[cs+2])/2);
        for (k = cs+2; k <= k1-1; k++)
        {
          cureveto (canvas, matrix, 
            (xcoord[k-1]+5*xcoord[k])/6, (ycoord[k-1]+5*ycoord[k])/6,
            (5*xcoord[k]+xcoord[k+1])/6, (5*ycoord[k]+ycoord[k+1])/6,
            (xcoord[k]+xcoord[k+1])/2, (ycoord[k]+ycoord[k+1])/2);
        }
        cureveto (canvas, matrix, 
            (xcoord[k1-1]+5*xcoord[k1])/6, (ycoord[k1-1]+5*ycoord[k1])/6,
            (2*xcoord[k1]+xcoord[ce])/3, (2*ycoord[k1]+ycoord[ce])/3,
            xcoord[ce], ycoord[ce]);
        break;
      } /* end switch (nguide) */
    }
    if (i >= contour_end)
    {
      first = true;
      i = contour_end + 1;
      canvas->closepath ();
      j++;
    }
    else
    {
      i++;
    }
  }
  if (kludge!=0) delete kludge;
  return;
}

/**
 * The moveto
 */
static void
moveto (SCanvas* canvas, const SS_Matrix2D &m,
  SHORT _x, SHORT _y)
{
  double x = m.x0 * double (_x) + m.y0 * double (_y) + m.t0;
  double y = m.x1 * double (_x) + m.y1 * double (_y) + m.t1;
  canvas->moveto (x, y);
//fprintf (stderr, "moveto: %g,%g\n", x, y);
}
/**
 * The lineto
 */
static void
lineto (SCanvas* canvas, const SS_Matrix2D &m,
  SHORT _x, SHORT _y)
{
  double x = m.x0 * double (_x) + m.y0 * double (_y) + m.t0;
  double y = m.x1 * double (_x) + m.y1 * double (_y) + m.t1;
  canvas->lineto (x, y);
//fprintf (stderr, "lineto: %g,%g\n", x, y);
}

/**
 * The curveto
 */
static void
cureveto (SCanvas* canvas, const SS_Matrix2D &m,
  SHORT _x0, SHORT _y0,
  SHORT _x1, SHORT _y1,
  SHORT _x2, SHORT _y2)
{
  double x0 = m.x0 * double (_x0) + m.y0 * double (_y0) + m.t0;
  double y0 = m.x1 * double (_x0) + m.y1 * double (_y0) + m.t1;

  double x1 = m.x0 * double (_x1) + m.y0 * double (_y1) + m.t0;
  double y1 = m.x1 * double (_x1) + m.y1 * double (_y1) + m.t1;

  double x2 = m.x0 * double (_x2) + m.y0 * double (_y2) + m.t0;
  double y2 = m.x1 * double (_x2) + m.y1 * double (_y2) + m.t1;
  canvas->curveto (x0, y0, x1, y1, x2, y2);
//fprintf (stderr, "curveto: %g,%g %g,%g %g,%g\n", x0, y0, x1, y1, x2, y2);
}

/**
 * Create double numbers between -2 .. 1.99994 
 * from a packed short.
 */
static double
f2dot14 (short x)
{
  short y = ntohs(x);
  return (y >> 14) + ((y & 0x3fff) / 16384.0);
}

static void
debugChars (const char* msg, const SS_GlyphIndex* gchars, unsigned int len)
{
#ifdef DEBUG_LIGATURE
  fprintf (stderr, "SFontTTF.cpp: %s", msg);
  for (unsigned int i=0; i<len; i++)
  {
    fprintf (stderr, " %04X", gchars[i]);
  }
  fprintf (stderr, "\n");
#endif
}
