/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include "stoolkit/sencoder/SB_Shape.h"
#include "stoolkit/SString.h"
#include "stoolkit/SStringVector.h"
#include "stoolkit/STextData.h"

#define SS_ESC 27

/**
 * @author: Gaspar Sinai <gsinai@yudit.org>
 * @version: 2000-05-12
 * This converter takes utf-8 text and converts characters
 * to Presentation Forms, similar way than Roman Czyborra's arabjoin
 * script.
 */
SB_Shape::SB_Shape() : SBEncoder ("\n,\r\n,\r"), shape ("shape")
{
  ok = shape.isOK();
}

SB_Shape::~SB_Shape ()
{
}

/**
 * return false if this generic encoder does not exist.
 */
bool
SB_Shape::isOK() const
{
  return ok;
}

/**
 * This is encoding a unicode string into a bytestring
 * @param input is a unicode string.
 */
const SString&
SB_Shape::encode (const SV_UCS4& input)
{
  return interface.encode (convert (input));
}

/**
 * Decode an input string into a unicode string.
 * @param input is a string.
 *   he output can be null, in this case a line is not
 *   read fully. If input size is zero output will be flushed.
 */
const SV_UCS4&
SB_Shape::decode (const SString& _input)
{
  SV_UCS4 decd = interface.decode (_input);
  return convert (decd);
}

/**
 * Decode an input string into a unicode string.
 * @param input is a string.
 *   he output can be null, in this case a line is not
 *   read fully. If input size is zero output will be flushed.
 */
const SV_UCS4&
SB_Shape::convert (const SV_UCS4& decd)
{
  SV_UCS4 before; bool wsbefore=false;
  SV_UCS4 after; bool wsnow=false;
  SV_UCS4 now; bool wsafter=false;
  SS_UCS4 fbnow=0; 
  SS_UCS4 fbafter=0; 

  for (unsigned int i=0; i<decd.size(); )
  {
    SV_UCS4 ret;
    unsigned int wsret = (decd[i] == 0x20 || decd[0] == 0x3000);
    unsigned int n = shape.lift (decd, i, false, &ret);
    /* the composition comes at the end  - if any */
    if (n>=i+1 && ret.size()==4)
    {
       /* shift them */
       before = now; now = after; after = ret;
       wsbefore = wsnow; wsnow = wsafter; wsafter = wsret;
       fbnow = fbafter; fbafter = (n == i+1) ? decd[i] : 0;
       if (now.size()==4)
       {
          ucs4string.append ( reShape(fbnow, before, now, after, wsbefore, wsafter));
       }
       i=n;
    }
    else
    {
      before = now; now = after; after.clear();
      wsbefore = wsnow; wsnow = wsafter; wsafter = false;
      fbnow = fbafter; fbafter = 0;
      if (now.size()==4)
      {
         ucs4string.append (reShape(fbnow, before, now, after, wsbefore, wsafter));
      }
      ucs4string.append (decd[i]);
      before.clear(); now.clear(); after.clear(); 
      i++;
    }
  }
  before = now; now = after; after.clear();
  wsbefore = wsnow; wsnow = wsafter; wsafter = false;
  fbnow = fbafter; fbafter = 0;
  if (now.size()==4)
  {
    ucs4string.append (reShape(fbnow, before, now, after, wsbefore, wsafter));
  }
  return ucs4string;
}

/**
 * Apply shaping algorithm.
 */
SS_UCS4
SB_Shape::reShape (SS_UCS4 fbnow, const SV_UCS4& _before, 
  const SV_UCS4& _now, const SV_UCS4& _after, 
  bool beforeTransp, bool afterTransp) const
{
  static SS_UCS4 initials[4] = {0x0, 0x0, 0x0, 0x0};

  const SS_UCS4* before = initials;
  const SS_UCS4* after = initials;
  const SS_UCS4* now = _now.array();

  if (_before.size() == 4) 
  {
    before = _before.array();
  }
  if (_after.size() == 4) 
  {
    after = _after.array();
  }

  /* Fuzzy eh? */
  if ((before[(unsigned int)SGlyph::INITIAL] || beforeTransp)
     && now[(unsigned int)SGlyph::MEDIAL]
     && (after[(unsigned int)SGlyph::FINAL] || afterTransp))
  {
    return now[(unsigned int)SGlyph::MEDIAL];
  }

  if (after[(unsigned int)SGlyph::FINAL]
    && now[(unsigned int)SGlyph::INITIAL])
  {
    return now[(unsigned int)SGlyph::INITIAL];
  }
  if (before[(unsigned int)SGlyph::INITIAL]
     && now[(unsigned int)SGlyph::FINAL])
  {
    return now[(unsigned int)SGlyph::FINAL];
  }
  if (now[(unsigned int)SGlyph::ISOLATED])
  {
    return now[(unsigned int)SGlyph::ISOLATED];
  }

  if (fbnow) return fbnow;

  /* FIXME: THIS IS SLIGHTLY DIFFERENT FROM WHAT WE REALLY NEED */
  /* IN FACT WE SHOULD RETURN THE ORIGINAL CHARACTERS. */
  /* Now we are in sync with STextData.cpp */

  /* But wait.. will this ever happen? */
  if (now[(unsigned int)SGlyph::INITIAL])
  {
    return now [(unsigned int)SGlyph::INITIAL];
  }
  if (now[(unsigned int)SGlyph::FINAL])
  {
    return now [(unsigned int)SGlyph::FINAL];
  }
  return now [(unsigned int)SGlyph::MEDIAL];
}


/**
 * These methods guess the line delimiters for the input
 * The one without arguments is giving the 'first approximation'
 * It returns an inclusive list of all possibilities.
 */
const SStringVector&
SB_Shape::delimiters ()
{
  return realDelimiters;
}

/**
 * These methods guess the line delimiters for the input
 * The one without arguments is giving the 'first approximation'
 * It returns an exact list
 */
const SStringVector&
SB_Shape::delimiters (const SString& sample)
{
  return sampleDelimiters;
}
