/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef STextData_h
#define STextData_h

#include "stoolkit/SVector.h"
#include "stoolkit/STypes.h"
#include "stoolkit/SString.h"
#include "stoolkit/STextIndex.h"

/**
 * Line breaking characters
 */
#define SS_LB_DOS "\r\n"
#define SS_LB_MAC "\r"
#define SS_LB_UNIX "\n"
#define SS_LB_LS "\342\200\250"
#define SS_LB_PS "\342\200\251"

class STextDataEvent
{
public:
  /* line per line events */
  STextDataEvent (void);
  STextDataEvent (const STextIndex& start, bool attribute=false);
  ~STextDataEvent ();

  void clear();
  void add (const STextDataEvent& evt);
  void setRemaining (const STextIndex& remain);

  bool          valid;
  bool          attribute; // only attribute of text has changed.

  STextIndex    start;
  /* a reversely calcualted value */
  STextIndex    remaining;

};

class STextDataListener
{
public:
  virtual void textChanged(void* src, const STextDataEvent& event) = 0;
};

#define SD_MAX_COMPOSE 100

class SGlyphShared
{
public:
  char    cluster;
  char    type;
  bool    lineend;
  bool    tab;
  bool    shaped;

  SS_UCS4 mcomposition;
  SS_UCS4 mirror;
  /* this array is multi-functional */
  // 1. composition. - shaped == false && cluster == 0
  // 2. shapes[4] + composition. - shaped == true
  // 3. memory-cluster + unicode-cluster - cluster > 0
  SV_UCS4 ucs4v;
};

class SGlyph : public SObject
{
public:
  enum SType {
       CTRL = 0,
       LF = '\n',
       CR = '\r',
       TAB = '\t',
       RLO = 0x202e, /* right-to-left override */
       LRO = 0x202d,/* leftto-right override */
       PDF = 0x202c,  /* pop directional format */
       PS = 0x2029, /* paragraph separator */
       LS = 0x2028 /* line separator */
  };

  enum SShape {
      ISOLATED=0,
      INITIAL=1,
      MEDIAL=2,
      FINAL=3,
  };

  enum SLineEnd {
      ELF=0,
      ECR,
      ECRLF,
      EPS,
      ELS
  };

  SGlyph (char type, const SV_UCS4 &decomp, SS_UCS4 comp, 
          SS_UCS4 mirror, bool shaped, unsigned int clusterindex); 

  SGlyph (const SGlyph& glyph);
  SGlyph (char type, SGlyphShared* _shared);

  SGlyph operator=(const SGlyph& glyph);
  virtual ~SGlyph ();
  virtual SObject*  clone() const;

  bool operator == (const SGlyph& g2) const;
  bool operator != (const SGlyph& g2) const;

  bool operator == (const SType& type) const;
  bool operator != (const SType& type) const;

  bool isWhiteSpace() const;
  bool isTransparent() const;
  bool isLineEnd() const;
  bool isTab() const;

  bool isCluster() const;
  bool isYuditLigature() const;
  bool isYuditComposition() const;
  bool isSpecial()  const;
  bool isMirrorable() const;
  unsigned int getType()  const;

  SString charKey() const;

  /* Work on characters */
  const SS_UCS4 getChar() const;
  const SS_UCS4 getMirroredChar() const;
  const SS_UCS4 getShapedChar() const;
  const SS_UCS4 getFirstChar () const;

  /* Work on decompositions */
  SV_UCS4 getChars() const; /* returns decomposed or precomposed  */

  unsigned int size() const;

  const SS_UCS4* getShapeArray() const; /* fast */
  const SS_UCS4* getDecompArray() const; /* fast */
  /* array operator works on memory represenation */
  SS_UCS4 operator[] (unsigned int index) const; /* fast */

  /* These should be constants */
  bool    underlined;
  bool    selected;
  bool    useComposition;
  bool    lr;
  char    currentShape;

private:
  SGlyphShared * shared;
};


typedef SVector<SGlyph> SGlyphLine;

void bidiToVisual (SGlyphLine* gl);
unsigned int split (const SV_UCS4& ucs4, SGlyphLine* gl, unsigned int from);

/**
 * This STextData has a notion about glyphs and compositions.
 */
class STextData
{
public:
  STextData (void);
  STextData (const SString& utf8);
  STextData operator = (const STextData & data);
  virtual ~STextData ();

  SString getText () const;
  SString getText (const STextIndex& index) const;
  SString getText (const STextIndex& begin, const STextIndex& end) const;
  STextIndex find (const SString& string);
  const SGlyph& glyphAt (const STextIndex& index) const;

  /* these, till fireEvent all work with events */
  void move (const STextIndex& index);

  /* moves the cursor to the end */
  void setText (const SString& utf8);

  /* insert moves the cursor */
  void insert (const SGlyph& glyph, STextIndex* ret);
  STextIndex insert (const SString& data);

  void insert (const STextData& data);

  /* remove moves the cursor */
  void remove (const STextIndex& index);

  /* These just change the text */
  void select (const STextIndex& index, bool is=true);
  void underline (const STextIndex& index, bool is=true);
  bool setLineType (const SString& str);

  void clear ();
  void fireEvent ();
  void clearEvent ();

  unsigned int  size() const;
  unsigned int  size(unsigned int line) const;

  STextIndex getTextIndex (int charOffset=0) const;
  STextIndex getTextIndex(const STextIndex& base, int charOffset=0) const;
  STextIndex getMaxTextIndex(const STextDataEvent& evt) const;
  void addTextDataListener (STextDataListener* listener);
  bool isProperLine (unsigned int line) const;
  bool isWhiteSpace (const STextIndex& index) const;
  bool isTransparent (const STextIndex& index) const;

private:

  int getNonTransparentLeft (SGlyphLine* gl, unsigned int index);
  int getNonTransparentRight (SGlyphLine* gl, unsigned int index);

  bool setLineType (SGlyph::SLineEnd type);
  bool setLineType (const SGlyph& glyph);
  void reShape (const STextIndex& index);
  void reShapeOne (const STextIndex& index);

  void reShapeSegment (SGlyphLine* gl);

  bool reShapeSegment (SGlyphLine* gl, const SGlyph* before,
         const SGlyph* after);

  char STextData::getShape(const SGlyph& gl,
     const SGlyph* before, const SGlyph* after);

  /* moves the cursor to the end */
  void setText (const SV_UCS4& ucs4);

  SString getText (unsigned int beginLine, unsigned int beginIndex,
   unsigned int endIndex) const;
  void setMaxLimits (STextDataEvent* evt, const STextIndex& index);
  STextIndex reorder (const STextIndex& index);

  /* These belong to the event being generated */
  STextDataEvent     event;
  STextIndex         textIndex;

  /* should be a linked list */
  SBinVector<SGlyphLine*>  lines;
  STextDataListener*     listener;
};

#endif /* STextData_h */
