#include <stdio.h>
#ifdef _AIX
# include <stdlib.h>
#endif
#include <malloc.h>
#include <string.h>
#include <sys/types.h>

#include <gtk/gtk.h>

#include "guiutils.h"

#include "pdialog.h"

/* Message icons. */
#include "images/icon_error_32x32.xpm"
#include "images/icon_info_32x32.xpm"
#include "images/icon_question_32x32.xpm"
#include "images/icon_warning_32x32.xpm"

/* Button icons. */
#include "images/icon_browse_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_help_20x20.xpm"


/* Callbacks. */
static void PDialogDestroyCB(GtkObject *object, gpointer data);
static gint PDialogCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void PDialogBrowseButtonCB(GtkWidget *widget, gpointer data);
static void PDialogButtonCB(GtkWidget *widget, gpointer data);

/* Private utils. */
static void PDialogSetIcon(pdialog_struct *d, u_int8_t **icon_data);
static pdialog_prompt_struct *PDialogPromptNew(
	const u_int8_t **icon_data,
	const char *label,
	const char *value,
	pdialog_struct *d
);
static pdialog_prompt_struct *PDialogPromptNewWithBrowse(
	const u_int8_t **icon_data,
	const char *label,
	const char *value,
	pdialog_struct *d,
	void *client_data,
        char *(*browse_cb)(void *, void *, int)
);
static void PDialogPromptDelete(pdialog_prompt_struct *p);

/* Public. */
int PDialogInit(void);
gbool PDialogIsQuery(void);
void PDialogBreakQuery(void);
void PDialogAddPrompt(
        const u_int8_t **icon_data,     /* Can be NULL. */
        const char *label,              /* Can be NULL. */
        const char *value               /* Can be NULL. */
);
void PDialogAddPromptWithBrowse(
        const u_int8_t **icon_data,     /* Can be NULL. */
        const char *label,              /* Can be NULL. */
        const char *value,              /* Can be NULL. */
        void *client_data,              /* Can be NULL. */
        char *(*browse_cb)(void *, void *, int) /* Can be NULL. */
);
void PDialogSetPromptValue(
        int prompt_num,
        const u_int8_t **icon_data,
        const char *label,
        const char *value
);
char *PDialogGetPromptValue(int prompt_num);
void PDialogDeleteAllPrompts(void);
char **PDialogGetResponse(
	const char *title,
        const char *message,
        const char *explaination,
        int icon_code,
        const char *submit_label,
        const char *cancel_label,
        unsigned int show_buttons,
        unsigned int default_button,
        int *nvalues
);
void PDialogMap(pdialog_struct *d);
void PDialogUnmap(pdialog_struct *d);
void PDialogShutdown(void);


#define PDIALOG_BTN_WIDTH	(100 + (2 * 3))
#define PDIALOG_BTN_HEIGHT	(30 + (2 * 3))

#define PDIALOG_ENTRY_MAX	256


static int response_code = PDIALOG_RESPONSE_NOT_AVAILABLE;
static int block_loop_level;
static char **response_val = NULL;
static int response_nvals = 0;
static pdialog_struct pdialog;


/*
 *	Destroy callback.
 */
static void PDialogDestroyCB(GtkObject *object, gpointer data)
{
	return;
}

/*
 *	Dialog close callback.
 */
static gint PDialogCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
        pdialog_struct *d = (pdialog_struct *)data;
        if((widget == NULL) ||
           (d == NULL)  
        )
            return(TRUE);

	response_code = PDIALOG_RESPONSE_CANCEL;
        gtk_main_quit();
	block_loop_level--;

	return(TRUE);
}

/*
 *	Dialog browse button callback.
 */
static void PDialogBrowseButtonCB(GtkWidget *widget, gpointer data)
{
	int i;
	GtkWidget *w;
	char *strptr;
	pdialog_prompt_struct *p;
        pdialog_struct *d = (pdialog_struct *)data;
        if((widget == NULL) ||
           (d == NULL)
        )
            return;

	/* Check which browse button was pressed. */
	for(i = 0; i < d->total_prompts; i++)
	{
	    p = d->prompt[i];
	    if(p == NULL)
		continue;

	    if(widget != p->browse_btn)
	        continue;

	    if(p->browse_cb != NULL)
	    {
		strptr = (p->browse_cb)(
		    d,			/* Prompt dialog. */
		    p->client_data,	/* Client data. */
		    i			/* Prompt number. */
		);
		w = p->entry;
		if((w != NULL) && (strptr != NULL))
		{
		    gtk_entry_set_text(GTK_ENTRY(w), strptr);
		}
	    }
	    break;
	}

	return;
}

/*
 *      Dialog prompt entry enter callback.
 */
static void PDialogEntryEnterCB(GtkWidget *widget, gpointer data)
{
        pdialog_struct *d = (pdialog_struct *)data;
        if((widget == NULL) ||
           (d == NULL)
        )
            return;

	/* Call button callback and pass submit button as the widget. */
	PDialogButtonCB(d->submit_btn, d);

	return;
}

/*
 *	Dialog button (not browse buttons) callback.
 */
static void PDialogButtonCB(GtkWidget *widget, gpointer data)
{
	pdialog_struct *d = (pdialog_struct *)data;
	if((widget == NULL) ||
           (d == NULL)
	)
	    return;

	if(widget == d->submit_btn)
	{
	    int i;


	    response_code = PDIALOG_RESPONSE_SUBMIT;

	    /* Clear responses values list if any. */
	    for(i = 0; i < response_nvals; i++)
		free(response_val[i]);
	    free(response_val);
	    response_val = NULL;
	    response_nvals = 0;

	    /* Fetch values from each prompt and copy them to local
             * global response_val. Number of response values is
	     * the number of prompts on the dialog.
             */
	    response_nvals = d->total_prompts;
	    response_val = (char **)malloc(response_nvals * sizeof(char *));
	    if(response_val == NULL)
	    {
		response_nvals = 0;
	    }
	    else
	    {
		pdialog_prompt_struct *p;

		/* Fetch values for each response value, thus going
		 * through each prompt.
		 */
		for(i = 0; i < response_nvals; i++)
		{
		    p = d->prompt[i];
		    if(p == NULL)
		    {
			response_val[i] = strdup("");
		    }
		    else
		    {
			char *strptr = NULL;
			GtkWidget *w = p->entry;

			if(w != NULL)
			    strptr = gtk_entry_get_text(GTK_ENTRY(w));
			if(strptr == NULL)
			    strptr = "";

			response_val[i] = strdup(strptr);
		    }
		}
	    }

	}
        else if(widget == d->cancel_btn)
	{
            response_code = PDIALOG_RESPONSE_CANCEL;
	}
        else if(widget == d->help_btn)
	{
	    response_code = PDIALOG_RESPONSE_HELP;
	    return;	/* Return, do not break out of block loop. */
	}

	/* Need to break out of the blocked loop. */
	gtk_main_quit();
	block_loop_level--;

	return;
}


/*
 *	Updates the icon for the dialog. Updates the member icon_pm
 *	as needed.
 */
static void PDialogSetIcon(pdialog_struct *d, u_int8_t **icon_data)
{
        GdkGC *gc;
        GtkWidget *w, *window, *pixmap;
        GdkPixmap *gdk_pixmap;
        GdkBitmap *mask;
        GtkStyle *style;
        gint width, height;


        if((d == NULL) ||
           (icon_data == NULL)
        )
            return;

        w = d->icon_fixed;
        if(w == NULL)
            return;

        window = d->toplevel;
        if(window == NULL)
            return;

        style = gtk_widget_get_default_style();
        gc = style->black_gc;


        /* Create new pixmap. */
        gdk_pixmap = gdk_pixmap_create_from_xpm_d(
            window->window,
            &mask,
            &style->bg[GTK_STATE_NORMAL],
            (gchar **)icon_data
        );
        pixmap = gtk_pixmap_new(gdk_pixmap, mask);
        gdk_window_get_size((GdkWindow *)gdk_pixmap, &width, &height);

        /* Adjust size of fixed widget to fit pixmap. */
        gtk_widget_set_usize(w, width, height);

        /* Put pixmap into fixed widget. */
        gtk_fixed_put(GTK_FIXED(w), pixmap, 0, 0);
	gtk_widget_shape_combine_mask(w, mask, 0, 0);
        gtk_widget_show(pixmap);

	gdk_pixmap_unref(gdk_pixmap);
	if(mask != NULL)
	    gdk_bitmap_unref(mask);


        /* Destroy the previous pixmap. */
        if(d->icon_pm != NULL)
            gtk_widget_destroy(d->icon_pm);

	/* Record new pixmap. */
        d->icon_pm = pixmap;


        /* Set WM icon for toplevel. */
        GUISetWMIcon(window->window, icon_data);                    

        return;
}


/*
 *	Allocates and creates a new prompt structure, but does not
 *	add it to the prompt dialog.
 */
static pdialog_prompt_struct *PDialogPromptNew(
	const u_int8_t **icon_data,
	const char *label,
        const char *value,
        pdialog_struct *d
)
{
	return(
	    PDialogPromptNewWithBrowse(
		icon_data, label, value, d, NULL, NULL
	    )
	);
}

/*
 *	Allocates and creates a new prompt structure, but does not 
 *      add it to the prompt dialog.
 */
static pdialog_prompt_struct *PDialogPromptNewWithBrowse(
	const u_int8_t **icon_data,
        const char *label,
        const char *value,
	pdialog_struct *d,
	void *client_data,
        char *(*browse_cb)(void *, void *, int)
)
{
        GtkWidget *w, *parent;
	GdkWindow *window = NULL;
        pdialog_prompt_struct *p = (pdialog_prompt_struct *)calloc(
            1,
            sizeof(pdialog_prompt_struct)
        );
        if(p == NULL)
            return(p);

	/* Get window of dialog toplevel. */
	w = d->toplevel;
	if(w != NULL)
	{
	    if(!GTK_WIDGET_NO_WINDOW(w))
		window = w->window;
	}

	/* Create main toplevel table, give the number of columns
	 * depending on which widgets have been specified to be created.
	 */
	w = gtk_table_new(
	    1,
	    MAX(
                ((icon_data == NULL) ? 0 : 1) +
		((label == NULL) ? 0 : 1) +
		(1) +
                ((browse_cb == NULL) ? 0 : 1),
		1
	    ),
	    FALSE
	);
        p->toplevel = w;
	/* Do not show toplevel just yet. */
        parent = w;

	/* Create icon? */
	if(icon_data != NULL)
	{
            int attach_x = 0;
            GdkGC *gc;
            GtkWidget *pixmap;
            GdkPixmap *gdk_pixmap;
            GdkBitmap *mask;
            GtkStyle *style;
            gint width, height;

	    /* Create fixed widget. */
	    w = gtk_fixed_new();
	    p->icon_fixed = w;
            gtk_table_attach(GTK_TABLE(parent), w,
                attach_x, attach_x + 1,
                0, 1,
                0,
                0,
                2, 0
            );
            gtk_widget_show(w);

	    /* Get style from dialog toplevel. */
	    if(d->toplevel == NULL)
		style = gtk_widget_get_default_style();
	    else
		style = gtk_widget_get_style(d->toplevel);

	    if(style != NULL)
		gc = style->black_gc;

	    /* Create the pixmap. */
	    gdk_pixmap = gdk_pixmap_create_from_xpm_d(
		window,
		&mask,
		((style == NULL) ? NULL : &style->bg[GTK_STATE_NORMAL]),
		(gchar **)icon_data  
	    );
	    pixmap = gtk_pixmap_new(gdk_pixmap, mask);
	    gdk_window_get_size((GdkWindow *)gdk_pixmap, &width, &height);

	    /* Adjust size of fixed widget to fit pixmap. */
	    gtk_widget_set_usize(w, width, height);

	    /* Put pixmap into fixed widget. */
	    gtk_fixed_put(GTK_FIXED(w), pixmap, 0, 0);
            gtk_widget_shape_combine_mask(w, mask, 0, 0);
            gtk_widget_show(pixmap);

            gdk_pixmap_unref(gdk_pixmap);
            if(mask != NULL)
                gdk_bitmap_unref(mask);

	    /* Record new pixmap. */
	    p->icon_pm = pixmap;
	}

	/* Create label? */
        if(label != NULL)
        {
	    int attach_x = ((icon_data == NULL) ? 0 : 1);

            w = gtk_label_new(label);
            p->label = w; 
            gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
            gtk_table_attach(GTK_TABLE(parent), w,
                attach_x, attach_x + 1,
                0, 1,
                0,
                0,
                2, 0
            );
            gtk_widget_show(w);
        }
 
	/* Text entry. */
	if(1)
	{
	    int attach_x = ((icon_data == NULL) ? 0 : 1) +
                           ((label == NULL) ? 0 : 1);

            w = gtk_entry_new_with_max_length(PDIALOG_ENTRY_MAX);
            p->entry = w;
	    if(value != NULL)
	        gtk_entry_set_text(GTK_ENTRY(w), value);
            gtk_signal_connect(
                GTK_OBJECT(w),
                "activate",
                GTK_SIGNAL_FUNC(PDialogEntryEnterCB),
                d		/* Pass the dialog structure. */
            );
            gtk_table_attach(GTK_TABLE(parent), w,
                attach_x, attach_x + 1,
                0, 1,
                GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                0,
                2, 0
            );
            gtk_widget_show(w);
	}

	/* Create browse button? */
	if(browse_cb != NULL)
	{
            int attach_x = ((icon_data == NULL) ? 0 : 1) +
                           ((label == NULL) ? 0 : 1) +
			   (1);

	    /* Set browse callback function and client data. */
	    p->client_data = client_data;
	    p->browse_cb = browse_cb;

	    /* Create browse button. */
	    w = GUIButtonPixmap((u_int8_t **)icon_browse_20x20_xpm);
	    p->browse_btn = w;
            gtk_signal_connect(
                GTK_OBJECT(w), "clicked",
                GTK_SIGNAL_FUNC(PDialogBrowseButtonCB),
                d		/* Pass the dialog structure. */
            );
            gtk_table_attach(GTK_TABLE(parent), w,
                attach_x, attach_x + 1,
                0, 1,
                0,
                0,
                2, 0
            );
            gtk_widget_show(w);
	}

        return(p); 
}

/*
 *	Deletes all widgets in the prompt structure and deallocates
 *	the prompt structure itself.
 */
static void PDialogPromptDelete(pdialog_prompt_struct *p)
{
	GtkWidget **w;


	if(p == NULL)
	    return;

#define DO_DESTROY_WIDGET	\
{ \
 if((*w) != NULL) \
 { \
  gtk_widget_destroy(*w); \
  (*w) = NULL; \
 } \
}

        w = &p->scale;
        DO_DESTROY_WIDGET

        w = &p->spin;
        DO_DESTROY_WIDGET

        w = &p->browse_btn;
	DO_DESTROY_WIDGET

        w = &p->entry;
        DO_DESTROY_WIDGET  

        w = &p->label;
        DO_DESTROY_WIDGET  

        w = &p->icon_pm;
        DO_DESTROY_WIDGET

        w = &p->icon_fixed;
        DO_DESTROY_WIDGET

        w = &p->toplevel;
        DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET

	free(p);

	return;
}


/*
 *	Initializes the prompt dialog.
 */
int PDialogInit(void)
{
	void *label_rtn;
	GtkWidget *w, *parent, *parent2, *parent3;
	GdkWindow *window;
        pdialog_struct *d = &pdialog;


        /* Reset globals. */
	response_code = PDIALOG_RESPONSE_NOT_AVAILABLE;
	block_loop_level = 0;
	response_val = NULL;
	response_nvals = 0;


	/* Reset values. */
	memset(d, 0x00, sizeof(pdialog_struct));

	d->initialized = TRUE;
	d->map_state = FALSE;
	d->last_icon_code = PDIALOG_ICON_QUESTION;

        /* Toplevel. */
        w = gtk_window_new(GTK_WINDOW_DIALOG);
        d->toplevel = w;
        gtk_widget_realize(w);
        window = w->window;
        if(window != NULL)
        {
            gdk_window_set_decorations(
                window,
                GDK_DECOR_TITLE | GDK_DECOR_MENU | GDK_DECOR_MINIMIZE
            );
            gdk_window_set_functions(
                window,
                GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
            );
        }
        gtk_signal_connect(
            GTK_OBJECT(w),
            "destroy",   
            GTK_SIGNAL_FUNC(PDialogDestroyCB),
            (gpointer)d
        );
	gtk_signal_connect(
            GTK_OBJECT(w), 
            "delete_event",
            GTK_SIGNAL_FUNC(PDialogCloseCB),
            (gpointer)d
        );
        gtk_container_set_border_width(GTK_CONTAINER(w), 0);
        parent = w;


	/* Vbox. */
	w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


	/* Hbox for holding the vboxes of icon and prompts. */
	w = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 5);
	gtk_widget_show(w);
	parent2 = w;

	/* Icon vbox and icon. */
	w = gtk_vbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_widget_show(w);
        parent3 = w;

        w = gtk_fixed_new();
        d->icon_fixed = w;
        d->icon_pm = NULL;
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
        gtk_widget_realize(w);
        gtk_widget_show(w);


	/* Vbox for prompt widgets. */
	w = gtk_vbox_new(FALSE, 0);
	d->main_vbox = w;
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
	gtk_widget_show(w);


	/* Separator. */
	w = gtk_hseparator_new();
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


	/* Hbox for buttons. */
	w = gtk_hbox_new(TRUE, 0);
        gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 5);
        gtk_widget_show(w);
        parent2 = w;

	/* Submit button. */
        w = GUIButtonPixmapLabelH(
	    (u_int8_t **)icon_ok_20x20_xpm, "Submit", &label_rtn
	);
	d->submit_btn = w;
	d->submit_btn_label = GTK_WIDGET(label_rtn);
        gtk_widget_set_usize(w, PDIALOG_BTN_WIDTH, PDIALOG_BTN_HEIGHT);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(PDialogButtonCB),
            d
        );

	/* Cancel button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_cancel_20x20_xpm, "Cancel", &label_rtn
        );
        d->cancel_btn = w;
        d->cancel_btn_label = GTK_WIDGET(label_rtn);
        gtk_widget_set_usize(w, PDIALOG_BTN_WIDTH, PDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(PDialogButtonCB),
            d
        );

        /* Help button. */
        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_help_20x20_xpm, "Help", NULL
        );
        d->help_btn = w;
        gtk_widget_set_usize(w, PDIALOG_BTN_WIDTH, PDIALOG_BTN_HEIGHT);
        GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 5);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(PDialogButtonCB),
            d
        );

	/* Set default icon. */
	PDialogSetIcon(d, (u_int8_t **)icon_question_32x32_xpm);

	return(0);
}

/*
 *      Returns TRUE if currently blocking for query.
 */
gbool PDialogIsQuery(void)
{
        if(block_loop_level > 0)
            return(TRUE);
        else
            return(FALSE);
}

/*
 *      Ends query if any and returns a not available response.
 */
void PDialogBreakQuery(void)
{
        response_code = PDIALOG_RESPONSE_NOT_AVAILABLE;

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;
  
        return;
}

/*
 *	Adds a prompt to the dialog.
 */
void PDialogAddPrompt(
	const u_int8_t **icon_data,
        const char *label,
        const char *value
)
{
        GtkWidget *w, *parent;
        int n;
        pdialog_prompt_struct *p;
        pdialog_struct *d = &pdialog;


        p = PDialogPromptNew(
            icon_data, label, value, d
        );
        if(p == NULL)
            return;

        if(d->total_prompts < 0)
            d->total_prompts = 0;

        for(n = 0; n < d->total_prompts; n++)
        {
            if(d->prompt[n] == NULL) 
                break; 
        }
        if(n < d->total_prompts)
        {
            d->prompt[n] = p;
        } 
        else
        {
            n = d->total_prompts;
            d->total_prompts++;
            d->prompt = (pdialog_prompt_struct **)realloc(
                d->prompt,
                d->total_prompts * sizeof(pdialog_prompt_struct *)
            );
            if(d->prompt == NULL)
            {
                PDialogPromptDelete(p);
                d->total_prompts = 0;
                return;
            }

            d->prompt[n] = p;
        }

        parent = d->main_vbox;
        w = p->toplevel;
        if((parent != NULL) && (w != NULL))
        {
            gtk_box_pack_start(GTK_BOX(parent), w, TRUE, FALSE, 2);
            gtk_widget_show(w);
        }

	return;
}

/*
 *	Adds a prompt with browse to the dialog.
 */
void PDialogAddPromptWithBrowse(
	const u_int8_t **icon_data,
        const char *label,
        const char *value,
        void *client_data,
        char *(*browse_cb)(void *, void *, int)
)
{
	GtkWidget *w, *parent;
	int n;
	pdialog_prompt_struct *p;
	pdialog_struct *d = &pdialog;


	p = PDialogPromptNewWithBrowse(
	    icon_data, label, value, d, client_data, browse_cb
	);
	if(p == NULL)
	    return;

	if(d->total_prompts < 0)
	    d->total_prompts = 0;

	for(n = 0; n < d->total_prompts; n++)
	{
	    if(d->prompt[n] == NULL)
		break;
	}
	if(n < d->total_prompts)
	{
	    d->prompt[n] = p;
	}
	else
	{
	    n = d->total_prompts;
	    d->total_prompts++;
	    d->prompt = (pdialog_prompt_struct **)realloc(
		d->prompt,
		d->total_prompts * sizeof(pdialog_prompt_struct *)
	    );
	    if(d->prompt == NULL)
	    {
		PDialogPromptDelete(p);
		d->total_prompts = 0;
		return;
	    }

	    d->prompt[n] = p;
	}

	parent = d->main_vbox;
	w = p->toplevel;
	if((parent != NULL) && (w != NULL))
	{
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, FALSE, 2);
	    gtk_widget_show(w);
	}

	return;
}

/*
 *	Changes the value in the prompt's text or spin widget, changes
 *	the prompt's icon and changes the label. If any input is NULL
 *	then that value will be left unchanged.
 *
 *	If prompt_num is invalid (ie unallocated), then no operation
 *	will be performed.
 */
void PDialogSetPromptValue(
        int prompt_num,
        const u_int8_t **icon_data,
        const char *label,
        const char *value 
)
{
        GtkWidget *w, *parent;
	GdkWindow *window = NULL;
        pdialog_prompt_struct *p;
        pdialog_struct *d = &pdialog;
            

	/* Prompt number valid on prompt dialog? */
	if((prompt_num < 0) || (prompt_num >= d->total_prompts))
	    return;
	else
	    p = d->prompt[prompt_num];

	if(p == NULL)
	    return;

        /* Get window of dialog toplevel. */
        w = d->toplevel;
        if(w != NULL)
        {   
            if(!GTK_WIDGET_NO_WINDOW(w))
                window = w->window;
        }

	/* Change icon? */
	parent = p->icon_fixed;
	if((icon_data != NULL) && (parent != NULL))
	{
            GtkWidget *pixmap;
            GdkPixmap *gdk_pixmap;
            GdkBitmap *mask;
            GtkStyle *style;
            gint width, height;


            /* Get style from dialog toplevel. */
            if(d->toplevel == NULL)
                style = gtk_widget_get_default_style();
            else
                style = gtk_widget_get_style(d->toplevel);

            /* Create the pixmap. */
            gdk_pixmap = gdk_pixmap_create_from_xpm_d(
                window,
                &mask,
                ((style == NULL) ? NULL : &style->bg[GTK_STATE_NORMAL]),
                (gchar **)icon_data
            );
            pixmap = gtk_pixmap_new(gdk_pixmap, mask);
            gdk_window_get_size((GdkWindow *)gdk_pixmap, &width, &height);

            /* Adjust size of fixed widget to fit pixmap. */
            gtk_widget_set_usize(parent, width, height);

            /* Put pixmap into fixed widget. */
            gtk_fixed_put(GTK_FIXED(parent), pixmap, 0, 0);
            gtk_widget_shape_combine_mask(parent, mask, 0, 0);
            gtk_widget_show(pixmap);

	    gdk_pixmap_unref(gdk_pixmap);
	    if(mask != NULL)
		gdk_bitmap_unref(mask);


	    /* Destroy the previous pixmap. */
	    if(p->icon_pm != NULL)
		gtk_widget_destroy(p->icon_pm);

	    /* Record new icon. */
	    p->icon_pm = pixmap;
	}

	/* Change label? */
	w = p->label;
	if((label != NULL) && (w != NULL))
	{
	    gtk_label_set_text(GTK_LABEL(w), label);
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	}

	/* Change entry text? */
	w = p->entry;
	if((value != NULL) && (w != NULL))
	{
	    gtk_entry_set_text(GTK_ENTRY(w), value);
	    gtk_entry_set_position(GTK_ENTRY(w), 0);
	}

	return;
}

/*
 *	Fetches the value of the specified prompt as a string.
 *	Can return NULL on error or if the prompt does not have
 *	a text entry. The returned string must not be
 *	free()'ed by the calling function.
 */
char *PDialogGetPromptValue(
        int prompt_num
)
{
        GtkWidget *w;
        pdialog_prompt_struct *p;
        pdialog_struct *d = &pdialog;


        /* Prompt number valid on prompt dialog? */
        if((prompt_num < 0) || (prompt_num >= d->total_prompts))
            return(NULL);
        else
            p = d->prompt[prompt_num];

        if(p == NULL)
            return(NULL);

	w = p->entry;
	if(w != NULL)
	{
	    /* Fetch value from entry. */
	    return(gtk_entry_get_text(GTK_ENTRY(w)));
	}

	return(NULL);
}


/*
 *	Destroys all prompts on prompt dialog.
 */
void PDialogDeleteAllPrompts(void)
{
	int i;
        pdialog_struct *d = &pdialog;


	for(i = 0; i < d->total_prompts; i++)
	    PDialogPromptDelete(d->prompt[i]);
	free(d->prompt);
	d->prompt = NULL;
	d->total_prompts = 0;

	return;
}


/*
 *	Block input and wait for a response.
 *
 *	Returns an array of string values from the given prompts
 *	which must not be free'ed. If NULL is returned then it should
 *	be considered that the user clicked on cancel.
 *
 *	If any values are set NULL then that value will not be modified
 *	from since the last usage.
 */
char **PDialogGetResponse(
        const char *title,		/* Can be NULL. */
        const char *message,		/* Can be NULL. */
        const char *explaination,	/* Can be NULL. */
        int icon_code,			/* One of PDIALOG_ICON_*. */
        const char *submit_label,	/* Can be NULL. */
        const char *cancel_label,	/* Can be NULL. */
        unsigned int show_buttons,	/* Any of PDIALOG_FLAG_*. */
        unsigned int default_button,	/* One of PDIALOG_FLAG_*. */
        int *nvalues			/* Number of string values return. */
)
{
	int i;
	GtkWidget *w;
	pdialog_struct *d = &pdialog;


        /* Do not handle response if already waiting for a response,
         * return with a not available response code.
         */
        if(block_loop_level > 0)
	{
	    if(nvalues != NULL)
		(*nvalues) = 0;

            return(NULL);
	}

	/* Reset responses. */
	response_code = PDIALOG_RESPONSE_NOT_AVAILABLE;
	for(i = 0; i < response_nvals; i++)
	    free(response_val[i]);
	free(response_val);
	response_val = NULL;
	response_nvals = 0;


	/* Reset number of values return. */
	if(nvalues != NULL)
	    (*nvalues) = response_nvals;

	/* Is dialog initialized? */
        if(!d->initialized)
            return(response_val);

	/* Change title. */
	if(title != NULL)
	{
	    w = d->toplevel;
	    if(w != NULL)
		gtk_window_set_title(GTK_WINDOW(w), title);
	}

	/* Set message label text. */
	if(message != NULL)
	{

	}

	/* Update icon. */
	if(icon_code != d->last_icon_code)
	{
	    d->last_icon_code = icon_code;
	    switch(icon_code)
	    {
	      case PDIALOG_ICON_ERROR:
		PDialogSetIcon(d, (u_int8_t **)icon_error_32x32_xpm);
		break;

              case PDIALOG_ICON_QUESTION:
                PDialogSetIcon(d, (u_int8_t **)icon_question_32x32_xpm);
                break;

              case PDIALOG_ICON_WARNING:
                PDialogSetIcon(d, (u_int8_t **)icon_warning_32x32_xpm);
                break;

              default:
                PDialogSetIcon(d, (u_int8_t **)icon_info_32x32_xpm);
                break;
	    }
	}

	/* Change button labels. */
	if(submit_label != NULL)
	{
	    w = d->submit_btn_label;
	    if(w != NULL)
		gtk_label_set_text(GTK_LABEL(w), submit_label);
	}
        if(cancel_label != NULL)
        {
            w = d->cancel_btn_label;
            if(w != NULL)
                gtk_label_set_text(GTK_LABEL(w), cancel_label);
        }


	/* Show/hide buttons. */
#define DO_MAP_BUTTON	\
{ \
 if(w != NULL) \
  gtk_widget_show(w); \
}
#define DO_UNMAP_BUTTON	\
{ \
 if(w != NULL) \
  gtk_widget_hide(w); \
}
#define DO_DEFAULT_BUTTON	\
{ \
 if(w != NULL) \
 { \
  GTK_WIDGET_SET_FLAGS(w, GTK_HAS_DEFAULT); \
  GTK_WIDGET_SET_FLAGS(w, GTK_RECEIVES_DEFAULT); \
 } \
}
#define DO_UNDEFAULT_BUTTON	\
{ \
 if(w != NULL) \
 { \
  GTK_WIDGET_UNSET_FLAGS(w, GTK_HAS_DEFAULT); \
  GTK_WIDGET_UNSET_FLAGS(w, GTK_RECEIVES_DEFAULT); \
 } \
}       

	w = d->submit_btn;
	if(show_buttons & PDIALOG_BTNFLAG_SUBMIT)
	    DO_MAP_BUTTON
	else
	    DO_UNMAP_BUTTON
	if(default_button & PDIALOG_BTNFLAG_SUBMIT)
	    DO_DEFAULT_BUTTON
	else
	    DO_UNDEFAULT_BUTTON

	w = d->cancel_btn;
        if(show_buttons & PDIALOG_BTNFLAG_CANCEL)
            DO_MAP_BUTTON
	else
	    DO_UNMAP_BUTTON
        if(default_button & PDIALOG_BTNFLAG_CANCEL)
            DO_DEFAULT_BUTTON
        else
            DO_UNDEFAULT_BUTTON

	w = d->help_btn;
        if(show_buttons & PDIALOG_BTNFLAG_HELP)
            DO_MAP_BUTTON
        else
            DO_UNMAP_BUTTON
        if(default_button & PDIALOG_BTNFLAG_HELP)
            DO_DEFAULT_BUTTON
        else
            DO_UNDEFAULT_BUTTON

#undef DO_MAP_BUTTON
#undef DO_UNMAP_BUTTON
#undef DO_DEFAULT_BUTTON
#undef DO_UNDEFAULT_BUTTON

	/* Map dialog. */
	PDialogMap(d);

	/* Block GUI untill response. */
	block_loop_level++;
	gtk_main();

	/* Unmap dialog. */
	PDialogUnmap(d);

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;


	/* Update number of values return. */
	if(nvalues != NULL)
	    (*nvalues) = response_nvals;

	return(response_val);
}


/*
 *	Maps the prompt dialog.
 */
void PDialogMap(pdialog_struct *d)
{
        GtkWidget *w;


        if(d == NULL)
            return;

        if(!d->initialized)
            return;

        if(!d->map_state)
        {
            w = d->toplevel;
            if(w != NULL)
                gtk_widget_show(w);

            d->map_state = TRUE;
        }

        return;
}

/*
 *	Unmaps the prompt dialog.
 */
void PDialogUnmap(pdialog_struct *d)
{
	GtkWidget *w;


	if(d == NULL)
	    return;

	if(!d->initialized)
	    return;

	if(d->map_state)
	{
	    w = d->toplevel;
	    if(w != NULL)
		gtk_widget_hide(w);

	    d->map_state = FALSE;
	}

	return;
}

/*
 *	Shuts down the prompt dialog.
 */
void PDialogShutdown(void)
{
	int i;
	GtkWidget *w;
	pdialog_struct *d = &pdialog;


        /* Reset globals. */
        response_code = PDIALOG_RESPONSE_NOT_AVAILABLE;
        for(i = 0; i < response_nvals; i++)
            free(response_val[i]);
        free(response_val);
        response_val = NULL;
        response_nvals = 0;

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0) 
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;

	/* Unmap dialog. */
        PDialogUnmap(d);

	/* Delete all prompts on dialog. */
        PDialogDeleteAllPrompts();

	/* Is dialog initialized? */
	if(d->initialized)
	{
	    /* Begin destroy dialog widgets. */

#define DO_DESTROY_WIDGET       \
{ \
 if(w != NULL) \
  gtk_widget_destroy(w); \
}

            w = d->icon_pm;
	    d->icon_pm = NULL;
            DO_DESTROY_WIDGET

            w = d->toplevel;
            if(w != NULL)
            {
                if(GTK_IS_WIDGET(w))
                    gtk_widget_destroy(w);
	    }

#undef DO_DESTROY_WIDGET
        }

        memset(d, 0x00, sizeof(pdialog_struct));

	return;
}
