/**
 * @file    color.c
 * @brief   Routines for manipulating plot colours.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ygraph.h"

/**
 * @brief    Set a GdkColor given RGB values.
 *
 * @param    colormap  The colormap of the current Plot.
 * @param    red       Red intensity, 0...65535.
 * @param    green     Green intensity, 0...65535.
 * @param    blue      Blue intensity, 0...65535.
 * @returns  A colour corresponding to the specified RGB values.
 */
GdkColor*
color_set_rgb(GdkColormap* colormap, gulong red, gulong green, gulong blue)
{  
  GdkColor* color;

  color = g_malloc(sizeof(GdkColor));

  color->red = red;
  color->green = green;
  color->blue = blue;

  color->pixel = (gulong) ((red&0xff00)<<8) + (green&0xff00) + (blue>>8);

  gdk_color_alloc(colormap, color);

  return color;
}

/**
 * @brief    Set the line colour for SHOW_ALL display mode.
 *
 *           Set the colour of a line depending on it's frame_nbr. Lines
 *           corresponding to a single data set are assigned a 'start colour'
 *           (the colour of the first frame) and an 'end colour' (the colour
 *           of the last frame) and the frames have a colour on the line in
 *           RGB-space between the two.
 *
 * @param    plot       The Plot for which the colour is being set.
 * @param    plot_data  The particular data set which is being coloured.
 * @param    frame_nbr  The frame to be coloured.
 * @param    nframes    The total number of frames in the data set.
 * @returns  A graphics context specifying an appropriate line colour.
 */
GdkGC*
line_set_gc(Plot* plot, PlotLine* plot_data, gint frame_nbr, gint nframes)
{
  GdkGC* gc;
  GdkColor* start_color;
  GdkColor* end_color;
  GdkColor* color;
  gdouble s;
  gulong red;
  gulong green;
  gulong blue;

  gc = gdk_gc_new(plot->plot_area->window);

  start_color = plot_data->start_color;
  end_color = plot_data->end_color;

  s = (gdouble) frame_nbr / nframes;

  red = start_color->red + s*(end_color->red - start_color->red);
  green = start_color->green + s*(end_color->green - start_color->green);
  blue = start_color->blue + s*(end_color->blue - start_color->blue);

  color = color_set_rgb(plot->colormap, red, green, blue);

  gdk_gc_set_foreground(gc, color);
  g_free(color);

  return gc;
}

/**
 * @brief    Retrieves a colour from the base global_base_color table.
 *
 *           Gets a colour along the line connecting a pair of entries
 *           in the global_base_color table. The particular base colours
 *           chosen are given by the data set index.
 *
 * @param    colormap  The colormap of the current Plot.
 * @param    i         The data set index.
 *
 * @note     The number of uniquely coloured plots will correspond to the
 *           number of entries in that table, so after that many plots are
 *           loaded then colouring is repeated.
 */
GdkColor*
base_color_set(GdkColormap* colormap, gint i)
{  
  GdkColor* color;

  color = g_malloc(sizeof(GdkColor));

  color->red = global_base_color[i][0] * (65535/255);
  color->green = global_base_color[i][1] * (65535/255);
  color->blue = global_base_color[i][2] * (65535/255);

  color->pixel = global_base_color[i][0]*65535 + global_base_color[i][1]*256 
    + global_base_color[i][2];

  gdk_color_alloc(colormap, color);

  return color;
}

/**
 * @brief    Set the start and end colours of a data set.
 *
 * @param    plot      The Plot for which colours are being chosen.
 * @param    plotline  The plotline whose colours are set.
 * @param    i         The index of the data set which is being coloured.
 */
void
plotline_set_colors(Plot* plot, PlotLine* plotline, gint i)
{
  plotline->start_color = base_color_set(plot->colormap, i%NCOLORS);
  plotline->end_color = base_color_set(plot->colormap, (i+NCOLORS/2)%NCOLORS);
}
