/* This is for emacs: -*-Mode: C++;-*- */
/*  
  Copyright 2002 Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#if !defined(_INC_YEHIA_NET_SOCKET_H)
#define _INC_YEHIA_NET_SOCKET_H

#include <string>
#include <stdexcept>

#include <sigc++/object.h>

namespace Yehia
{

namespace Net
{

/** Base class for socket addresses. */
class Address
{
    friend class Socket;
  public:
    Address(const void *data, int size);
    Address(const Address& addr);
    ~Address();
    
    int size() const { return size_; }
    const void *data() const { return data_; }
  protected:
    Address();
    
    void set_data(void *data, int size);
    void copy_data(const void *data, int size);
  private:
    int size_;
    void *data_;
};

class InetAddress : public Address
{
  public:
    enum Type
    {
      IPV4,
      IPV6
    };
    class Invalid : public std::domain_error
    {
      public:
        Invalid(const std::string& msg) : domain_error(msg) { }
    };
    InetAddress(Type type = IPV4); // create unspecified address
    InetAddress(const std::string& address, unsigned short port);
    InetAddress(const Address& addr);

    Type type() const;
    std::string address() const;
    unsigned short port() const;
    
    static int port_by_name(const std::string& name);
    static int proto_by_name(const std::string& name);
    static std::string host_by_name(const std::string& name);
};

/** Basic socket class. */
class Socket
{
  public:
    /** Thrown on socket errors. */
    class Error : public std::runtime_error
    {
      public:
        Error(const std::string& s) : runtime_error(s) { }
    };
    enum Direction
    {
      RECEIVE,
      TRANSMIT,
      BOTH
    };
    enum Style
    {
      STREAM,
      DATAGRAM,
      RAW
    };
    
    Socket(const Address& address, Style style = STREAM, int proto = 0);
    ~Socket();

    ssize_t read(void *buffer, size_t size);
    ssize_t write(const void *buffer, size_t size);

    int send(const void *buffer, size_t size, int flags);
    int recv(void *buffer, size_t size, int flags);
    
    int sendto(const void *buffer, size_t size, int flags, 
                const Address& address);
    int recvfrom(void *buffer, size_t size, int flags, Address& address);
    
    void connect(const Address& address) throw (Error);
    
    /** Close the socket. */
    void close();
    /* Shutdown communication in direction \a d. */
    void shutdown(Direction d);
    /** Listen to socket (make it a server socket). */
    void listen(int n = 1);
    /** Accept a new connection. */
    Socket accept(Address& address) throw (Error);
    Socket accept() throw (Error);

    bool set_nonblocking(bool nonblock);
    bool set_keepalive(bool keepalive);
    
    /** Check if open. */
    bool is_open() const { return fd_ >= 0; }
    /** Return file descriptor. */
    int fd() const { return fd_; }
    /** Get style of socket. */
    Style style() const;
    Address address() const;

    /** Return file descriptor. */
    operator int() const { return fd_; }
  protected:
    Socket(int fd);
  private:
    void ensure_open(const Address& addr, Style style, int proto);
    
    int fd_;
};

}

}

#endif
