#include <iostream>

#include <sigc++/tunnel.h>

#include "uc++/script.h"
#include "uc++/plugin.h"

namespace
{

using namespace uC;
using namespace uC::Script;
using namespace SigC;

class PluginManagerClass : public ClassObject
{
  public:
    PluginManagerClass(ObjectFactory& fact)
        : ClassObject(fact, false) {
    }
    virtual SigC::Object& instantiate(const ObjectContainer& args) {
      return *SigC::manage(new PluginManager);
    }
};

class scriptPlugin : public Plugin
{
  public:
    scriptPlugin(PluginManager *);
    virtual ~scriptPlugin() {
      //cout << "~scriptPlugin()" << endl;
      if (scriptobj_)
        scriptobj_->unreference();
    }
    virtual string description() const { return "uC++ scripting plugin"; }
    void set_scriptobject(Script::Object& obj) { scriptobj_ = &obj; }
  private:
    Script::Object *scriptobj_;
};

scriptPlugin::scriptPlugin(PluginManager *mgr) : Plugin(mgr)
{
  scriptobj_ = 0;
  //cout << "script plugin constr" << endl;
}

class PluginClass : public ClassObject
{
  public:
    PluginClass(PluginManager *mgr, ObjectFactory& fact)
        : ClassObject(fact, false), mgr_(mgr) {
    }
    virtual SigC::Object& instantiate(const ObjectContainer& args) {
      return *SigC::manage(new scriptPlugin(mgr_));
    }
    virtual void associate(SigC::Object& instance, Script::Object& obj) {
      dynamic_cast<scriptPlugin&>(instance).set_scriptobject(obj);
    }
  private:
    PluginManager *mgr_;
};

class ucxxPlugin : public Plugin
{
  public:
    ucxxPlugin(PluginManager *mgr);
    virtual string description() const { return "uC++ scripting support"; }

    void got_new_language(const string& name);
    void install_extensions(Language *lang);
};

ucxxPlugin::ucxxPlugin(PluginManager *mgr) : Plugin(mgr)
{
  mgr->language_registered.connect(SigC::slot(this,
					      &ucxxPlugin::got_new_language));
}

void ucxxPlugin::got_new_language(const string& name)
{
  Script::Language *lang = manager()->language(name);
  
  if (lang)
  {
    emit_log(ErrorHandler::LOG_INFO, "--registering: " + name);
  
    pack(slot(this, &ucxxPlugin::install_extensions),
	 lang)->tunnel(lang->tunnel());
  }
}

void ucxxPlugin::install_extensions(Language *lang)
{
  try
  {
    Namespace& root_ns = lang->root_namespace();
    ObjectFactory& fact = lang->factory();

    Script::Object& obj = fact.create_namespace();
    root_ns.insert("ucxx", obj);

    Script::Namespace& myns = obj.namespace_interface();
    
    myns.insert("PluginManager", fact.create_class(
                        *(SigC::manage(new PluginManagerClass(fact)))));
    myns.insert("Plugin", fact.create_class(
                        *(SigC::manage(new PluginClass(manager(), fact)))));
  }
  catch (Script::Exception& e)
  {
    std::cerr << "exception while registering: " << e.what() << endl;
  }
  
  emit_log(ErrorHandler::LOG_INFO, "--registered.");
}

extern "C" uC::Plugin *ucxx_ucxx_plugin_init(uC::PluginManager *mgr)
{
  try
  {
    ucxxPlugin *plugin = SigC::manage(new ucxxPlugin(mgr));
    return(plugin);
  }
  catch (...)
  {
    mgr->set_error("ucxxPlugin initialisation failed");
    return 0;
  }
}

}
