/* This is for emacs: -*-Mode: C++;-*- */
#if !defined(__LIBUCXX_PLUGIN_H)
#define __LIBUCXX_PLUGIN_H

#include <glib.h>

#include <string>
#include <list>
#include <map>
#include <stdexcept>

#include <uc++/error-handler.h>
#include <uc++/script.h>

namespace uC
{

class Plugin;
class PluginManager;
class PluginNode;

template <class T> class G_Node_Iterator
{
  public:
    typedef T value_type;
    typedef T* pointer;
    typedef T& reference;

    GNode *node;
    
    typedef G_Node_Iterator<T> Self;
    
    G_Node_Iterator(GNode *n = 0) : node(n) { }
    G_Node_Iterator(const PluginNode& n) : node(n.node_) { }
    G_Node_Iterator(const G_Node_Iterator<T>& it) : node(it.node) { }
    
    bool operator ==(const Self& x) const { return node == x.node; }
    bool operator !=(const Self& x) const { return node != x.node; }

    Self&  operator++() {
      if (node)
        node = g_node_next_sibling(node);
      return *this;
    }

    
    Self operator++(int) {
      Self tmp = *this;
      ++*this;
      return tmp;
    }
    
    Self&  operator--() {
      if (node)
        node = g_node_previous_sibling(node);
      return *this;
    }

    Self operator--(int) {
      Self tmp = *this;
      --*this;
      return tmp;
    }

    T operator*() const {
      return T(node);
    }
};

template <class T> class G_Node_Recursive_Iterator : public G_Node_Iterator<T>
{
  public:
    typedef G_Node_Recursive_Iterator<T> Self;
    typedef G_Node_Iterator<T> Parent;
    
    G_Node_Recursive_Iterator(GNode *n) : Parent(n) { }
    G_Node_Recursive_Iterator(const PluginNode& n) : Parent(n) { }
    G_Node_Recursive_Iterator(const G_Node_Iterator<T>& it) : Parent(it) { }
    
    Self& operator++() {
      if (node)
      {
        if (node->children) node = g_node_first_child(node);
        else if (node->next) node = g_node_next_sibling(node);
        else if (node->parent)
        {
          do
          {
            node = node->parent;
          } while (node && !node->next);
          if (node)
            node = g_node_next_sibling(node);
        }
        else node = 0;
      }
      return *this;
    }
    
    Self operator++(int) {
      Self tmp = *this;
      ++*this;
      return tmp;
    }
    
    Self&  operator--() {
      if (node)
        node = g_node_previous_sibling(node);
      return *this;
    }

    Self operator--(int) {
      Self tmp = *this;
      --*this;
      return tmp;
    }
};

class PluginNode
{
    friend class G_Node_Iterator<PluginNode>;

  public:
    typedef G_Node_Iterator<PluginNode> iterator;
    typedef G_Node_Iterator<PluginNode> const_iterator;
    typedef G_Node_Recursive_Iterator<PluginNode> recursive_iterator;
    typedef GNode *pointer;
    typedef GNode& reference;

    PluginNode();
    PluginNode(const reference r) {  node_ = &r; }
    PluginNode(const string& name, Plugin *p = 0);
    virtual ~PluginNode();

    pointer operator&() const { return(node_); }

    string full_name() const;
    string name() const;
    Plugin *plugin() const;
    void set_plugin(Plugin *p);

    bool is_ancestor_of(const PluginNode& descendant) const;
    
    iterator find(const string& name) { return(find_(name)); }
    const_iterator find(const string& name) const { return(find_(name)); }

    Plugin* operator[](const string& name) const {
      const_iterator it = find(name);
      return(it != end() ? (*it).plugin() : 0);
    }

    iterator begin() { return(begin_()); }
    const_iterator begin() const { return(begin_()); }
    
    iterator end() { return(end_()); }
    const_iterator end() const { return(end_()); }

    iterator insert(const string& name, Plugin *plugin = 0);
    bool erase(const string& name);
    void erase(iterator it);


    void clear();

    void print(int indent = 0) const;

    // internal!
    explicit PluginNode(GNode *node);
  private:
    iterator begin_() const { return(iterator(g_node_first_child(node_))); }
    iterator end_() const { return(iterator(0)); }
    iterator find_(const string& name) const;
    
    GNode *node_;
};

class PluginLoader : virtual public ErrorHandler
{
  public:
    PluginLoader(ErrorHandler *parent) : ErrorHandler(parent) {
      set_sink();
    }

    virtual string id() const = 0;
    virtual Plugin *load(PluginManager *, const string& name) = 0;
    
    virtual void scan(PluginManager *) const = 0;
};

class PluginManager : public ErrorHandler
{
  public:
    PluginManager(ErrorHandler *parent = 0);
    ~PluginManager();
    
    const list<string>& arch_indep_paths();
    const list<string>& arch_dep_paths();
    
    void register_plugin_loader(PluginLoader *loader);
    void unregister_plugin_loader(PluginLoader *loader);

    Plugin *load_plugin(const string& name);
    void release_plugin(const string& name);

    void set_error(const string& err = string());
    
    const PluginNode& plugins() const { return(plugin_tree_); }
    
    void scan();

    void plugin_found(const string& name);

    // Scripting
    void register_language(const string& name, const Script::Language& lang);
    
    Script::Language *language(const string& name) {
      map<string, Script::Language>::iterator it = languages_.find(name);
      return it == languages_.end() ? 0 : &(*it).second;
    }
    
    
    /** Emitted when a plugin is loaded.
     *  \param id The number of the \c Plugin.
     */
    SigC::Signal1<void, const string&> plugin_loaded;

    /** Emitted when a plugin is unloaded.
     *  \param id The number of the \c Plugin.
     */
    SigC::Signal1<void, const string&> plugin_unloaded;

    /** Emitted when a new plugin becomes available for loading.
     *  \param id The number of the \c Plugin.
     */
    SigC::Signal1<void, const string&> plugin_available;

    /** Emitted when a plugin becomes unavailable.
     *  \param id The number of the \c Plugin.
     */
    SigC::Signal1<void, const string&> plugin_unavailable;

    SigC::Signal1<void, const string&> language_registered;
  private:

    PluginNode plugin_tree_;
    map<string, Script::Language> languages_;
    list<PluginLoader *> loaders_;
    list<string> ad_paths_, ai_paths_;
    string last_error_;
    bool scan_time_;
};

class Plugin : public ErrorHandler
{
  public:
    Plugin(PluginManager *parent)
        : ErrorHandler(parent) {
      set_sink();
    }
    virtual ~Plugin() { }

    virtual string description() const = 0;

    PluginManager *manager() {
      return dynamic_cast<PluginManager *>(parent());
    }
 
};

inline PluginNode operator*(const PluginNode::reference r) {
  return(PluginNode(r));
}


}

#endif
