#include "python-script.h"

namespace uC
{

namespace Script
{

class pythonClassType
{
  public:
    static PyTypeObject *type();
  private:
    static void instance_destroy(PyObject *);
    static PyObject *instance_repr(PyObject *);
    static PyObject *instance_str(PyObject *);
    static PyObject *instance_getattr(PyObject *, PyObject *name);
    static int instance_setattr(PyObject *, PyObject *, PyObject *value);
};

pythonObjectFactory::pythonObjectFactory()
{
  memchunk_ = g_mem_chunk_new("uC++ python object mem chunks",
                              sizeof(pythonObject),
                              sizeof(pythonObject) * 256,
                              G_ALLOC_AND_FREE);
  
  gpointer mem = g_mem_chunk_alloc(memchunk_);
  null_obj_ = (new (mem) pythonObject(*this));
}

pythonObjectFactory::~pythonObjectFactory()
{
  g_mem_chunk_destroy(memchunk_);
}


Object& pythonObjectFactory::create_value(const Any& v)
{
  PyObject *pyobj;

  switch (v.typecode())
  {
    case TC_LONG:
    {
      long l;
      v >>= l;
      pyobj = PyInt_FromLong(l);
      break;
    }
    case TC_ULONG:
    {
      unsigned long ul;
      v >>= ul;
      pyobj = PyLong_FromUnsignedLong(ul);
      break;
    }
    case TC_STRING:
    {
      const char *s;
      v >>= s;
      pyobj = PyString_FromString(s);
      if (!pyobj)
	throw Exception("can't create pystr");
      break;
    }
    case TC_NULL:
    default:
      return null_object();
  }
  return create_object(pyobj);
}

Object& pythonObjectFactory::create_method(const Slot& s,
                                           const Signature& sig)
{
  return null_object();
}

static PyObject *py_string_from_pointer(const void *p)
{
  char address[64];
  sprintf(address, "%p", p);
  return PyString_FromString(address);
}

static ClassObject *py_get_class(PyObject *obj)
{
  ClassObject *classobj;
  PyObject *pyaddr = obj ? PyObject_GetAttrString(obj, "__ucxxclass__") : 0;
  char *addr = 0;
  
  addr = pyaddr && PyString_Check(pyaddr) ? PyString_AsString(pyaddr) : 0;
  
  if (!addr || sscanf(addr, "%p", &classobj) != 1)
    return 0;

  return classobj;
}

static SigC::Object *py_get_instance(PyObject *obj)
{
  ClassObject *instance;
  PyObject *pyaddr =
    obj ? PyObject_GetAttrString(obj, "__ucxxinstance__") : 0;
  char *addr = 0;
  
  addr = pyaddr && PyString_Check(pyaddr) ? PyString_AsString(pyaddr) : 0;
  
  if (!addr || sscanf(addr, "%p", &instance) != 1)
    return 0;

  return instance;
}

static PyObject *py_init_instance(PyObject *self, PyObject *args)
{
  ClassObject *classobj;
  SigC::Object *instance = 0;

  if (PyTuple_Check(args) && PyTuple_Size(args) == 1)
  {
    PyObject *pyaddr = PyTuple_GetItem(args, 0);
    char *addr = 0;
    
    if (PyString_Check(pyaddr))
      addr = PyString_AsString(pyaddr);
    if (!addr || sscanf(addr, "%p", &instance) != 1)
      return NULL;
  }

  if ((classobj = py_get_class(self)) == 0)
    return NULL;
  
  if (!instance)
    instance = &classobj->instantiate(ObjectContainer());
  
  PyObject_SetAttrString(self, "__ucxxinstance__",
                         py_string_from_pointer(instance));

  if (classobj->script_referenced())
    instance->reference();

  classobj->associate(*instance,
                      static_cast<pythonObjectFactory&>(
                              classobj->factory()).create_object(self));
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *py_del_instance(PyObject *self, PyObject *args)
{
  SigC::Object *instance;
  ClassObject *classobj;
  
  if ((instance = py_get_instance(self)) == 0)
    return NULL;

  if ((classobj = py_get_class(self)) == 0)
    return NULL;
  
  if (classobj->script_referenced())
    instance->unreference();
  
  Py_INCREF(Py_None);
  return Py_None;
}

Object& pythonObjectFactory::create_class(const ClassObject& c,
                                          const Signature& supers)
{
  static PyMethodDef methods[] = {
    { "__init__", py_init_instance, METH_VARARGS, "uC++ constructor" },
    { "__del__", py_del_instance, METH_VARARGS, "uC++ destructor" },
    { NULL }
  };
  
  PyObject *pysupers, *dict, *name, *pyclass, *pystr;
  
  // Create and fill tuple of superclasses
  pysupers = PyTuple_New(supers.size());
  for (Signature::size_type i = 0; i < supers.size(); i++)
  {
    PyObject *pysuper = dynamic_cast<const pythonObject&>(supers[i]).pyobj();
    PyTuple_SET_ITEM(pysupers, i, pysuper);
    Py_DECREF(pysuper);
  }
  
  dict = PyDict_New();
  name = PyString_FromString("unnamed uC++ class");
  
  pyclass = PyClass_New(pysupers, dict, name);
  
  Py_DECREF(dict);
  Py_DECREF(name);
  Py_DECREF(pysupers);
  
  // insert methods
  for (PyMethodDef *def = methods; def->ml_name != NULL; def++)
  {
    PyObject *func = PyCFunction_New(def, pyclass);
    PyDict_SetItemString(dict, def->ml_name, func);
    Py_DECREF(func);
  }
  pystr = py_string_from_pointer(&c);
  PyDict_SetItemString(dict, "__ucxxclass__", pystr);
  Py_DECREF(pystr);
    
  return create_object(pyclass);
}

Object& pythonObjectFactory::create_namespace()
{
  return create_object(0);
}

Object& pythonObjectFactory::wrap_instance(const Object& klass,
                                           SigC::Object& obj)
{
  return null_object();
}

pythonObject& pythonObjectFactory::create_object(PyObject *pyobj)
{
  gpointer mem = g_mem_chunk_alloc(memchunk_);
  return *(new (mem) pythonObject(*this, pyobj));
}

Object& pythonObjectFactory::null_object()
{
  return *null_obj_;
}

void pythonObjectFactory::destroy(Object& obj)
{
  pythonObject& pyobj = dynamic_cast<pythonObject&>(obj);
  pyobj.~pythonObject();
  g_mem_chunk_free(memchunk_, &pyobj);
}

pythonObject::pythonObject(pythonObjectFactory& factory, PyObject *pyobj)
    : Script::Object(factory)
{
  //Py_XINCREF(pyobj);
  
  pyobj_ = pyobj;
}

pythonObject::~pythonObject()
{
  Py_XDECREF(pyobj_);
}

inline PyObject *pythonObject::pyobj() const
{
  if (!pyobj_)
    throw Exception("oops");

  return pyobj_;
}

inline bool pythonObject::is_null()
{
  return pyobj_ == 0;
}

inline pythonObject& pythonObject::operator=(PyObject *pyobj)
{
  Py_XDECREF(pyobj_);
  Py_XINCREF(pyobj);
  pyobj_ = pyobj;
  return *this;
}

inline void pythonObject::ensure(bool cond)
{
  if (!cond) throw Exception("hooo");
}

Namespace& pythonObject::namespace_interface()
{
  return(*this);
}

Callable& pythonObject::callable_interface()
{
  ensure(PyCallable_Check(pyobj()));
  return(*this);
}

Instance& pythonObject::instance_interface()
{
  ensure(PyClass_Check(pyobj()));
  return(*this);
}

Any pythonObject::value() const
{
  Any retval;
  
  // FIXME
  
  return retval;
}

SigC::Object& pythonObject::object()
{
  throw 0;
}

Object& pythonObject::get_attribute(const std::string& name)
{
  PyObject *attr = PyObject_GetAttrString(pyobj(), (char *)name.c_str());
  
  if (attr)
  {
    return factory().create_object(attr);
  }
  throw 0;
}

void pythonObject::set_attribute(const std::string& name, Object& value)
{
  PyObject_SetAttrString(pyobj(), (char *)name.c_str(),
			 dynamic_cast<pythonObject&>(value).pyobj());
}

Object& pythonObject::call(const ObjectContainer& args)
{
  throw 0;
}

Object& pythonObject::member(const std::string& name)
{
#if 0  
  if (ptr() == Py_None)
  {
    Py::Mapping modules(Py::Module("sys").getAttr("modules").ptr());
    pyobj = modules.getItem("name").ptr();
  }
  else
    pyobj = getAttr(name).ptr();

#endif
  return factory().create_object(0);
}

void pythonObject::insert(const std::string& name, uC::Script::Object& object)
{
  pythonObject& obj = static_cast<pythonObject&>(object);
  PyObject *ns = 0;
  
  if (is_null())
  {
    if (obj.is_null())
    {
      // We need to create a new module in the module dictionary
      obj = PyImport_AddModule(const_cast<char *>(name.c_str()));
      return;
    }
    ns = PyImport_ImportModule("__main__");
  }
  else if (obj.is_null())
  {
    // New module in this namespace - if we are a module, we are in
    // fact a package ;-), else this means an error
    if (PyModule_Check(pyobj_))
    {
      // We need to create a new module in the module dictionary
      obj = PyImport_AddModule(
              const_cast<char *>((std::string(PyModule_GetName(pyobj_)) +
                                  name).c_str()));
      return;
    }
  }
  else
  {
    if (PyClass_Check(obj.pyobj()))
    {
      PyObject *pyname = PyString_FromString(name.c_str());
      PyObject_SetAttrString(obj.pyobj_, "__name__", pyname);
      if (PyModule_Check(pyobj_))
      {
        char *ourname = PyModule_GetName(pyobj_);
        if (!ourname)
          throw Exception("how");
        pyname = PyString_FromString(ourname);
        PyObject_SetAttrString(obj.pyobj_, "__module__", pyname);
      }
    }
    ns = pyobj_;
  }
  
  if (!ns)
    throw Exception("!");
  
  PyObject_SetAttrString(ns, const_cast<char *>(name.c_str()), obj.pyobj_);
}

}

}

