/* $Header: /fridge/cvs/xscorch/sgame/saccessory.c,v 1.17 2001/08/15 04:55:57 jacob Exp $ */
/*

   xscorch - saccessory.c     Copyright(c) 2000 Jacob Luna Lundberg
                              Copyright(c) 2000 Justin David Smith
   jacob(at)chaos2.org        http://chaos2.org/~jacob
   justins(at)chaos2.org      http://chaos2.org/

   Scorched tank accessories


   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <saccessory.h>    /* Get accessory data */
#include <sinventory.h>    /* Need inventory constants */
#include <saddconf.h>      /* Adding accessories */
#include <sconfig.h>       /* Config constants */
#include <sstr.h>          /* String functions */
#include <sutil/shash.h>



/***  Accessory Data    ***/



/*
 * How much accessory selection information do ya wanna see?
 * (bit)  (entertainment)
 *   0  None but deadly errors.
 *   1  Contents of accessory_info structs.
 *   2  Entry and exit values from functions.
 *   4  Contents of accessory_info structs in _sc_accessory_viewable.
 */
#define  SC_ACCESSORY_INFO_DEBUG   0



inline void _sc_accessory_dump_info(const sc_accessory_info *info) {
/* _sc_accessory_dump_info
   Waddaya think it does, eh? */

   if(info == NULL)
      printf("NULL");
   else
      printf("ident=%i,armslevel=%i,price=%i,bundle=%i,shield=%i,fuel=%i,repulsion=%i,state=0x%.6x,useless=%s,indirect=%s,name=%s\n",
             info->ident,info->armslevel,info->price,info->bundle,info->shield,info->fuel,info->repulsion,info->state,info->useless?"true":"false",info->indirect?"true":"false",info->name);

   fflush(stdout);

}



inline bool _sc_accessory_viewable(const sc_accessory_config *ac, const sc_accessory_info *info, int flags) {
/* sc_accessory_viewable
   Is the accessory viewable under the current rules? */

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      printf("DEBUG -   entering _sc_accessory_viewable(ac(%p), info(%p), flags=%i)\n", ac, info, flags);
      fflush(stdout);
   #endif

   if(ac == NULL || info == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG -   exit early, returning false\n");
         fflush(stdout);
      #endif
      return(false);
   }

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      printf("DEBUG -     info->armslevel(%p)=%i\n", &info->armslevel, info->armslevel);
      printf("DEBUG -     info->useless(%p)=%i\n", &info->useless, info->useless);
      printf("DEBUG -     info->indirect(%p)=%i\n", &info->indirect, info->indirect);
      fflush(stdout);
   #endif
   #if (SC_ACCESSORY_INFO_DEBUG & 4)
      printf("DEBUG -     dumping info struct (%p) ...\n", info);
      _sc_accessory_dump_info(info);
   #endif

   if( (!(flags & SC_ACCESSORY_LIMIT_ARMS)     || info->armslevel <= ac->armslevel)   &&
       (!(flags & SC_ACCESSORY_LIMIT_USELESS)  || ac->uselessitems || !info->useless) &&
       (!(flags & SC_ACCESSORY_LIMIT_INDIRECT) || !info->indirect)                    ) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG -   exit, returning true\n");
         fflush(stdout);
      #endif
      return(true);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG -   exit, returning false\n");
         fflush(stdout);
      #endif
      return(false);
   }

}



int sc_accessory_count(const sc_accessory_config *ac, int flags) {
/* sc_accessory_count
   Counts the number of accessories in an sc_accessory_config struct's accessorylist.
   This is optimized but of course if you're calling it a lot you should cache the data.
   The only exception is that the count may change if a new accessory list file is appended. */

   sc_accessory_list_item *iter = ac->accessorylist->defs_head;
   int counter = 0;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_count(ac(%p), flags=%i) pass %i\n", ac, flags, ++track);
      fflush(stdout);
   #endif

   /* Count the accessories */
   while(iter != NULL) {
      if(_sc_accessory_viewable(ac, iter->item, flags))
         ++counter;
      iter = iter->next;
   }

   /* Return the count */
   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      printf("DEBUG - exit, returning counter=%i\n", counter);
      fflush(stdout);
   #endif
   return(counter);

}



sc_accessory_info *sc_accessory_first(const sc_accessory_config *ac, int flags) {
/* sc_accessory_first
   Return the first viable accessory in the accessory config space. */

   sc_accessory_list_item *iter;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_first(ac(%p), flags=%i) pass %i\n", ac, flags, ++track);
      fflush(stdout);
   #endif

   if(ac == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }
   iter = ac->accessorylist->defs_head;

   while(iter != NULL && !_sc_accessory_viewable(ac, iter->item, flags))
      iter = iter->next;

   if(iter != NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_accessory_dump_info(iter->item);
      #endif
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}


sc_accessory_info *sc_accessory_last(const sc_accessory_config *ac, int flags) {
/* sc_accessory_last
   Return the last viable accessory in the accessory config space. */

   sc_accessory_list_item *iter;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_last(ac(%p), flags=%i) pass %i\n", ac, flags, ++track);
      fflush(stdout);
   #endif

   if(ac == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }
   iter = ac->accessorylist->defs_tail;

   while(iter != NULL && !_sc_accessory_viewable(ac, iter->item, flags))
      iter = iter->prev;

   if(iter != NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_accessory_dump_info(iter->item);
      #endif
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_accessory_info *sc_accessory_lookup(const sc_accessory_config *ac, int id, int flags) {
/* sc_accessory_lookup
   Returns the accessory information for index given.  If the item is
   not found or out-of-bounds, then NULL is returned. */

   sc_accessory_list_item *iter;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_lookup(ac(%p), id=%i, flags=%i) pass %i\n", ac, id, flags, ++track);
      fflush(stdout);
   #endif

   if(ac == NULL || id < 0 || id > SC_ACCESSORY_MAX_ID) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = ac->accessorylist->defs_hash[shash(SC_ACCESSORY_HASH_BITS, id)];
   while(iter != NULL && iter->item->ident != id)
      iter = iter->chain;

   if(iter != NULL && _sc_accessory_viewable(ac, iter->item, flags)) {
      #if (SC_ACCESSORY_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_accessory_dump_info(iter->item);
      #endif
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



inline bool _sc_accessory_compare_names(const char *A, const char *B) {
/* _sc_accessory_compare_names
   Basically, make sure the names are the same, allowing for case and punctuation. */

   if(A == NULL || B == NULL)
      return(false);

   while(*A != '\0' && B != '\0') {
      /* next valid A char */
      while(!( (*A == '\0') || (*A >= '0' && *A <= '9') ||
               (*A >= 'A' && *A <= 'Z') || (*A >= 'a' && *A <= 'z') ))
         ++A;
      /* next valid B char */
      while(!( (*B == '\0') || (*B >= '0' && *B <= '9') ||
               (*B >= 'A' && *B <= 'Z') || (*B >= 'a' && *B <= 'z') ))
         ++B;
      /* test for disparate strings at this char pair */
      if( ((*A >= 'a' && *A <= 'z') ? (*A - ('a' - 'A')) : *A) !=
          ((*B >= 'a' && *B <= 'z') ? (*B - ('a' - 'A')) : *B) )
         break;
      ++A;
      ++B;
   }

   /* If both made it to the end then they're ``equal'' enough for us. */
   if(*A == '\0' && *B == '\0')
      return(true);
   else
      return(false);

}



sc_accessory_info *sc_accessory_lookup_by_name(const sc_accessory_config *ac, char *name, int flags) {
/* sc_accessory_lookup
   Returns the accessory information for index given.  If the item is
   not found or out-of-bounds, then NULL is returned. */

   sc_accessory_list_item *iter;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_lookup_by_name(ac(%p), name(%p)=\"%s\", flags=%i) pass %i\n", ac, name, name, flags, ++track);
      fflush(stdout);
   #endif

   if(ac == NULL || name == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = ac->accessorylist->defs_head;
   while(iter != NULL && !_sc_accessory_compare_names(name, iter->item->name))
      iter = iter->next;

   if(iter != NULL && _sc_accessory_viewable(ac, iter->item, flags)) {
      #if (SC_ACCESSORY_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_accessory_dump_info(iter->item);
      #endif
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_accessory_info *sc_accessory_prev(const sc_accessory_config *ac, const sc_accessory_info *info, int flags) {
/* sc_accessory_prev
   Advance to the previous accessory in the list (with wrapping).  */

   sc_accessory_list_item *iter;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_prev(ac(%p), info(%p), flags=%i) pass %i\n", ac, info, flags, ++track);
      fflush(stdout);
   #endif

   if(ac == NULL || info == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = ac->accessorylist->defs_hash[shash(SC_ACCESSORY_HASH_BITS, info->ident)];
   while(iter != NULL && iter->item->ident != info->ident)
      iter = iter->chain;

   if(iter == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early (no valid accessories), returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   do {
      iter = iter->prev;
      if(iter == NULL) iter = ac->accessorylist->defs_tail;
   } while(iter->item->ident != info->ident && !_sc_accessory_viewable(ac, iter->item, flags));

   if(iter != NULL && iter->item->ident != info->ident) {
      #if (SC_ACCESSORY_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_accessory_dump_info(iter->item);
      #endif
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



sc_accessory_info *sc_accessory_next(const sc_accessory_config *ac, const sc_accessory_info *info, int flags) {
/* sc_accessory_next
   Advance to the next accessory in the list (with wrapping).  */

   sc_accessory_list_item *iter;

   #if (SC_ACCESSORY_INFO_DEBUG & 2)
      static int track = 0;
      printf("\nDEBUG - entering sc_accessory_next(ac(%p), info(%p), flags=%i) pass %i\n", ac, info, flags, ++track);
      fflush(stdout);
   #endif

   if(ac == NULL || info == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   iter = ac->accessorylist->defs_hash[shash(SC_ACCESSORY_HASH_BITS, info->ident)];
   while(iter != NULL && iter->item->ident != info->ident)
      iter = iter->chain;

   if(iter == NULL) {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit early (no valid accessories), returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

   do {
      iter = iter->next;
      if(iter == NULL) iter = ac->accessorylist->defs_head;
   } while(iter->item->ident != info->ident && !_sc_accessory_viewable(ac, iter->item, flags));

   if(iter != NULL && iter->item->ident != info->ident) {
      #if (SC_ACCESSORY_INFO_DEBUG & 1)
         printf("DEBUG -   dumping info struct (%p) ...\n", iter->item);
         _sc_accessory_dump_info(iter->item);
      #endif
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning iter->item(%p)\n", iter->item);
         fflush(stdout);
      #endif
      return(iter->item);
   } else {
      #if (SC_ACCESSORY_INFO_DEBUG & 2)
         printf("DEBUG - exit, returning NULL\n");
         fflush(stdout);
      #endif
      return(NULL);
   }

}



void sc_accessory_info_line(const sc_accessory_config *ac, const sc_accessory_info *info, char *buffer, int buflen) {
/* sc_accessory_info_line
   Create a line of information about the accessory. */

   int moving;

   if(buffer == NULL || buflen <= 0)
      return;

   if(info == NULL) {
      buffer[0] = '\0';
      return;
   }

   /* Clear the buffer. */
   memsetn(buffer, 0, buflen * sizeof(char));
   /* Terminating NULL character. */
   --buflen;

   /* Add the name to the buffer. */
   snprintfn(buffer, buflen, "%s:", info->name);
   moving = strnlen(info->name, SC_INVENTORY_MAX_NAME_LEN - 1) + 1;
   buffer += moving;
   buflen -= moving;

   /* Add spaces out to the end of the name area. */
   while(++moving < 20 && --buflen > 0)
      *(buffer++) = ' ';

   /* Display the accessory's arms level. */
   snprintfn(buffer, buflen, "arms = %1i, ", info->armslevel);
   moving = 10;
   buffer += moving;
   buflen -= moving;

   /* We display much different stuff for shields than for other accessories. */
   if(SC_ACCESSORY_IS_SHIELD(info)) {
      /* Display the shield level. */
      moving = info->shield;
      snprintfn(buffer, buflen, "shield = %-5i", moving);
      moving = 9 + (moving ? (int)log10(moving) : 1);
      buffer += moving;
      buflen -= moving;

      /* Add the comma. */
      if(buflen-- > 0)
         *(buffer++) = ',';

      /* Add spaces out to the end of the yield area, plus two. */
      while(++moving < 16 && --buflen > 0)
         *(buffer++) = ' ';

      /* Write out the shield short name. */
      snprintfn(buffer, buflen, "%3i%1c ", info->shield / 100, SC_ACCESSORY_SHIELD_CHAR(info));
      moving = 5;
      buffer += moving;
      buflen -= moving;

      /* And the shield type, of course. */
      snprintfn(buffer, buflen, "%s",
                SC_ACCESSORY_SHIELD_IS_FORCE(info) ? "force" :
                (SC_ACCESSORY_SHIELD_IS_MAGNETIC(info) ? "magnetic" :
                 (SC_ACCESSORY_SHIELD_IS_STANDARD(info) ? "standard" : "unknown")));
      moving = 8;
      buffer += moving;
      buflen -= moving;

      /* If magnetic, also give repulsion. */
      if(SC_ACCESSORY_SHIELD_IS_MAGNETIC(info))
         snprintfn(buffer, buflen, " (repulsion = %i)", info->repulsion);
   } else {
      /* Write out some accessory info flags. */
      snprintfn(buffer, buflen, " %10s %8s %9s %7s",
                SC_ACCESSORY_IS_CONSUMABLE(info) ? "consumable" : "",
                SC_ACCESSORY_IS_INFINITE(info) ? "infinite" : "",
                SC_ACCESSORY_IS_PERMANENT(info) ? "permanent" : "",
                info->useless ? "useless" : "");
   }

}



void sc_accessory_info_free(sc_accessory_info **ai) {
/* sc_accessory_info_free
   Invalidate memory used by an sc_accessory_info. */

   /* Make sure there is an item to free */
   if(ai == NULL || *ai == NULL) return;
   /* Free the item's name if it has one */
   if((*ai)->name != NULL) free((*ai)->name);
   /* Free the item */
   free(*ai);
   *ai = NULL;

}



void sc_accessory_list_item_free(sc_accessory_list_item **ali) {
/* sc_accessory_list_item_free
   Invalidate a list item. */

   if(ali == NULL || *ali == NULL) return;
   sc_accessory_info_free(&(*ali)->item);
   free(*ali);
   *ali = NULL;

}



void sc_accessory_list_free(sc_accessory_list **al) {
/* sc_accessory_list_free
   Invalidate an accessory list and subordinate chain. */

   sc_accessory_list_item *item = (*al)->defs_head;

   if(al == NULL || *al == NULL) return;
   memsetn((*al)->defs_hash, 0, sizeof(void *) * SC_ACCESSORY_HASH_SIZE);
   while(item != NULL) {
      (*al)->defs_head = item->next;
      sc_accessory_list_item_free(&item);
      item = (*al)->defs_head;
   }
   free(*al);
   *al = NULL;

}



void sc_accessory_config_free(sc_accessory_config **ac) {
/* sc_accessory_config_free
   Invalidate memory used by an accessory config struct. */

   if(ac == NULL || *ac == NULL) return;
   sc_accessory_list_free(&(*ac)->accessorylist);
   free(*ac);
   *ac = NULL;

}



void sc_accessory_inventory_clear(sc_accessory_config *ac) {
/* sc_accessory_inventory_clear
   Clear out the player accessory inventories. */

   int count = sc_accessory_count(ac, SC_ACCESSORY_LIMIT_NONE);
   sc_accessory_info *info = sc_accessory_first(ac, SC_ACCESSORY_LIMIT_NONE);

   for(; count > 0; --count) {
      if(!SC_ACCESSORY_IS_INFINITE(info))
         memsetn(info->inventories, 0, SC_MAX_PLAYERS * sizeof(int));
      info = sc_accessory_next(ac, info, SC_ACCESSORY_LIMIT_NONE);
   }

}



sc_accessory_list_item *sc_accessory_list_item_new() {
/* sc_accessory_list_item_new
   Allocate and set to NULL a new sc_accessory_list_item struct. */

   sc_accessory_list_item *ali;

   ali = (sc_accessory_list_item *)malloc(sizeof(sc_accessory_list_item));
   if(ali == NULL) return(NULL);

   ali->chain = NULL;
   ali->prev = NULL;
   ali->next = NULL;

   ali->item = (sc_accessory_info *)malloc(sizeof(sc_accessory_info));
   if(ali->item == NULL) {
      free(ali);
      return(NULL);
   }

   ali->item->ident = -1;
   ali->item->name = NULL;

   return(ali);

}



sc_accessory_list *sc_accessory_list_new(void) {
/* sc_accessory_list_new
   Allocate space and set defaults (NULL, heh) on a new sc_accessory_list struct. */

   sc_accessory_list *al;

   al = (sc_accessory_list *)malloc(sizeof(sc_accessory_list));
   if(al == NULL) return(NULL);

   /* Not necessary on Linux but I don't trust some others (*cough* Solaris). */
   memsetn(al->defs_hash, 0, sizeof(void *) * SC_ACCESSORY_HASH_SIZE);

   al->defs_head = NULL;
   al->defs_tail = NULL;

   return(al);

}



sc_accessory_config *sc_accessory_config_new(void) {
/* sc_accessory_config
   Allocate space and set defaults on a new sc_accessory_config struct. */

   const char *filename;
   sc_accessory_config *ac;

   ac = (sc_accessory_config *)malloc(sizeof(sc_accessory_config));
   if(ac == NULL) return(NULL);

   ac->armslevel = SC_ARMS_LEVEL_DEF;
   ac->uselessitems = true;
   ac->accessorylist = NULL;

   ac->accessorylist = sc_accessory_list_new();
   if(ac->accessorylist == NULL) {
      free(ac);
      return(NULL);
   }

   /* Read in the accessory info list */
   filename = SC_GLOBAL_DIR "/" SC_ACCESSORY_FILE;
   if(!sc_accessory_append_file(ac, filename)) {
      /* This is the root accessory list...  Die! */
      sc_accessory_list_free(&ac->accessorylist);
      free(ac);
      return(NULL);
   }

   return(ac);

}



/* Summary of the accessory state bits, for use in saddconf.c */
static const char *_sc_accessory_state_bit_names[] = {
   "consumable",
   "permanent",
   "standard",
   "magnetic",
   "force",
   "shield",
   "triple_turret",
   "auto_defense",
   "contact_trigger",
   "fuel",
   "battery",
   "recharge",
   NULL
};
static const int _sc_accessory_state_bit_items[] = {
   SC_ACCESSORY_STATE_CONSUMABLE,
   SC_ACCESSORY_STATE_PERMANENT,
   SC_ACCESSORY_SHIELD_STANDARD,
   SC_ACCESSORY_SHIELD_MAGNETIC,
   SC_ACCESSORY_SHIELD_FORCE,
   SC_ACCESSORY_STATE_SHIELD,
   SC_ACCESSORY_STATE_TRIPLE,
   SC_ACCESSORY_STATE_AUTO_DEF,
   SC_ACCESSORY_STATE_TRIGGER,
   SC_ACCESSORY_STATE_FUEL,
   SC_ACCESSORY_STATE_BATTERY,
   SC_ACCESSORY_STATE_RECHARGE,
   0
};



const char **sc_accessory_state_bit_names(void) {

   return(_sc_accessory_state_bit_names);

}



const int *sc_accessory_state_bit_items(void) {

   return(_sc_accessory_state_bit_items);

}
