/*-
# X-BASED RUBIK'S CUBE(tm)
#
#  Rubik3d.c
#
###
#
#  Copyright (c) 1994 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Rubik3d */

#include "RubikP.h"
#include "Rubik3dP.h"

#ifndef WINVER
static Boolean SetValuesRubik3D(Widget current, Widget request, Widget renew);
static void ResizeRubik3D(Rubik3DWidget w);
static void InitializeRubik3D(Widget request, Widget renew);
static void ExposeRubik3D(Widget renew, XEvent * event, Region region);
static void MoveRubik3DTl(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DTop(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DTr(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DLeft(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DRight(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DBl(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DBottom(Rubik3DWidget w, XEvent * event, char **args, int nArgs);
static void MoveRubik3DBr(Rubik3DWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsRubik3D[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>p: Practice()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>x: IncrementX()\n\
 <KeyPress>y: IncrementY()\n\
 <KeyPress>z: IncrementZ()\n\
 <KeyPress>o: Orientize()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListRubik3D[] =
{
	{(char *) "Quit", (XtActionProc) QuitRubik},
	{(char *) "Hide", (XtActionProc) HideRubik},
	{(char *) "MoveCcw", (XtActionProc) MoveRubikCcw},
	{(char *) "MoveTl", (XtActionProc) MoveRubik3DTl},
	{(char *) "MoveTop", (XtActionProc) MoveRubik3DTop},
	{(char *) "MoveTr", (XtActionProc) MoveRubik3DTr},
	{(char *) "MoveLeft", (XtActionProc) MoveRubik3DLeft},
	{(char *) "MoveCw", (XtActionProc) MoveRubikCw},
	{(char *) "MoveRight", (XtActionProc) MoveRubik3DRight},
	{(char *) "MoveBl", (XtActionProc) MoveRubik3DBl},
	{(char *) "MoveBottom", (XtActionProc) MoveRubik3DBottom},
	{(char *) "MoveBr", (XtActionProc) MoveRubik3DBr},
	{(char *) "Select", (XtActionProc) SelectRubik},
	{(char *) "Release", (XtActionProc) ReleaseRubik},
	{(char *) "Practice", (XtActionProc) PracticeRubik},
	{(char *) "PracticeMaybe", (XtActionProc) PracticeRubikMaybe},
	{(char *) "Practice2", (XtActionProc) PracticeRubik2},
	{(char *) "Randomize", (XtActionProc) RandomizeRubik},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizeRubikMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizeRubik2},
	{(char *) "Get", (XtActionProc) GetRubik},
	{(char *) "Write", (XtActionProc) WriteRubik},
	{(char *) "Clear", (XtActionProc) ClearRubik},
	{(char *) "Undo", (XtActionProc) UndoRubik},
	{(char *) "Solve", (XtActionProc) SolveRubik},
	{(char *) "Decrement", (XtActionProc) DecrementRubik},
	{(char *) "Increment", (XtActionProc) IncrementRubik},
	{(char *) "IncrementX", (XtActionProc) IncrementXRubik},
	{(char *) "IncrementY", (XtActionProc) IncrementYRubik},
	{(char *) "IncrementZ", (XtActionProc) IncrementZRubik},
	{(char *) "Orientize", (XtActionProc) OrientizeRubik},
	{(char *) "Enter", (XtActionProc) EnterRubik},
	{(char *) "Leave", (XtActionProc) LeaveRubik}
};

static XtResource resourcesRubik3D[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.username),
	 XtRString, (caddr_t) "nobody"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.frameColor),
	 XtRString, (caddr_t) "cyan" /*XtDefaultForeground*/},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[0]),
	 XtRString, (caddr_t) "red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[1]),
	 XtRString, (caddr_t) "yellow"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[2]),
	 XtRString, (caddr_t) "white"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[3]),
	 XtRString, (caddr_t) "green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[4]),
	 XtRString, (caddr_t) "orange"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[5]),
	 XtRString, (caddr_t) "blue"},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNsizex, XtCSizeX, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizex),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNsizey, XtCSizeY, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizey),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNsizez, XtCSizeZ, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizez),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULTORIENT */
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.delay),
	 XtRString, (caddr_t) "10"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.currentControl),
	 XtRString, (caddr_t) "FALSE"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.practice),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULTPRACTICE */
	{XtNfont, XtCFont, XtRString, sizeof (String),
         XtOffset(RubikWidget, rubik.font),
         XtRString, (caddr_t) "9x15bold"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(RubikWidget, rubik.select),
	 XtRCallback, (caddr_t) NULL}
};

Rubik3DClassRec rubik3dClassRec =
{
	{
		(WidgetClass) & rubikClassRec,	/* superclass */
		(char *) "Rubik3D",	/* class name */
		sizeof (Rubik3DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeRubik3D,		/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListRubik3D,	/* actions */
		XtNumber(actionsListRubik3D),	/* num actions */
		resourcesRubik3D,	/* resources */
		XtNumber(resourcesRubik3D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizeRubik3D,	/* resize */
		(XtExposeProc) ExposeRubik3D,	/* expose */
		(XtSetValuesFunc) SetValuesRubik3D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsRubik3D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass rubik3dWidgetClass = (WidgetClass) & rubik3dClassRec;
#endif

static Point faceLoc3D[MAXFACES][MAXORIENT];
static Point cubeLoc3D[MAXFACES][MAXORIENT + 1];
static Point letter3DList[MAXFACES];
static Point orient3DList[MAXFACES][MAXORIENT][2];
static RowNext rotateToRow[MAXFACES] =	/* CW to min face */
{
	{1, LEFT, TOP},
	{0, BOTTOM, RIGHT},
	{0, RIGHT, BOTTOM},
	{0, TOP, LEFT},
	{1, RIGHT, BOTTOM},
	{0, LEFT, TOP}
};

static void
CubeOffset3D(Rubik3DWidget w, int face, int x, int y, int *dx, int *dy)
{
	char *buf;
	if (w->rubik.vertical)
		switch (face) {
			case 0:
				*dx = w->rubik3d.viewMiddle.x + w->rubik.delta - 1 +
					y * (w->rubik3d.cubeSizex.x + w->rubik.delta) -
					x * (w->rubik3d.cubeDiagonalz + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y - w->rubik.delta - 2 -
					x * (w->rubik3d.cubeSizez.y + w->rubik.delta);
				break;
			case 1:
				*dx = w->rubik3d.viewMiddle.x - 2 * w->rubik.delta -
					x * (w->rubik3d.cubeDiagonaly + w->rubik.delta) -
					y * (w->rubik3d.cubeDiagonalz + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y +
					x * (w->rubik3d.cubeSizey.y + w->rubik.delta) -
					y * (w->rubik3d.cubeSizez.y + w->rubik.delta);
				break;
			case 2:
				*dx = w->rubik3d.viewMiddle.x + w->rubik.delta - 1 +
					x * (w->rubik3d.cubeSizex.x + w->rubik.delta) -
					y * (w->rubik3d.cubeDiagonaly + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y + 2 * w->rubik.delta - 1 +
					y * (w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			case 3:
				*dx = w->rubik3d.viewMiddle.x + 2 * w->rubik.delta +
					x * (w->rubik3d.cubeDiagonalz + w->rubik.delta) +
					y * (w->rubik3d.cubeDiagonaly + w->rubik.delta);
				*dy = w->rubik3d.viewSize.y + w->rubik3d.viewMiddle.y +
					w->rubik.delta - 1 -
					x * (w->rubik3d.cubeSizez.y + w->rubik.delta) +
					y * (w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			case 4:
				*dx = w->rubik3d.viewMiddle.x - w->rubik.delta + 1 -
					x * (w->rubik3d.cubeSizex.x + w->rubik.delta) +
					y * (w->rubik3d.cubeDiagonalz + w->rubik.delta);
				*dy = w->rubik3d.viewSize.y + w->rubik3d.viewMiddle.y -
					w->rubik.delta + 1 -
					y * (w->rubik3d.cubeSizez.y + w->rubik.delta);
				break;
			case 5:
				*dx = w->rubik3d.viewMiddle.x - 2 -
					y * (w->rubik3d.cubeSizex.x + w->rubik.delta) +
					x * (w->rubik3d.cubeDiagonaly + w->rubik.delta);
				*dy = w->rubik3d.viewSize.y + w->rubik3d.viewMiddle.y +
					2 * w->rubik.delta + 2 +
					x * (w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			default:
				intCat(&buf, "CubeOffset3D: face ", face);
				DISPLAY_WARNING(buf);
				free(buf);
	} else
		switch (face) {
			case 0:
				*dx = w->rubik3d.viewMiddle.x +
					y * (w->rubik3d.cubeSizex.x + w->rubik.delta) -
					x * (w->rubik3d.cubeSizez.x + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y - 2 * w->rubik.delta + 1 -
					y * (w->rubik3d.cubeDiagonalx + w->rubik.delta) -
					x * (w->rubik3d.cubeDiagonalz + w->rubik.delta);
				break;
			case 1:
				*dx = w->rubik3d.viewMiddle.x - w->rubik.delta - 2 -
					y * (w->rubik3d.cubeSizez.x + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y + w->rubik.delta +
					x * (w->rubik3d.cubeSizey.y + w->rubik.delta) -
					y * (w->rubik3d.cubeDiagonalz + w->rubik.delta);
				break;
			case 2:
				*dx = w->rubik3d.viewMiddle.x + 2 * w->rubik.delta - 1 +
					x * (w->rubik3d.cubeSizex.x + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y + w->rubik.delta +
					y * (w->rubik3d.cubeSizey.y + w->rubik.delta) -
					x * (w->rubik3d.cubeDiagonalx + w->rubik.delta);
				break;
			case 3:
				*dx = w->rubik3d.viewSize.x + w->rubik3d.viewMiddle.x -
					w->rubik.delta + 1 -
					x * (w->rubik3d.cubeSizez.x + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y - w->rubik.delta -
					y * (w->rubik3d.cubeSizey.y + w->rubik.delta) +
					x * (w->rubik3d.cubeDiagonalz + w->rubik.delta);
				break;
			case 4:
				*dx = w->rubik3d.viewSize.x + w->rubik3d.viewMiddle.x +
					w->rubik.delta - 1 -
					y * (w->rubik3d.cubeSizez.x + w->rubik.delta) +
					x * (w->rubik3d.cubeSizex.x + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y + 2 * w->rubik.delta +
					y * (w->rubik3d.cubeDiagonalz + w->rubik.delta) +
					x * (w->rubik3d.cubeDiagonalx + w->rubik.delta);
				break;
			case 5:
				*dx = w->rubik3d.viewSize.x + w->rubik3d.viewMiddle.x +
					2 * w->rubik.delta + 2 +
					y * (w->rubik3d.cubeSizex.x + w->rubik.delta);
				*dy = w->rubik3d.viewMiddle.y - w->rubik.delta -
					x * (w->rubik3d.cubeSizey.y + w->rubik.delta) +
					y * (w->rubik3d.cubeDiagonalx + w->rubik.delta);
				break;
			default:
				intCat(&buf, "CubeOffset3D: face ", face);
				DISPLAY_WARNING(buf);
				free(buf);
		}
}

static void
MapTo3D(Rubik3DWidget w, int face, int i, int j, int *x, int *y)
{
	switch (face) {
		case 0:
			*x = w->rubik.sizez - 1 - j;
			*y = i;
			break;
		case 1:
			*x = j;
			*y = w->rubik.sizez - 1 - i;
			break;
		case 2:
			*x = i;
			*y = j;
			break;
		case 3:
			*x = w->rubik.sizez - 1 - i;
			*y = w->rubik.sizey - 1 - j;
			break;
		case 4:
			*x = w->rubik.sizex - 1 - i;
			*y = w->rubik.sizez - 1 - j;
			break;
		case 5:
			*x = j;
			*y = w->rubik.sizex - 1 - i;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "MapTo3D: face ", face);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
}

#ifdef DEBUG
static void
MapFrom3D(Rubik3DWidget w, int face, int x, int y, int *i, int *j)
{
	switch (face) {
		case 0:
			*i = y;
			*j = w->rubik.sizez - 1 - x;
			break;
		case 1:
			*i = w->rubik.sizez - 1 - y;
			*j = x;
			break;
		case 2:
			*i = x;
			*j = y;
			break;
		case 3:
			*i = w->rubik.sizez - 1 - x;
			*j = w->rubik.sizey - 1 - y;
			break;
		case 4:
			*i = w->rubik.sizex - 1 - x;
			*j = w->rubik.sizez - 1 - y;
			break;
		case 5:
			*i = w->rubik.sizex - 1 - y;
			*j = x;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "MapFrom3D: face ", face);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
}

#endif

static void
MapOrientFrom3D(Rubik3DWidget w, int face, int corner, int *side)
{
	switch (face) {
		case 0:
			*side = (corner + 2) % MAXORIENT;
			break;
		case 1:
			*side = corner;
			break;
		case 2:
			*side = (corner + 3) % MAXORIENT;
			break;
		case 3:
			*side = (corner + 1) % MAXORIENT;
			break;
		case 4:
			*side = (corner + 1) % MAXORIENT;
			break;
		case 5:
			*side = corner;
			break;
		default:
			{
				char *buf;

				intCat(&buf, "MapOrientFrom3D: face ", face);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
}

void
DrawSquare3D(Rubik3DWidget w, int face, int position, int offset)
{
	GC          faceGC, borderGC;
	int         x, y, dx, dy, i, j, sizeOfRow;

	sizeOfRow = sizeRow((RubikWidget) w, face);
	i = position % sizeOfRow;
	j = position / sizeOfRow;
	MapTo3D(w, face, i, j, &x, &y);
	CubeOffset3D(w, face, x, y, &dx, &dy);
	cubeLoc3D[face][0].x = dx;
	cubeLoc3D[face][0].y = dy;
	if (offset) {
		borderGC = w->rubik.faceGC[(int) w->rubik.cubeLoc[face][position].face];
		if (w->rubik.mono) {
			faceGC = w->rubik.inverseGC;
		} else {
			faceGC = w->rubik.borderGC;
		}
	} else {
		faceGC = w->rubik.faceGC[(int) w->rubik.cubeLoc[face][position].face];
		borderGC = w->rubik.borderGC;
	}
	POLYGON((RubikWidget) w, faceGC, borderGC,
		cubeLoc3D[face], MAXORIENT, False);
	if (w->rubik.mono) {
		int         letterX, letterY;
		char        buf[2];

		buf[0] =
#ifdef WINVER
			w->rubik.faceChar[(int) w->rubik.cubeLoc
				[face][position].face];
#else
			w->rubik.faceName[(int) w->rubik.cubeLoc
				[face][position].face][0];
#endif
		buf[1] = '\0';
		letterX = dx + letter3DList[face].x + w->rubik.letterOffset.x;
		letterY = dy + letter3DList[face].y + w->rubik.letterOffset.y;
		if (offset) {
			borderGC = w->rubik.borderGC;
		} else {
			borderGC = w->rubik.inverseGC;
		}
		DRAWTEXT(w, borderGC,
			    letterX, letterY, buf, 1);
	}
	if (w->rubik.orient) {
		DRAWLINE(w, borderGC,
			  dx + orient3DList[face][(int) w->rubik.cubeLoc[face][position].rotation][0].x,
			  dy + orient3DList[face][(int) w->rubik.cubeLoc[face][position].rotation][0].y,
			  dx + orient3DList[face][(int) w->rubik.cubeLoc[face][position].rotation][1].x,
			  dy + orient3DList[face][(int) w->rubik.cubeLoc[face][position].rotation][1].y);
	}
}

void
DrawFrame3D(Rubik3DWidget w, Boolean focus)
{
	int         face, dx, dy;
	GC gc = (focus) ? w->rubik.frameGC : w->rubik.borderGC;

	dx = w->rubik3d.viewSize.x + w->rubik.puzzleOffset.x;
	dy = w->rubik3d.viewSize.y + w->rubik.puzzleOffset.y;
	if (w->rubik.vertical) {
		DRAWLINE(w, w->rubik.frameGC,
			  0, dy, dx + w->rubik.puzzleOffset.x + 1, dy);
		DRAWTEXT(w, w->rubik.borderGC,
			    (int) (2 * w->rubik.delta),
			(int) (3 * w->rubik.delta + w->rubik.letterOffset.y),
			    "Front", 5);
		DRAWTEXT(w, w->rubik.borderGC, (int)
			    (-4 * w->rubik.delta + 2 * 6 * w->rubik.letterOffset.x + w->core.width),
			    (int) (-w->rubik.delta - 2 * w->rubik.letterOffset.y + w->core.height),
			    "Back", 4);
	} else {
		DRAWLINE(w, w->rubik.frameGC,
			  dx, 0, dx, dy + w->rubik.puzzleOffset.y + 1);
		DRAWTEXT(w, w->rubik.borderGC,
			    (int) (2 * w->rubik.delta),
			(int) (3 * w->rubik.delta + w->rubik.letterOffset.y),
			    "Front", 5);
		DRAWTEXT(w, w->rubik.borderGC, (int)
			    (-4 * w->rubik.delta + 2 * 6 * w->rubik.letterOffset.x + w->core.width),
			    (int) (-w->rubik.delta - 2 * w->rubik.letterOffset.y + w->core.height),
			    "Back", 4);
	}
	for (face = 0; face < MAXFACES; face++)
		POLYLINE((RubikWidget) w, gc,
			faceLoc3D[face], MAXORIENT, False);
}

static void
ResizePieces(Rubik3DWidget w)
{
	int         face, orient, side, corner, sizex, sizey;
	Point      subcubeLoc3D[MAXFACES][MAXORIENT];
	Point      diamondLoc3D[MAXFACES][MAXORIENT];
	Point      subdiamondLoc3D[MAXFACES][MAXORIENT];

	w->rubik.letterOffset.x = -2;
	w->rubik.letterOffset.y = 4;
	w->rubik3d.viewMiddle.x = w->rubik3d.faceSize.x +
		w->rubik.puzzleOffset.x;
	w->rubik3d.viewMiddle.y = w->rubik3d.faceSize.y +
		w->rubik.puzzleOffset.y;
	for (face = 0; face < MAXFACES; face++) {
		faceLoc3D[face][0].x = w->rubik3d.viewMiddle.x;
		faceLoc3D[face][0].y = w->rubik3d.viewMiddle.y;
		for (orient = 1; orient < MAXORIENT; orient++) {
			faceLoc3D[face][orient].x = w->rubik3d.faceSize.x;
			faceLoc3D[face][orient].y = w->rubik3d.faceSize.y;
		}
	}
	if (w->rubik.vertical) {
		faceLoc3D[0][1].x /= -2;
		faceLoc3D[0][1].y /= -1;
		faceLoc3D[0][2].y = 0;
		faceLoc3D[0][3].x /= 2;

		faceLoc3D[1][1].x /= -2;
		faceLoc3D[1][2].x /= -2;
		faceLoc3D[1][2].y /= -1;
		faceLoc3D[1][3].x /= 2;
		faceLoc3D[1][3].y /= -1;

		faceLoc3D[2][1].y = 0;
		faceLoc3D[2][2].x /= -2;
		faceLoc3D[2][3].x /= -1;
		faceLoc3D[2][3].y = 0;

		for (face = MAXFACES / 2; face < MAXFACES; face++)
			faceLoc3D[face][0].y += w->rubik3d.viewSize.y + 3;

		faceLoc3D[3][1].x /= 2;
		faceLoc3D[3][1].y /= -1;
		faceLoc3D[3][2].x /= 2;
		faceLoc3D[3][3].x /= -2;

		faceLoc3D[4][1].x /= -1;
		faceLoc3D[4][1].y = 0;
		faceLoc3D[4][2].x /= 2;
		faceLoc3D[4][2].y /= -1;
		faceLoc3D[4][3].y = 0;

		faceLoc3D[5][1].x /= 2;
		faceLoc3D[5][2].x /= -1;
		faceLoc3D[5][2].y = 0;
		faceLoc3D[5][3].x /= -2;
		faceLoc3D[5][3].y /= -1;
	} else {
		faceLoc3D[0][1].x /= -1;
		faceLoc3D[0][1].y /= -2;
		faceLoc3D[0][2].y /= -2;
		faceLoc3D[0][3].y /= 2;

		faceLoc3D[1][1].x = 0;
		faceLoc3D[1][2].x /= -1;
		faceLoc3D[1][2].y /= -2;
		faceLoc3D[1][3].x = 0;
		faceLoc3D[1][3].y /= -1;

		faceLoc3D[2][1].y /= -2;
		faceLoc3D[2][2].x = 0;
		faceLoc3D[2][3].y /= 2;
		faceLoc3D[2][3].x /= -1;

		for (face = MAXFACES / 2; face < MAXFACES; face++)
			faceLoc3D[face][0].x += w->rubik3d.viewSize.x + 3;

		faceLoc3D[3][1].x /= -1;
		faceLoc3D[3][1].y /= 2;
		faceLoc3D[3][2].x = 0;
		faceLoc3D[3][2].y /= -1;
		faceLoc3D[3][3].y /= -2;

		faceLoc3D[4][1].y /= 2;
		faceLoc3D[4][2].x /= -1;
		faceLoc3D[4][2].y /= 2;
		faceLoc3D[4][3].x /= -1;
		faceLoc3D[4][3].y /= -2;

		faceLoc3D[5][1].x = 0;
		faceLoc3D[5][1].y /= -1;
		faceLoc3D[5][2].y /= 2;
		faceLoc3D[5][3].x = 0;
	}
	for (face = 0; face < MAXFACES; face++) {
		cubeLoc3D[face][0].x = faceLoc3D[face][0].x;
		cubeLoc3D[face][0].y = faceLoc3D[face][0].y;
		subcubeLoc3D[face][0].x = faceLoc3D[face][0].x;
		subcubeLoc3D[face][0].y = faceLoc3D[face][0].y;
		if (face == 0 || face == 4) {
			sizex = w->rubik.sizex;
			sizey = w->rubik.sizez;
		} else if (face == 1 || face == 3) {
			sizex = w->rubik.sizez;
			sizey = w->rubik.sizey;
		} else {	/* if (face == 2 || face == 5) */
			sizex = w->rubik.sizex;
			sizey = w->rubik.sizey;
		}
		for (orient = 1; orient < MAXORIENT; orient++) {
			if (((orient & 1) && (face == 2 || face == 3 || face == 4)) ||
			    (!(orient & 1) && !(face == 2 || face == 3 || face == 4)))
				cubeLoc3D[face][orient].x =
					(faceLoc3D[face][orient].x - (3 + sizex) *
				 w->rubik.delta * faceLoc3D[face][orient].x /
					 w->rubik3d.faceSize.x) / sizex;
			else
				cubeLoc3D[face][orient].x =
					(faceLoc3D[face][orient].x - (3 + sizey) *
				 w->rubik.delta * faceLoc3D[face][orient].x /
					 w->rubik3d.faceSize.x) / sizey;
			if (((orient & 1) && !(face == 2 || face == 3 || face == 4)) ||
			    (!(orient & 1) && (face == 2 || face == 3 || face == 4)))
				cubeLoc3D[face][orient].y =
					(faceLoc3D[face][orient].y - (3 + sizey) *
				 w->rubik.delta * faceLoc3D[face][orient].y /
					 w->rubik3d.faceSize.y) / sizey;
			else
				cubeLoc3D[face][orient].y =
					(faceLoc3D[face][orient].y - (3 + sizex) *
				 w->rubik.delta * faceLoc3D[face][orient].y /
					 w->rubik3d.faceSize.y) / sizex;
			if (((orient & 1) && (face == 2 || face == 3 || face == 4)) ||
			    (!(orient & 1) && !(face == 2 || face == 3 || face == 4)))
				subcubeLoc3D[face][orient].x =
					(faceLoc3D[face][orient].x - (5 + sizex) *
				 w->rubik.delta * faceLoc3D[face][orient].x /
					 w->rubik3d.faceSize.x) / (2 * sizex);
			else
				subcubeLoc3D[face][orient].x =
					(faceLoc3D[face][orient].x - (5 + sizey) *
				 w->rubik.delta * faceLoc3D[face][orient].x /
					 w->rubik3d.faceSize.x) / (2 * sizey);
			if (((orient & 1) && !(face == 2 || face == 3 || face == 4)) ||
			    (!(orient & 1) && (face == 2 || face == 3 || face == 4)))
				subcubeLoc3D[face][orient].y =
					(faceLoc3D[face][orient].y - (5 + sizey) *
				 w->rubik.delta * faceLoc3D[face][orient].y /
					 w->rubik3d.faceSize.y) / (2 * sizey);
			else
				subcubeLoc3D[face][orient].y =
					(faceLoc3D[face][orient].y - (5 + sizex) *
				 w->rubik.delta * faceLoc3D[face][orient].y /
					 w->rubik3d.faceSize.y) / (2 * sizex);
		}
		cubeLoc3D[face][MAXORIENT].x = -cubeLoc3D[face][1].x -
			cubeLoc3D[face][2].x - cubeLoc3D[face][3].x;
		cubeLoc3D[face][MAXORIENT].y = -cubeLoc3D[face][1].y -
			cubeLoc3D[face][2].y - cubeLoc3D[face][3].y;
	}
	w->rubik3d.cubeSizex.x = MAX(w->rubik3d.faceSize.x /
				     w->rubik.sizex - w->rubik.delta, 0);
	w->rubik3d.cubeSizey.x = MAX(w->rubik3d.faceSize.x /
				     w->rubik.sizey - w->rubik.delta, 0);
	w->rubik3d.cubeSizez.x = MAX(w->rubik3d.faceSize.x /
				     w->rubik.sizez - w->rubik.delta, 0);
	w->rubik3d.cubeSizex.y = MAX(w->rubik3d.faceSize.y /
				     w->rubik.sizex - w->rubik.delta, 0);
	w->rubik3d.cubeSizey.y = MAX(w->rubik3d.faceSize.y /
				     w->rubik.sizey - w->rubik.delta, 0);
	w->rubik3d.cubeSizez.y = MAX(w->rubik3d.faceSize.y /
				     w->rubik.sizez - w->rubik.delta, 0);
	w->rubik3d.cubeDiagonalx = MAX((w->rubik3d.faceDiagonal - w->rubik.delta) /
		w->rubik.sizex - w->rubik.delta, 1);
	w->rubik3d.cubeDiagonaly = MAX((w->rubik3d.faceDiagonal - w->rubik.delta) /
		w->rubik.sizey - w->rubik.delta, 1);
	w->rubik3d.cubeDiagonalz = MAX((w->rubik3d.faceDiagonal - w->rubik.delta) /
		w->rubik.sizez - w->rubik.delta, 1);
	if (w->rubik.vertical) {
		letter3DList[0].x = w->rubik3d.cubeSizex.x / 4;
		letter3DList[0].y = -w->rubik3d.cubeSizez.y / 2;
		letter3DList[1].x = -w->rubik3d.cubeDiagonalz;
		letter3DList[1].y = 0;
		letter3DList[2].x = w->rubik3d.cubeSizex.x / 4;
		letter3DList[2].y = w->rubik3d.cubeSizey.y / 2;
		letter3DList[3].x = w->rubik3d.cubeDiagonalz;
		letter3DList[3].y = 0;
		letter3DList[4].x = -w->rubik3d.cubeSizex.x / 4;
		letter3DList[4].y = -w->rubik3d.cubeSizez.y / 2;
		letter3DList[5].x = -w->rubik3d.cubeSizex.x / 4;
		letter3DList[5].y = w->rubik3d.cubeSizey.y / 2;
	} else {
		letter3DList[0].x = 0;
		letter3DList[0].y = -w->rubik3d.cubeDiagonalz;
		letter3DList[1].x = -w->rubik3d.cubeSizez.x / 2;
		letter3DList[1].y = w->rubik3d.cubeSizey.y / 4;
		letter3DList[2].x = w->rubik3d.cubeSizex.x / 2;
		letter3DList[2].y = w->rubik3d.cubeSizey.y / 4;
		letter3DList[3].x = -w->rubik3d.cubeSizez.x / 2;
		letter3DList[3].y = -w->rubik3d.cubeSizey.y / 4;
		letter3DList[4].x = 0;
		letter3DList[4].y = w->rubik3d.cubeDiagonalz;
		letter3DList[5].x = w->rubik3d.cubeSizex.x / 2;
		letter3DList[5].y = -w->rubik3d.cubeSizey.y / 4;
	}
	/* The following figures out where to put the orient lines */
	for (face = 0; face < MAXFACES; face++) {
		for (orient = 0; orient < MAXORIENT - 1; orient++) {
			diamondLoc3D[face][orient].x = (cubeLoc3D[face][orient].x +
					  cubeLoc3D[face][orient + 1].x) / 2;
			diamondLoc3D[face][orient].y = (cubeLoc3D[face][orient].y +
					  cubeLoc3D[face][orient + 1].y) / 2;
			subdiamondLoc3D[face][orient].x = (subcubeLoc3D[face][orient].x +
				       subcubeLoc3D[face][orient + 1].x) / 2;
			subdiamondLoc3D[face][orient].y = (subcubeLoc3D[face][orient].y +
				       subcubeLoc3D[face][orient + 1].y) / 2;
		}
		/* Its a parallelagram so take advantage of that */
		diamondLoc3D[face][orient].x = (cubeLoc3D[face][MAXORIENT - 1].x -
				       cubeLoc3D[face][MAXORIENT / 2].x) / 2;
		diamondLoc3D[face][orient].y = (cubeLoc3D[face][MAXORIENT - 1].y -
				       cubeLoc3D[face][MAXORIENT / 2].y) / 2;
		subdiamondLoc3D[face][orient].x = (subcubeLoc3D[face][MAXORIENT - 1].x -
				    subcubeLoc3D[face][MAXORIENT / 2].x) / 2;
		subdiamondLoc3D[face][orient].y = (subcubeLoc3D[face][MAXORIENT - 1].y -
				    subcubeLoc3D[face][MAXORIENT / 2].y) / 2;

		MapOrientFrom3D(w, face, 1, &corner);
		orient3DList[face][corner][0].x = cubeLoc3D[face][1].x / 2;
		orient3DList[face][corner][0].y = cubeLoc3D[face][1].y / 2;
		orient3DList[face][corner][1].x = orient3DList[face][corner][0].x +
			(cubeLoc3D[face][2].x - subcubeLoc3D[face][2].x) / 2;
		orient3DList[face][corner][1].y = orient3DList[face][corner][0].y +
			(cubeLoc3D[face][2].y - subcubeLoc3D[face][2].y) / 2;
		for (orient = 1; orient < MAXORIENT; orient++) {
			side = corner;
			MapOrientFrom3D(w, face, (orient + 1) % MAXORIENT,
				&corner);
			orient3DList[face][corner][0].x =
				orient3DList[face][side][0].x + diamondLoc3D[face][orient].x;
			orient3DList[face][corner][0].y =
				orient3DList[face][side][0].y + diamondLoc3D[face][orient].y;
			orient3DList[face][corner][1].x =
				orient3DList[face][side][1].x + subdiamondLoc3D[face][orient].x;
			orient3DList[face][corner][1].y =
				orient3DList[face][side][1].y + subdiamondLoc3D[face][orient].y;
		}
	}
}

Boolean
SelectPieces3D(Rubik3DWidget w, int x, int y, int *face, int *position)
{
	int        u, v, front, tl, ur, ul, i, j;
	int        tempx = x, tempy = y, sizeOfRow, sizeOfColumn;

	if (w->rubik.vertical) {
		tempx -= w->rubik3d.viewMiddle.x;
		front = (tempy < w->rubik3d.viewSize.y + w->rubik.puzzleOffset.y);
		if (!front)
			tempy -= (w->rubik3d.viewSize.y);
		tl = (tempy < w->rubik3d.viewMiddle.y);
		tempy -= w->rubik3d.viewMiddle.y;
		u = -w->rubik3d.faceSize.y * tempx + w->rubik3d.faceDiagonal * tempy;
		v = w->rubik3d.faceSize.y * tempx + w->rubik3d.faceDiagonal * tempy;
		ur = (u < 0);
		ul = (v < 0);
		if (front) {
			if (tl)
				*face = (ur) ? 0 : 1;
			else
				*face = (ul) ? 1 : 2;
		} else {
			if (tl)
				*face = (ul) ? 4 : 3;
			else
				*face = (ur) ? 3 : 5;
		}
	} else {
		tempy -= w->rubik3d.viewMiddle.y;
		front = (tempx < w->rubik3d.viewSize.x + w->rubik.puzzleOffset.x);
		if (!front)
			tempx -= (w->rubik3d.viewSize.x);
		tl = (tempx < w->rubik3d.viewMiddle.x);
		tempx -= w->rubik3d.viewMiddle.x;
		u = -w->rubik3d.faceSize.x * tempy + w->rubik3d.faceDiagonal * tempx;
		v = w->rubik3d.faceSize.x * tempy + w->rubik3d.faceDiagonal * tempx;
		ur = (u < 0);
		ul = (v < 0);
		if (front) {
			if (tl)
				*face = (ur) ? 1 : 0;
			else
				*face = (ul) ? 0 : 2;
		} else {
			if (tl)
				*face = (ul) ? 3 : 4;
			else
				*face = (ur) ? 4 : 5;
		}
	}
	faceSizes((RubikWidget) w, *face, &sizeOfRow, &sizeOfColumn);
	if (w->rubik.vertical)
		switch (*face) {
			case 0:
				i = (tempx - 2 -
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(w->rubik3d.cubeSizex.x + w->rubik.delta);
				j = (tempy + 2) / (w->rubik3d.cubeSizez.y + w->rubik.delta) +
					sizeOfColumn - 1;
				break;
			case 1:
				i = (tempx + 4 +
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(2 * (w->rubik3d.cubeDiagonalz + w->rubik.delta)) + sizeOfRow - 1;
				j = (-tempx - 6 +
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(2 * (w->rubik3d.cubeDiagonaly + w->rubik.delta));
				break;
			case 2:
				i = (tempx - 4 +
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(w->rubik3d.cubeSizex.x + w->rubik.delta);
				j = (tempy - 4) / (w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			case 3:
				i = (-tempx + 5 +
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(2 * (w->rubik3d.cubeDiagonalz + w->rubik.delta)) + sizeOfRow - 1;
				j = (-tempx + 7 -
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(2 * (w->rubik3d.cubeDiagonaly + w->rubik.delta)) + sizeOfColumn - 1;
				break;
			case 4:
				i = (tempx +
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(w->rubik3d.cubeSizex.x + w->rubik.delta) + sizeOfRow - 1;
				j = (tempy - 2) / (w->rubik3d.cubeSizez.y + w->rubik.delta) +
					sizeOfColumn - 1;
				break;
			case 5:
				i = (tempx + 2 -
					    (tempy * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.y)) /
					(w->rubik3d.cubeSizex.x + w->rubik.delta) + sizeOfRow - 1;
				j = (tempy - 6) / (w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			default:
				return False;
	} else
		switch (*face) {
			case 0:
				i = (-tempy - 3 +
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(2 * (w->rubik3d.cubeDiagonalx + w->rubik.delta));
				j = (tempy + 1 +
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(2 * (w->rubik3d.cubeDiagonalz + w->rubik.delta)) + sizeOfColumn - 1;
				break;
			case 1:
				i = (tempx + 2) / (w->rubik3d.cubeSizez.x + w->rubik.delta) +
					sizeOfRow - 1;
				j = (tempy - 3 -
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			case 2:
				i = (tempx - 4) / (w->rubik3d.cubeSizex.x + w->rubik.delta);
				j = (tempy - 6 +
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			case 3:
				i = tempx / (w->rubik3d.cubeSizez.x + w->rubik.delta) +
					sizeOfRow - 1;
				j = (tempy +
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(w->rubik3d.cubeSizey.y + w->rubik.delta) + sizeOfColumn - 1;
				break;
			case 4:
				i = (-tempy + 9 -
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(2 * (w->rubik3d.cubeDiagonalx + w->rubik.delta)) + sizeOfRow - 1;
				j = (-tempy + 7 +
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(2 * (w->rubik3d.cubeDiagonalz + w->rubik.delta)) + sizeOfColumn - 1;
				break;
			case 5:
				i = (-tempx + 7) / (w->rubik3d.cubeSizex.x + w->rubik.delta) +
					sizeOfRow - 1;
				j = (-tempy - 4 +
					    (tempx * w->rubik3d.faceDiagonal / w->rubik3d.faceSize.x)) /
					(w->rubik3d.cubeSizey.y + w->rubik.delta);
				break;
			default:
				return False;
		}
	if (i < 0 || j < 0 || i >= sizeOfRow || j >= sizeOfColumn)
		return False;
	*position = j * sizeOfRow + i;
	return True;
}

Boolean
NarrowSelection3D(Rubik3DWidget w, int *face, int *position, int *direction)
{
	switch (*direction) {
		case TOP:
		case RIGHT:
		case BOTTOM:
		case LEFT:
			if (w->rubik.vertical) {
				if (*face == 1 || *face == 3)
					return False;
			} else {
				if (*face == 0 || *face == 4)
					return False;
				if (*face == 5)
					*direction = (*direction + 2) % MAXORIENT;
			}
			break;
		case CCW:
		case CW:
			break;
		case TR:
			if (w->rubik.vertical) {
				if (*face == 0 || *face == 5)
					return False;
				else if (*face == 1 || *face == 2 || *face == 4)
					*direction = TOP;
				else	/* (*face == 3) */
					*direction = LEFT;
			} else {
				if (*face == 1 || *face == 5)
					return False;
				else if (*face == 0 || *face == 2 || *face == 3)
					*direction = RIGHT;
				else	/* (*face == 4) */
					*direction = BOTTOM;
			}
			break;
		case BR:
			if (w->rubik.vertical) {
				if (*face == 2 || *face == 4)
					return False;
				else if (*face == 0 || *face == 5)
					*direction = BOTTOM;
				else if (*face == 1)
					*direction = RIGHT;
				else	/* (*face == 3) */
					*direction = TOP;
			} else {
				if (*face == 2 || *face == 3)
					return False;
				else if (*face == 4 || *face == 5)
					*direction = LEFT;
				else if (*face == 0)
					*direction = BOTTOM;
				else	/* (*face == 1) */
					*direction = RIGHT;
			}
			break;
		case BL:
			if (w->rubik.vertical) {
				if (*face == 0 || *face == 5)
					return False;
				else if (*face == 1 || *face == 2 || *face == 4)
					*direction = BOTTOM;
				else	/* (*face == 3) */
					*direction = RIGHT;
			} else {
				if (*face == 1 || *face == 5)
					return False;
				else if (*face == 0 || *face == 2 || *face == 3)
					*direction = LEFT;
				else	/* (*face == 4) */
					*direction = TOP;
			}
			break;
		case TL:
			if (w->rubik.vertical) {
				if (*face == 2 || *face == 4)
					return False;
				else if (*face == 0 || *face == 5)
					*direction = TOP;
				else if (*face == 1)
					*direction = LEFT;
				else	/* (*face == 3) */
					*direction = BOTTOM;
			} else {
				if (*face == 2 || *face == 3)
					return False;
				else if (*face == 4 || *face == 5)
					*direction = RIGHT;
				else if (*face == 0)
					*direction = TOP;
				else	/* (*face == 1) */
					*direction = LEFT;
			}
			break;
		default:
			return False;
	}
	/* Remap to row movement */
	if (*direction == CW || *direction == CCW) {
		int         i, j, sizeOfRow, sizeOfColumn, newFace;

		newFace = rotateToRow[*face].face;
		faceSizes((RubikWidget) w, newFace, &sizeOfRow, &sizeOfColumn);
		*direction = (*direction == CCW) ?
			(rotateToRow[*face].direction + 2) % MAXORIENT :
			rotateToRow[*face].direction;
		if (rotateToRow[*face].sideFace == LEFT ||
		    rotateToRow[*face].sideFace == BOTTOM) {
			i = sizeOfRow - 1;
			j = sizeOfColumn - 1;
		} else {
			i = j = 0;
		}
		*face = newFace;
		*position = j * sizeOfRow + i;
	}
	return True;
}

#ifndef WINVER
static      Boolean
SetValuesRubik3D(Widget current, Widget request, Widget renew)
{
	Rubik3DWidget c = (Rubik3DWidget) current, w = (Rubik3DWidget) renew;
	Boolean     redraw = False;

	if (w->rubik.sizex != c->rubik.sizex ||
	    w->rubik.sizey != c->rubik.sizey ||
	    w->rubik.sizez != c->rubik.sizez) {
		ResetPieces((RubikWidget) w);
		ResizeRubik3D(w);
		redraw = True;
	}
	if (w->rubik3d.cubeSizex.x != c->rubik3d.cubeSizex.x ||
	    w->rubik3d.cubeSizey.x != c->rubik3d.cubeSizey.x ||
	    w->rubik3d.cubeSizez.x != c->rubik3d.cubeSizez.x ||
	    w->rubik3d.cubeSizex.y != c->rubik3d.cubeSizex.y ||
	    w->rubik3d.cubeSizey.y != c->rubik3d.cubeSizey.y ||
	    w->rubik3d.cubeSizez.y != c->rubik3d.cubeSizez.y) {
		ResizeRubik3D(w);
		redraw = True;
	}
	return (redraw);
}
#endif

#ifndef WINVER
static
#endif
void
ResizeRubik3D(Rubik3DWidget w)
{
#ifdef WINVER
	RECT        rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->rubik.delta = 4;
	w->rubik.vertical = (w->core.height >= w->core.width);
	/* Unlike the 2d implementation... lets have variable size cubies and
	   static sized cube */
	if (w->rubik.vertical) {
		w->rubik3d.viewSize.y = w->core.height / MAXVIEWS;
		w->rubik3d.viewSize.x = w->core.width;
		if (w->rubik3d.viewSize.x >= DIVIDE(w->rubik3d.viewSize.y)) {
			w->rubik3d.viewSize.y = MAX(w->rubik3d.viewSize.y, 0);
			w->rubik3d.viewSize.x = DIVIDE(w->rubik3d.viewSize.y);
		} else {
			w->rubik3d.viewSize.x = MAX(w->rubik3d.viewSize.x, 0);
			w->rubik3d.viewSize.y = MULTIPLY(w->rubik3d.viewSize.x);
		}
		w->rubik3d.faceSize.x = MAX(w->rubik3d.viewSize.x / 2 - 2, 1);
		w->rubik3d.faceSize.y = MAX(w->rubik3d.viewSize.y / 2 - 2, 1);
		w->rubik3d.faceDiagonal = w->rubik3d.faceSize.x / 2;
		w->rubik.puzzleSize.x = w->rubik3d.viewSize.x;
		w->rubik.puzzleSize.y = MAXVIEWS * w->rubik3d.viewSize.y;
	} else {
		w->rubik3d.viewSize.x = w->core.width / MAXVIEWS;
		w->rubik3d.viewSize.y = w->core.height;
		if (w->rubik3d.viewSize.y >= DIVIDE(w->rubik3d.viewSize.x)) {
			w->rubik3d.viewSize.x = MAX(w->rubik3d.viewSize.x, 0);
			w->rubik3d.viewSize.y = DIVIDE(w->rubik3d.viewSize.x);
		} else {
			w->rubik3d.viewSize.y = MAX(w->rubik3d.viewSize.y, 0);
			w->rubik3d.viewSize.x = MULTIPLY(w->rubik3d.viewSize.y);
		}
		w->rubik3d.faceSize.x = MAX(w->rubik3d.viewSize.x / 2 - 2, 1);
		w->rubik3d.faceSize.y = MAX(w->rubik3d.viewSize.y / 2 - 2, 1);
		w->rubik3d.faceDiagonal = w->rubik3d.faceSize.y / 2;
		w->rubik.puzzleSize.y = w->rubik3d.viewSize.y;
		w->rubik.puzzleSize.x = MAXVIEWS * w->rubik3d.viewSize.x;
	}
	w->rubik.puzzleOffset.x = ((int) w->core.width - w->rubik.puzzleSize.x) / 2;
	w->rubik.puzzleOffset.y = ((int) w->core.height - w->rubik.puzzleSize.y) / 2;
	ResizePieces(w);
}

#ifndef WINVER
static
#endif
void
InitializeRubik3D(
#ifdef WINVER
Rubik3DWidget w
#else
Widget request, Widget renew
#endif
)
{
#ifndef WINVER
	Rubik3DWidget w = (Rubik3DWidget) renew;

	SetAllColors((RubikWidget) w);
#endif
	w->rubik.dim = 3;
	ResizeRubik3D(w);
}

#ifndef WINVER
static
#endif
void
ExposeRubik3D(
#ifdef WINVER
Rubik3DWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	Rubik3DWidget w = (Rubik3DWidget) renew;

	if (!w->core.visible)
		return;
#endif
	FILLRECTANGLE(w, w->rubik.inverseGC,
		0, 0, w->core.width, w->core.height);
	DrawFrame3D(w, w->rubik.focus);
	DrawAllPieces((RubikWidget) w);
}

#ifndef WINVER
static void
MoveRubik3DTl(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, TL,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DTop(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, TOP,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DTr(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, TR,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DLeft(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, LEFT,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DRight(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, RIGHT,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DBl(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, BL,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DBottom(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, BOTTOM,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik3DBr(Rubik3DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, BR,
		       (int) (event->xkey.state & ControlMask));
}
#endif
