/****************************************************************************
*  Copyright (C) 1996-98 by Leo Khramov
*  email:     leo@xnc.dubna.su
*  
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
 ****************************************************************************/
#include "panel.h"
#include "ftpcfg.h"
#include "query_windows.h"
#include "au/au.h"

#include "pwdseq.h"

//This file contains functions for read/write xnc.ftp file
//and create new host records.

char*   hosts_list[FTPCFG_MAX_HOSTS];
int     max_hosts=0;

static FTPCFG_S* fcur=NULL;
static char c;

//Return NO if frec1 don't match frec2 for flags
int ftpcfg_match_host(FTPCFG_S *frec1, FTPCFG_S *frec2, int flags)
{
  if((flags & FCFG_FBOOK)!=0 && strcmp(frec1->ftpbookmark, frec2->ftpbookmark)!=0 )
          return NO;
  if((flags & FCFG_FHOST)!=0 && strcmp(frec1->ftphost, frec2->ftphost)!=0 )
          return NO;
  if((flags & FCFG_FPATH)!=0 && strcmp(frec1->ftppath, frec2->ftppath)!=0 )
          return NO;
  if((flags & FCFG_FLOGIN)!=0 && strcmp(frec1->ftplogin, frec2->ftplogin)!=0 )
          return NO;
  if((flags & FCFG_FPASSWD)!=0 && strcmp(frec1->ftppasswd, frec2->ftppasswd)!=0 )
          return NO;
  if(frec1->bool_prox==0 || frec2->bool_prox==0)
          return YES;
  if((flags & FCFG_FPHOST)!=0 && strcmp(frec1->proxhost, frec2->proxhost)!=0 )
          return NO;
  if((flags & FCFG_FPLOGIN)!=0 && strcmp(frec1->proxlogin, frec2->proxlogin)!=0 )
          return NO;
  if((flags & FCFG_FPPASSWD)!=0 && strcmp(frec1->proxpasswd, frec2->proxpasswd)!=0 )
          return NO;
  return YES;
}

FTPCFG_S* ftpcfg_findhost(FTPCFG_S *frec, int flags)
{
  int i;
  for(i=0;i<max_hosts;i++)
   if(ftpcfg_match_host(frec, (FTPCFG_S*)hosts_list[i], flags))
          return (FTPCFG_S*)hosts_list[i];
  return 0;
}

//Add record to hosts and save hosts to ~/.xnc/xnc.ftp
//return YES if success and NO if add failed
int ftpcfg_addhost(FTPCFG_S *frec)
{
 FTPCFG_S *newrec;
 if(max_hosts>=FTPCFG_MAX_HOSTS)
    return NO;
 if(ftpcfg_findhost(frec, FCFG_FBOOK)!=NULL)
    return NO;        //Already in host_list
      
 newrec=new FTPCFG_S;
 if(newrec==NULL)
      return NO;
 *newrec=*frec;
 hosts_list[max_hosts++]=(char*) newrec;
 return ftpcfg_save_all_hosts();
}

void ftpcfg_str_to_pwd(char *dest, char *src)
{
  *dest++='[';
  while(*src)
  {
      if(*src>=PWD_CHAR_START && (unsigned char)(*src)<=PWD_CHAR_END)
        *dest++=pwdseq_chr[*src-PWD_CHAR_START];
      else
      {
        *dest++=QUOTE_CHAR;
        *dest++=*src;
      }
      src++;
  }
  *dest++=']';
  *dest=0;
}

//Return 0 if not a pwd format :)
int ftpcfg_pwd_to_str(char *dest, char *src)
{
  if(*src!='[')
    return 0;
  src++;
  while(*src)
  {
      if(*src==QUOTE_CHAR)
      {
        src++;
        *dest++=*src;
      }
      else
      if(*src>=PWD_CHAR_START && (unsigned char)(*src)<=PWD_CHAR_END)
        *dest++=(char)(strchr(pwdseq_chr, *src)-pwdseq_chr+PWD_CHAR_START);
      else
        return 0;
      src++;
  }
  if(*(src-1)!=']')
     return 0;
  *(dest-1)=0;
  return 1;
}

void ftpcfg_deinit_hosts()
{
  int i;
  for(i=0;i<max_hosts;i++)
    delete hosts_list[i];
  max_hosts=0;
}

int ftpcfg_save_all_hosts()
{
  char tmp[L_MAXPATH];
  char tmp2[100];
  FILE *fp;
  int i;
  FTPCFG_S *frec;
  if(max_hosts==0)
    return NO;
  sprintf(tmp,"%s/.xnc/xnc.ftp", getenv("HOME"));
  fp=fopen(tmp, "w");
  if(fp==NULL)
       return NO;
  fprintf(fp,">LSF-ftp\n");
  fprintf(fp,"#This file was autosaved by X Northern Captain\n");
  fprintf(fp,"#WARNING: Do NOT EDIT this file. It will be overwritten...\n");
  for(i=0;i<max_hosts;i++)
  {
    frec=(FTPCFG_S*)hosts_list[i];
    fprintf(fp,"URL_start\n");
    fprintf(fp,"session_name:  '%s';\n", frec->ftpbookmark);
    fprintf(fp,"ftphost:       %s;\n", frec->ftphost);
    fprintf(fp,"ftppath:       '%s';\n", frec->ftppath);
    fprintf(fp,"ftplogin:      %s;\n", frec->ftplogin);
    ftpcfg_str_to_pwd(tmp2, frec->ftppasswd);
    fprintf(fp,"ftppasswd:     %s;\n", tmp2);
    fprintf(fp,"ftpport:       %d;\n", frec->ftpport);
    fprintf(fp,"proxy_enabled: %s;\n", frec->bool_prox ? "yes" : "no");
    if(frec->bool_prox)
    {
    if(!blank(frec->proxhost))
      fprintf(fp,"proxyhost:     %s;\n", frec->proxhost);
    if(!blank(frec->proxlogin))
      fprintf(fp,"proxylogin:    %s;\n", frec->proxlogin);
    if(!blank(frec->proxpasswd))
    {
      ftpcfg_str_to_pwd(tmp2, frec->proxpasswd);
      fprintf(fp,"proxypasswd:   %s;\n", tmp2);
    }
    }
    fprintf(fp,"URL_end\n");
  }
  fprintf(fp,">endLSF\n");
  fclose(fp);
  chmod(tmp,0600);
  return YES;
}

//////////////////Functions for compiling FTP support////////////////////

int ftpcfg_urlstart()
{
  if(fcur)
  {
    serror("Previous support not closed");
    return 0;
  }
  fcur=new FTPCFG_S;
  if(fcur==NULL)
  {
    serror("Memory allocation");
    return 0;
  }
  fcur->ftpbookmark[0]=0;
  hosts_list[max_hosts]=(char*)fcur;
  return 1;
}
  
int ftpcfg_urlend()
{
  if(fcur==NULL)
  {
    serror("No active support");
    return 0;
  }
  if(fcur->ftpbookmark[0]==0)
    strncpy(fcur->ftpbookmark, fcur->ftphost, 80);
  max_hosts++;
  fcur=NULL;
  if(au_out)
    fprintf(stderr,".");
  return 1;
}

int ftpcfg_stamp()
{
  if(fcur==NULL)
  {
    serror("No active support");
    return 0;
  }
  c=dm->process_lex();
  if(c!=':')
  {
   serror("Missing separator ':'");
   dm->unlex(c);
   return 0;
  }
  c=dm->process_lex();
  if(c!='s')
  {
   dm->unlex(c);
   serror("Must be character string");
   return 0;
  }
  return 1;
}  

int ftpcfg_ftphost()
{
  if(!ftpcfg_stamp())
       return 0;
  strcpy(fcur->ftphost, dm->wlast);
  c=dm->process_lex();
  return 1;
}
  
int ftpcfg_ftplogin()
{
  if(!ftpcfg_stamp())
       return 0;
  strcpy(fcur->ftplogin, dm->wlast);
  c=dm->process_lex();
  return 1;
}
  
int ftpcfg_ftppath()
{
  if(!ftpcfg_stamp())
       return 0;
  strcpy(fcur->ftppath, dm->wlast);
  c=dm->process_lex();
  return 1;
}
  
int ftpcfg_ftppasswd()
{
  if(!ftpcfg_stamp())
       return 0;
  c=dm->process_lex();
  if(ftpcfg_pwd_to_str(fcur->ftppasswd, dm->wlast)==0 && au_out!=0)
  {
    fprintf(stderr, "Password is not in native format!\n");
    return 0;
  }
  return 1;
}
  
int ftpcfg_proxhost()
{
  if(!ftpcfg_stamp())
       return 0;
  strcpy(fcur->proxhost, dm->wlast);
  c=dm->process_lex();
  return 1;
}
  
int ftpcfg_proxlogin()
{
  if(!ftpcfg_stamp())
       return 0;
  strcpy(fcur->proxlogin, dm->wlast);
  c=dm->process_lex();
  return 1;
}

int ftpcfg_proxpasswd()
{
  if(!ftpcfg_stamp())
       return 0;
  c=dm->process_lex();
  if(ftpcfg_pwd_to_str(fcur->proxpasswd, dm->wlast)==0 && au_out!=0)
  {
    fprintf(stderr, "Password is not in native format!\n");
    return 0;
  }
  return 1;
}
  
int ftpcfg_boolprox()
{
  if(!ftpcfg_stamp())
       return 0;
  if(strcmp(dm->wlast, "yes")==0)
    fcur->bool_prox=YES;
  else
    fcur->bool_prox=NO;
  c=dm->process_lex();
  return 1;
}
  
int ftpcfg_ftpport()
{
  if(!ftpcfg_stamp())
       return 0;
  fcur->ftpport=atoi(dm->wlast);
  c=dm->process_lex();
  return 1;
}
  
int ftpcfg_bookmark()
{
  if(!ftpcfg_stamp())
       return 0;
  strncpy(fcur->ftpbookmark, dm->wlast, 80);
  c=dm->process_lex();
  return 1;
}
  
static struct lexdata_s term[] = {
 { "URL_start",            1  },
 { "ftphost",              2  },
 { "ftppath",              3  },
 { "ftplogin",             4  },
 { "ftppasswd",            5  },
 { "proxy_enabled",        6  },
 { "proxyhost",            7  },
 { "proxylogin",           8  },
 { "URL_end",              9  },
 { "proxypasswd",          10 },
 { "ftpport",              11 },
 { "session_name",         12 },
 { ":",                    ':'},
 { ";",                    ';'},
 { ">LSF-ftp",             'H'},
 { ">endLSF",              'E'},
 { NULL, 0 }
};
      

static int (*aufuncs[])() = {
/*  0 */  NULL,
/*  1 */  ftpcfg_urlstart,
/*  2 */  ftpcfg_ftphost,
/*  3 */  ftpcfg_ftppath,
/*  4 */  ftpcfg_ftplogin,
/*  5 */  ftpcfg_ftppasswd,
/*  6 */  ftpcfg_boolprox,
/*  7 */  ftpcfg_proxhost,
/*  8 */  ftpcfg_proxlogin,
/*  9 */  ftpcfg_urlend,
/* 10 */  ftpcfg_proxpasswd,
/* 11 */  ftpcfg_ftpport,
/* 12 */  ftpcfg_bookmark,
/* 13 */  NULL,
/* 14 */  NULL,
/* 15 */  NULL,
/* 16 */  NULL
};

int ftp_sup_init(char *sbuf)
{
 dm = new LexDemon(term);
 dm->set_input_stream(sbuf);
 dm->set_terminators(";:");
 dm->disable_const_detection();
 char c=dm->process_lex();
 if(c!='H') {
   fprintf(stderr,"Error: Not a FTP support file...\n");
   delete dm;
   return 0;
 }
 return 1;
}


void compile_FTP_supports(char *supname)
{
 char str[1024];
 supwar=superr=0;
 sprintf(str,"%s/%s",home_files_dir,supname);
 comp_sup("Loading FTP info..", str, ftp_sup_init, aufuncs);
}


void ftpcfg_reinit_hosts()
{
  ftpcfg_deinit_hosts();
  compile_FTP_supports("xnc.ftp");
}


//Init ftp config structure
//Parsing input str and fill structure fields
//str must be in format:
// [user[:password]@]host[:port][/path]
const char* const anonftp_passwd="xnc_user@lc.ld.net";
const char* const anonftp_user1="anonymous";
const char* const anonftp_user2="ftp";

int FTPCFG_S::init(char* str)
{
  char *c,*c1;
  char tmp[L_MAXPATH];
  ftpbookmark[0]=0;
  ftphost[0]=0;
  strcpy(ftppath,"/");
  strcpy(ftplogin,anonftp_user1);
  strcpy(ftppasswd,anonftp_passwd);
  ftpport=21;
  bool_prox=0;
  proxhost[0]=0;
  proxlogin[0]=0;
  proxpasswd[0]=0;

  strcpy(tmp,str);

  c=strchr(tmp,'@');
  if(c)  //We have user filled
  {
    *c=0;
    c1=strchr(tmp,':');
    if(c1) //We have user password
      *c1=0;
    strcpy(ftplogin,tmp);
    if(c1)
    {
      c1++;
      strcpy(ftppasswd,c1);
    }
    c++;
  } else
  {
    c=tmp;
  }
  c1=strchr(c,'/'); //search '/path' entry
  if(c1)
  {
    strcpy(ftppath,c1);
    *c1=0;
  }
  c1=strchr(c,':');
  if(c1) //Found port number
  {
    *c1=0;
    c1++;
    ftpport=atoi(c1);
    if(ftpport==0)
      ftpport=21;
  }
  strcpy(ftphost,c);
  xncdprintf(("Parsed ftp string: [%s[:%s]@%s[:%d][%s]",
	      ftplogin,
	      ftppasswd,
	      ftphost,
	      ftpport,
	      ftppath));
  if(strcmp(ftplogin,anonftp_user1)!=0 &&
     strcmp(ftplogin,anonftp_user2)!=0 &&
     strcmp(ftppasswd, anonftp_passwd)==0)
    return 0;    //We are logging to ftp as normal user but don't give passwd

  return 1;
}

