/****************************************************************************
*  Copyright (C) 1996-98 by Leo Khramov
*  email:     leo@xnc.dubna.su
*  
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
 ****************************************************************************/

//This file contains routines for FList class.
//This is base strcture used for storing directory listings inside XNC.

#include "xh.h"
#include "gui.h"
#include "flist.h"

int        sort_order=1;        //If 1 then sorting in alphabetical order else in reverse order

char  *getext(char *str)
{
  int    l = strlen(str) - 1;
  if (l == 1 && str[l] == str[l - 1] && str[l] == '.')
    return NULL;
  while (l > 0 && str[l] != '.')
    l--;
  if (l > 0)
    return str + l;
  return NULL;
}


//*******************************************************************//
//Compare functions used for sorting dir listing in different ways.
//These functions Return the same values as 'strcmp' does.

//Compare two files by extensions.
//If extensions equal then compare by names.
int    extcmp(char *str1, int aa1, int aa2, char *str2, int aa3, int aa4)
{
  char  *d1, *d2;
  int    k;
  if(*str2==0)
    return 1;
  d1 = getext(str1);
  d2 = getext(str2);
  if (d1 == NULL)
  {
    if (d2 == NULL)
      return sort_order ? strcmp(str1, str2) : -strcmp(str1, str2);
    else
      return sort_order ? -1 : 1;
  }
  else if (d2 == NULL)
    return sort_order ? 1 : -1;
  k = sort_order ? strcmp(d1, d2) : -strcmp(d1, d2);
  if (k == 0)
  {
    *d1 = 0;
    *d2 = 0;
    k = extcmp(str1, aa1, aa2, str2, aa3, aa4);
    *d1 = '.';
    *d2 = '.';
    return k;
  }
  return k;
}

//Name compare i.e -> strcmp.
int    mystrcmp(char *str1, int aa1, int aa2, char *str2, int aa3, int aa4)
{
  if(*str2==0)
    return 1;
  return sort_order ? strcmp(str1, str2) : -strcmp(str1, str2);
}

//First compare file sizes,
//if equal then name compare.
int    sizecmp(char *str1, int aa1, int aa2, char *str2, int aa3, int aa4)
{
  if (str1[0] == 0)
    return -1;
  if (str2[0] == 0)
    return 1;
  else if (str1[0] == '.' && str1[1] == 0)
  {
    if (str2[0] == '.' && str2[1] == 0)
      return 0;
    return sort_order ? -1 : 1;
  }
  else if (str1[0] == '.' && str1[1] == '.' && str1[2] == 0)
  {
    if (str2[0] == '.' && str2[1] == 0)
      return sort_order ? 1 : -1;
    if (str2[0] == '.' && str2[1] == '.' && str2[2] == 0)
      return 0;
    return sort_order ? -1 : 1;
  }
  else if (str2[0] == '.' && str2[1] == 0)
  {
    if (str1[0] == '.' && str1[1] == 0)
      return 0;
    return sort_order ? 1 : -1;
  }
  else if (str2[0] == '.' && str2[1] == '.' && str2[2] == 0)
    return sort_order ? 1 : -1;
  if (aa1 == aa3)
    return sort_order ? strcmp(str1, str2) : -strcmp(str1, str2);
  return aa1 < aa3 ? (sort_order ? -1 : 1) : (sort_order ? 1 : -1);
}

//Compare last access time of files.
int    timecmp(char *str1, int aa1, int aa2, char *str2, int aa3, int aa4)
{
  if (str1[0] == 0)
    return -1;
  if (str2[0] == 0)
    return 1;
  else if (str1[0] == '.' && str1[1] == 0)
  {
    if (str2[0] == '.' && str2[1] == 0)
      return 0;
    return sort_order ? -1 : 1;
  }
  else if (str1[0] == '.' && str1[1] == '.' && str1[2] == 0)
  {
    if (str2[0] == '.' && str2[1] == 0)
      return 1;
    if (str2[0] == '.' && str2[1] == '.' && str2[2] == 0)
      return 0;
    return sort_order ? -1 : 1;
  }
  else if (str2[0] == '.' && str2[1] == 0)
  {
    if (str1[0] == '.' && str1[1] == 0)
      return 0;
    return sort_order ? 1 : -1;
  }
  else if (str2[0] == '.' && str2[1] == '.' && str2[2] == 0)
    return sort_order ? 1 : -1;
  if (aa2 == aa4)
    return sort_order ? strcmp(str1, str2) : -strcmp(str1, str2);
  return aa2 > aa4 ? (sort_order ? -1 : 1) : (sort_order ? 1 : -1);
}

//Unsorted - no comparision.
int    unsortcmp(char *str1, int aa1, int aa2, char *str2, int aa3, int aa4)
{
  if(*str2==0)
    return 1;
  return sort_order ? 1 : -1;
}




//Pointer to compare functions. Store current compare function.
int    (*compare) (char *, int, int, char *, int, int) = extcmp;



//**********************************************************************//


void find_user_and_group(char *touser, int uid, char* togroup, int gid)
{
  static int rem_uid=-1, rem_gid=-1;
  static char *rem_user=NULL, *rem_group=NULL;
  
  if(rem_uid!=uid)
  {
    rem_user=finduser(uid);
    rem_uid=uid;
  }
  strncpy(touser, rem_user, L_USER);

  if(rem_gid!=gid)
  {
    rem_group=findgroup(gid);
    rem_gid=gid;
  }
  strncpy(togroup, rem_group, L_GROUP);
}

//------------------------------------------------------------------------------

//FList class functions. FList is a two way sorted list which stores dir listing.
//Constructor and initialization functions.
// *INDENT-OFF*        
FList::FList(char *iname, int imode, int is, int iuid, int igid, time_t t, 
	     struct stat* ist)
  // *INDENT-ON*        
{
  init(iname,imode,is,iuid,igid,t,ist);
};

void   FList::init(char *iname, int imode, int is, int iuid, int igid, 
		   time_t t, struct stat *ist)
{
  struct tm *ttm;
  next = prev = link = NULL;
  gid = igid;
  uid = iuid;
  options=0;
  time = t;
  ttm = localtime(&t);
  strftime(chr_time, FILE_TIMESTAMP, "%d-%b-%Y/%H:%M", ttm);
  strcpy(name, iname);
  mode = imode;
  size = is;
  if(ist)
    st=*ist;
  else
  {
    memset(&st, 0, sizeof(st));
    st.st_mode=mode;
    st.st_size=size;
    st.st_uid=uid;
    st.st_gid=gid;
    st.st_mtime=time;
  }
  magic[0]='\0';
  
  find_user_and_group(user, uid, group, gid);

};

//Add new Flist element to the list using current compare function for sorting
void   FList::add(FList * it)
{
  FList *cur = this;
  while (compare(it->name, it->size, it->time, cur->name, cur->size, cur->time) > 0)
  {
    if (cur->next)
      cur = cur->next;
    else
    {
      cur->next = it;
      it->prev = cur;
      break;
    };
  };
  if (compare(it->name, it->size, it->time, cur->name, cur->size, cur->time) <= 0)
  {
    it->next = cur;
    it->prev = cur->prev;
    if (cur->prev)
      cur->prev->next = it;
    cur->prev = it;
  }
}

void   FList::set_user_group_names(char *fromuser, char *fromgroup)
{
  strncpy(user, fromuser, L_USER);
  strncpy(group, fromgroup, L_GROUP);
}

void   FList::out()
{
}

void   FList::add_magic(char* mag)
{
  strncpy(magic, mag, L_MAGIC);
}

FList* flist_find_by_name(FList *list, char *name, struct stat *st)
{
  while(list)
  {
    if(strcmp(list->name,name)==0)
    {
      if(st)
      {
	st->st_mode=list->mode;
	st->st_size=list->size;
	st->st_mtime=list->time;
	st->st_uid=list->uid;
	st->st_gid=list->gid;
      }
      return list;
    }
    list=list->next;
  }
  return NULL;
}

//---------------------------------------------------------------------------------

//Break string 'something1 -> something2' to two strings 'something1' and 'something2'
//return 1 if success and 0 otherwise
int breakdown_link(char *from, char *to1, char *to2)
{
  char *linkid[3]={" -> ", " == ", " link to "};
  char *s;
  int i;
  for(i=0;i<3;i++)
  {
    s=strstr(from, linkid[i]);
    if(s)
    {
      memcpy(to1,from,s-from);
      to1[s-from]=0;
      strcpy(to2,s+strlen(linkid[i]));
      return 1;
    }
  }
  strcpy(to1,from);
  return 0;
}

int check_for_link(char *from)
{
  char *linkid[3]={" -> ", " == ", " link to "};
  char *s;
  int i;
  for(i=0;i<3;i++)
  {
    s=strstr(from, linkid[i]);
    if(s)
    {
      return 1;
    }
  }
  return 0;
}

void   simple_flist_add(FList * base, FList * it)
{
  it->next = base->next;
  if (base->next)
    base->next->prev = it;
  it->prev = base;
  base->next = it;
}



//----------------------------End of file----------------------//


