/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <glib.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_ttf.h>

#include "SDL_prim.h"

#include "singit/displayer_plugin.h"
#include "displayer_sdl_whitecap.h"

#include "sdl_whitecap_config.h"
#include "sdl_whitecap_status.h"
#include "sdl_lyrics.h"

#include "sdl_whitecap_handlers.h"
#include "sdl_helpers.h"

typedef struct _SDLLyrics {

	SingitSong *song;
	GList *last_token;
	gint last_line;

	SDL_Surface *first_line;
	SDL_Surface *second_line;
	SDL_Surface *blurred_line;
	SDL_Surface *progress_line;

	SDL_Surface **active_line;

	SDL_Color active_text_color;
	gint active_text_color_mapped;

	SDL_Color passive_text_color;
	gint passive_text_color_mapped;

	SDL_Color blurred_text_color;
	gint blurred_text_color_mapped;

	SDL_Color background_color;
	gint background_color_mapped;

	SDL_Rect clip_rect;

	TTF_Font *font;
	gchar* font_file_name;
	gint font_size;
	gint font_size_optimized;
	gint font_line_skip;

	gboolean color_changed;
	gboolean real_color_change;

	gint longest_line;
	gint longest_line_length;
	gint ball_diameter;
	gint min_width;

	gboolean draw_mode;
	gboolean optimize_font_size;

	pthread_mutex_t lyrics_draw_mutex;
}
SDLLyrics;

#define MIN_BORDER(font) (TTF_FontHeight(font))
#define LINE_WIDTH(font) (STATUS->screen->w - 2 * MIN_BORDER(font))

static SDLLyrics sdl_lyrics;

extern DisplayerPlugin sdl_dp;

static void lyrics_optimize_font(gboolean lock);
static void sdl_lyrics_resize_real(gboolean lock);

static inline void set_sdl_color
	(SDL_Color *col, Uint8 r, Uint8 g, Uint8 b, Uint8 a)
{
	col->r = r;
	col->g = g;
	col->b = b;
	col->unused = a;
}

static inline void set_sdl_rect
	(SDL_Rect *rect, Sint16 x, Sint16 y, Uint16 w, Uint16 h)
{
	rect->x = x;
	rect->y = y;
	rect->w = w;
	rect->h = h;
}

void sdl_lyrics_init(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [lyrics_init]\n"));
#endif

	sdl_lyrics.song = NULL;
	sdl_lyrics.last_token = NULL;
	sdl_lyrics.last_line = -1;

	sdl_lyrics.first_line = NULL;
	sdl_lyrics.second_line = NULL;
	sdl_lyrics.blurred_line = NULL;
	sdl_lyrics.progress_line = NULL;

	sdl_lyrics.active_line = NULL;

	set_sdl_color(&sdl_lyrics.active_text_color, 0, 0, 0, 0);
	sdl_lyrics.active_text_color_mapped = 0;

	set_sdl_color(&sdl_lyrics.passive_text_color, 0, 0, 0, 0);
	sdl_lyrics.passive_text_color_mapped = 0;

	set_sdl_color(&sdl_lyrics.blurred_text_color, 0, 0, 0, 0);
	sdl_lyrics.blurred_text_color_mapped = 0;

	set_sdl_color(&sdl_lyrics.background_color, 0, 0, 0, 0);
	sdl_lyrics.background_color_mapped = 0;

	set_sdl_rect(&sdl_lyrics.clip_rect, 0, 0, 0, 0);

	sdl_lyrics.font = NULL;
	sdl_lyrics.font_file_name = NULL;
	sdl_lyrics.font_size = 0;
	sdl_lyrics.font_size_optimized = -1;
	sdl_lyrics.font_line_skip = -1;

	sdl_lyrics.color_changed = FALSE;
	sdl_lyrics.real_color_change = FALSE;

	sdl_lyrics.longest_line = 0;
	sdl_lyrics.longest_line_length = 0;
	sdl_lyrics.ball_diameter = 0;
	sdl_lyrics.min_width = 0;

	sdl_lyrics.draw_mode = FALSE;
	sdl_lyrics.optimize_font_size = FALSE;

	pthread_mutex_init(&sdl_lyrics.lyrics_draw_mutex, NULL);
}

void sdl_lyrics_finish(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [lyrics_finish]\n"));
#endif

	pthread_mutex_destroy(&sdl_lyrics.lyrics_draw_mutex);

	singit_song_detach(&sdl_lyrics.song);

	if (sdl_lyrics.first_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.first_line);
		sdl_lyrics.first_line = NULL;
	}

	if (sdl_lyrics.second_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.second_line);
		sdl_lyrics.second_line = NULL;
	}

	if (sdl_lyrics.blurred_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.blurred_line);
		sdl_lyrics.blurred_line = NULL;
	}

	if (sdl_lyrics.progress_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.progress_line);
		sdl_lyrics.progress_line = NULL;
	}

	if (sdl_lyrics.font != NULL) {
		TTF_CloseFont(sdl_lyrics.font);
		sdl_lyrics.font = NULL;
	}

	if (sdl_lyrics.font_file_name != NULL) {
		g_free(sdl_lyrics.font_file_name);
		sdl_lyrics.font_file_name = NULL;
	}
}

void sdl_lyrics_remap_colors()
{
	sdl_lyrics.active_text_color_mapped = SDL_MapRGB
		(STATUS->screen->format,
		sdl_lyrics.active_text_color.r, sdl_lyrics.active_text_color.g,
		sdl_lyrics.active_text_color.b);

	sdl_lyrics.passive_text_color_mapped = SDL_MapRGB
		(STATUS->screen->format,
		sdl_lyrics.passive_text_color.r, sdl_lyrics.passive_text_color.g,
		sdl_lyrics.passive_text_color.b);

	sdl_lyrics.blurred_text_color_mapped = SDL_MapRGB
		(STATUS->screen->format,
		sdl_lyrics.blurred_text_color.r, sdl_lyrics.blurred_text_color.g,
		sdl_lyrics.blurred_text_color.b);

	sdl_lyrics.background_color_mapped = SDL_MapRGB
		(STATUS->screen->format,
		sdl_lyrics.background_color.r, sdl_lyrics.background_color.g,
		sdl_lyrics.background_color.b);

	sdl_lyrics.color_changed = TRUE;
}

static guint get_max_line_width(SingitSong *cur_song, gint *line)
{
	gint i = 0, strlength, h;
	guint line_width = 0;

	if ((cur_song == NULL) ||
		(cur_song->lyrics == NULL) ||
		(sdl_lyrics.font == NULL))
	{
		return 0;
	}

	while (cur_song->lyrics[i]) {
		strlength = strlen(cur_song->lyrics[i]);
		if (strlength > 0) {
			TTF_SizeText(sdl_lyrics.font,
				cur_song->lyrics[i], &strlength, &h);
  			if ((guint) strlength > line_width) {
				line_width = strlength;
				if (line != NULL) { *line = i; }
			}
		}
		i++;
	}

	return line_width;
}

static gboolean lyrics_set_font_int(const gchar *font_file, gint ptsize,
	gboolean refresh, gboolean lock, gboolean optimized)
{
	TTF_Font *new_font;
	gint new_font_size = ptsize;
	const gchar *new_font_file = font_file;

	g_return_val_if_fail(ptsize != 0, FALSE);

//	g_print("opt: %s (%i / %i / %i)\n", (optimized) ? "TRUE" : "FALSE",
//		ptsize, sdl_lyrics.font_size, sdl_lyrics.font_size_optimized);

	if (optimized == TRUE) {
		if (ptsize < 0) {
			new_font_size = sdl_lyrics.font_size;
		}
		else if (new_font_size == sdl_lyrics.font_size_optimized) {
			return TRUE;
		}
		new_font_file = sdl_lyrics.font_file_name;
	}
	else {
		g_return_val_if_fail(font_file != NULL, FALSE);

		if ((sdl_lyrics.font != NULL) &&
			(new_font_size == sdl_lyrics.font_size) &&
			(sdl_lyrics.font_file_name != NULL) &&
			(strcmp(sdl_lyrics.font_file_name, new_font_file) == 0))
		{
			return TRUE;
		}
	}

	new_font = TTF_OpenFont(new_font_file, new_font_size);

	if (new_font != NULL) {
		if (lock)
			pthread_mutex_lock(&sdl_lyrics.lyrics_draw_mutex);

		if (sdl_lyrics.font != NULL)
			{ TTF_CloseFont(sdl_lyrics.font); }
		sdl_lyrics.font = new_font;

		if (optimized) {
			sdl_lyrics.font_size_optimized = ptsize;
		}
		else {
			if (sdl_lyrics.font_file_name != NULL)
				{ g_free(sdl_lyrics.font_file_name); }
			sdl_lyrics.font_file_name = g_strdup(font_file);
			sdl_lyrics.font_size = ptsize;
		}
		sdl_lyrics.font_line_skip =
			(TTF_FontLineSkip(sdl_lyrics.font) >=
			TTF_FontHeight(sdl_lyrics.font) + 3) ?
			TTF_FontLineSkip(sdl_lyrics.font) :
			(TTF_FontHeight(sdl_lyrics.font) + 3);

		sdl_lyrics.longest_line_length =
			get_max_line_width(sdl_lyrics.song, &sdl_lyrics.longest_line);
		sdl_lyrics.ball_diameter = (TTF_FontHeight(sdl_lyrics.font) / 2.5);
		sdl_lyrics.min_width = sdl_lyrics.ball_diameter * 16;

		if (refresh)
			sdl_lyrics_resize_real(FALSE);
		if (lock)
			pthread_mutex_unlock(&sdl_lyrics.lyrics_draw_mutex);
	}

	return (new_font != NULL);
}

gboolean sdl_lyrics_set_font(const gchar *font_file, gint ptsize, gboolean lock)
{
	return lyrics_set_font_int(font_file, ptsize, TRUE, lock, FALSE);
}

static inline gboolean color_changed(SDL_Color cur_color, SDL_Color new_color)
{
	return ((cur_color.r != new_color.r) ||
		(cur_color.g != new_color.g) ||
		(cur_color.b != new_color.b));
}

gboolean sdl_lyrics_set_text_colors(SDL_Color text_color, SDL_Color blurred_color, SDL_Color background_color)
{
	if (color_changed(sdl_lyrics.active_text_color, text_color) ||
		color_changed(sdl_lyrics.blurred_text_color, blurred_color) ||
		color_changed(sdl_lyrics.background_color, background_color))
	{
		sdl_lyrics.active_text_color = text_color;
		sdl_lyrics.blurred_text_color = blurred_color;
		sdl_lyrics.background_color = background_color;

		sdl_lyrics.passive_text_color.r = (int) (text_color.r * 0.6);
		sdl_lyrics.passive_text_color.g = (int) (text_color.g * 0.6);
		sdl_lyrics.passive_text_color.b = (int) (text_color.b * 0.6);

		sdl_lyrics_remap_colors();

		return TRUE;
	}
	return FALSE;
}

void sdl_lyrics_set_song(SingitSong *cur_song)
{
	SingitSong *new_song = singit_song_attach(cur_song);

	singit_song_detach(&sdl_lyrics.song);
	sdl_lyrics.song = new_song;

	sdl_lyrics.last_token = (GList*) -1;
	sdl_lyrics.longest_line_length =
		get_max_line_width(sdl_lyrics.song, &sdl_lyrics.longest_line);
	sdl_lyrics.active_line = NULL;

	pthread_mutex_lock(&sdl_lyrics.lyrics_draw_mutex);
	if (sdl_lyrics.first_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.first_line);
		sdl_lyrics.first_line = NULL;
	}

	if (sdl_lyrics.second_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.second_line);
		sdl_lyrics.second_line = NULL;
	}

	if (sdl_lyrics.blurred_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.blurred_line);
		sdl_lyrics.blurred_line = NULL;
	}

	if (sdl_lyrics.progress_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.progress_line);
		sdl_lyrics.progress_line = NULL;
	}

	sdl_lyrics_resize_real(FALSE);
	pthread_mutex_unlock(&sdl_lyrics.lyrics_draw_mutex);
}

static void draw_ball(SDL_Surface *surface, gint pos_x, gint pos_y, gboolean active)
{
#define pixel(col) ((255 << 24) | (col.r << 16) | (col.g << 8) | col.b)
	gint radius = sdl_lyrics.ball_diameter / 2;
	Uint32 color;
	SDL_Rect rect;

	rect.x = pos_x;
	rect.y = pos_y;
	rect.w = radius * 2;
	rect.h = radius * 2;

	SDL_FillRect(surface, &rect, pixel(sdl_lyrics.background_color));

	if (active == TRUE) { color = pixel(sdl_lyrics.active_text_color); }
	else {
		color = pixel(sdl_lyrics.passive_text_color);
		radius -= sdl_lyrics.ball_diameter / 10;
	}

	SDL_drawCircle_TG(surface, pos_x, pos_y, radius, color,
		SDL_ALPHA_OPAQUE, SDL_TG_FILL | SDL_TG_ANTIALIAS);

#undef pixel
}

static gboolean should_draw_timeline(SingitSong *song, gint time, GList *current,
	GList *next_item, gint *ret_balls, gint *ret_light_balls)
{
	static gint last_balls = 0, last_light_balls = 0;
	gint balls, light_balls;

	if (
		(song == NULL) ||
		(current == NULL) ||
		(next_item == NULL) ||
		(singit_song_is_empty_item(song, current, TRUE) == FALSE) ||
		((balls = ((tTime(next_item) - tTime(current)) / 1000)) < 2)
		)
	{
		if (sdl_lyrics.progress_line != NULL) {
//			g_print("Free\n");
			SDL_FreeSurface(sdl_lyrics.progress_line);
			sdl_lyrics.progress_line = NULL;
		}
		last_balls = 0;
		last_light_balls = 0;
		return FALSE;
	}

	// We cheat a little bit about the timje - feels better
	light_balls = (tTime(next_item) - time + 100) / 1000;
//	if (light_balls < 0) { light_balls = 0; }
//	light_balls = balls - (time + 100 - tTime(current)) / 1000;

	if ((balls != last_balls) || (sdl_lyrics.progress_line == NULL)) {
		if (sdl_lyrics.progress_line != NULL)
			{ SDL_FreeSurface(sdl_lyrics.progress_line); }

//		g_print("Create\n");
		sdl_lyrics.progress_line = SDL_CreateRGBSurface(SDL_SWSURFACE | SDL_SRCALPHA,
			sdl_lyrics.clip_rect.w, TTF_FontHeight(sdl_lyrics.font), 32,
			0x00FF0000, 0x0000FF00, 0x000000FF, 0xFF000000);
	}
	else if ((light_balls == last_light_balls) &&
		(sdl_lyrics.real_color_change == FALSE)) { return FALSE; }

	last_balls = balls;
	last_light_balls = light_balls;

	if (ret_balls != NULL)
		{ *ret_balls = balls; }
	if (ret_light_balls != NULL)
		{ *ret_light_balls = light_balls; }

	return TRUE;
}

void sdl_lyrics_render_text(guint time)
{
	SingitSong *my_song;
	GList *current, *next;
	SDL_Surface *swap_surface;
	gboolean switch_lines = FALSE;
	gboolean refresh_text = FALSE, refresh_progress = FALSE;

// Progress
	gint balls, light_balls;
	gint pos_x, pos_y;

	my_song = singit_song_attach(sdl_lyrics.song);
	if (my_song == NULL)
		{ return; }

	if (singit_song_text_found(my_song) == FALSE)
		{ goto detach_song; }

	if (sdl_lyrics.color_changed == TRUE) {
		sdl_lyrics.color_changed = FALSE;
		sdl_lyrics.real_color_change = TRUE;
	}

	refresh_text = ((sdl_lyrics.last_token != my_song->active_token) ||
		sdl_lyrics.real_color_change);

	if (my_song->active_token == NULL)
		{ current = sdl_lyrics.last_token = my_song->first_token; }
	else {
		switch_lines = (sdl_lyrics.last_token != my_song->active_token);
		current = sdl_lyrics.last_token = my_song->active_token;
	}
	next = singit_song_find_next_lyric_line(my_song, current, FALSE, NULL);

	refresh_progress =
		should_draw_timeline(my_song, time, current, next, &balls, &light_balls);

	if (!(refresh_text || refresh_progress))
		{ goto detach_song; }

	pthread_mutex_lock(&sdl_lyrics.lyrics_draw_mutex);

	if (refresh_text == TRUE)
	{
		if (sdl_lyrics.first_line != NULL)
			{ SDL_FreeSurface(sdl_lyrics.first_line); }
		if (sdl_lyrics.second_line != NULL)
			{ SDL_FreeSurface(sdl_lyrics.second_line); }
		if (sdl_lyrics.blurred_line != NULL)
			{ SDL_FreeSurface(sdl_lyrics.blurred_line); }

		if (current && (tPos(current) < strlen(tText(my_song, current)))) {
			sdl_lyrics.first_line = TTF_RenderText_Blended
				(sdl_lyrics.font, tText(my_song, current),
				sdl_lyrics.active_text_color);
			sdl_lyrics.blurred_line = TTF_RenderText_Blended
				(sdl_lyrics.font, tText(my_song, current),
				sdl_lyrics.blurred_text_color);
//			displayer_sdl_set_active_song_line(tText(my_song, current));
		}
		else {
			sdl_lyrics.first_line = NULL;
			sdl_lyrics.blurred_line = NULL;
//			displayer_sdl_set_active_song_line(NULL);
		}

		if (next && !singit_song_is_empty_item(my_song, next, FALSE)) {
			sdl_lyrics.second_line = TTF_RenderText_Blended
				(sdl_lyrics.font, tText(my_song, next),
				sdl_lyrics.passive_text_color);
		}
		else { sdl_lyrics.second_line = NULL; }

		if (sdl_lyrics.draw_mode == TRUE) {
			if (switch_lines && (sdl_lyrics.last_line != (gint) tLine(current)))
			{
				if (sdl_lyrics.active_line == &sdl_lyrics.second_line) {
					sdl_lyrics.active_line = &sdl_lyrics.first_line;
				}
				else {
					sdl_lyrics.active_line = &sdl_lyrics.second_line;
				}
			}
			if (sdl_lyrics.active_line == &sdl_lyrics.second_line) {
				swap_surface = sdl_lyrics.first_line;
				sdl_lyrics.first_line = sdl_lyrics.second_line;
				sdl_lyrics.second_line = swap_surface;
			}
		}
		else { sdl_lyrics.active_line = &sdl_lyrics.first_line; }

		sdl_lyrics.last_line = (current == NULL) ? -1 : (gint) tLine(current);
	}

	if (refresh_progress == TRUE)
	{
		pos_x = sdl_lyrics.ball_diameter + 1;
		pos_y = TTF_FontHeight(sdl_lyrics.font) / 2;

		// If we have to pass more then 10 seconds just print 10 balls
		// And blink with the first one
		if (balls > 10) {
			if (light_balls > 10)
				light_balls = (light_balls % 2 != 0) ? 9 : 10;
			balls = 10;
		}

		while (balls > 0) {
			draw_ball(sdl_lyrics.progress_line, pos_x, pos_y,
				(balls > light_balls));
			balls--;
			pos_x += sdl_lyrics.ball_diameter * 2;
		}
	}

	sdl_lyrics.real_color_change = FALSE;

	pthread_mutex_unlock(&sdl_lyrics.lyrics_draw_mutex);

detach_song:
	singit_song_detach(&my_song);
}

static void draw_border_rect(SDL_Surface *surface, SDL_Rect *rect, guint border, Uint32 color)
{
	guint x2 = rect->x + rect->w;
	guint y2 = rect->y + rect->h;

	SDL_drawLine(surface, rect->x - border - 1, rect->y - border - 1, x2 + border, rect->y - border - 1, color);
	SDL_drawLine(surface, rect->x - border - 1, rect->y - border - 1, rect->x - border - 1, y2 + border, color);
	SDL_drawLine(surface, rect->x - border - 1, y2 + border, x2 + border, y2 + border, color);
	SDL_drawLine(surface, x2 + border, rect->y - border, x2 + border, y2 + border, color);
}

void sdl_lyrics_draw_solid(SDL_Surface *surface)
{
	SDL_Rect fill_rect, blit_rect, text_rect;

	pthread_mutex_lock(&sdl_lyrics.lyrics_draw_mutex);
	if ((sdl_lyrics.first_line != NULL) || (sdl_lyrics.second_line != NULL))
	{
		blit_rect = fill_rect = sdl_lyrics.clip_rect;
		fill_rect.x--;
		fill_rect.w += 2;

		if (sdl_lyrics.first_line != NULL) {
			fill_rect.h = blit_rect.h = sdl_lyrics.first_line->h;
			SDL_FillRect(surface, &fill_rect,
				sdl_lyrics.background_color_mapped);
			if (sdl_lyrics.active_line == &sdl_lyrics.first_line)
				draw_border_rect(surface, &fill_rect, 1,
					sdl_lyrics.active_text_color_mapped);
			text_rect = sdl_lyrics.first_line->clip_rect;
			text_rect.w = blit_rect.w;
			SDL_BlitSurface(sdl_lyrics.first_line, &text_rect,
				surface, &blit_rect);
		}
		else if (sdl_lyrics.second_line != NULL) {
			fill_rect.h = sdl_lyrics.second_line->h;
			SDL_FillRect(surface, &fill_rect,
				sdl_lyrics.background_color_mapped);
			if (sdl_lyrics.active_line == &sdl_lyrics.first_line)
				draw_border_rect(surface, &fill_rect, 1,
					sdl_lyrics.active_text_color_mapped);

			if (sdl_lyrics.progress_line != NULL) {
				blit_rect.h = sdl_lyrics.progress_line->h;
				text_rect = sdl_lyrics.progress_line->clip_rect;
				text_rect.w = blit_rect.w;
				SDL_BlitSurface(sdl_lyrics.progress_line, &text_rect,
					surface, &blit_rect);
			}
		}

		if (sdl_lyrics.second_line != NULL) {
			blit_rect.y += sdl_lyrics.font_line_skip;
			fill_rect.y = blit_rect.y;

			fill_rect.h = blit_rect.h = sdl_lyrics.second_line->h;
			blit_rect.w = sdl_lyrics.clip_rect.w;
			SDL_FillRect(surface, &fill_rect,
				sdl_lyrics.background_color_mapped);
			if (sdl_lyrics.active_line == &sdl_lyrics.second_line) {
				draw_border_rect(surface, &fill_rect, 1,
					sdl_lyrics.active_text_color_mapped);
			}
			text_rect = sdl_lyrics.second_line->clip_rect;
			text_rect.w = blit_rect.w;
			SDL_BlitSurface(sdl_lyrics.second_line, &text_rect,
				surface, &blit_rect);
		}
	}
	pthread_mutex_unlock(&sdl_lyrics.lyrics_draw_mutex);
}

void sdl_lyrics_draw_blurred(SDL_Surface *surface)
{
	SDL_Rect clip_rect, text_rect;

	pthread_mutex_lock(&sdl_lyrics.lyrics_draw_mutex);
	if (sdl_lyrics.blurred_line != NULL)
	{
		clip_rect = sdl_lyrics.clip_rect;
		clip_rect.h = sdl_lyrics.blurred_line->h;
		text_rect = sdl_lyrics.blurred_line->clip_rect;
		text_rect.w = sdl_lyrics.clip_rect.w;
		if (sdl_lyrics.active_line == &sdl_lyrics.second_line)
			clip_rect.y += sdl_lyrics.font_line_skip;
		SDL_BlitSurface(sdl_lyrics.blurred_line, NULL, surface, &clip_rect);
	}
	pthread_mutex_unlock(&sdl_lyrics.lyrics_draw_mutex);
}

static void sdl_lyrics_resize_real(gboolean lock)
{
	gint w;

	if ((sdl_lyrics.font == NULL) ||
		(sdl_lyrics.song == NULL))
	{
		return;
	}

	if (sdl_lyrics.optimize_font_size) {
		lyrics_optimize_font(lock);
	}
	else if (sdl_lyrics.font_size_optimized != -1) {
		lyrics_set_font_int(sdl_lyrics.font_file_name,
			-1, FALSE, lock, TRUE);
	}

	sdl_lyrics.clip_rect.x =
		(int) (STATUS->screen->w - sdl_lyrics.longest_line_length) / 2;
	if (sdl_lyrics.clip_rect.x < MIN_BORDER(sdl_lyrics.font)) {
		sdl_lyrics.clip_rect.x = MIN_BORDER(sdl_lyrics.font);
		w = LINE_WIDTH(sdl_lyrics.font);
	}
	else {
		w = sdl_lyrics.longest_line_length;
	}

	if (w < sdl_lyrics.min_width)
		{ w = sdl_lyrics.min_width; }

	sdl_lyrics.clip_rect.w = w;
	sdl_lyrics.clip_rect.y = STATUS->screen->h -
		sdl_lyrics.font_line_skip -
		TTF_FontHeight(sdl_lyrics.font) - MIN_BORDER(sdl_lyrics.font);
	sdl_lyrics.clip_rect.h = sdl_lyrics.font_line_skip;

	if (sdl_lyrics.progress_line != NULL) {
		SDL_FreeSurface(sdl_lyrics.progress_line);
		sdl_lyrics.progress_line = NULL;
	}
}

void sdl_lyrics_resize()
{
	sdl_lyrics_resize_real(TRUE);

	sdl_lyrics_remap_colors();
}

void sdl_lyrics_draw_mode(gboolean switch_mode)
{
	sdl_lyrics.draw_mode = switch_mode;
}

static void lyrics_optimize_font(gboolean lock)
{
#define MIN_FONT_SIZE 8
#define MAX_FONT_SIZE 50
	gint font_size, tested_font_size, last_tested_font_size;
	gint width, h, line_width;
	TTF_Font *test_font;
	SingitSong *my_song;

	if (sdl_lyrics.font == NULL)
		{ return; }

	my_song = singit_song_attach(sdl_lyrics.song);
	if (my_song == NULL)
		{ return; }

	if (singit_song_text_found(my_song) == FALSE)
		{ goto detach_song; }

	TTF_SizeText(sdl_lyrics.font, my_song->lyrics[sdl_lyrics.longest_line], &width, &h);
	line_width = LINE_WIDTH(sdl_lyrics.font);

	if (sdl_lyrics.font_size_optimized == -1)
		{ font_size = tested_font_size = last_tested_font_size = sdl_lyrics.font_size; }
	else
		{ font_size = tested_font_size = last_tested_font_size = sdl_lyrics.font_size_optimized; }

//	g_print("%i : %i / %i (%s)\n", font_size, width, line_width, sdl_lyrics.font_file_name);
	while (width > line_width) {
		font_size--;
		test_font = TTF_OpenFont(sdl_lyrics.font_file_name, font_size);
		if (test_font != NULL) {
			TTF_SizeText(test_font, my_song->lyrics[sdl_lyrics.longest_line], &width, &h);
			line_width = LINE_WIDTH(test_font);
			tested_font_size = font_size;
			TTF_CloseFont(test_font);
		}
		if ((width <= line_width) || (font_size <= MIN_FONT_SIZE)) {
			lyrics_set_font_int(sdl_lyrics.font_file_name,
				tested_font_size, FALSE, lock, TRUE);
			goto detach_song;
		}
	}

	while (width < line_width) {
		font_size++;
		test_font = TTF_OpenFont(sdl_lyrics.font_file_name, font_size);
		if (test_font != NULL) {
			TTF_SizeText(test_font, my_song->lyrics[sdl_lyrics.longest_line], &width, &h);
			line_width = LINE_WIDTH(test_font);
			last_tested_font_size = tested_font_size;
			tested_font_size = font_size;
			TTF_CloseFont(test_font);
		}
		if ((width >= line_width) || (font_size >= MAX_FONT_SIZE)) {
			if (width > line_width)
				{ tested_font_size = last_tested_font_size; }
			lyrics_set_font_int(sdl_lyrics.font_file_name,
				tested_font_size, FALSE, lock, TRUE);
			goto detach_song;
		}
	}

detach_song:
	singit_song_detach(&my_song);
#undef MIN_FONT_SIZE
#undef MAX_FONT_SIZE
}

void sdl_lyrics_set_optimize_font(gboolean optimize)
{
	if (optimize != sdl_lyrics.optimize_font_size) {
		sdl_lyrics.optimize_font_size = optimize;
		if (optimize == TRUE)
			{ sdl_lyrics_resize_real(TRUE); }
	}
}
