/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "singit_debug.h"

#include "singit/karaoke_data.h"
#include "singit/wgt_karaoke.h"
#include "singit_tools.h"

#define left_right_line_border ((SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter / 2) + 1)
#define font_height(a) ((a)->ascent + (a)->descent)
#define SKW_FONT ((SINGIT_KARAOKE_DATA(skw->skd)->font) ? SINGIT_KARAOKE_DATA(skw->skd)->font : GTK_WIDGET(skw)->style->font)

// *
// * Uncomment to enable extended debug messages
// *
// #define WIDGET_DEBUG

// *
// * All sizes in the sizes attribute
// *
enum {

	LEFT_RIGHT_WIDGET_BORDER,	// Left and right border around the widget
	TOP_BOTTOM_WIDGET_BORDER,       // Top and bottom border around the widget
	LAST_VALUE
};

void singit_karaoke_widget_build_lyric_pixmaps(SingitKaraokeWidget *skw);
void singit_karaoke_widget_update_ball(SingitKaraokeWidget *skw, gint calced_offset);
gint singit_karaoke_widget_update_progess_bar(SingitKaraokeWidget *skw, gchar *text);
static void skw_draw_line_rects(SingitKaraokeWidget *skw, GdkPixmap *pixmap, gboolean restore_background);

gint singit_karaoke_widget_get_centered_line_offset(SingitKaraokeWidget *skw, gchar *text);

static GtkWidgetClass *parent_class = NULL;

static void singit_karaoke_widget_class_init (SingitKaraokeWidgetClass *klass);
static void singit_karaoke_widget_init (SingitKaraokeWidget *skw);

static gboolean singit_karaoke_expose_event (GtkWidget *widget, GdkEventExpose *event);
static void singit_karaoke_widget_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void singit_karaoke_widget_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void singit_karaoke_widget_realize (GtkWidget *widget);
static void singit_karaoke_widget_unrealize(GtkWidget *widget);

static void singit_karaoke_widget_destroy(GtkObject *object);
static void singit_karaoke_widget_finalize(GtkObject *object);

GtkType singit_karaoke_widget_get_type (void)
{
	static GtkType singit_karaoke_widget_type = 0;

	if (!singit_karaoke_widget_type) {

		static const GtkTypeInfo singit_karaoke_widget_info =
		{
			(gchar*) "SingitKaraokeWidget",
			sizeof (SingitKaraokeWidget),
			sizeof (SingitKaraokeWidgetClass),
			(GtkClassInitFunc) singit_karaoke_widget_class_init,
			(GtkObjectInitFunc) singit_karaoke_widget_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_karaoke_widget_type = gtk_type_unique(GTK_TYPE_WIDGET, &singit_karaoke_widget_info);
	}

	return singit_karaoke_widget_type;
}

static void singit_karaoke_widget_class_init (SingitKaraokeWidgetClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	GtkWidgetClass *widget_class = (GtkWidgetClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_WIDGET);

	object_class->destroy = singit_karaoke_widget_destroy;
	object_class->finalize = singit_karaoke_widget_finalize;

	widget_class->expose_event = singit_karaoke_expose_event;
	widget_class->size_request = singit_karaoke_widget_size_request;
	widget_class->size_allocate = singit_karaoke_widget_size_allocate;
	widget_class->realize = singit_karaoke_widget_realize;
	widget_class->unrealize = singit_karaoke_widget_unrealize;
}

// Forward declaration of the SingitDisplayerData event function to keep init together
static gint singit_karaoke_data_render
	(SingitKaraokeData *skd, gpointer font, gchar* text);
static gboolean singit_karaoke_data_expose
	(SingitKaraokeData *skd, GdkRectangle *area, gint event, gint time, gpointer user_data);
static void singit_karaoke_data_new_ball
	(SingitKaraokeData *skd, gint ball_diameter, gpointer user_data);
static void singit_karaoke_data_new_visual
	(SingitKaraokeData *skd, gint wx, gint wy, gpointer user_data);

static void singit_karaoke_widget_init(SingitKaraokeWidget *skw)
{
	static gint color_presets[skwc_last][3] = {
		{     0,     0,     0 }, // Background
		{     0,     0,     0 }, // Normal text
		{ 65535, 65535, 65535 }, // Normal bar
		{     0,     0,     0 }, // Progress text
		{ 65535, 65535,     0 }, // Progress bar
	};

	gint i;

#ifdef WIDGET_DEBUG
	g_print("skw_init\n");
#endif

	GTK_WIDGET_UNSET_FLAGS(skw, GTK_NO_WINDOW);
//	GTK_WIDGET_SET_FLAGS(skw, GTK_CAN_FOCUS);

	skw->mode = skwm_line_scroll;
	skw->double_buffer_pixmap = NULL;

	for (i = 0; i < skwc_last; i++) {
		skw->private_color[i].red = color_presets[i][0];
		skw->private_color[i].green = color_presets[i][1];
		skw->private_color[i].blue = color_presets[i][2];
	}

	skw->last_time = -1;

	skw->timeIndicators = TRUE;

	skw->sizes = g_new(gint, LAST_VALUE);
	memset(&skw->sizes[0], 0, LAST_VALUE * sizeof(gint));

	g_get_current_time(&skw->last_rtc_time);

	skw->private_font_name = NULL;

	skw->update_background = TRUE;

	skw->skd = (gpointer) singit_karaoke_data_new();
	gtk_signal_connect(GTK_OBJECT(skw->skd), "render",
		GTK_SIGNAL_FUNC(singit_karaoke_data_render), NULL);
//	gtk_signal_connect(GTK_OBJECT(skw->skd), "optimize",
//		GTK_SIGNAL_FUNC(singit_karaoke_data_optimize), NULL);
	gtk_signal_connect(GTK_OBJECT(skw->skd), "expose",
		GTK_SIGNAL_FUNC(singit_karaoke_data_expose), skw);
	gtk_signal_connect(GTK_OBJECT(skw->skd), "new_ball",
		GTK_SIGNAL_FUNC(singit_karaoke_data_new_ball), skw);
	gtk_signal_connect(GTK_OBJECT(skw->skd), "new_visual",
		GTK_SIGNAL_FUNC(singit_karaoke_data_new_visual), skw);
}

static gint singit_karaoke_data_render(SingitKaraokeData *skd, gpointer font, gchar* text)
{
	return gdk_string_width((GdkFont*) font, text);
}

static void draw_timeline(SingitKaraokeWidget *skw, gint time, GdkRectangle *area);

static gboolean singit_karaoke_data_expose(SingitKaraokeData *skd,
	GdkRectangle *area, gint event, gint time, gpointer user_data)
{
	gint offset = 0;
	SingitKaraokeWidget *skw;
	GdkRectangle combined_area;

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(user_data)))
		{ return FALSE; }

	if (event == 0)
		{ return TRUE; }

	skw = SINGIT_KARAOKE_WIDGET(user_data);

	if (event & SKD_EXPOSE_ALL) {
		singit_karaoke_widget_build_lyric_pixmaps(skw);
	}
	if (event & SKD_EXPOSE_PROGRESS) {
		offset =  singit_karaoke_widget_update_progess_bar
			(skw, tText(skd->song, skd->current));
		offset -= ((skd->ball_diameter / 2) + 1);
	}
	if ((offset >= 0) && (event & SKD_EXPOSE_BALL)) {
		singit_karaoke_widget_update_ball(skw, offset);
	}

	if ((skw->timeIndicators == FALSE) || (event & SKD_EXPOSE_TIMELINE) != event) {
		gtk_widget_queue_draw_area(GTK_WIDGET(user_data),
			area->x + skw->sizes[LEFT_RIGHT_WIDGET_BORDER],
			area->y + skw->sizes[TOP_BOTTOM_WIDGET_BORDER],
			area->width, area->height);
	}

	if ((skw->timeIndicators == TRUE) && (event & SKD_EXPOSE_TIMELINE)) {
		if ((event & SKD_EXPOSE_TIMELINE) == event) {
			combined_area = *area;
			draw_timeline(skw, time, &combined_area);
			gtk_widget_queue_draw_area(GTK_WIDGET(user_data),
				combined_area.x + skw->sizes[LEFT_RIGHT_WIDGET_BORDER],
				combined_area.y + skw->sizes[TOP_BOTTOM_WIDGET_BORDER],
				combined_area.width, combined_area.height);
		}
	}

	return TRUE;
}

static void draw_ball(SingitKaraokeWidget *skw, GdkPixmap *pixmap, gint pos_x,
	gint pos_y, gboolean active, gboolean small_active)
{
	GdkGC *draw_gc;
	gint ball_frame;
	gboolean filled_ball = (SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter > 5);

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(skw)))
		{ return; }

	if (filled_ball == TRUE) {
		draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_text]);
	}
	else {
		if ((small_active == TRUE) && (active == TRUE))
			draw_gc = gdk_gc_ref(skw->private_gc[skwc_progress_bar]);
		else
			draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_text]);
	}

	gdk_draw_arc (pixmap, draw_gc, TRUE, pos_x, pos_y,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter, 0, 360 * 64);

	if ((filled_ball == TRUE) && (active == TRUE)) {
		gdk_gc_unref(draw_gc);
		draw_gc = gdk_gc_ref(skw->private_gc[skwc_progress_bar]);
		ball_frame = SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter / 10;
#ifdef WIDGET_DEBUG
		g_print("  Ball-Sizes: %i / %i / %i\n",
			SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter,
			ball_frame, ball_frame * 2);
#endif
		gdk_draw_arc (pixmap, draw_gc, TRUE, ball_frame + pos_x, ball_frame + pos_y,
			SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter - ball_frame * 2,
			SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter - ball_frame * 2,
			0, 360 * 64);
	}

	gdk_gc_unref(draw_gc);
}

void draw_timeline(SingitKaraokeWidget *skw, gint time, GdkRectangle *area)
{
	GList *next_item, *current;
	gint balls, light_balls;
	gint pos_x = SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter / 2 + 1;
	gint pos_y = SINGIT_KARAOKE_DATA(skw->skd)->top_lines_height +
		(SINGIT_KARAOKE_DATA(skw->skd)->text_line_height - SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter) / 2;
	GdkRectangle area_update;

	if (skw->double_buffer_pixmap == NULL) { return; }

	current = SINGIT_KARAOKE_DATA(skw->skd)->song->active_token;
	if (current == NULL) { return; }
	next_item = inl_singit_song_get_next_token(SINGIT_KARAOKE_DATA(skw->skd)->song);
	if (next_item == NULL) { return; }
	balls = (tTime(next_item) - tTime(current)) / 1000;
	if (balls < 2) { return; }
	light_balls = balls - (time + 100 - tTime(current)) / 1000;
//	g_print("%i / %i / %i\n", balls, light_balls, time);

	// If we have to pass more then 10 seconds just print 10 balls
	// And blink with the first one
	if (balls > 10) {
		if (light_balls > 10)
			light_balls = (light_balls % 2 != 0) ? 9 : 10;
		balls = 10;
	}

	if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
		pos_y += SINGIT_KARAOKE_DATA(skw->skd)->ball_line_height +
			SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high;
	}

	area_update.x = pos_x;
	area_update.y = pos_y;
	area_update.width = SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter * 2 * balls;
	area_update.height = SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter;
	gdk_rectangle_union(area, &area_update, area);

	while (balls > 0) {
		draw_ball(skw, skw->double_buffer_pixmap, pos_x, pos_y,
			(balls > light_balls), FALSE);
		balls--;
		pos_x += SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter * 2;
	}
}

static void singit_karaoke_data_new_ball
	(SingitKaraokeData *skd, gint ball_diameter, gpointer user_data)
{
	GdkGC *draw_gc = NULL;
	SingitKaraokeWidget *skw;

	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(user_data));

	skw = SINGIT_KARAOKE_WIDGET(user_data);

	if (!GTK_WIDGET_REALIZED(skw))
		{ return; }

#ifdef WIDGET_DEBUG
	g_print("New ball pixmap\n");
#endif

	if (skw->ball_pixmap != NULL)
		{ gdk_pixmap_unref(skw->ball_pixmap); }

	if (SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter <= 0) {
		skw->ball_pixmap = NULL;
		return;
	}

	draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_bar]);

	skw->ball_pixmap = gdk_pixmap_new(GTK_WIDGET(skw)->window,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter, -1);
	gdk_draw_rectangle(skw->ball_pixmap, draw_gc, TRUE, 0, 0,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter);

	draw_ball(skw, skw->ball_pixmap, 0, 0, TRUE, FALSE);

	gdk_gc_unref(draw_gc);

	gtk_widget_set_usize((GtkWidget*) skw,
		SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width,
		SINGIT_KARAOKE_DATA(skw->skd)->visual_min_height);
}

static void skw_draw_line_rects(SingitKaraokeWidget *skw, GdkPixmap *pixmap, gboolean restore_background)
{
#define RESTORE_BG(height) \
	do { \
		if (restore_background) { \
			gdk_draw_rectangle(pixmap, draw_bg_gc, \
				TRUE, 0, pos_y, \
				SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width, \
				SINGIT_KARAOKE_DATA(skw->skd)->height); \
		} \
		pos_y += SINGIT_KARAOKE_DATA(skw->skd)->height; \
	} \
	while (0)

#define DRAW_BAR(height) \
	do { \
		gdk_draw_rectangle(pixmap, draw_bar_gc, \
			TRUE, 0, pos_y, SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width, \
			SINGIT_KARAOKE_DATA(skw->skd)->height); \
		pos_y += SINGIT_KARAOKE_DATA(skw->skd)->height; \
	} \
	while (0)

	gint pos_y, i;
	GdkGC *draw_bar_gc, *draw_bg_gc;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));
	g_return_if_fail(pixmap != NULL);

#ifdef WIDGET_DEBUG
	g_print("skw_draw_line_rects\n");
#endif

	if (!GTK_WIDGET_REALIZED(GTK_WIDGET(skw)))
		{ return; }

	draw_bar_gc = gdk_gc_ref(skw->private_gc[skwc_normal_bar]);
	draw_bg_gc = gdk_gc_ref(skw->private_gc[skwc_background]);

	pos_y = 0;
	for (i = 0; i < (gint) SINGIT_KARAOKE_DATA(skw->skd)->top_lines; i++) {
		DRAW_BAR(text_line_height);
		RESTORE_BG(line_seperator_high);
	}

	RESTORE_BG(active_line_seperator_high);

	if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
		DRAW_BAR(ball_line_height);
		RESTORE_BG(line_seperator_high);
	}

	for (i = 0; i < (gint) (SINGIT_KARAOKE_DATA(skw->skd)->lines - SINGIT_KARAOKE_DATA(skw->skd)->top_lines); i++) {
		DRAW_BAR(text_line_height);
		RESTORE_BG(line_seperator_high);
		if (i == 0) {
			RESTORE_BG(active_line_seperator_high);
		}
	}

	gdk_gc_unref(draw_bg_gc);
	gdk_gc_unref(draw_bar_gc);
#undef DRAW_BAR
#undef RESTORE_BG
}

static void singit_karaoke_data_new_visual
	(SingitKaraokeData *skd, gint wx, gint wy, gpointer user_data)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail(skd != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skd));
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(user_data));

	if (!GTK_WIDGET_REALIZED(user_data))
		{ return; }

	skw = SINGIT_KARAOKE_WIDGET(user_data);

#ifdef WIDGET_DEBUG
	g_print("New doublebuffer pixmap\n");
#endif

	if (skw->double_buffer_pixmap)
		{ gdk_pixmap_unref(skw->double_buffer_pixmap); }

	if ((wx <= 0) || (wy <= 0)) {
		skw->double_buffer_pixmap = NULL;
		return;
	}

	skw->double_buffer_pixmap = gdk_pixmap_new(GTK_WIDGET(skw)->window, wx, wy, -1);
	gdk_draw_rectangle(skw->double_buffer_pixmap,
		skw->private_gc[skwc_background], TRUE,
		0, 0, wx, wy);

	skw_draw_line_rects(skw, skw->double_buffer_pixmap, FALSE);
}

static void singit_karaoke_widget_destroy (GtkObject *object)
{
	SingitKaraokeWidget *skw;

#ifdef WIDGET_DEBUG
	g_print("skw_destroy\n");
#endif

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET (object));

	skw = SINGIT_KARAOKE_WIDGET (object);

	// Cleanup connected signals
//	if (skw->adjustment)
//		gtk_signal_disconnect_by_data (GTK_OBJECT(skw->adjustment), skw);

	if (skw->private_font_name != NULL) {
		if (SINGIT_KARAOKE_DATA(skw->skd)->font)
			{ gdk_font_unref(SINGIT_KARAOKE_DATA(skw->skd)->font); }
		g_free(skw->private_font_name);
	}

	gtk_object_unref(GTK_OBJECT(skw->skd));

	if (skw->double_buffer_pixmap != NULL)
		{ gdk_pixmap_unref(skw->double_buffer_pixmap); }

	if (skw->ball_pixmap != NULL)
		{ gdk_pixmap_unref(skw->ball_pixmap); }

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy (object);
}

static void singit_karaoke_widget_finalize (GtkObject *object)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET (object));

#ifdef WIDGET_DEBUG
	g_print("skw_finalize\n");
#endif

	skw = SINGIT_KARAOKE_WIDGET (object);

	if (skw->sizes)
		{ g_free(skw->sizes); }

	if (GTK_OBJECT_CLASS(parent_class)->finalize)
		GTK_OBJECT_CLASS(parent_class)->finalize (object);
}

static void singit_karaoke_widget_realize (GtkWidget *widget)
{
	SingitKaraokeWidget *skw;
	GdkWindowAttr attributes;
	gint attributes_mask;
	gint i;
	gboolean status[skwc_last];

	g_return_if_fail(widget != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(widget));

#ifdef WIDGET_DEBUG
	g_print("skw_realize\n");
#endif

	skw = SINGIT_KARAOKE_WIDGET(widget);

	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
	widget->window = gdk_window_new
		(gtk_widget_get_parent_window (widget), &attributes, attributes_mask);

	widget->style = gtk_style_attach (widget->style, widget->window);

	gdk_window_set_user_data(widget->window, widget);

	gtk_style_set_background (widget->style, widget->window, GTK_STATE_ACTIVE);

	gdk_colormap_alloc_colors(attributes.colormap,
		skw->private_color, skwc_last, FALSE, TRUE, status);

	for (i = 0; i < skwc_last; i++) {
		skw->private_gc[i] = gdk_gc_new(widget->window);
		gdk_gc_set_foreground (skw->private_gc[i], &skw->private_color[i]);
		gdk_gc_set_fill(skw->private_gc[i], GDK_SOLID);
	}

	gdk_window_set_background(widget->window, &skw->private_color[skwc_background]);

	GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);
}

static void singit_karaoke_widget_unrealize(GtkWidget *widget)
{
	SingitKaraokeWidget *skw;
	gint i;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET(widget));

#ifdef WIDGET_DEBUG
	g_print("Unrealize\n");
#endif

	skw = SINGIT_KARAOKE_WIDGET(widget);

	for (i = 0; i < skwc_last; i++) {
		if (skw->private_gc[i] != NULL) {
			gdk_gc_destroy(skw->private_gc[i]);
			skw->private_gc[i] = NULL;
		}
	}

	gdk_colormap_free_colors(gtk_widget_get_colormap(GTK_WIDGET(skw)),
		skw->private_color, skwc_last);

	if (GTK_WIDGET_CLASS (parent_class)->unrealize)
		(* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

static void singit_karaoke_widget_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(widget));
	g_return_if_fail(requisition != NULL);

#ifdef WIDGET_DEBUG
	g_print("skw_size_request\n");
#endif

	skw = SINGIT_KARAOKE_WIDGET(widget);

	if (SINGIT_KARAOKE_DATA(skw->skd)->song) {
		requisition->width = SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width;
		requisition->height = SINGIT_KARAOKE_DATA(skw->skd)->visual_min_height;
	}
	else {
		if (GTK_WIDGET_CLASS(parent_class)->size_request)
			GTK_WIDGET_CLASS(parent_class)->size_request(widget, requisition);
	}

}

static void singit_karaoke_widget_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	SingitKaraokeWidget *skw;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_KARAOKE_WIDGET(widget));
	g_return_if_fail (allocation != NULL);

#ifdef WIDGET_DEBUG
	g_print("skw_size_allocate\n");
#endif

	widget->allocation = *allocation;

	skw = SINGIT_KARAOKE_WIDGET(widget);

	skw->sizes[LEFT_RIGHT_WIDGET_BORDER] =
		MAX(0, (GTK_WIDGET(skw)->allocation.width - SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width) / 2);

	skw->sizes[TOP_BOTTOM_WIDGET_BORDER] =
		MAX(0, (GTK_WIDGET(skw)->allocation.height - SINGIT_KARAOKE_DATA(skw->skd)->visual_min_height) / 2);

	if (GTK_WIDGET_REALIZED (widget))
	{
		gdk_window_move_resize (widget->window,
			allocation->x, allocation->y,
			allocation->width, allocation->height);
	}
}

GtkWidget *singit_karaoke_widget_new (void)
{
	GtkWidget *karaoke_widget;

#ifdef WIDGET_DEBUG
	g_print("skw_new\n");
#endif

	karaoke_widget = gtk_type_new(TYPE_SINGIT_KARAOKE_WIDGET);

	return karaoke_widget;
}

typedef enum {

	XLFD_FOUNDRY        = 0,
	XLFD_FAMILY         = 1,
	XLFD_WEIGHT         = 2,
	XLFD_SLANT          = 3,
	XLFD_SET_WIDTH      = 4,
	XLFD_ADD_STYLE      = 5,
	XLFD_PIXELS         = 6,
	XLFD_POINTS         = 7,
	XLFD_RESOLUTION_X   = 8,
	XLFD_RESOLUTION_Y   = 9,
	XLFD_SPACING        = 10,
	XLFD_AVERAGE_WIDTH  = 11,
	XLFD_CHARSET        = 12
}
FontField;

#define GTK_NUM_STYLE_PROPERTIES 5
#define XLFD_MAX_FIELD_LEN 64

typedef struct _FontStyle {

	guint16  properties[GTK_NUM_STYLE_PROPERTIES];
	gint	 pixel_sizes_index;
	guint16  npixel_sizes;
	gint	 point_sizes_index;
	guint16  npoint_sizes;
	guint8   flags;
}
FontStyle;

static gboolean is_xlfd_font_name (const gchar *fontname)
{
	gint i = 0;
	gint field_len = 0;

	while (*fontname) {
		if (*fontname++ == '-') {
			if (field_len > XLFD_MAX_FIELD_LEN) { return FALSE; }
			field_len = 0;
			i++;
		}
		else
			field_len++;
	}
	return (i == 14) ? TRUE : FALSE;
}

static gchar* get_xlfd_field (const gchar *fontname, FontField field_num, gchar *buffer)
{
	const gchar *t1, *t2;
	gint countdown, len, num_dashes;

	if (!fontname) { return NULL; }

	countdown = field_num;
	t1 = fontname;
	while (*t1 && (countdown >= 0))
		if (*t1++ == '-')
			countdown--;

	num_dashes = (field_num == XLFD_CHARSET) ? 2 : 1;
	for (t2 = t1; *t2; t2++) {
		if (*t2 == '-' && --num_dashes == 0) { break; }
	}

	if (t1 != t2) {
		// Check we don't overflow the buffer
		len = (long) t2 - (long) t1;
		if (len > XLFD_MAX_FIELD_LEN - 1) { return NULL; }
		strncpy (buffer, t1, len);
		buffer[len] = 0;

		// Convert to lower case.
		g_strdown (buffer);
	}
	else
		strcpy(buffer, "(nil)");

	return buffer;
}

/*
 * Returns the line number of the longest line
 */
static inline gint inl_get_max_line_width_nbr (SingitKaraokeWidget *skw, GdkFont *font)
{
	gint i = 0, strlength = 0, temp = 0, result = -1;
	SingitSong *my_song = singit_song_attach(SINGIT_KARAOKE_DATA(skw->skd)->song);

	if (my_song) {
		if (my_song->lyrics) {
			while (my_song->lyrics[i]) {
	  			temp = gdk_string_width(font, my_song->lyrics[i]);
	  			if (temp > strlength) {
					strlength = temp;
					result = i;
				}
				i++;
			}
		}
		singit_song_detach(&my_song);
	}
	return result;
}

typedef struct _TestFontInfo {

	gint      longest_line_nbr;
	gint      font_pixel_size;
	gint      line_width;
	gchar    *base_font_name;
	gchar   **split_font_name;
	guint8    flags;
	gchar    *xfont_name;
}
TestFontInfo;

static void set_optimal_font(SingitKaraokeWidget *skw)
{
	gchar **xfontnames;
	gchar **attrib_list;
	gchar *search_fond_str, *result_font = NULL;
	gint i = 0;
	gint num_fonts;
	gint pixels, points, res_x, res_y;
	gchar field_buffer[XLFD_MAX_FIELD_LEN];
	gchar *field;
	guint8 flags = 0;
	GdkFont *test_font = NULL;
	SingitSong *my_song = NULL;
	gint test_width = 0, test_height = 0;
	TestFontInfo font_info = { -1, 0, 0, NULL, NULL, 0, NULL };
	gchar *font_start, *font_end, *font_temp;
	gchar font_size[5] = { 0 };
	gint switch_it;

	if (skw->private_font_name == NULL) { return; }

	my_song = singit_song_attach(SINGIT_KARAOKE_DATA(skw->skd)->song);
	if (my_song == NULL) { return; }

	attrib_list = g_strsplit(skw->private_font_name, "-", 16);
	while (attrib_list[i]) { i++; }
	if (i < 11) {
		g_strfreev(attrib_list);
		singit_song_detach(&my_song);
		return;
	}
	g_free(attrib_list[7]);
	g_free(attrib_list[8]);
	attrib_list[7] = g_strdup("*");
	attrib_list[8] = g_strdup("*");
	search_fond_str = g_strjoinv("-", attrib_list);
	g_strfreev(attrib_list);

	field = get_xlfd_field (skw->private_font_name, XLFD_PIXELS, field_buffer);
	font_info.font_pixel_size = atoi(field);
	field = get_xlfd_field (skw->private_font_name, XLFD_POINTS, field_buffer);
	points = atoi(field);
	if (!font_info.font_pixel_size) { font_info.font_pixel_size = points * 10; }

	xfontnames = XListFonts (GDK_DISPLAY(), search_fond_str, 1000, &num_fonts);

#ifdef WIDGET_DEBUG
	g_print("\nOld    Font: %s\nSearch Font: %s\nResults :\n",
		skw->private_font_name, search_fond_str);
#endif
	g_free(search_fond_str);

	// Find the best test font by FONT_TYPE
	i = 0;
	while ((i < num_fonts) && !(flags & GTK_FONT_SCALABLE)) {
		if (is_xlfd_font_name(xfontnames[i])) {
			field = get_xlfd_field (xfontnames[i], XLFD_PIXELS, field_buffer);
			pixels = atoi(field);

			field = get_xlfd_field (xfontnames[i], XLFD_POINTS, field_buffer);
			points = atoi(field);

			field = get_xlfd_field (xfontnames[i], XLFD_RESOLUTION_X, field_buffer);
			res_x = atoi(field);

			field = get_xlfd_field (xfontnames[i], XLFD_RESOLUTION_Y, field_buffer);
			res_y = atoi(field);

			if (pixels == 0 && points == 0) {
				if (res_x == 0 && res_y == 0)
					flags = GTK_FONT_SCALABLE;
				else
					flags = GTK_FONT_SCALABLE_BITMAP;
			}
			else
				flags = GTK_FONT_BITMAP;

			if (font_info.flags < flags) {
				g_free(font_info.base_font_name);
				font_info.base_font_name = g_strdup(xfontnames[i]);
				font_info.flags = flags;
				font_info.font_pixel_size = pixels;
			}
		}
		i++;
	}
	XFreeFontNames(xfontnames);

	// Check which size fits best
	if (font_info.base_font_name && !(font_info.flags & GTK_FONT_BITMAP))
	{
		// Split base font at '-' and construct font_start and font_end
		font_info.split_font_name = g_strsplit(font_info.base_font_name, "-", 16);
		font_temp = font_info.split_font_name[7];
		font_info.split_font_name[7] = NULL;
		result_font = font_start = g_strjoinv("-", font_info.split_font_name);
		font_info.split_font_name[7] = font_temp;
		font_start = g_strconcat(result_font, "-", NULL);
		g_free(result_font);

		result_font = g_strjoinv("-", &font_info.split_font_name[8]);
		font_end = g_strconcat("-", result_font, NULL);
		g_free(result_font);

		i = 7;
		do
		{
			g_snprintf(&font_size[0], 5, "%i", i);
			result_font = g_strconcat(font_start, &font_size, font_end, NULL);
			if ((test_font = gdk_font_load(result_font)) != NULL) {

				test_width = font_height(test_font) / 2.5;
				if ((test_width % 2) == 0) { test_width++; }
				test_width = GTK_WIDGET(skw)->allocation.width - test_width;
/*
				test_height = SINGIT_KARAOKE_DATA(skw->skd)->lines * (font_height(test_font) + font_height(test_font) / 6 + 3)
					+ 2 * (font_height(test_font) / 3 + 1);
*/
				test_height = SINGIT_KARAOKE_DATA(skw->skd)->lines * (font_height(test_font) + 2);
				switch_it = (SINGIT_KARAOKE_DATA(skw->skd)->lines > 3) ? 4 : SINGIT_KARAOKE_DATA(skw->skd)->lines;
				switch (switch_it) {
				case 4:
					test_height += (SINGIT_KARAOKE_DATA(skw->skd)->lines - 3) * (font_height(test_font) / 6 + 1);
				case 3:
					test_height += (font_height(test_font) / 3 + 1);
				case 2:
					test_height += (font_height(test_font) / 3 + 1);
				case 1:
					if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
						test_height += font_height(test_font) +
							(font_height(test_font) / 6 + 1);
					}
				}

				if (font_info.longest_line_nbr < 0)
					font_info.longest_line_nbr = inl_get_max_line_width_nbr(skw, test_font);
				font_info.line_width = gdk_string_width
					(test_font, my_song->lyrics[font_info.longest_line_nbr]);
				gdk_font_unref(test_font);

				if (font_info.line_width < test_width)
				{
					g_free(font_info.xfont_name);
					font_info.xfont_name = result_font;
				}
//				g_print("%i: %i / %i\n", switch_it, font_info.line_width, test_height);
			}
			else { g_free(result_font); }
			i++;
		}
		while ((font_info.line_width < test_width) && (test_height < GTK_WIDGET(skw)->allocation.height) && (test_font != NULL) && (i < 500));


		g_strfreev(font_info.split_font_name);
	}
	if (font_info.xfont_name) {
#ifdef WIDGET_DEBUG
		g_print("Size: %i / Width: %i\n", i--, font_info.line_width);
#endif
		singit_karaoke_widget_set_font(skw, font_info.xfont_name);
	}

	g_free(font_info.base_font_name);
	g_free(font_info.xfont_name);

	singit_song_detach(&my_song);
}

/*
 * Substracts two regions (rectangles) and return the intersection and
 * eventually all 4 part rectangles from the first rectangle.
 * The funtion is used to do a full repaint of the widget, if the widget has borders
 * to prevent clear on used areas (doesn't flicker)
 */
static gint substract_region(GdkRectangle full, GdkRectangle part, GdkRectangle *result_top,
	GdkRectangle *result_left, GdkRectangle *result_right, GdkRectangle *result_bottom)
{
	gboolean result = 0;
	if (result_top && (full.y < part.y)) {
		(*result_top).x = full.x;
		(*result_top).y = full.y;
		(*result_top).width = full.width;
		(*result_top).height = part.y - full.y;
		result += 1;
	}
	if (result_left && (full.x < part.x)) {
		(*result_left).x = full.x;
		(*result_left).y = part.y;
		(*result_left).width = part.x - full.x;
		(*result_left).height = full.height + full.y - part.y;
		result += 2;
	}
	if (result_right && (full.x + full.width > part.x + part.width)) {
		(*result_right).x = part.x + part.width;
		(*result_right).y = part.y ;
		(*result_right).width = full.x + full.width - part.x - part.width;
		(*result_right).height = full.height + full.y - part.y;
		result += 4;
	}
	if (result_bottom && (full.y + full.height > part.y + part.height)) {
		(*result_bottom).x = part.x;
		(*result_bottom).y = part.y + part.height;
		(*result_bottom).width = part.width;
		(*result_bottom).height = full.y + full.height - part.y - part.height;
		result += 8;
	}
	return result;
}

static gboolean singit_karaoke_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	SingitKaraokeWidget *skw;
	GdkRectangle intersect_rectangle, widget_rectangle;
	GdkRectangle result_top, result_left, result_right, result_bottom;
	gint draw_sides;

	g_return_val_if_fail(widget != NULL, FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(widget), FALSE);
	g_return_val_if_fail(event != NULL, FALSE);

	skw = SINGIT_KARAOKE_WIDGET (widget);

//	Check if we are ready to draw
	if (!GTK_WIDGET_DRAWABLE(widget) || (skw->double_buffer_pixmap == NULL))
	{
		return FALSE;
	}

	if (skw->freezers > 0)
		{ return FALSE; }

	widget_rectangle.x = skw->sizes[LEFT_RIGHT_WIDGET_BORDER];
	widget_rectangle.y = skw->sizes[TOP_BOTTOM_WIDGET_BORDER];
	widget_rectangle.width = SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width;
	widget_rectangle.height = SINGIT_KARAOKE_DATA(skw->skd)->visual_min_height;

	gdk_rectangle_intersect(&event->area, &widget_rectangle, &intersect_rectangle);

	// Redraw all widget borders if needed
	if ((draw_sides = substract_region(event->area, intersect_rectangle,
		&result_top, &result_left, &result_right, &result_bottom)) != 0) {
		if (draw_sides & (1 << 0)) {
			gdk_window_clear_area(widget->window, result_top.x, result_top.y,
				result_top.width, result_top.height);
		}
		if (draw_sides & (1 << 1)) {
			gdk_window_clear_area(widget->window, result_left.x, result_left.y,
				result_left.width, result_left.height);
		}
		if (draw_sides & (1 << 2)) {
			gdk_window_clear_area(widget->window, result_right.x, result_right.y,
				result_right.width, result_right.height);
		}
		if (draw_sides & (1 << 3)) {
			gdk_window_clear_area(widget->window, result_bottom.x, result_bottom.y,
				result_bottom.width, result_bottom.height);
		}
	}

	// Redraw the real widget
	gdk_draw_pixmap(widget->window, widget->style->fg_gc[widget->state],
		skw->double_buffer_pixmap,
		intersect_rectangle.x - skw->sizes[LEFT_RIGHT_WIDGET_BORDER],
		intersect_rectangle.y - skw->sizes[TOP_BOTTOM_WIDGET_BORDER],
		intersect_rectangle.x,
		intersect_rectangle.y,
		intersect_rectangle.width,
		intersect_rectangle.height);

	return FALSE;
}

/*
 * This creates a new a double_buffer_pixmap dependent on
 * the current song position
 */
void singit_karaoke_widget_set_time (SingitKaraokeWidget *skw, guint time)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));
	g_return_if_fail(IS_SINGIT_KARAOKE_DATA(skw->skd));

	singit_karaoke_data_set_time(SINGIT_KARAOKE_DATA(skw->skd), time);
}

/*
 * Updates the ball part of the double_buffer_pixmap and return the
 * united area which has to be repaintet
 */
void singit_karaoke_widget_update_ball(SingitKaraokeWidget *skw, gint calced_offset)
{
	GdkGC *draw_gc = NULL;
	gint pos_x, height;
	gint xpos[2];

	if ((GTK_WIDGET(skw)->window == NULL) ||
		(skw->double_buffer_pixmap == NULL) ||
		(skw->ball_pixmap == NULL)) { return; }

	draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_bar]);

	pos_x = SINGIT_KARAOKE_DATA(skw->skd)->pbp_start_last +
		SINGIT_KARAOKE_DATA(skw->skd)->pbp_offset_last + calced_offset;
	xpos[0] = xpos[1] = pos_x;
	height = SINGIT_KARAOKE_DATA(skw->skd)->top_lines_height;

	gdk_draw_rectangle(skw->double_buffer_pixmap, draw_gc, TRUE, pos_x,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_y_pos_last + height,
		SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter, SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter);

	if (inl_singit_song_get_next_token(SINGIT_KARAOKE_DATA(skw->skd)->song)) {
		pos_x = SINGIT_KARAOKE_DATA(skw->skd)->pbp_start +
			SINGIT_KARAOKE_DATA(skw->skd)->pbp_offset + calced_offset;
		xpos[1] = pos_x;
		gdk_draw_pixmap(skw->double_buffer_pixmap, draw_gc, skw->ball_pixmap, 0, 0,
			pos_x, SINGIT_KARAOKE_DATA(skw->skd)->ball_y_pos + height,
			SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter, SINGIT_KARAOKE_DATA(skw->skd)->ball_diameter);
	}

	gdk_gc_unref(draw_gc);
}

/*
 * Updates the progress bar part of the double_buffer_pixmap and return the
 * united area which has to be repaintet
 */
gint singit_karaoke_widget_update_progess_bar(SingitKaraokeWidget *skw, gchar *text)
{
	GdkGC *draw_gc = NULL;
	GdkFont *my_font = SKW_FONT;
	gint height;
	gint ranges[4];
	gint string_left_offset;

	if (strlen(text) < 1) { return 0; }
	if (!(GTK_WIDGET(skw)->window && skw->double_buffer_pixmap)) { return 0; }

	height = SINGIT_KARAOKE_DATA(skw->skd)->top_lines_height;
	if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball)
		{ height += (SINGIT_KARAOKE_DATA(skw->skd)->ball_line_height + SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high); }

	string_left_offset = singit_karaoke_data_get_line_offset(SINGIT_KARAOKE_DATA(skw->skd), FALSE);

	ranges[0] = SINGIT_KARAOKE_DATA(skw->skd)->pbp_start_last + SINGIT_KARAOKE_DATA(skw->skd)->pbp_offset_last;
	ranges[1] = SINGIT_KARAOKE_DATA(skw->skd)->pbp_start + SINGIT_KARAOKE_DATA(skw->skd)->pbp_offset;
	ranges[2] = MIN(ranges[0], ranges[1]); // x
	ranges[3] = MAX(ranges[0], ranges[1]) - ranges[2]; // width

	if (ranges[0] > ranges[1]) {
		draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_bar]);
		gdk_draw_rectangle(skw->double_buffer_pixmap,
			draw_gc, TRUE, string_left_offset + ranges[2], height, ranges[3],
			SINGIT_KARAOKE_DATA(skw->skd)->text_line_height);
		gdk_gc_unref(draw_gc);
	}
	else if (ranges[0] < ranges[1]) {
		draw_gc = gdk_gc_ref(skw->private_gc[skwc_progress_bar]);
		gdk_draw_rectangle(skw->double_buffer_pixmap,
			draw_gc, TRUE, string_left_offset + ranges[2], height, ranges[3],
			SINGIT_KARAOKE_DATA(skw->skd)->text_line_height);
		gdk_gc_unref(draw_gc);
	}

	draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_text]);
	gdk_draw_string (skw->double_buffer_pixmap, my_font,
		draw_gc, string_left_offset, height + my_font->ascent + 1, text);
	gdk_gc_unref(draw_gc);

	return string_left_offset;
}

/*
 * Creates all text lines
 */
void singit_karaoke_widget_build_lyric_pixmaps (SingitKaraokeWidget *skw)
{
	GList *item = 0;
	GdkGC *draw_gc = NULL;
	SingitSong *my_song = NULL;
	GdkFont *my_font = SKW_FONT;
	gint i, height;
	gint string_left_offset = left_right_line_border;

	if (!(GTK_WIDGET(skw)->window && skw->double_buffer_pixmap &&
		GTK_WIDGET_REALIZED(GTK_WIDGET(skw)))) { return; }

	draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_bar]);

	skw_draw_line_rects(skw, skw->double_buffer_pixmap, skw->update_background);
	skw->update_background = FALSE;

	my_song = singit_song_attach(SINGIT_KARAOKE_DATA(skw->skd)->song);
	if (my_song == NULL) {
		gdk_gc_unref(draw_gc);
		return;
	}

	if (skw->mode == skwm_line_scroll) {
		item = my_song->active_token;

		i = SINGIT_KARAOKE_DATA(skw->skd)->top_lines;
		height = SINGIT_KARAOKE_DATA(skw->skd)->top_lines_height;
		if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
			height += (SINGIT_KARAOKE_DATA(skw->skd)->ball_line_height +
				SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high);
		}

		if (item) {
			if ((!SINGIT_KARAOKE_DATA(skw->skd)->show_empty_lines) && (singit_song_is_empty_item(my_song, item, FALSE)))
				item = singit_song_find_next_lyric_line(my_song, item, FALSE, NULL);

			if (item) {
				while ((i > 0) && item) {
					item = singit_song_find_prev_lyric_line(my_song, item,
						SINGIT_KARAOKE_DATA(skw->skd)->show_empty_lines, NULL);
					i--;
				}
				if (!item) {
					item = my_song->first_token;
					i++;
				}
				if (i < (gint) SINGIT_KARAOKE_DATA(skw->skd)->top_lines) {
					height = i * (SINGIT_KARAOKE_DATA(skw->skd)->text_line_height +
						SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high);
				}
				else if (i > (gint) SINGIT_KARAOKE_DATA(skw->skd)->top_lines) {
					height = i * (SINGIT_KARAOKE_DATA(skw->skd)->text_line_height +
						SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high) +
						2 * SINGIT_KARAOKE_DATA(skw->skd)->active_line_seperator_high;
					if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
						height += (SINGIT_KARAOKE_DATA(skw->skd)->ball_line_height +
							SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high);
					}
				}
			}
		}
		else {
			item = my_song->first_token;
			if (SINGIT_KARAOKE_DATA(skw->skd)->show_empty_lines) {
				i++;
				height = i * (SINGIT_KARAOKE_DATA(skw->skd)->text_line_height +
					SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high) +
					2 * SINGIT_KARAOKE_DATA(skw->skd)->active_line_seperator_high;
				if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
					height += (SINGIT_KARAOKE_DATA(skw->skd)->ball_line_height +
						SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high);
				}
			}
		}

		gdk_gc_unref(draw_gc);
		draw_gc = gdk_gc_ref(skw->private_gc[skwc_normal_text]);

		while (i < (gint) SINGIT_KARAOKE_DATA(skw->skd)->lines) {
			if (item) {
				string_left_offset = singit_karaoke_data_calc_line_offset
					(SINGIT_KARAOKE_DATA(skw->skd), FALSE, tText(my_song, item));
				gdk_draw_string (skw->double_buffer_pixmap, my_font,
					draw_gc, string_left_offset, height + my_font->ascent + 1,
					tText(my_song, item));
				item = singit_song_find_next_lyric_line(my_song, item,
					SINGIT_KARAOKE_DATA(skw->skd)->show_empty_lines, NULL);
			}
			if (i == (gint) (SINGIT_KARAOKE_DATA(skw->skd)->top_lines - 1)) {
				height += SINGIT_KARAOKE_DATA(skw->skd)->active_line_seperator_high;
				if (SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {
					height += (SINGIT_KARAOKE_DATA(skw->skd)->ball_line_height +
						SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high);
				}
			}
			else if (i == (gint) SINGIT_KARAOKE_DATA(skw->skd)->top_lines) {
				height += SINGIT_KARAOKE_DATA(skw->skd)->active_line_seperator_high;
			}

			i++;
			height += (SINGIT_KARAOKE_DATA(skw->skd)->text_line_height +
				SINGIT_KARAOKE_DATA(skw->skd)->line_seperator_high);
		}
	}

	singit_song_detach(&my_song);
	gdk_gc_unref(draw_gc);
}

static void singit_karaoke_widget_refresh (SingitKaraokeWidget *skw)
{
	if (skw->freezers != 0) { return; }
	if (!SINGIT_KARAOKE_DATA(skw->skd)->song) { return; }

#ifdef WIDGET_DEBUG
	g_print("skw_refresh\n");
#endif


	singit_karaoke_widget_build_lyric_pixmaps(skw);
	if (SINGIT_KARAOKE_DATA(skw->skd)->song &&
		SINGIT_KARAOKE_DATA(skw->skd)->song->active_token)
	{
		singit_karaoke_widget_update_progess_bar
			(skw, tText(SINGIT_KARAOKE_DATA(skw->skd)->song,
			SINGIT_KARAOKE_DATA(skw->skd)->song->active_token));
	}

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(skw)))
		gtk_widget_set_usize((GtkWidget*) skw,
			SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width,
			SINGIT_KARAOKE_DATA(skw->skd)->visual_min_height);
}


/*****************************
 *
 * API - Setters and Getters
 *
 *****************************/

void singit_karaoke_widget_set_song (SingitKaraokeWidget *skw, SingitSong *new_song)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_song\n");
#endif

	singit_karaoke_data_set_song(SINGIT_KARAOKE_DATA(skw->skd), new_song);
}

void singit_karaoke_widget_set_lines(SingitKaraokeWidget *skw, guint value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_lines\n");
#endif

	singit_karaoke_data_set_lines(SINGIT_KARAOKE_DATA(skw->skd), value);
}

guint singit_karaoke_widget_get_lines(SingitKaraokeWidget *skw)
{
#ifdef WIDGET_DEBUG
	g_print("skw_get_lines\n");
#endif

	g_return_val_if_fail(skw != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), 0);

	return SINGIT_KARAOKE_DATA(skw->skd)->lines;
}

void singit_karaoke_widget_set_toplines(SingitKaraokeWidget *skw, guint value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_toplines\n");
#endif

	singit_karaoke_data_set_toplines(SINGIT_KARAOKE_DATA(skw->skd), value);
}

guint singit_karaoke_widget_get_toplines(SingitKaraokeWidget *skw)
{
#ifdef WIDGET_DEBUG
	g_print("skw_get_toplines\n");
#endif

	g_return_val_if_fail(skw != NULL, 0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), 0);

	return SINGIT_KARAOKE_DATA(skw->skd)->top_lines;
}

void singit_karaoke_widget_set_color
	(SingitKaraokeWidget *skw, SingitKaraokeWidgetColor item, gchar *color)
{
	GdkColor new_color;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));
	g_return_if_fail(item < skwc_last);
	g_return_if_fail(color != NULL);

#ifdef WIDGET_DEBUG
	g_print("skw_set_color : ");
#endif

	if (gdk_color_parse(color, &new_color) == TRUE) {
#ifdef WIDGET_DEBUG
		g_print("Ok\n");
#endif

		// If the color is equal we don't have to change it
		if (gdk_color_equal(&new_color, &skw->private_color[item]) == TRUE)
			{ return; }

		if (GTK_WIDGET_REALIZED(GTK_WIDGET(skw))) {
			gdk_colormap_free_colors(gtk_widget_get_colormap(GTK_WIDGET(skw)),
				&skw->private_color[item], 1);
			gdk_colormap_alloc_color(gtk_widget_get_colormap(GTK_WIDGET(skw)),
				&new_color, FALSE, TRUE);

			if (item == skwc_background)
				{ gdk_window_set_background(GTK_WIDGET(skw)->window, &new_color); }

			gdk_gc_set_foreground(skw->private_gc[item], &new_color);
		}

		skw->private_color[item] = new_color;

		if (item == skwc_background)
			{ skw->update_background = TRUE; }
	}
#ifdef WIDGET_DEBUG
	else {
		g_print("Failed");
	}
#endif
}

void singit_karaoke_widget_set_color_gdk
	(SingitKaraokeWidget *skw, SingitKaraokeWidgetColor item, GdkColor *new_color)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));
	g_return_if_fail(item < skwc_last);
	g_return_if_fail(new_color != NULL);

#ifdef WIDGET_DEBUG
	g_print("skw_set_color_gdk\n");
#endif

	// If the color is equal we don't have to change it
	if (gdk_color_equal(new_color, &skw->private_color[item]) == TRUE)
		{ return; }

	if (GTK_WIDGET_REALIZED(GTK_WIDGET(skw))) {
		gdk_colormap_free_colors(gtk_widget_get_colormap(GTK_WIDGET(skw)),
			&skw->private_color[item], 1);
		gdk_colormap_alloc_color(gtk_widget_get_colormap(GTK_WIDGET(skw)),
			new_color, FALSE, TRUE);

		if (item == skwc_background)
			{ gdk_window_set_background(GTK_WIDGET(skw)->window, new_color); }

		gdk_gc_set_foreground(skw->private_gc[item], new_color);
	}

	skw->private_color[item] = (*new_color);

	if (item == skwc_background)
		{ skw->update_background = TRUE; }
}

void singit_karaoke_widget_set_font (SingitKaraokeWidget *skw, const gchar *font)
{
	GdkFont *new_font = NULL;
	SingitKaraokeData *skd;

	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_font : ");
#endif

	g_return_if_fail(font != NULL);

	if ((new_font = gdk_font_load(font)) != NULL) {
		skd = SINGIT_KARAOKE_DATA(skw->skd);

		if (skd->font) { gdk_font_unref((GdkFont*) (skd->font)); }
		singit_karaoke_data_set_font(skd, (gpointer) new_font,
			new_font->ascent + new_font->descent);

		if (skw->private_font_name != NULL)
			g_free(skw->private_font_name);
		skw->private_font_name = g_strdup(font);

#ifdef WIDGET_DEBUG
		g_print("Ok\n");
#endif
	}
#ifdef WIDGET_DEBUG
	else {
		g_print("Failed\n");
	}
#endif
}

void singit_karaoke_widget_set_show_empty_lines (SingitKaraokeWidget *skw, const gboolean value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_show_empty_lines\n");
#endif

	if (value != SINGIT_KARAOKE_DATA(skw->skd)->show_empty_lines) {

		SINGIT_KARAOKE_DATA(skw->skd)->show_empty_lines = value;
	}
}

void singit_karaoke_widget_set_jumping_ball (SingitKaraokeWidget *skw, const gboolean value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_jumping_ball\n");
#endif

	if (value != SINGIT_KARAOKE_DATA(skw->skd)->use_ball) {

		SINGIT_KARAOKE_DATA(skw->skd)->use_ball = value;
		singit_karaoke_widget_refresh(skw);
	}
}

void singit_karaoke_widget_freeze(SingitKaraokeWidget *skw)
{
	skw->freezers++;
	singit_karaoke_data_freeze(SINGIT_KARAOKE_DATA(skw->skd));
}

void singit_karaoke_widget_thaw(SingitKaraokeWidget *skw)
{
	if (skw->freezers > 0) { skw->freezers--; }
	singit_karaoke_data_thaw(SINGIT_KARAOKE_DATA(skw->skd));
	if (skw->freezers == 0) { singit_karaoke_widget_refresh(skw); }
}

void singit_karaoke_widget_set_optimal_font (SingitKaraokeWidget *skw)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_optimal_font\n");
#endif

	if ((SINGIT_KARAOKE_DATA(skw->skd)->song == NULL) ||
		(singit_song_lyrics_found(SINGIT_KARAOKE_DATA(skw->skd)->song) == FALSE))
		{ return; }

	singit_karaoke_widget_freeze(skw);
	set_optimal_font(skw);
	singit_karaoke_widget_thaw(skw);
}

void singit_karaoke_widget_set_centered_lines (SingitKaraokeWidget *skw, gboolean value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_centered_lines\n");
#endif

	if (value != SINGIT_KARAOKE_DATA(skw->skd)->centerLines) {
		singit_karaoke_data_set_centered_lines
			(SINGIT_KARAOKE_DATA(skw->skd), value);
		singit_karaoke_widget_refresh(skw);
	}
}

gboolean singit_karaoke_widget_get_centered_lines (SingitKaraokeWidget *skw)
{
	g_return_val_if_fail((skw != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), FALSE);

#ifdef WIDGET_DEBUG
	g_print("skw_get_centered_lines\n");
#endif

	return SINGIT_KARAOKE_DATA(skw->skd)->centerLines;
}

gdouble singit_karaoke_widget_get_frames_per_second (SingitKaraokeWidget *skw)
{
#ifdef WIDGET_DEBUG
	g_print("skw_get_frames_per_second\n");
#endif

	g_return_val_if_fail(skw != NULL, 0.0);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), 0.0);

	return skw->fps;
}

void singit_karaoke_widget_set_time_indicators (SingitKaraokeWidget *skw, gboolean value)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

#ifdef WIDGET_DEBUG
	g_print("skw_set_time_indicators\n");
#endif

	if (value != skw->timeIndicators) {
		skw->timeIndicators = value;
		singit_karaoke_widget_refresh(skw);
	}
}

gboolean singit_karaoke_widget_get_time_indicators (SingitKaraokeWidget *skw)
{
	g_return_val_if_fail((skw != NULL), FALSE);
	g_return_val_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw), FALSE);

#ifdef WIDGET_DEBUG
	g_print("skw_get_time_indicators\n");
#endif

	return skw->timeIndicators;
}

void singit_karaoke_widget_get_minimal_sizes(SingitKaraokeWidget *skw, gint *width, gint *height)
{
	g_return_if_fail(skw != NULL);
	g_return_if_fail(IS_SINGIT_KARAOKE_WIDGET(skw));

	if (width != NULL) { *width = SINGIT_KARAOKE_DATA(skw->skd)->visual_min_width; }
	if (height != NULL) { *height = SINGIT_KARAOKE_DATA(skw->skd)->visual_min_height; }
}
