#include "MemFileSysDir.h"

#include "XLongList.h"


#define		MEM_FILE_SYS_HT_DUPE	0x1


MemFileSysDir::MemFileSysDir() :
	Hashtable() {
	
}


void MemFileSysDir::Catalog( XLongList& outList ) {
	long i;
	KEntry*	entry;
	
	outList.Dim( outList.Count() + NumEntries() );
	
	// Step thru the dict table values...
	for ( i = 0; i < mTableSize; i++ ) {
		for ( entry = mTable[ i ]; entry; entry = entry -> mNext ) {
		
			// If we encounter a dupe list, put each dupe ID in the outList
			if ( entry -> mFlags & MEM_FILE_SYS_HT_DUPE )
				GetValues( outList );
			
			// Proceed to delete the sub item from the FileSys
			else
				outList.Add( (FileObj) entry -> mValue );
		}
	}
}






long MemFileSysDir::LookupName( const UtilStr* inName, FileObj& outID ) {
	long result, flags;
	
	// Lookup the flags and item ID in O(1) time...
	flags = Get( inName, (void**) &outID );
	
	// Flags is non-zero if the key was found
	if ( flags ) {
		
		// This flags tells us outID is a ptr to a hashtable containing our dupe list
		// MEM_FILE_SYS_HT_DUPE is set only if there's more then one item in this folder w/ the given name.
		if ( flags & MEM_FILE_SYS_HT_DUPE ) {
			Hashtable* dupeList = (Hashtable*) outID;

			// The spec for LookupName says we have to return any file with that name--it doesn't matter which			
			outID = (FileObj) dupeList -> GetValue( 0 );
			result = MEM_FILE_SYS_DUPE; }
		else
			result = MEM_FILE_SYS_UNIQUE;
		}
	else
		result = MEM_FILE_SYS_FNF;
		
	return result;	
}




const UtilStr* MemFileSysDir::AddName( const UtilStr* inName, FileObj inID ) {
	long flags;
	FileObj outID;
	Hashtable* dupeList;
	const UtilStr* retName;
	
	// Lookup the flags and item ID in O(1) time...
	flags = Get( inName, (void**) &outID );

	// If the title doesn't already exist...
	if ( flags == 0 ) {
		
		// Put it in the title lookup table and we're done!
		retName = new UtilStr( inName );
		Put( retName, (void*) inID, HT_KEY_OWNED ); }

	// If one (or more) items already share this name...		
	else {
			
		// We use a hashtable to keep track of what IDs share this name
		if ( flags & MEM_FILE_SYS_HT_DUPE )
	
			// MEM_FILE_SYS_HT_DUPE tells us we've alredy allocated a hashtable
			dupeList = (Hashtable*) outID;
	
		// If one (and only one) item shared the given name (two or more would have had the MEM_FILE_SYS_HT_DUPE flag set)
		else {
	
			// Start a new dupe list and store it in the title lookup table under the dupe name
			// Also flag that the value for the name is a dupe list (vs. a file item ID)
			dupeList = new Hashtable;
			PutOwned( inName, dupeList, MEM_FILE_SYS_HT_DUPE );

			// Put the item that had the same name into out dupe list
			dupeList -> Put( outID, (void*) outID, HT_NO_FLAGS ); 
		}
		
		// Put the new item in the dupe list		
		dupeList -> Put( inID, (void*) inID, HT_NO_FLAGS );

		// We've already allocated a key that matches the given name so return that as the name ref
		retName = (const UtilStr*) GetKey( inName );
	}
	
	// The ptr returned is what ptr file item is going to store as the name b/c
	// it's always guarunteed to exist while the file is around.
	return retName;
}



void MemFileSysDir::RemoveName( const UtilStr* inName, FileObj inID ) {
	long flags;
	FileObj itemID;
	Hashtable* dupeList;

	// Lookup the flags and item ID in O(1) time...
	flags = Get( inName, (void**) &itemID );
	
	// If two (or more) items already share this name...	
	if ( flags & MEM_FILE_SYS_HT_DUPE ) {
	
		// MEM_FILE_SYS_HT_DUPE means there's more than one item with the given name...
		dupeList = (Hashtable*) itemID;
		dupeList -> Remove( inID );
		
		// If we no longer have duplicate named items, do away with the dupe list
		if ( dupeList -> NumEntries() == 1 ) {
		
			// Get the ID of the last remaining item
			itemID = (FileObj) dupeList -> GetValue( 0 );
			
			// This will clear the MEM_FILE_SYS_HT_DUPE flag and will delete dupeList (b/c it's owned)
			Put( inName, (void*) itemID, HT_NO_FLAGS );
		} }
		
	// We know there was just one item with that name so we chuck it (this will also delete the UtilStr holding the name)
	else
		Remove( inName );
}





void MemFileSysDir::RemoveAll( MemFileSys* inFileSys ) {
	long i;
	KEntry*	entry;
	
	// Step thru the dict table and delete all file items from the FileSys...
	for ( i = 0; i < mTableSize; i++ ) {
		for ( entry = mTable[ i ]; entry; entry = entry -> mNext ) {
		
			// If we encounter a dupe list, remove each item in the list from the FileSys
			if ( entry -> mFlags & MEM_FILE_SYS_HT_DUPE )
				( (MemFileSysDupes*) entry -> mValue ) -> DeleteSubsFromFileSys( inFileSys );
			
			// If it was a uniquely named item....
			else
				// Go to MemFileSys to delete that item and its subs, but prevent 
				inFileSys -> internalDelete( (FileObj) entry -> mValue, false );
		}
	}
	
	// Since all the sub's names and any dupe tables are owned, this will delete all the names and any dupe tables used
	Hashtable::RemoveAll();
}




void MemFileSysDupes::DeleteSubsFromFileSys( MemFileSys* inFileSys ) {
	long i;
	KEntry*	entry;
	
	// Step thru the dict table and delete all file items from the FileSys...
	for ( i = 0; i < mTableSize; i++ ) {
		for ( entry = mTable[ i ]; entry; entry = entry -> mNext ) {
		
			// Go to MemFileSys to delete that item and its subs
			inFileSys -> internalDelete( (FileObj) entry -> mValue, false );
		}
	}
}