#include <string.h>
#include <gtk/gtk.h>
#include <xmms/configfile.h>

#include "config.h"
#include "infconfig.h"
#include "gettext.h"
#include "config-dialog.h"


#define DEFAULT_WIDTH 320
#define DEFAULT_HEIGHT 240
#define DEFAULT_TIME_EFFECT 100
#define DEFAULT_TIME_PALETTE 100
#define DEFAULT_SCALE_FACTOR 1
#define DEFAULT_FPS 30
#define DEFAULT_SHOW_TITLE TRUE


typedef struct t_config {
    gint32 x, y;
    gint32 xres;
    gint32 yres;
    gint32 sres;
    gint32 teff;
    gint32 tcol;
    gint32 fps;
    gboolean show_title;
} t_config;


static t_config config = {-1, -1, 0, 0, 0, 0, 0, 0, DEFAULT_SHOW_TITLE};
static ConfigDialog *configure_dialog = NULL;

/*
 * Here we save values selected on configure dialog, but which
 * have not been yet applied, so on cancel click we just forget
 * them.
 */
static t_config temp_config;

/*
 *
 * Private functions
 *
 */


/*
 * Callbacks for config_plugin_config_window
 */

static void scale_value_changed(GtkWidget *widget, gpointer data)
{
    temp_config.sres = ((gint32)(GTK_ADJUSTMENT(widget)->value));
}

static void teff_value_changed(GtkWidget *widget, gpointer data)
{
    temp_config.teff = ((gint32)(GTK_ADJUSTMENT(widget)->value));
}

static void tcol_value_changed(GtkWidget *widget, gpointer data)
{
    temp_config.tcol = ((gint32)(GTK_ADJUSTMENT(widget)->value));
}

static void on_fps_changed (GtkWidget *widget, gpointer data)
{
    gint32 value;

    value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(configure_dialog->spinbutton_fps));
    temp_config.fps = value;
    g_message ("fps changed to %d", value);
}

static void on_show_title_toggled (GtkWidget *widget, gpointer data)
{
    temp_config.show_title = !temp_config.show_title;
}

static gboolean check_cfg_version (ConfigFile *f)
{
    gchar *vstr;

    if (xmms_cfg_read_string (f, "infinity", "version", &vstr))
	if (!strcmp (vstr, PACKAGE_VERSION))
	    return TRUE;
    return FALSE;
}

static void sync_options ()
{
  /*g_message ("Synchronizing options: sres %d, teff %d, tcol %d, fps %d",
    temp_config.sres, temp_config.teff, temp_config.tcol, temp_config.fps);*/
    config_set_sres (temp_config.sres);
    config_set_teff (temp_config.teff);
    config_set_tcol (temp_config.tcol);
    config_set_fps (temp_config.fps);
    config_set_show_title (temp_config.show_title);
}

static void apply_clicked (GtkWidget *w, GtkWidget *window)
{
    g_message ("Applying changes ...");
    sync_options ();
    config_plugin_save_prefs();
}

static void ok_clicked (GtkWidget *w, GtkWidget *window)
{
    apply_clicked (w, window);
    config_dialog_hide (configure_dialog);
}

static void cancel_clicked (GtkWidget *w, GtkWidget *window)
{
    g_message ("Cancelling changes...");
    config_dialog_hide (configure_dialog);
}

static void on_config_dialog_destroy (GtkWidget *widget, gpointer data)
{
    config_dialog_destroy (configure_dialog);
    configure_dialog = NULL;
}

/*
 * This function set widgets on configure dialog according with
 * saved values. This way after cancel on a configure dialog, if
 * we reopen it we get the saved values.
 */
static void set_config_values (ConfigDialog *config_dialog)
{
    GtkAdjustment *teadj, *tcadj, *scale_adj;

    g_return_if_fail (configure_dialog != NULL);

    teadj = gtk_range_get_adjustment (GTK_RANGE(config_dialog->hscale_effect));
    gtk_adjustment_set_value (teadj, config_get_teff());

    tcadj = gtk_range_get_adjustment (GTK_RANGE(config_dialog->hscale_palette));
    gtk_adjustment_set_value (tcadj, config_get_tcol());

    scale_adj = gtk_range_get_adjustment (GTK_RANGE(config_dialog->hscale_scale_factor));    
    gtk_adjustment_set_value (scale_adj, config_get_sres());

    gtk_spin_button_set_value (GTK_SPIN_BUTTON(config_dialog->spinbutton_fps), config_get_fps());

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(config_dialog->checkbutton_show_title), config_get_show_title());

    /* Memorize current values */
    temp_config.sres = config_get_sres();
    temp_config.teff = config_get_teff();
    temp_config.tcol = config_get_tcol();
    temp_config.fps = config_get_fps();
    temp_config.show_title = config_get_show_title();
}

static void connect_callbacks (ConfigDialog *configure_dialog)
{
    GtkAdjustment *teadj, *tcadj, *scale_adj;

    g_return_if_fail (configure_dialog != NULL);

    gtk_signal_connect (GTK_OBJECT(configure_dialog->window_main), "destroy", 
                        GTK_SIGNAL_FUNC(on_config_dialog_destroy), NULL);

    teadj = gtk_range_get_adjustment (GTK_RANGE(configure_dialog->hscale_effect));
    tcadj = gtk_range_get_adjustment (GTK_RANGE(configure_dialog->hscale_palette));
    scale_adj = gtk_range_get_adjustment (GTK_RANGE(configure_dialog->hscale_scale_factor));

    gtk_signal_connect (GTK_OBJECT(teadj), "value-changed", 
                       GTK_SIGNAL_FUNC(teff_value_changed), NULL);
    gtk_signal_connect (GTK_OBJECT(tcadj), "value-changed", 
                       GTK_SIGNAL_FUNC(tcol_value_changed), NULL);
    gtk_signal_connect (GTK_OBJECT(scale_adj), "value-changed", 
                       GTK_SIGNAL_FUNC(scale_value_changed), NULL);
    gtk_signal_connect (GTK_OBJECT(configure_dialog->spinbutton_fps), "changed", 
                       GTK_SIGNAL_FUNC(on_fps_changed), NULL);
    gtk_signal_connect (GTK_OBJECT(configure_dialog->checkbutton_show_title), "toggled", 
                       GTK_SIGNAL_FUNC(on_show_title_toggled), NULL);

    gtk_signal_connect (GTK_OBJECT(configure_dialog->button_ok), "clicked",
                        GTK_SIGNAL_FUNC(ok_clicked), NULL);
    gtk_signal_connect (GTK_OBJECT(configure_dialog->button_cancel), "clicked",
                        GTK_SIGNAL_FUNC(cancel_clicked), NULL);
    gtk_signal_connect (GTK_OBJECT(configure_dialog->button_apply), "clicked",
                        GTK_SIGNAL_FUNC(apply_clicked), NULL);
}

/*
 *
 * Public functions
 *
 */


void config_plugin_load_prefs (void)
{
    gint32 value;
    gboolean bvalue;
    ConfigFile *f;
    gboolean error, must_update;

    /* FIXME finish this */
    error = must_update = FALSE;
    if ((f = xmms_cfg_open_default_file())) {
        if (check_cfg_version (f)) {
            g_message("version Ok");
            if (xmms_cfg_read_int (f, "infinity", "xres", &value))
                config_set_xres (value);
            else {
                config_set_xres (DEFAULT_WIDTH);
                error = TRUE;
            }

            if (xmms_cfg_read_int (f, "infinity", "yres", &value))
                config_set_yres (value);
            else {
                config_set_yres (DEFAULT_HEIGHT);
                error = TRUE;
            }

            if (xmms_cfg_read_int (f, "infinity", "teff", &value))
                config_set_teff (value);
            else {
                config_set_teff (DEFAULT_TIME_EFFECT);
                error = TRUE;
            }

            if (xmms_cfg_read_int (f, "infinity", "tcol", &value))
                config_set_tcol (value);
            else {
                config_set_tcol (DEFAULT_TIME_PALETTE);
                error = TRUE;
            }

            if (xmms_cfg_read_int (f, "infinity", "sres", &value))
                config_set_sres (value);
            else {
                config_set_sres (DEFAULT_SCALE_FACTOR);
                error = TRUE;
            }

            if (xmms_cfg_read_int (f, "infinity", "fps", &value))
                config_set_fps (value);
            else {
                config_set_fps (DEFAULT_FPS);
                error = TRUE;
            }

            if (xmms_cfg_read_boolean (f, "infinity", "show_title", &bvalue))
                config_set_show_title (bvalue);
            else {
                config_set_show_title (DEFAULT_SHOW_TITLE);
                error = TRUE;
            }

            /* FIXME for now these are not used */
            xmms_cfg_read_int (f, "infinity", "xorig", &value);
            config_set_x (value);
            xmms_cfg_read_int (f, "infinity", "yorig", &value);
            config_set_y (value);
        } else {
            config_set_default_values ();
            must_update = TRUE;
        }
        xmms_cfg_free (f);
    } else {
        config_set_default_values ();
    }

    if (error || must_update) {
        if (error)
            g_message ("Fixing errors on Infinity's configure file...");
        else
            g_message ("Updating Infinity's configure file...");

        config_plugin_save_prefs();
    }
}


void config_plugin_save_prefs (void)
{
    gchar *name;
    gint32 orig;
    ConfigFile *f;
    
    name = g_strconcat (g_get_home_dir(), "/.xmms/config", 0);
    if	(!(f=xmms_cfg_open_file (name)))
        f = xmms_cfg_new();
    xmms_cfg_write_string (f, "infinity", "version", PACKAGE_VERSION);
    if (!config_is_initialized ())
	config_set_default_values ();
    xmms_cfg_write_int (f, "infinity", "xres", config_get_xres());
    xmms_cfg_write_int (f, "infinity", "yres", config_get_yres());
    xmms_cfg_write_int (f, "infinity", "teff", config_get_teff());
    xmms_cfg_write_int (f, "infinity", "tcol", config_get_tcol());
    xmms_cfg_write_int (f, "infinity", "sres", config_get_sres());
    xmms_cfg_write_int (f, "infinity", "fps", config_get_fps());
    xmms_cfg_write_boolean (f, "infinity", "show_title", config_get_show_title());
    orig = (config.x < 0 ? -1 : config.x);
    xmms_cfg_write_int (f, "infinity", "xorig", orig);
    orig = (config.y < 0 ? -1 : config.y);
    xmms_cfg_write_int (f, "infinity", "yorig", orig);
    xmms_cfg_write_file (f, name);
    xmms_cfg_free (f);
    g_free (name);
}


void config_set_default_values (void)
{
    config_set_x (20);
    config_set_y (10);
    config_set_xres (DEFAULT_WIDTH);
    config_set_yres (DEFAULT_HEIGHT);
    config_set_teff (DEFAULT_TIME_EFFECT);
    config_set_tcol (DEFAULT_TIME_PALETTE);
    config_set_sres (DEFAULT_SCALE_FACTOR);
    config_set_fps (DEFAULT_FPS);
    config_set_show_title (DEFAULT_SHOW_TITLE);
}


gboolean config_is_initialized (void)
{
    /* FIXME mhm, ugly thing... */
    return ((config.x > -1) && (config.y > -1) &&
        (config.xres != 0) && (config.yres != 0) && 
        (config.sres != 0) && (config.teff != 0) &&
        (config.tcol != 0) && (config.fps != 0));
}


void config_plugin_config_window (void)
{
#if ENABLE_NLS
    setlocale (LC_MESSAGES, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#endif

    if (configure_dialog) {
        set_config_values (configure_dialog);
        config_dialog_show (configure_dialog);
        return;
    }

    config_plugin_load_prefs ();
    if (!config_is_initialized ())
        config_set_default_values ();

    configure_dialog = config_dialog_new ();

    set_config_values (configure_dialog);

    connect_callbacks (configure_dialog);

    config_dialog_show (configure_dialog);
}

void config_set_x (gint32 value)
{
    config.x = value;
}

void config_set_y (gint32 value)
{
    config.y = value;
}

void config_set_xres (gint32 value)
{
    config.xres = value;
}

void config_set_yres (gint32 value)
{
    config.yres = value;
}

void config_set_teff (gint32 value)
{
    config.teff = value;
}

void config_set_tcol (gint32 value)
{
    config.tcol = value;
}

void config_set_sres (gint32 value)
{
    config.sres = value;
}

void config_set_fps (gint32 value)
{
    config.fps = value;
}

void config_set_show_title (gboolean value)
{
    config.show_title = value;
}

gint32 config_get_x (void)
{
    return config.x;
}

gint32 config_get_y (void)
{
    return config.y;
}

gint32 config_get_xres (void)
{
    return config.xres;
}

gint32 config_get_yres (void)
{
    return config.yres;
}

gint32 config_get_teff (void)
{
    return config.teff;
}

gint32 config_get_tcol (void)
{
    return config.tcol;
}

gint32 config_get_sres (void)
{
    return config.sres;
}

gint32 config_get_fps (void)
{
    return config.fps;
}

gboolean config_get_show_title (void)
{
    return config.show_title;
}
