/* 
 * Copyright (C) 2000, 2001 the xine project, 
 *                          Rich Wareham <richwareham@users.sourceforge.net>
 * 
 * This file is part of xine, a unix video player.
 * 
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: input_dvdnav.c,v 1.112 2002/01/13 22:16:07 jcdutton Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* Standard includes */
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <errno.h>

/* Xine includes */
#include <xine/xineutils.h>
#include <xine/buffer.h>
#include <xine/input_plugin.h>
#include <xine/video_out.h>
#include <xine/events.h>
#include <xine/spu_decoder_api.h>
#include <xine/xine_internal.h>

/* DVDNAV includes */
#include "libdvdnav/dvdnav.h"
#include "extra_ifo_types.h"
 
/* libdvdread includes */
#include <dvdread/nav_read.h>

/* Print debug messages? */
#define INPUT_DEBUG 1

/* Print trace messages? */
#define INPUT_DEBUG_TRACE 0

/* Current play mode (title only or menus?) */
#define MODE_NAVIGATE 0
#define MODE_TITLE 1

/* Is seeking enabled? 1 - Yes, 0 - No */
#define CAN_SEEK 0

/* The default DVD device on Solaris is not /dev/dvd */
#if defined(__sun)
#define DVD_PATH "/vol/dev/aliases/cdrom0"
#else
#define DVD_PATH "/dev/dvd"
#endif 

/* Some misc. defines */
#define DVD_BLOCK_SIZE 2048
#ifndef BUF_DEMUX_BLOCK
#define BUF_DEMUX_BLOCK 0x05000000
#endif
#define VIDEO_FILL_THROTTLE 5

/* Debugging macros */
#if INPUT_DEBUG
#define dprint(s, args...) fprintf(stderr, __FUNCTION__ ": " s, ##args);
#else
#define dprint(s, args...) /* Nowt */
#endif

#if INPUT_DEBUG_TRACE
#define trace_print(s, args...) fprintf(stdout, __FUNCTION__ ": " s, ##args);
#else
#define trace_print(s, args...) /* Nothing */
#endif

/* Globals */
extern int errno;

/* Array to hold MRLs returned by get_autoplay_list */
#define MAX_DIR_ENTRIES 250
#define MAX_STR_LEN     255  
char    filelist[MAX_DIR_ENTRIES][MAX_STR_LEN];
char   *filelist2[MAX_DIR_ENTRIES];

/* A Temporary string (FIXME: May cause problems if multiple
 * dvdnavs in multiple threads). */
char    temp_str[256];
#define TEMP_STR_LEN 255

typedef struct {
  input_plugin_t    input_plugin; /* Parent input plugin type        */

  int               pause_timer;  /* Cell stil-time timer            */
  int               pause_counter;
  time_t	    pause_end_time;

  /* Flags */
  int               opened;       /* 1 if the DVD device is already open */
  
  /* Xine specific variables */
  config_values_t  *config;       /* Pointer to XineRC config file   */  
  char		   *dvd_device;	  /* Default DVD device		     */
  char             *mrl;          /* Current MRL                     */
  int               mode;

  dvdnav_t         *dvdnav;       /* Handle for libdvdnav            */
  xine_t           *xine;         
} dvdnav_input_plugin_t;

static uint32_t dvdnav_plugin_get_capabilities (input_plugin_t *this_gen) {
  trace_print("Called\n");

  return INPUT_CAP_AUTOPLAY | INPUT_CAP_BLOCK | INPUT_CAP_CLUT |
#if CAN_SEEK
         INPUT_CAP_SEEKABLE | INPUT_CAP_VARIABLE_BITRATE | 
#endif
         INPUT_CAP_AUDIOLANG | INPUT_CAP_SPULANG; 
}

void region_changed_cb(void *this_gen, cfg_entry_t *entry) {
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen;

  if(!this)
   return;

  if(!this->dvdnav)
   return;

  if((entry->num_value >= 1) && (entry->num_value <= 8)) {
    /* FIXME: Remove debug message */
    printf("Setting region code to %i (0x%x)\n",
	   entry->num_value, 1<<(entry->num_value-1));
    dvdnav_set_region_mask(this->dvdnav, 1<<(entry->num_value-1));
  }
}

/*
 * Opens the DVD plugin. The MRL takes the following form:
 *
 * dvdnav://[dvd_path][:vts[.program]]
 *
 * e.g.
 *   dvdnav://                    - Play (navigate) /dev/dvd
 *   dvdnav:///dev/dvd2           - Play (navigate) /dev/dvd2
 *   dvdnav:///dev/dvd2:1         - Play Title 1 from /dev/dvd2
 *   dvdnav://:1.3                - Play Title 1, program 3 from /dev/dvd
 */
static int dvdnav_plugin_open (input_plugin_t *this_gen, char *mrl) {
  char                  *locator;
  int                    colon_point;
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t *) this_gen;
  dvdnav_status_t ret;
  cfg_entry_t *region_entry;
    
  trace_print("Called\n");
  
  this->dvdnav                 = NULL;
  this->mrl                    = mrl;
  this->pause_timer            = 0;

  /* Check we can handle this MRL */
  if (!strncasecmp (mrl, "dvdnav://",9))
    locator = &mrl[9];
  else {
    return 0;
  }

  /* Attempt to parse MRL */
  colon_point=0;
  while((locator[colon_point] != '\0') && (locator[colon_point] != ':')) {
    colon_point++;
  }

  if(locator[colon_point] == ':') {
    this->mode = MODE_TITLE; 
  } else {
    this->mode = MODE_NAVIGATE;
  }

  locator[colon_point] = '\0';
  ret = DVDNAV_STATUS_OK;
  if(colon_point == 0) {
    /* Use default device */

    ret = dvdnav_open(&this->dvdnav, this->dvd_device);
  } else {
    /* Use specified device */

    ret = dvdnav_open(&this->dvdnav, locator);
  }
  
  if(ret == DVDNAV_STATUS_ERR) {
    fprintf(stderr, "Error opening DVD device\n");
    return 0;
  }

  this->opened = 1;
  
  /* Set region code */
  region_entry = this->config->lookup_entry(this->config,
					    "dvd.region");
  if(region_entry) {
    region_changed_cb(this, region_entry);
  }
  
  if(this->mode == MODE_TITLE) {
    int tt, i, pr, found;
    int titles;
    
    /* A program and/or VTS was specified */
    locator += colon_point + 1;

    if(locator[0] == '\0') {
      /* Empty specifier */
      fprintf(stderr, "Incorrect MRL format.\n");
      dvdnav_close(this->dvdnav);
      return 0;
    }

    /* See if there is a period. */
    found = -1;
    for(i=0; i<strlen(locator); i++) {
      if(locator[i] == '.') {
	found = i;
	locator[i] = '\0';
      }
    }
    tt = strtol(locator, NULL,10);

    dvdnav_get_number_of_titles(this->dvdnav, &titles);
    if((tt <= 0) || (tt > titles)) {
      fprintf(stderr, "Title %i is out of range (1 to %i).\n", tt,
	      titles);
      dvdnav_close(this->dvdnav);
      return 0;
    }

    /* If there was a program specified, get that too. */
    pr = -1;
    if(found != -1) {
      pr = strtol(locator+found+1, NULL,10);
    }

    dprint("Jumping to VTS >%i<, prog >%i<\n", tt, pr);
    if(pr != -1) {
      dvdnav_part_play(this->dvdnav, tt, pr);
    } else {
      dvdnav_title_play(this->dvdnav, tt);
    }
  }
  
  dprint("DVD device successfully opened.\n");
  
  return 1;
}

static buf_element_t *dvdnav_plugin_read_block (input_plugin_t *this_gen, 
						fifo_buffer_t *fifo, off_t nlen) {
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen;
  buf_element_t      *buf;
  dvdnav_status_t     result;
  int                 event, len;
  int                 finished = 0;

  if(fifo == NULL) {
    dprint("values of \\beta will give rise to dom!\n");
    return NULL;
  }

  /* Read buffer */
  buf = fifo->buffer_pool_alloc (fifo);

  while(!finished) {
    result = dvdnav_get_next_block (this->dvdnav, buf->mem, &event, &len);
    if(result == DVDNAV_STATUS_ERR) {
      fprintf(stderr, "Error getting next block from DVD (%s)\n",
	      dvdnav_err_to_string(this->dvdnav));
      buf->free_buffer(buf);
      return NULL;
    }
    switch(event) {
     case DVDNAV_BLOCK_OK: 
       {
	buf->content = buf->mem;
	buf->type = BUF_DEMUX_BLOCK;

	/* Make sure we don't think we are still paused */
	this->pause_timer = 0;
	
	finished = 1;
       }
      break;
     case DVDNAV_NOP:
       {
	/* Nothing */
       }
      break;
     case DVDNAV_STILL_FRAME:
       {
	dvdnav_still_event_t *still_event =
	 (dvdnav_still_event_t*)(buf->mem);
	
	/* Xine's method of doing still-frames */
	if(this->pause_timer) {
	  if (this->pause_counter < VIDEO_FILL_THROTTLE) {
	    this->pause_counter++;
	  } else if (fifo->size (fifo) >= VIDEO_FILL_THROTTLE) {
	    /*
	     * approximate sleep for half the number of VIDEO_FILL_THROTTLE
	     * frames (assuming we use a frame rate of 30fps)
	     */
	    xine_usec_sleep(1000000 / 30 * (VIDEO_FILL_THROTTLE / 2));
	  }
	  
	  buf->type = BUF_VIDEO_FILL;
	  if (this->pause_timer != 0xFF && 
	      time(NULL) >= this->pause_end_time){
	    this->pause_timer = 0;
	    this->pause_end_time = 0;
	    dvdnav_still_skip(this->dvdnav);
	  }

	  finished = 1;
	} else {	
	  printf("dvdnav:input_dvdnav.c:Stillframe! (pause time = 0x%02x)\n",
		 still_event->length);
	  this->pause_timer = still_event->length;
	  this->pause_end_time = time(NULL) + this->pause_timer;
	  this->pause_counter = 0;
	}
       }
      break;
     case DVDNAV_SPU_STREAM_CHANGE:
       {
	dvdnav_stream_change_event_t *stream_event = 
	 (dvdnav_stream_change_event_t*) (buf->mem);
        buf->content = buf->mem;
        buf->type = BUF_CONTROL_SPU_CHANNEL;
        buf->decoder_info[0] = stream_event->physical;
	dprint("FIXME: SPU stream %d\n",stream_event->physical);
	finished = 1;
       }
      break;
     case DVDNAV_AUDIO_STREAM_CHANGE:
       {
	dvdnav_stream_change_event_t *stream_event = 
	 (dvdnav_stream_change_event_t*) (buf->mem);
        buf->content = buf->mem;
        buf->type = BUF_CONTROL_AUDIO_CHANNEL;
        buf->decoder_info[0] = stream_event->physical;
	dprint("FIXME: AUDIO stream %d\n",stream_event->physical);
	finished = 1;
       }
      break;
     case DVDNAV_HIGHLIGHT:
       {
	dvdnav_highlight_event_t *hevent = 
	 (dvdnav_highlight_event_t*) (buf->mem);
	spu_button_t spu_button;
	xine_spu_event_t spu_event;

	/* Do we want to show or hide the button? */
      	spu_event.event.type = XINE_EVENT_SPU_BUTTON;
	spu_event.data = &spu_button;
	if(hevent->display == 0) {
	  dprint("Highlight display = %i\n",
	       hevent->display);
	  spu_button.show = 0;
	  
	  spu_button.left   = 0;
	  spu_button.right  = 0;
	  spu_button.top    = 0;
	  spu_button.bottom = 0;
	} else {
          int i;
	  dprint("Highlight area is (%u,%u)-(%u,%u), display = %i, button = %u\n",
	       hevent->sx, hevent->sy,
	       hevent->ex, hevent->ey,
	       hevent->display,
	       hevent->buttonN);
	  spu_button.show = 1;
	  spu_button.left   = hevent->sx;
	  spu_button.right  = hevent->ex;
	  spu_button.top    = hevent->sy;
	  spu_button.bottom = hevent->ey;
          spu_button.pts    = hevent->pts;
          spu_button.buttonN  = hevent->buttonN;
          
          for (i = 0;i < 4; i++) {
            spu_button.color[i] = 0xf & (hevent->palette >> (16 + 4*i));
            spu_button.trans[i] = 0xf & (hevent->palette >> (4*i));
          }
	}
		
	xine_send_event(this->xine, &spu_event.event);
       }
      break;
     case DVDNAV_VTS_CHANGE:
       {
	dprint("FIXME: VTS change\n");
       }
      break;
     case DVDNAV_CELL_CHANGE:
       {
	dprint("FIXME: Cell change\n");
       }
      break;
     case DVDNAV_NAV_PACKET:
       {
	buf->content = buf->mem;
	buf->type = BUF_DEMUX_BLOCK;
/*********************
    { 
    int i;
    uint8_t *p = buf->content;
    printf("input_dvdnav.c:NAV:");
    for(i = 0;i<50;i++) {
      printf(" %02x",p[i]);
    }
    printf("\n");
    }
*******************8*/

	finished = 1;
       }
      break;
     case DVDNAV_SPU_CLUT_CHANGE:
       {
	buf->content = buf->mem;
	buf->type = BUF_SPU_CLUT;
	finished = 1;
       }
      break;
     case DVDNAV_STOP:
       {
	/* return NULL to indicate end of stream */
	return NULL;
       }
     default:
      dprint("FIXME: Unknown event (%i)\n", event);
      break;
    }
  }
 
  return buf;
}

static off_t dvdnav_plugin_read (input_plugin_t *this_gen, char *ch_buf, off_t len) {
/*  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen; */

  /* FIXME: Implement somehow */

  return 0;
}
  
static off_t dvdnav_plugin_seek (input_plugin_t *this_gen, off_t offset, int origin) {
  /* dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen; */
 
  trace_print("Called\n");

  return -1;
}

static off_t dvdnav_plugin_get_current_pos (input_plugin_t *this_gen){
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen;
  uint32_t pos=0;
  uint32_t length=1;
  dvdnav_status_t result;
  trace_print("Called\n");

  if(!this || !this->dvdnav) {
    return 0;
  }
  result = dvdnav_get_position(this->dvdnav, &pos, &length);
  return pos;
}

static off_t dvdnav_plugin_get_length (input_plugin_t *this_gen) {
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen;
  uint32_t pos=0;
  uint32_t length=1;
  dvdnav_status_t result;
 
  trace_print("Called\n");

  if(!this || !this->dvdnav) {
    return 0;
  }

  result = dvdnav_get_position(this->dvdnav, &pos, &length);
  return length;
}

static uint32_t dvdnav_plugin_get_blocksize (input_plugin_t *this_gen) {
  trace_print("Called\n");

  return DVD_BLOCK_SIZE;
}

static mrl_t **dvdnav_plugin_get_dir (input_plugin_t *this_gen, 
				    char *filename, int *nFiles) {
  trace_print("Called\n");

  /* Borwsing is not yet supported. In future allow browsing chapters as sub-dirs of 
   * titles. */

  return NULL;
}

static int dvdnav_plugin_eject_media (input_plugin_t *this_gen) {
  trace_print("Called\n");

  /* FIXME: Support ejecting the DVD device (copy from stock Xine DVD plugin) */

  return 1; 
}

static char* dvdnav_plugin_get_mrl (input_plugin_t *this_gen) {
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen;
  
  trace_print("Called\n");

  return this->mrl;
}

static void dvdnav_plugin_stop (input_plugin_t *this_gen) {
  /* dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*) this_gen; */
}

static void dvdnav_plugin_close (input_plugin_t *this_gen) {
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen;
  
  trace_print("Called\n");
  
  if(this->dvdnav) 
    dvdnav_close(this->dvdnav);
  this->dvdnav = NULL;
}

static char *dvdnav_plugin_get_description (input_plugin_t *this_gen) {
  trace_print("Called\n");

  return "DVD Navigator plugin";
}

static char *dvdnav_plugin_get_identifier (input_plugin_t *this_gen) {
  trace_print("Called\n");

  return "NAV";
}

static void dvdnav_event_listener (void *this_gen, xine_event_t *event) {

  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t *) this_gen; 

  if(!this->dvdnav) {
    return;
  }

  switch(event->type) {
   case XINE_EVENT_INPUT_MENU1:
    dvdnav_menu_call(this->dvdnav, DVD_MENU_Root);
    break;
   case XINE_EVENT_INPUT_MENU2:
    dvdnav_menu_call(this->dvdnav, DVD_MENU_Title);
    break;
   case XINE_EVENT_INPUT_MENU3:
    dvdnav_menu_call(this->dvdnav, DVD_MENU_Audio);
    break;
   case XINE_EVENT_INPUT_NEXT:
    dvdnav_next_pg_search(this->dvdnav);
    break;
   case XINE_EVENT_INPUT_PREVIOUS:
    dvdnav_prev_pg_search(this->dvdnav);
    break;
   case XINE_EVENT_INPUT_SELECT:
    dvdnav_button_activate(this->dvdnav);
    break;
   case XINE_EVENT_MOUSE_BUTTON: 
     {
      xine_input_event_t *input_event = (xine_input_event_t*) event;      
      dvdnav_mouse_activate(this->dvdnav, input_event->x, 
			    input_event->y);
     }
    break;
   case XINE_EVENT_MOUSE_MOVE: 
     {
      xine_input_event_t *input_event = (xine_input_event_t*) event;      
      /* printf("Mouse move (x,y) = (%i,%i)\n", input_event->x,
	     input_event->y); */
      dvdnav_mouse_select(this->dvdnav, input_event->x, input_event->y);
     }
    break;
   case XINE_EVENT_INPUT_UP:
    dvdnav_upper_button_select(this->dvdnav);
    break;
   case XINE_EVENT_INPUT_DOWN:
    dvdnav_lower_button_select(this->dvdnav);
    break;
   case XINE_EVENT_INPUT_LEFT:
    dvdnav_left_button_select(this->dvdnav);
    break;
   case XINE_EVENT_INPUT_RIGHT:
    dvdnav_right_button_select(this->dvdnav);
    break;
  }
   
  return;
}

static int dvdnav_plugin_get_optional_data (input_plugin_t *this_gen, 
					    void *data, int data_type) {
  /* dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t*)this_gen; */

  return INPUT_OPTIONAL_UNSUPPORTED;
}

static char **dvdnav_plugin_get_autoplay_list (input_plugin_t *this_gen, 
					       int *nFiles) {
  dvdnav_input_plugin_t *this = (dvdnav_input_plugin_t *) this_gen;
  dvdnav_status_t res;
  int titles, i;
  
  /* Close the plugin is opened */
  if(this->opened) {
    dvdnav_close(this->dvdnav);
    this->opened = 0;
  }
  
  /* Use default path */
  res = dvdnav_open(&(this->dvdnav), this->dvd_device);
  if(res == DVDNAV_STATUS_ERR) {
    return NULL;
  }

  this->opened = 1;
   
  /* Return a list of all titles */
  snprintf (&(filelist[0][0]), MAX_STR_LEN, "dvdnav://");
  filelist2[0] = &(filelist[0][0]);

  dvdnav_get_number_of_titles(this->dvdnav, &titles);
  for(i=1; i<=titles; i++) {
    snprintf (&(filelist[i][0]), MAX_STR_LEN, "dvdnav://:%i", i);
    filelist2[i] = &(filelist[i][0]);
  }
  *nFiles=titles+1;
  filelist2[*nFiles] = NULL;

  return filelist2;
}


#ifdef	__sun
/* 
 * Check the environment, if we're running under sun's
 * vold/rmmount control.
 */
static void
check_solaris_vold_device(dvdnav_input_plugin_t *this)
{
  char *volume_device;
  char *volume_name;
  char *volume_action;
  char *device;
  struct stat stb;

  if ((volume_device = getenv("VOLUME_DEVICE")) != NULL &&
      (volume_name   = getenv("VOLUME_NAME"))   != NULL &&
      (volume_action = getenv("VOLUME_ACTION")) != NULL &&
      strcmp(volume_action, "insert") == 0) {

    device = malloc(strlen(volume_device) + strlen(volume_name) + 2);
    if (device == NULL)
      return;
    sprintf(device, "%s/%s", volume_device, volume_name);
    if (stat(device, &stb) != 0 || !S_ISCHR(stb.st_mode)) {
      free(device);
      return;
    }
    this->dvd_device = device;
  }
}
#endif

input_plugin_t *init_input_plugin (int iface, xine_t *xine) {
  dvdnav_input_plugin_t *this;
  config_values_t *config = xine->config;

  trace_print("Called\n");

  switch (iface) {
  case 5:
    this = (dvdnav_input_plugin_t *) malloc (sizeof (dvdnav_input_plugin_t));

    this->input_plugin.interface_version  = INPUT_PLUGIN_IFACE_VERSION;
    this->input_plugin.get_capabilities   = dvdnav_plugin_get_capabilities;
    this->input_plugin.open               = dvdnav_plugin_open;
    this->input_plugin.read               = dvdnav_plugin_read;
    this->input_plugin.read_block         = dvdnav_plugin_read_block;
    this->input_plugin.seek               = dvdnav_plugin_seek;
    this->input_plugin.get_current_pos    = dvdnav_plugin_get_current_pos;
    this->input_plugin.get_length         = dvdnav_plugin_get_length;
    this->input_plugin.get_blocksize      = dvdnav_plugin_get_blocksize;
    this->input_plugin.get_dir            = dvdnav_plugin_get_dir;
    this->input_plugin.eject_media        = dvdnav_plugin_eject_media;
    this->input_plugin.get_mrl            = dvdnav_plugin_get_mrl;
    this->input_plugin.stop               = dvdnav_plugin_stop;
    this->input_plugin.close              = dvdnav_plugin_close;
    this->input_plugin.get_description    = dvdnav_plugin_get_description;
    this->input_plugin.get_identifier     = dvdnav_plugin_get_identifier;
    this->input_plugin.get_autoplay_list  = dvdnav_plugin_get_autoplay_list;
    this->input_plugin.get_optional_data  = dvdnav_plugin_get_optional_data;
    this->input_plugin.is_branch_possible = NULL;

    this->config                 = config;
    this->xine                   = xine;
    this->dvdnav                 = NULL;
    this->opened                 = 0;
    
    xine_register_event_listener(this->xine, dvdnav_event_listener, this);
    this->dvd_device = config->register_string(config,
                                         "input.dvd_device",
                                         DVD_PATH,
                                         "device used for dvd drive",
                                         NULL,
                                         NULL,
                                         NULL);
    config->register_num(config, "dvdnav.region",
		      	 1,
			 "Region that DVD player claims "
			 "to be (1 -> 8)",
			 "This only needs to be changed "
			 "if your DVD jumps to a screen "
			 "complaining about region code ",
			 region_changed_cb,
			 this);
    
#ifdef __sun
    check_solaris_vold_device(this);
#endif

    return (input_plugin_t *) this;
    break;
  default:
    fprintf(stderr,
	    "DVD Navigator input plugin doesn't support plugin API version %d.\n"
	    "PLUGIN DISABLED.\n"
	    "This means there's a version mismatch between xine and this input"
	    "plugin.\nInstalling current input plugins should help.\n",
	    iface);
    return NULL;
  }
}

/*
 * $Log: input_dvdnav.c,v $
 * Revision 1.112  2002/01/13 22:16:07  jcdutton
 * Change logging.
 *
 *
 */
