// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_pref.h
// *
// * Purpose: Preferences editor
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 28.5.2000
// *************************************************************************

#ifndef XGSM_PREF_H
#define XGSM_PREF_H

#include "xgsm_util.h"
#include "xgsm_dialogs.h"
#include "xgsm_device.h"

#include <gnome--/entry.h>
#include <gtk--/frame.h>
#include <gtk--/menuitem.h>
#include <gtk--/checkbutton.h>
#include <gnome--/propertybox.h>
#include <gnome--/file-entry.h>
#include <sigc++/signal_system.h>
#include <gsmlib/gsm_util.h>
#include <gsmlib/gsm_error.h>
#include <gsmlib/gsm_map_key.h>
#include <vector>

// Xgsm configuration path prefix
#define XGSM_CONFIG_PATH "Xgsm"
#define XGSM_DEFAULT_DIR ".xgsm"

namespace Xgsm
{

  // Configuration data types
  enum DataSource {FromFile = 0, FromDevice = 1};
  enum LeftOrRight {Left = 0, Right = 1};

  struct EditorDefault
  {
    DataSource _dataSource;
    string _storeName;
    gsmlib::SortOrder _sortOrder;
  };

  // global configuration class and object
  class Configuration : public gsmlib::NoCopy
  {
    // used by the load() routine
    bool _loadError;

    // device page
    string _phoneDevice;
    string _baudRate;
    string _initString;
    bool _softwareHandshake;
    string _characterSet;
    int _characterEncoding;

    // phonebooks page (left and right phonebook defaults)
    EditorDefault _phoneEditorDefault[2];

    // SMS stores page (left and right phonebook defaults)
    EditorDefault _smsEditorDefault[2];

    // call forwarding page
    bool _voice;
    bool _fax;
    bool _data;

    // SMS page
    bool _showLogWindow;
    bool _longSMS;
    string _sentSMSColor;
    string _receivedSMSColor;
    string _longSMSColor;
    bool _incomingSMSNotification;

    // gui page
    bool _multipleListSelectionStyle;
    string _warningColor;

    // check and convert string to DataSource
    DataSource checkDataSource(string title, string dataSource);

    // check and convert string to SortOrder
    gsmlib::SortOrder checkSortOrder(string title, string sortOrder,
                                     bool isSMS);
    
    // dialog sizes (-1 == not set)
    int _mainWidth, _mainHeight;
    int _pbWidth, _pbHeight;
    int _smsWidth, _smsHeight;
    int _newSmsWidth, _newSmsHeight;
    int _chatSmsWidth, _chatSmsHeight;
    int _incomingSmsWidth, _incomingSmsHeight;
    int _smsPanedPosition;

  public:
    Configuration();

    // set defaults and load from configuration file
    void load(bool loadDialogSizes = true) throw(gsmlib::GsmException);
    
    // synchronize with file
    void sync();

    // drop all changes and reload from configuration file
    void drop();

    // device page get and set functions
    string getPhoneDevice() const {return _phoneDevice;}
    void setPhoneDevice(string phoneDevice);

    string getBaudRate() const {return _baudRate;}
    void setBaudRate(string baudRate);

    bool getSoftwareHandshake() const {return _softwareHandshake;}
    void setSoftwareHandshake(bool softwareHandshake);

    string getInitString() const {return _initString;}
    void setInitString(string initString);

    string getCharacterSet() const {return _characterSet;}
    void setCharacterSet(string characterSet);

    int getCharacterEncoding() const {return _characterEncoding;}
    void setCharacterEncoding(int characterEncoding);

    // phonebook page get and set functions
    DataSource getPhonebookDataSource(LeftOrRight lr) const
      {return _phoneEditorDefault[(int)lr]._dataSource;}
    void setPhonebookDataSource(LeftOrRight lr, DataSource dataSource);

    string getPhonebookStoreName(LeftOrRight lr) const
      {return _phoneEditorDefault[(int)lr]._storeName;}
    void setPhonebookStoreName(LeftOrRight lr, string storeName);
    
    gsmlib::SortOrder getPhonebookSortOrder(LeftOrRight lr) const
      {return _phoneEditorDefault[(int)lr]._sortOrder;}
    void setPhonebookSortOrder(LeftOrRight lr, gsmlib::SortOrder sortOrder);

    // SMS store page get and set functions
    DataSource getSMSDataSource(LeftOrRight lr) const
      {return _smsEditorDefault[(int)lr]._dataSource;}
    void setSMSDataSource(LeftOrRight lr, DataSource dataSource);

    string getSMSStoreName(LeftOrRight lr) const
      {return _smsEditorDefault[(int)lr]._storeName;}
    void setSMSStoreName(LeftOrRight lr, string storeName);
    
    gsmlib::SortOrder getSMSSortOrder(LeftOrRight lr) const
      {return _smsEditorDefault[(int)lr]._sortOrder;}
    void setSMSSortOrder(LeftOrRight lr, gsmlib::SortOrder sortOrder);

    // call forwarding page get and set functions
    bool getVoice() const {return _voice;}
    void setVoice(bool voice);

    bool getFax() const {return _voice;}
    void setFax(bool voice);

    bool getData() const {return _voice;}
    void setData(bool voice);

    // SMS page
    bool getShowLogWindow() const {return _showLogWindow;}
    void setShowLogWindow(bool showLogWindow);

    string getSentSMSColor() const {return _sentSMSColor;}
    void setSentSMSColor(string sentSMSColor);

    string getReceivedSMSColor() const {return _receivedSMSColor;}
    void setReceivedSMSColor(string receivedSMSColor);

    string getLongSMSColor() const {return _longSMSColor;}
    void setLongSMSColor(string longSMSColor);

    bool getLongSMS() const {return _longSMS;}
    void setLongSMS(bool longSMS);

    bool getIncomingSMSNotification() const {return _incomingSMSNotification;}
    void setIncomingSMSNotification(bool incomingSMSNotification);

    // gui page
    string getWarningColor() const {return _warningColor;}
    void setWarningColor(string warningColor);
    
    bool getMultipleListSelectionStyle() const
      {return _multipleListSelectionStyle;}
    void setMultipleListSelectionStyle(bool multipleListSelectionStyle);

    // dialog sizes (get functions return false if not set)
    bool getMainSizes(int &width, int &height);
    void setMainSizes(int width, int height);
    
    bool getPbSizes(int &width, int &height);
    void setPbSizes(int width, int height);
    
    bool getSmsSizes(int &width, int &height);
    void setSmsSizes(int width, int height);

    bool getNewSmsSizes(int &width, int &height);
    void setNewSmsSizes(int width, int height);
    
    bool getChatSmsSizes(int &width, int &height);
    void setChatSmsSizes(int width, int height);
    
    bool getIncomingSmsSizes(int &width, int &height);
    void setIncomingSmsSizes(int width, int height);

    bool getSmsPanedPosition(int &position);
    void setSmsPanedPosition(int position);
    
    // save dialog sizes
    ~Configuration();
  };

  extern Configuration config;
  
  // GUI class
  class Preferences : public Gnome::PropertyBox, public DeviceHelper,
                      public ToplevelHelper
  {
    // dialog state
    bool _okToClose;            // true if user input is acceptable

    // device page objects
    Gtk::Entry *_phoneDeviceEntry;
    Gnome::FileEntry *_phoneDeviceFileEntry;
    string _oldPhoneDevice;
    Gnome::Entry *_initStringEntry;
    string _oldInitString;
    Gtk::CheckButton *_softwareHandshakeButton;
    DynamicOptionMenu _baudRateMenu;
    DynamicOptionMenu _characterSetMenu;
    Gtk::Menu *_characterEncodingMenu;

    // phonebooks page objects
    Gtk::Menu *_pbLeftFileOrDevice, *_pbLeftSortOrder;
    Gnome::Entry *_pbLeftPhonebookName;
    string _oldPhonebookStoreNameLeft;
    Gtk::Menu *_pbRightFileOrDevice, *_pbRightSortOrder;
    Gnome::Entry *_pbRightPhonebookName;
    string _oldPhonebookStoreNameRight;

    // SMS stores page objects
    Gtk::Menu *_smsLeftFileOrDevice, *_smsLeftSortOrder;
    Gnome::Entry *_smsLeftSMSStoreName;
    string _oldSSMSStoreNameLeft;
    Gtk::Menu *_smsRightFileOrDevice, *_smsRightSortOrder;
    Gnome::Entry *_smsRightSMSStoreName;
    string _oldSSMSStoreNameRight;

    // SMS page objects
    Gtk::CheckButton *_showMessageLogButton;
    Gtk::CheckButton *_longSMSButton;
    Gnome::Entry *_sentSMSColorEntry;
    string _oldSentSMSColor;
    Gnome::Entry *_receivedSMSColorEntry;
    string _oldReceivedSMSColor;
    Gnome::Entry *_longSMSColorEntry;
    string _oldLongSMSColor;
    Gtk::CheckButton *_newSMSNotifyButton;

    // GUI page objects
    Gtk::Menu *_listSelectionStyleMenu;
    Gnome::Entry *_warningColorEntry;
    string _oldWarningColor;

    // check for valid color string and show error dialog if not
    bool checkColor(string color);

    // info request callback
    void onInfoResponse(ResponseRef response);

    // DeviceHelper callbacks
    void onOpenDone(ResponseRef response);
    
    // device page callbacks
    void onPhoneDevice();
    void onBaudRateSet();
    void onSoftwareHandshake();
    void onInitString();
    void onCharacterSetSet();
    void onCharacterEncodingSet();

    // phonebooks page callbacks
    void onPbLeftFileOrDevice();
    void onPbLeftName();
    void onPbLeftSortOrder();
    void onPbRightFileOrDevice();
    void onPbRightName();
    void onPbRightSortOrder();

    // SMS stores page callbacks
    void onSmsLeftFileOrDevice();
    void onSmsLeftName();
    void onSmsLeftSortOrder();
    void onSmsRightFileOrDevice();
    void onSmsRightName();
    void onSmsRightSortOrder();
    
    // SMS page callbacks
    void onShowMessageLog();
    void onLongSMS();
    void onSentSMSColorChanged();
    void onSentSMSColorClicked();
    void onReveivedSMSColorChanged();
    void onReceivedSMSColorClicked();
    void onLongSMSColorChanged();
    void onLongSMSColorClicked();
    void onNewSMSNotify();

    // GUI page callbacks
    void onListSelectionStyle();
    void onWarningColorChanged();
    void onWarningColorClicked();

    // property box callbacks
    gboolean onWindowClose();
    void onApply(gint pageNum);
    void onHelp(gint pageNum);
    void onClicked(gint p0);

  public:
    Preferences(bool loadError = false);

    ~Preferences();
  };


};

#endif // XGSM_PREF_H
