/* -----------------------------------------------------------------------
   cd_control process

   part of XfreeCD

   Copyright (C) 1998 Brian C. Lane
   nexus@tatoosh.com
   http://www.tatoosh.com/nexus

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
   
   ==========================[ HISTORY ]==================================
   06/22/98   Bug in track length calculation. It was waiting until track
              #2 before calculating the lengths (holdover from the change
	      to 0 offset from 1 offset). FIXED.

   05/08/98   Added the control and init code from old version of xfreecd
              Added software control of the EJECT on CLOSE state - using
	      the CD_SET_EJECT command and an argument of 0 to disable
	      eject on exit and 1 to enable eject on exit.

   05/03/98   Wrote the synchronization code and basic switch statment.
              Testing different shutdown situations. No hungup child
	      processes yet.

	      Things this process needs to do:
	      x 1. Get the data on the current CD and send it to the
	           parent, in response to a request by the parent.
	      x 2. Play a CD
	      x 3. Pause a CD
	      x 4. Resume a CD
	      x 5. Go to next track
	      x 6. Go to Previous track
	      x 8. Eject a CD
	      x 9. Close the CD tray
	     10. Switch CDs for a CD changer
	     11. Seek forward (may need a kernel patch?)
	     12. Seek backwards (may need a kernel patch?)
	     13. Return error/status codes (like if a play is tried and
	         there is no CD in the player).
	     14. Shuffle play (on one Cd or among multiple CDs)
	     15. Repeat Song
	     16. Repeat whole CD, repeat group of CDs


   ----------------------------------------------------------------------- */
#include <stdio.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "child_sync.h"
#include "cd_control.h"
#include "cddb.h"

int cd_control( int );
int readline( register int, register char *, register int );

#undef DEBUG1
#undef DEBUG3

/*
   Fire up the cd_control process.
   Create a socket pair.
   Fork a child process
   Return the parent's file descriptor for the socket or -1 + errno
   fill in childpid with the child's PID
   Child calls cd_control 
*/
int start_cd_control( int *childpid )
{
  int  fd[2];

  if( socketpair( AF_UNIX, SOCK_STREAM, 0, fd ) < 0 )
    return(-1);

  if( ( *childpid = fork() ) < 0 )
  {
    perror("start_cd_control, cannot fork");
    return(-1);
  } else if( *childpid == 0 ) {
    close( fd[0] );

    /* Synchronize with the parent, exit if it fails */
    if( parent_sync( fd[1] ) == 0 )
      cd_control( fd[1] );
    
    close( fd[1] );
    exit(0);
  } 
  close( fd[1] );

  if( child_sync( fd[0] ) == 0 )
    return(fd[0]);  

  return(-1);
}


/*
   Load a new CD, read all of its info into the cdinfo array
*/
int cdrom_init( struct CDINFO *cdinfo, struct cdrom_volctrl *volctrl, char *cd_device )
{
  int  fd;
  int  x;

  /* Try and open the /dev/cdrom device */
  if( ( fd = open( cd_device, O_RDONLY ) ) < 0 )
    {
#ifdef DEBUG3
      g_print( "device = %s\n", cd_device);
#endif

      perror("CDROM device" );
      return -1;
    }
  
  /* Read the cdrom's header information */
  if( ioctl( fd, CDROMREADTOCHDR, &cdinfo->tochdr ) != 0 )
    {
      perror("cdrom_init(CDROMREADTOCHDR-2)" );
      return -1;
    }

  /* Get the time on the end of the last track */
  cdinfo->leadout.cdte_track  = CDROM_LEADOUT;
  cdinfo->leadout.cdte_format = CDROM_MSF;

  if( ioctl( fd, CDROMREADTOCENTRY, &cdinfo->leadout ) != 0 )
    {
      perror( "cdrom_init(CDROMREADTOCENTRY-2)" );
      return -1;
    }

  /* Calculate the total length of the CD */
  cdinfo->cd_length = cdinfo->leadout.cdte_addr.msf.second \
                     + (cdinfo->leadout.cdte_addr.msf.minute * 60);

  /* Read the current CD volume level - different from soundcard volume */
  if( ioctl( fd, CDROMVOLREAD, volctrl ) == 0 )
    { 
      cdinfo->volume = volctrl->channel0;
    } else {
      perror( "cdrom_init(CDROMVOLREAD-2)" );
      return -1;
    }

  /* Fill in the tracks[] array with info on all the tracks */
  for( x = 0; x < cdinfo->tochdr.cdth_trk1; x++ )
    {
      /* Read the info for the current track */
      cdinfo->track[x].te.cdte_track  = x+1;
      cdinfo->track[x].te.cdte_format = CDROM_MSF;
      if( ioctl( fd, CDROMREADTOCENTRY, &cdinfo->track[x].te ) != 0 )
	{
	  perror( "set_track_length(CDROMREADTOCENTRY)" );
	  return -1;
	}

      cdinfo->track[x].frame_offset = cdinfo->track[x].te.cdte_addr.msf.minute * 60 + cdinfo->track[x].te.cdte_addr.msf.second;

      /* Convert seconds to frames */
      cdinfo->track[x].frame_offset *= 75;
      cdinfo->track[x].frame_offset += cdinfo->track[x].te.cdte_addr.msf.frame;

      /* 
       * Set the length of the previous track
       *  Accomplished by taking the start of the current track - start
       *  of the previous track.
       */
      if( x > 0 )
	{
	  /* Set length to the start of the current one */
	  cdinfo->track[x-1].length = (cdinfo->track[x].te.cdte_addr.msf.minute * 60) + cdinfo->track[x].te.cdte_addr.msf.second;

	  /* Subtract the start of the previous track */
	  cdinfo->track[x-1].length = cdinfo->track[x-1].length - ((cdinfo->track[x-1].te.cdte_addr.msf.minute * 60) + cdinfo->track[x-1].te.cdte_addr.msf.second);

#ifdef DEBUG1
	  /* Show some debug data on the tracks we find */
	  g_print("track %d - length = %d seconds\n", x-1, cdinfo->track[x-1].length );
#endif
	}
    }

  /* 
   * Set the length of the last track
   * This is accomplished by subtracting the last track start from the
   * length of the CD.
   */
  x--;
  cdinfo->track[x].length = cdinfo->cd_length - ((cdinfo->track[x].te.cdte_addr.msf.minute * 60) + cdinfo->track[x].te.cdte_addr.msf.second );

#ifdef DEBUG1
  /* Show debug data on the last track */
  g_print("track %d - length = %d seconds\n", x, cdinfo->track[x].length );
  
  g_print("New CD loaded: ");
  g_print("%d tracks, cd_length = %d, ", cdinfo->tochdr.cdth_trk1, cdinfo->cd_length );
  g_print("volume = %d\n", cdinfo->volume );
#endif		

  /* Calculate the discid for this CD */
  cdinfo->discid = cddb_discid( cdinfo );

  cdinfo->revision = -1;
  cdinfo->extd = NULL;


  /* All is well */
  return fd;
}


/* ----------------------------------------------------------------------
   Control the low-level audio CD player

   Wait for commands from the parent, execute the command and send back
   a response if one is needed

   A command is 2 bytes of data - a command and an optional argument.
   The Commands are defined in cd_control.h as CD_*

   To init the cd, if status is called the first time it will try to init
   the CD. If this fails it sets cd_fd to -1, so status won't repeatedly
   try the CD. After the initial failure only a play will try.

   ----------------------------------------------------------------------- */
int cd_control( int control_fd )
{
  char   cmnd[2],
         cd_device[80];                 /* Device to talk to */
  struct cdrom_volctrl volctrl;  	/* Volume control 		 */
  struct CDINFO cdinfo;                 /* All the info on the CD        */
  int  fd_cd=0;

  /* Default cdrom device */
  strcpy( cd_device, "/dev/cdrom" );

  for(;;)
  {
    /* Get the command from the parent process */
    if( read( control_fd, &cmnd, 2 ) != 2 )
      return(-1);

    switch( cmnd[0] )
      {
      case CD_DIAG :
	puts("cd_control diagnostic\n");
	break;

      case CD_STATUS :
	if( fd_cd > 0 )
	  {
	    cdinfo.sc.cdsc_format = CDROM_MSF;     /* Info in MM:SS:FF format*/
	    if( ioctl( fd_cd, CDROMSUBCHNL, &cdinfo.sc ) != 0 )
	      {
		/* perror( "cd_control(CDROMSUBCHNL-2)" ); */
		cdinfo.sc.cdsc_audiostatus = CDROM_AUDIO_NO_STATUS;
	      }
  	  } else {
	    cdinfo.sc.cdsc_audiostatus = CDROM_AUDIO_NO_STATUS;
	  }

	/* Send the new data back to the main process */
	write( control_fd, &cdinfo, sizeof( struct CDINFO ) );
	break;


      case CD_PLAY :
	/* If we are not open yet, try to open it first */
	if( fd_cd < 1 )
	  {
	    fd_cd = cdrom_init( &cdinfo, &volctrl, cd_device );
	    
	    /* Start playing at the beginning*/
	    cmnd[1] = cdinfo.tochdr.cdth_trk0;
	  }

	if( fd_cd > 0 )
	  {
	    /* Setup the CD to play the indicated track */
	    cdinfo.ti.cdti_trk0 = cmnd[1];
	    cdinfo.ti.cdti_trk1 = cdinfo.tochdr.cdth_trk1;
	    cdinfo.ti.cdti_ind0 = cdinfo.ti.cdti_ind1 = 0;

	    /* Stop playing previous track and start playing the next */
	    /* Try this without the stop and see what happens... */
	    if( ioctl( fd_cd, CDROMSTOP ) != 0 )
	      {
		perror( "cd_control(CDROMSTOP-1)" );
	      } else if( ioctl( fd_cd, CDROMPLAYTRKIND, &cdinfo.ti ) != 0 ) {
		perror( "cd_control(CDROMPLAYTRKIND-1)" );
	      }
	  }
	break;

      case CD_PAUSE :
	if( fd_cd > 0 )
	  {
	    if( ioctl( fd_cd, CDROMPAUSE ) != 0 )
	      perror( "cd_control(CDROMPAUSE)" );
	  }
	break;

      case CD_RESUME :
	if( fd_cd > 0 )
	  {
	    if( ioctl( fd_cd, CDROMRESUME ) != 0 )
	      perror( "cd_control(CDROMRESUME)" );
	  }
	break;

      case CD_STOP :
	if( fd_cd > 0 )
	  {
	    if( ioctl( fd_cd, CDROMSTOP ) != 0 )
	      perror( "cd_control(CDROMSTOP-2)" );

	    close( fd_cd );
	    fd_cd = 0;
	  }
	break;

      case CD_VOLUME :
	if( fd_cd > 0 )
	  {
	    cdinfo.volume = (unsigned char) cmnd[1];

	    volctrl.channel0 = cdinfo.volume;
	    volctrl.channel1 = cdinfo.volume;
	    if( ioctl( fd_cd, CDROMVOLCTRL, &volctrl ) != 0 )
	      perror( "cd_control(CDROMVOLCTRL-3)" );
	  }
	break;

      case CD_EJECT :
	if( (fd_cd > 0) )
	  {
	    /* Stop playing, then eject the CD */
	    if( ioctl( fd_cd, CDROMSTOP ) != 0 )
	      {
		perror( "cd_control(CDROMSTOP-3)" );
	      } else  if( ioctl( fd_cd, CDROMEJECT ) != 0 ) {
		perror( "cd_control(CDROMEJECT)" );
	      }
	    close( fd_cd );
	    fd_cd = 0;
	  }
	break;

      case CD_SET_EJECT :
	if( fd_cd > 0 )
	  {
	    /* Set the software configurable eject on close state */
	    if( ioctl( fd_cd, CDROMEJECT_SW, cmnd[1] ) != 0 )
	      perror("cd_control(CDROMEJECT_SW)");
	  }
	break;

	/* Set the default device, close the current device and open the
	   new one */
      case CD_SET_DEVICE :

#ifdef DEBUG3
	printf("got CD_SET_DEVICE\n");
#endif

	/*
	   This routine is different from the others, it reads a string from
	   the sending process after receiving the 2 byte command
	*/
	if( readline( control_fd, cd_device, 80 ) < 0 )
	  {
	    perror("readline - 2");	    
	  }

	/* Strip trailing CR */
	if( cd_device[strlen(cd_device)-1] == '\n' )
	  cd_device[strlen(cd_device)-1] = 0;

#ifdef DEBUG3
	printf("Setting cd_control cd_device = %s\n", cd_device );
#endif
	/* If a device isn't open, then open one */
	if( fd_cd < 1 )
	  fd_cd = cdrom_init( &cdinfo, &volctrl, cd_device );

	break;


      case CD_QUIT :
	if( fd_cd )
	  close( fd_cd );
	return(0);
	break;
      }
  }
  return(0);
}
