/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk/gdkkeysyms.h>

#include <xfmedia/xfmedia-playlist-queue.h>
#include <xfmedia/xfmedia-settings.h>
#include "xfmedia-handle.h"

#define EXO_API_SUBJECT_TO_CHANGE 1
#include <exo/exo.h>

enum
{
    SIG_ENTRY_ADDED = 0,
    SIG_ENTRY_REMOVED,
    SIG_EMPTIED,
    N_SIGNALS
};

enum
{
    COL_DISPLAY_NAME = 0,
    COL_ENTRY_LENGTH,
    N_COLS
};

static void xfmedia_playlist_queue_class_init(XfmediaPlaylistQueueClass *klass);
static void xfmedia_playlist_queue_init      (XfmediaPlaylistQueue *playlist_q);
static void xfmedia_playlist_queue_finalize  (GObject *object);

static gboolean xfmedia_playlist_queue_button_press_cb(GtkWidget *w,
                                                       GdkEventButton *evt,
                                                       gpointer user_data);
static void xfmedia_playlist_queue_remove_mi_cb(GtkWidget *w,
                                                gpointer user_data);
static gboolean xfmedia_playlist_queue_handle_button_cb(GtkWidget *w,
                                                        GdkEventButton *evt,
                                                        gpointer user_data);
static gboolean xfmedia_playlist_queue_handle_drag_cb(GtkWidget *w,
                                                      GdkEventMotion *evt,
                                                      gpointer user_data);

struct _XfmediaPlaylistQueuePrivate
{
    GQueue *entries;
    
    GtkWidget *handle;
    gboolean dragging;
    gint last_x, last_y;
    
    GtkWidget *sw;
    GtkWidget *treeview;
    GtkListStore *list_store;
    GtkTreeSelection *sel;
    
    GtkWidget *rightclick_menu;
};

guint __signals[N_SIGNALS] = { 0, };


G_DEFINE_TYPE(XfmediaPlaylistQueue, xfmedia_playlist_queue, GTK_TYPE_WINDOW);


static void
xfmedia_playlist_queue_class_init(XfmediaPlaylistQueueClass *klass)
{
    GObjectClass *gobject_class = (GObjectClass *)klass;
    
    gobject_class->finalize = xfmedia_playlist_queue_finalize;
    
    __signals[SIG_ENTRY_ADDED] = g_signal_new("entry-added",
                                              XFMEDIA_TYPE_PLAYLIST_QUEUE,
                                              G_SIGNAL_RUN_LAST,
                                              G_STRUCT_OFFSET(XfmediaPlaylistQueueClass, entry_added),
                                              NULL, NULL,
                                              g_cclosure_marshal_VOID__VOID,
                                              G_TYPE_NONE, 0);
    
    __signals[SIG_ENTRY_REMOVED] = g_signal_new("entry-removed",
                                                XFMEDIA_TYPE_PLAYLIST_QUEUE,
                                                G_SIGNAL_RUN_LAST,
                                                G_STRUCT_OFFSET(XfmediaPlaylistQueueClass, entry_removed),
                                                NULL, NULL,
                                                g_cclosure_marshal_VOID__VOID,
                                                G_TYPE_NONE, 0);
    
    __signals[SIG_EMPTIED] = g_signal_new("emptied",
                                          XFMEDIA_TYPE_PLAYLIST_QUEUE,
                                          G_SIGNAL_RUN_LAST,
                                          G_STRUCT_OFFSET(XfmediaPlaylistQueueClass, emptied),
                                          NULL, NULL,
                                          g_cclosure_marshal_VOID__VOID,
                                          G_TYPE_NONE, 0);
}

static void
xfmedia_playlist_queue_init(XfmediaPlaylistQueue *playlist_q)
{
    GtkWidget *hbox, *mi;
    GtkCellRenderer *render;
    GtkTreeViewColumn *col;
    GtkAccelGroup *accel;
    
    playlist_q->priv = g_new0(XfmediaPlaylistQueuePrivate, 1);
    
    playlist_q->priv->entries = g_queue_new();
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(playlist_q), hbox);
    
    playlist_q->priv->handle = xfmedia_handle_new(XFMEDIA_HANDLE_STYLE_GRIP,
                                                  GTK_ORIENTATION_VERTICAL);
    gtk_widget_show(playlist_q->priv->handle);
    gtk_box_pack_start(GTK_BOX(hbox), playlist_q->priv->handle, FALSE, FALSE, 0);
    gtk_widget_add_events(playlist_q->priv->handle,
                          GDK_BUTTON_PRESS
                          | GDK_BUTTON_RELEASE
                          | GDK_MOTION_NOTIFY);
    g_signal_connect(G_OBJECT(playlist_q->priv->handle),
                     "button-press-event",
                     G_CALLBACK(xfmedia_playlist_queue_handle_button_cb),
                     playlist_q);
    g_signal_connect(G_OBJECT(playlist_q->priv->handle),
                     "button-release-event",
                     G_CALLBACK(xfmedia_playlist_queue_handle_button_cb),
                     playlist_q);
    g_signal_connect(G_OBJECT(playlist_q->priv->handle),
                     "motion-notify-event",
                     G_CALLBACK(xfmedia_playlist_queue_handle_drag_cb),
                     playlist_q);
    
    playlist_q->priv->sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(playlist_q->priv->sw),
                                   GTK_POLICY_NEVER,
                                   GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(playlist_q->priv->sw),
                                        GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(playlist_q->priv->sw);
    gtk_box_pack_start(GTK_BOX(hbox), playlist_q->priv->sw, TRUE, TRUE, 0);
    
    playlist_q->priv->list_store = gtk_list_store_new(N_COLS,
                                                      G_TYPE_STRING,
                                                      G_TYPE_STRING);
    
    playlist_q->priv->treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(playlist_q->priv->list_store));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(playlist_q->priv->treeview), FALSE);
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(playlist_q->priv->treeview), FALSE);
    
    render = exo_cell_renderer_ellipsized_text_new();
    g_object_set(G_OBJECT(render),
                 "ellipsize", EXO_PANGO_ELLIPSIZE_END,
                 "ellipsize-set", TRUE,
                 NULL);
    col = gtk_tree_view_column_new_with_attributes("display-name", render,
                                                   "text", COL_DISPLAY_NAME,
                                                   NULL);
    gtk_tree_view_column_set_expand(col, TRUE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(playlist_q->priv->treeview), col);
    
    render = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes("length", render,
                                                   "text", COL_ENTRY_LENGTH,
                                                   NULL);
    gtk_tree_view_column_set_expand(col, FALSE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(playlist_q->priv->treeview), col);
    
    gtk_widget_show(playlist_q->priv->treeview);
    gtk_container_add(GTK_CONTAINER(playlist_q->priv->sw),
                      playlist_q->priv->treeview);
    gtk_widget_add_events(playlist_q->priv->treeview, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(playlist_q->priv->treeview),
                     "button-press-event",
                     G_CALLBACK(xfmedia_playlist_queue_button_press_cb),
                     playlist_q);
    
    playlist_q->priv->sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(playlist_q->priv->treeview));
    
    accel = gtk_accel_group_new();
    gtk_window_add_accel_group(GTK_WINDOW(playlist_q), accel);
    gtk_accel_map_add_entry("<Xfmedia-Playlist-Queue>/Remove-Entry",
                            GDK_Delete, 0);
    
    playlist_q->priv->rightclick_menu = gtk_menu_new();
    gtk_widget_show(playlist_q->priv->rightclick_menu);
    
    mi = gtk_image_menu_item_new_from_stock(GTK_STOCK_REMOVE, accel);
    gtk_menu_item_set_accel_path(GTK_MENU_ITEM(mi),
                                 "<Xfmedia-Playlist-Queue>/Remove-Entry");
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(playlist_q->priv->rightclick_menu),
                          mi);
    g_signal_connect(G_OBJECT(mi),
                     "activate",
                     G_CALLBACK(xfmedia_playlist_queue_remove_mi_cb),
                     playlist_q);
    
}

static void
xfmedia_playlist_queue_finalize(GObject *object)
{
    XfmediaPlaylistQueue *playlist_q = XFMEDIA_PLAYLIST_QUEUE(object);
    XfmediaPlaylistEntryRef *ref;
    
    gtk_widget_destroy(playlist_q->priv->rightclick_menu);
    
    while((ref = g_queue_pop_head(playlist_q->priv->entries)))
        xfmedia_playlist_entry_ref_destroy(ref);
    g_queue_free(playlist_q->priv->entries);
    
    g_free(playlist_q->priv);
    playlist_q->priv = NULL;
    
    G_OBJECT_CLASS(xfmedia_playlist_queue_parent_class)->finalize(object);
}


/*
 * internal callbacks
 */

static gboolean
xfmedia_playlist_queue_button_press_cb(GtkWidget *w,
                                       GdkEventButton *evt,
                                       gpointer user_data)
{
    XfmediaPlaylistQueue *playlist_q = user_data;
    
    if(evt->button == 3) {
        GList *selected = gtk_tree_selection_get_selected_rows(playlist_q->priv->sel, NULL);
        if(selected) {
            g_list_free(selected);
            gtk_menu_popup(GTK_MENU(playlist_q->priv->rightclick_menu), NULL,
                           NULL, NULL, NULL, evt->button, evt->time);
        }
    }
    
    return FALSE;
}

static void
xfmedia_playlist_queue_remove_mi_cb(GtkWidget *w,
                                    gpointer user_data)
{
    XfmediaPlaylistQueue *playlist_q = user_data;
    GList *selected, *l;
    GtkTreeModel *model = NULL;
    
    selected = gtk_tree_selection_get_selected_rows(playlist_q->priv->sel,
                                                    &model);
    if(selected) {
        GList *to_remove = NULL;
        GtkTreeRowReference *rref;
        
        for(l = selected; l; l = l->next) {
            rref = gtk_tree_row_reference_new(model, (GtkTreePath *)l->data);
            to_remove = g_list_prepend(to_remove, rref);
        }
        g_list_free(selected);
        
        for(l = to_remove; l; l = l->next) {
            GtkTreeRowReference *rref = l->data;
            GtkTreePath *path = gtk_tree_row_reference_get_path(rref);
            gint *indices = gtk_tree_path_get_indices(path);
            XfmediaPlaylistEntryRef *eref = g_queue_pop_nth(playlist_q->priv->entries,
                                                            indices[0]);
            GtkTreeIter itr;
            
            gtk_tree_model_get_iter(model, &itr, path);
            gtk_list_store_remove(GTK_LIST_STORE(model), &itr);
            
            gtk_tree_path_free(path);
            gtk_tree_row_reference_free(rref);
            xfmedia_playlist_entry_ref_destroy(eref);
        }
        g_list_free(to_remove);
    }
}

static gboolean
xfmedia_playlist_queue_handle_button_cb(GtkWidget *w,
                                        GdkEventButton *evt,
                                        gpointer user_data)
{
    XfmediaPlaylistQueue *playlist_q = user_data;
    
    if(evt->type == GDK_BUTTON_PRESS) {
        playlist_q->priv->dragging = TRUE;
        playlist_q->priv->last_x = evt->x;
        playlist_q->priv->last_y = evt->y;
    } else
        playlist_q->priv->dragging = FALSE;
    
    return FALSE;
}

static gboolean
xfmedia_playlist_queue_handle_drag_cb(GtkWidget *w,
                                      GdkEventMotion *evt,
                                      gpointer user_data)
{
    XfmediaPlaylistQueue *playlist_q = user_data;
    gint cur_x, cur_y, new_x, new_y;
    
    if(!playlist_q->priv->dragging)
        return FALSE;
    
    gtk_window_get_position(GTK_WINDOW(playlist_q), &cur_x, &cur_y);
    
    new_x = cur_x + (evt->x - playlist_q->priv->last_x);
    new_y = cur_y + (evt->y - playlist_q->priv->last_y);
    
    gtk_window_move(GTK_WINDOW(playlist_q), new_x, new_y);
    
    return FALSE;
}


/*
 * public API
 */


/**
 * xfmedia_playlist_queue_new:
 *
 * Creates a new empty playlist queue widget.
 *
 * Returns: A new #XfmediaPlaylistQueue widget.
 **/
GtkWidget *
xfmedia_playlist_queue_new(GtkWindow *parent)
{
    GtkWidget *playlist_q;
    gint width, height;
    GdkGeometry geom;
    
    width = (xfmedia_settings_get_int("/xfmedia/general/width") * 2) / 3;
    height = xfmedia_settings_get_int("/xfmedia/general/height_large") / 2;
    
    DBG("setting queue window to %dx%d", width, height);
    
    playlist_q = g_object_new(XFMEDIA_TYPE_PLAYLIST_QUEUE,
                              "decorated", FALSE,
                              "default-width", width,
                              "default-height", height,
                              "destroy-with-parent", TRUE,
                              "resizable", FALSE,
                              "skip-pager-hint", TRUE,
                              "skip-taskbar-hint", TRUE,
                              "title", "Xfmedia",
                              "type-hint", GDK_WINDOW_TYPE_HINT_UTILITY,
                              NULL);
    
    if(parent)
        gtk_window_set_transient_for(GTK_WINDOW(playlist_q), parent);
    
    geom.min_width = geom.max_width = width;
    geom.min_height = geom.max_height = height;
    gtk_window_set_geometry_hints(GTK_WINDOW(playlist_q),
                                  playlist_q,
                                  &geom,
                                  GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
    
    return playlist_q;
}

void
xfmedia_playlist_queue_push_entry(XfmediaPlaylistQueue *playlist_q,
                                  const XfmediaPlaylistEntryRef *entry,
                                  const gchar *display_name,
                                  gint entry_length)
{
    XfmediaPlaylistEntryRef *new_ref;
    GtkTreeIter itr;
    gchar *length_str;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST_QUEUE(playlist_q) && entry);
    
    new_ref = xfmedia_playlist_entry_ref_copy((XfmediaPlaylistEntryRef *)entry);
    g_queue_push_tail(playlist_q->priv->entries, new_ref);
    
    if(entry_length < 0)
        length_str = g_strdup("??:??");
    else {
        length_str = g_strdup_printf("%02d:%02d",
                                     entry_length / 60,
                                     entry_length % 60);
    }
    
    gtk_list_store_append(playlist_q->priv->list_store, &itr);
    gtk_list_store_set(playlist_q->priv->list_store, &itr,
                       COL_DISPLAY_NAME, display_name,
                       COL_ENTRY_LENGTH, length_str,
                       -1);
    g_free(length_str);
    
    g_signal_emit(G_OBJECT(playlist_q), __signals[SIG_ENTRY_ADDED], 0);
}

XfmediaPlaylistEntryRef *
xfmedia_playlist_queue_pop_entry(XfmediaPlaylistQueue *playlist_q)
{
    XfmediaPlaylistEntryRef *ref = NULL;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST_QUEUE(playlist_q), NULL);
    
    ref = g_queue_pop_head(playlist_q->priv->entries);
    if(ref) {
        GtkTreeIter itr;
        gtk_tree_model_get_iter_first(GTK_TREE_MODEL(playlist_q->priv->list_store),
                                      &itr);
        gtk_list_store_remove(playlist_q->priv->list_store, &itr);
        
        g_signal_emit(G_OBJECT(playlist_q), __signals[SIG_ENTRY_REMOVED], 0);
    }
    
    return ref;
}

XfmediaPlaylistEntryRef *
xfmedia_playlist_queue_peek_entry(XfmediaPlaylistQueue *playlist_q)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST_QUEUE(playlist_q), NULL);
    
    return g_queue_peek_head(playlist_q->priv->entries);
}

void
xfmedia_playlist_queue_clear(XfmediaPlaylistQueue *playlist_q)
{
    XfmediaPlaylistEntryRef *ref;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST_QUEUE(playlist_q));
    
    if(g_queue_get_length(playlist_q->priv->entries)) {
        while((ref = g_queue_pop_head(playlist_q->priv->entries)))
            xfmedia_playlist_entry_ref_destroy(ref);
        
        gtk_list_store_clear(playlist_q->priv->list_store);
        
        g_signal_emit(G_OBJECT(playlist_q), __signals[SIG_EMPTIED], 0);
    }
}

guint
xfmedia_playlist_queue_n_entries(XfmediaPlaylistQueue *playlist_q)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST_QUEUE(playlist_q), 0);
    return g_queue_get_length(playlist_q->priv->entries);
}
