/*
	Create a number of circles used for the radio-button images.
*/

#include <glib.h>
#include <math.h>

/*
 *	Declarations
 */

/* buffer */
typedef struct {
	gint	width;
	gint	height;
	guchar	*data;
} Buffer;

#define PIXEL(buffer, x, y) \
	((buffer)->data[(y)*(buffer)->width + (x)])

Buffer *	alloc_buffer	(gint width, gint height);
void		free_buffer		(Buffer *buffer);
Buffer *	sample_buffer	(Buffer *buffer, guint new_width, guint new_height);
void		print_buffer	(Buffer *buffer, const gchar *name);

void		clear_buffer	(Buffer *buffer);
void		normalize_buffer(Buffer *buffer, guchar min, guchar max);
void		subtract_buffer	(Buffer *a, Buffer *b);
void		transpose_buffer(Buffer *buffer);
void		average_buffers	(Buffer *a, Buffer *b);
void		modulate_one_minus_buffer (Buffer *a, Buffer *b);

/* draw */
void		draw_circle		(Buffer *buffer, gint x, gint y, gint radius, guchar value);
void		fill_circle		(Buffer *buffer, gint x, gint y, gint radius, guchar value);
void		shade_circle	(Buffer *buffer, gint x, gint y, gint radius, gboolean bottom);


/* Main */

typedef enum {
	CIRCLE_OUTLINE			= 0,
	CIRCLE_FILLED			= 1,
	CIRCLE_SHADED_TOP		= 2,
	CIRCLE_SHADED_BOTTOM	= 3,
	CIRCLE_DIMPLE			= 4,
	
	CIRCLE_TYPE_MASK		= 0x0f,
	CIRCLE_NORMALIZE		= 0x30,
	CIRCLE_NORMALIZE_POS	= 0x20,
	CIRCLE_NORMALIZE_NEG	= 0x10
} CircleType;

static void
print_circle (const gchar	*name,
			  guint			size,
			  gdouble		outer_radius,
			  gdouble		inner_radius,
			  guint			sampling,
			  CircleType	circle_type)
{
	CircleType circle_type2;
	Buffer	*large, *small;
	Buffer	*small2 = NULL;
	guint	large_size;
	guint	pos;
	
	g_return_if_fail (name != NULL);
	g_return_if_fail (size > 0);
	g_return_if_fail (outer_radius > 0.0);
	g_return_if_fail (inner_radius <= outer_radius);
	g_return_if_fail (sampling >= 1.0);
	
	large_size	 = (size * sampling) | 1;
	pos			 = large_size / 2;
	
	circle_type2 = circle_type & CIRCLE_TYPE_MASK;
	
	large = alloc_buffer (large_size, large_size);
	switch (circle_type2) {
	  case CIRCLE_OUTLINE:
		fill_circle (large, pos, pos, outer_radius * sampling, 0xff);
		small = sample_buffer (large, size, size);
		
		clear_buffer (large);
		fill_circle (large, pos, pos, inner_radius * sampling, 0xff);
		small2 = sample_buffer (large, size, size);
		
		if (circle_type & CIRCLE_NORMALIZE_POS)
			normalize_buffer (small, 0, PIXEL(small, 0, size/2));
		
		if (circle_type & CIRCLE_NORMALIZE_NEG)
			normalize_buffer (small2, PIXEL(small2, 0, 0), PIXEL(small2, 1, size/2));
		
		subtract_buffer (small, small2);
		break;
		
	  case CIRCLE_FILLED:
		fill_circle (large, pos, pos, outer_radius * sampling, 0xff);
		small = sample_buffer (large, size, size);
		if (circle_type & CIRCLE_NORMALIZE_POS)
			normalize_buffer (small, PIXEL(small, 0,0), PIXEL(small, 0, size/2));
		break;
		
	  case CIRCLE_SHADED_TOP:
	  case CIRCLE_SHADED_BOTTOM:
		shade_circle (large, pos, pos, outer_radius * sampling, (circle_type2 == CIRCLE_SHADED_BOTTOM));
		small = sample_buffer (large, size, size);
		
		clear_buffer (large);
		fill_circle (large, pos, pos, inner_radius * sampling, 0xff);
		small2 = sample_buffer (large, size, size);
		
		if (circle_type & CIRCLE_NORMALIZE_POS) {
			guchar min, max, t;
			min = PIXEL(small, size/2, size - 1) ;
			max = PIXEL(small, 0, size/2);
			if (circle_type2 == CIRCLE_SHADED_BOTTOM) {
				t = min; min = max; max = t; 
			}
			normalize_buffer (small, min, max);
		}
		if (circle_type & CIRCLE_NORMALIZE_NEG)
			normalize_buffer (small2, PIXEL(small2, 0, 0), PIXEL(small2, 1, size/2));
		
		subtract_buffer (small, small2);
		break;
		
	  case CIRCLE_DIMPLE:
		shade_circle (large, pos, pos, outer_radius * sampling, FALSE);
		small = sample_buffer (large, size, size);
		
		if (circle_type & CIRCLE_NORMALIZE_POS) {
			guchar min, max, t;
			min = PIXEL(small, size/2, size - 1) ;
			max = PIXEL(small, 0, size/2);
			if (circle_type2 == CIRCLE_SHADED_BOTTOM) {
				t = min; min = max; max = t; 
			}
			normalize_buffer (small, min, max);
		}
	}
	
	free_buffer (large);
	if (small2)
		free_buffer (small2);
	
	print_buffer (small, name);
	free_buffer (small);
}

int main (int argc, char **argv) {
	gint sampling;
	
	sampling = 16;
	if (argc > 1) {
		sampling = (gint) g_strtod(argv[1], NULL);
		sampling = CLAMP(sampling, 0,1024);
	}
	
	/*
	// top15/bot15 .. top7/bot7
	print_circle ("circle_top15_data", 15, 7.5, 6.5, sampling, CIRCLE_SHADED_TOP);
	print_circle ("circle_bot15_data", 15, 7.5, 6.5, sampling, CIRCLE_SHADED_BOTTOM);
	
	print_circle ("circle_top13_data", 13, 6.5, 5.5, sampling, CIRCLE_SHADED_TOP);
	print_circle ("circle_bot13_data", 13, 6.5, 5.5, sampling, CIRCLE_SHADED_BOTTOM);
	
	print_circle ("circle_top11_data", 11, 5.5, 4.5, sampling, CIRCLE_SHADED_TOP);
	print_circle ("circle_bot11_data", 11, 5.5, 4.5, sampling, CIRCLE_SHADED_BOTTOM);
	
	print_circle ("circle_top9_data", 9, 4.5, 3.5, sampling, CIRCLE_SHADED_TOP);
	print_circle ("circle_bot9_data", 9, 4.5, 3.5, sampling, CIRCLE_SHADED_BOTTOM);
	
	print_circle ("circle_top7_data", 7, 3.5, 2.5, sampling, CIRCLE_SHADED_TOP);
	print_circle ("circle_bot7_data", 7, 3.5, 2.5, sampling, CIRCLE_SHADED_BOTTOM);
	
	// base11, base9, base7: rings, 2, 1.5 and 1 pixels thick.
	print_circle ("circle_ring11_data", 11, 5.5, 4.5, sampling, CIRCLE_OUTLINE);
	print_circle ("circle_ring9_data", 9, 4.5, 3.0, sampling, CIRCLE_OUTLINE);
	print_circle ("circle_ring7_data", 7, 3.5, 2.5, sampling, CIRCLE_OUTLINE);
	
	// bg11, bg9, bg7, bg6, bg5
	print_circle ("circle_solid11_data", 11, 5.5, 0.0, sampling, CIRCLE_FILLED);
	print_circle ("circle_solid9_data", 9, 4.5, 0.0, sampling, CIRCLE_FILLED);
	print_circle ("circle_solid7_data", 7, 3.5, 0.0, sampling, CIRCLE_FILLED);
	print_circle ("circle_solid7_6_data", 7, 3.0, 0.0, sampling, CIRCLE_FILLED);
	print_circle ("circle_solid6_data", 6, 3.0, 0.0, sampling, CIRCLE_FILLED);
	print_circle ("circle_solid5_data", 5, 2.5, 0.0, sampling, CIRCLE_FILLED);
	*/
	
	print_circle ("dimple_angle_data", 6, 3.0, 0.0, sampling, CIRCLE_DIMPLE);
	print_circle ("dimple_circle3_data", 3, 1.5, 0.0, sampling, CIRCLE_FILLED);
	print_circle ("dimple_circle5_data", 5, 2.5, 0.0, sampling, CIRCLE_FILLED);
	
	return 0;
}


/*
 *	Buffer
 */

#define PIXEL(buffer, x, y) \
	((buffer)->data[(y)*(buffer)->width + (x)])

Buffer *
alloc_buffer (gint width, gint height)
{
	Buffer	*buf;
	
	g_return_val_if_fail (width > 0, NULL);
	g_return_val_if_fail (height > 0, NULL);
	
	buf = g_new(Buffer, 1);
	buf->data = g_malloc0 (width * height);
	buf->width = width;
	buf->height = height;
	return buf;
}

void
free_buffer (Buffer *buf)
{
	g_return_if_fail (buf != NULL);
	
	if (buf->data)
		g_free (buf->data);
	
	g_free(buf);
}

static gdouble
sample_line (Buffer *src, gdouble sx, guint y, gdouble x_step)
{
	const guchar *line;
	gdouble	sx2, frac_sx, frac_sx2, v;
	gint	int_sx, int_sx2;
	guint	i;
	
	line = &src->data[y*src->width];
	
	sx2 = sx + x_step;
	
	int_sx	= (gint)floor(sx);
	int_sx2	= (gint)floor(sx2);
	frac_sx	 = sx - floor(sx);
	frac_sx2 = sx2 - floor(sx2);
	
	v = (line[int_sx]/255.0) * (1.0 - frac_sx);
	
	for (i = int_sx + 1; i < int_sx2; ++i)
		v += line[i] / 255.0;
	
	if (int_sx2 < src->width)
		v += (line[int_sx2] / 255.0) * frac_sx2;
	
	return v / x_step;
}

Buffer *
sample_buffer (Buffer *src, guint width, guint height)
{
	Buffer	*dst;
	
	gdouble	x_step, y_step;
	gdouble	v;
	
	gdouble	sx, sy, sy2, frac_sy, frac_sy2;
	gint	int_sy, int_sy2;
	gint	x, y, i;
	gint	c;
	
	g_return_val_if_fail (src != NULL, NULL);
	g_return_val_if_fail (width <= src->width, NULL);
	g_return_val_if_fail (height <= src->height, NULL);
	g_return_val_if_fail (width > 0, NULL);
	g_return_val_if_fail (height > 0, NULL);
	
	dst = alloc_buffer (width, height);
	x_step = ((gdouble)src->width)/width;
	y_step = ((gdouble)src->height)/height;
	
	for (y = 0, sy = 0.0; y < height; ++y, sy += y_step) {
		for (x = 0, sx = 0.0; x < width; ++x, sx += x_step) {
			sy2 = sy + y_step;
			
			int_sy	= (gint)floor(sy);
			int_sy2	= (gint)floor(sy2);
			
			frac_sy  = sy - floor(sy);
			frac_sy2 = sy2 - floor(sy2);
			
			v = sample_line (src, sx, int_sy, x_step) * (1.0 - frac_sy);
			
			for (i = int_sy + 1; i < int_sy2; ++i)
				v += sample_line (src, sx, i, x_step);
			
			if (int_sy2 < src->height)
				v += sample_line (src, sx, int_sy2, x_step) * frac_sy2;
			
			c = (gint)(v / y_step * 255.0);
			
			PIXEL(dst, x, y) = CLAMP(c, 0x00, 0xff);
		}
	}
	return dst;
}

void clear_buffer (Buffer *buffer)
{
	gint i, n;
	
	n = buffer->width * buffer->height;
	for (i = 0; i < n; ++i) {
		buffer->data[i] = 0;
	}
}

void normalize_buffer (Buffer *buffer, guchar min, guchar max)
{
	gint	i, n, f, t;
	
	//if (max <= min)
	//	return;
	
	n = buffer->width * buffer->height;
	// f = 0x0ff00000 / (max - min);
	// g_printerr ("f = 0x%08x\n", f);
	f = 0x0010d3ec;
	for (i = 0; i < n; ++i) {
		t = ((buffer->data[i] - min) * f + 0x00080000) >> 20;
		t = CLAMP(t, 0.0, 255.0);
		buffer->data[i] = (guchar)t;
	}
}

void modulate_one_minus_buffer (Buffer *a, Buffer *b)
{
	Buffer 	*tmp = NULL;
	gint	i, n;
	gdouble	t;
	
	if (b->width != a->width || b->height != a->height) {
		tmp = sample_buffer (b, a->width, a->height);
		b = tmp;
	}
	
	n = a->width * b->height;
	for (i = 0; i < n; ++i) {
		t = a->data[i] * (1.0 - b->data[i]/255.0);
		t = CLAMP(t, 0.0, 255.0);
		a->data[i] = (guchar)t;
	}
	if (tmp)
		free_buffer(tmp);
}

void subtract_buffer (Buffer *a, Buffer *b)
{
	Buffer 	*tmp = NULL;
	gint	i, n, t;
	
	if (b->width != a->width || b->height != a->height) {
		tmp = sample_buffer (b, a->width, a->height);
		b = tmp;
	}
	
	n = a->width * b->height;
	for (i = 0; i < n; ++i) {
		t = a->data[i] - b->data[i];
		t = CLAMP(t, 0, 255);
		a->data[i] = (guchar)t;
	}
	if (tmp)
		free_buffer(tmp);
}

void transpose_buffer (Buffer *buffer)
{
	guchar	a, b, *ap, *bp;
	gint	x, y, side;
	
	side = MIN(buffer->width, buffer->height);
	for (y = 0; y < side; ++y) {
		for (x = 0; x < side - y - 1; ++x) {
			ap = &PIXEL(buffer, x, y); a = *ap;
			bp = &PIXEL(buffer, side - 1 - y, side - 1 - x); b = *bp;
			*ap = b;
			*bp = a;
		}
	}
}

void average_buffers (Buffer *a, Buffer *b)
{
	gint i, n, t;
	
	n = a->width * a->height;
	for (i = 0; i < n; ++i) {
		t = a->data[i] + (gint)b->data[i];
		t = CLAMP(t, 0, 0xff);
		a->data[i] = t >> 1;
		b->data[i] = (t >> 1) + (t & 1);
	}
}


void print_buffer (Buffer *buffer, const gchar *name)
{
	gint x, y;

	g_return_if_fail (buffer != NULL);
	g_return_if_fail (name != NULL);
	
	g_print ("const static guchar %s[%d*%d] = {", name, buffer->width, buffer->height);
	for (y = 0; y < buffer->height; ++y) {
		// g_print ("\n\t{");
		g_print ("\n\t");
		for (x = 0; x < buffer->width; ++x) {
			g_print (" 0x%02x", PIXEL(buffer, x, y));
			if (x < buffer->width - 1)
				g_print (",");
		}
		// g_print (" }");
		if (y < buffer->width - 1)
			g_print (",");
	}
	g_print("\n};\n\n");
}


/* Draw */

#define VALID_CIRCLE(buffer, x, y, radius) \
	g_return_if_fail (buffer != NULL); \
	g_return_if_fail (x > 0 && y > 0); \
	g_return_if_fail (radius <= x && radius <= y); \
	g_return_if_fail (x + radius <= buffer->width); \
	g_return_if_fail (y + radius <= buffer->height)


void
draw_horizontal_line (Buffer *buffer,
					  gint x, gint y, gint width,
					  guchar value)
{
	guchar *p;
	
	g_return_if_fail (buffer != NULL);
	
	p = &buffer->data[y*buffer->width + x];
	while (width--)
		*(p++) = value;
}

void
draw_circle (Buffer *buffer,
			 gint x, gint y, gint radius,
			 guchar value)
{
	gint d, u, v;
	
	VALID_CIRCLE(buffer, x, y, radius);
	
	d = 3 - (2 * radius);
	for (u = 0, v = radius; u <= v; ++u) {
		PIXEL(buffer, x + u, y + v) = value;
		PIXEL(buffer, x + u, y - v) = value;
		PIXEL(buffer, x - u, y + v) = value;
		PIXEL(buffer, x - u, y - v) = value;
		PIXEL(buffer, x + v, y + u) = value;
		PIXEL(buffer, x + v, y - u) = value;
		PIXEL(buffer, x - v, y + u) = value;
		PIXEL(buffer, x - v, y - u) = value;
		
		if (d < 0) {
			d += (4*u) + 6;
		} else {
			d += 4*(u - v) + 10;
			--v;
		}
	}
}

void
fill_circle (Buffer *buffer,
			 gint x, gint y, gint radius,
			 guchar value)
{
	gint d, u, v;
	
	VALID_CIRCLE(buffer, x, y, radius);
	
	d = 3 - (2 * radius);
	for (u = 0, v = radius; u <= v; ++u) {
		draw_horizontal_line (buffer, x - v, y + u, v*2+1, value);
		draw_horizontal_line (buffer, x - v, y - u, v*2+1, value);
		if (d < 0) {
			d += (4*u) + 6;
		} else {
			draw_horizontal_line (buffer, x - u, y + v, u*2+1, value);
			draw_horizontal_line (buffer, x - u, y - v, u*2+1, value);
			d += 4*(u - v) + 10;
			--v;
		}
	}
}

static void
shade_line (Buffer *buffer,
			gint x, gint y,
			gint u, gint v,
			gboolean invert)
{
	gdouble	f;
	gint	i;
	guchar	c;
	
	g_return_if_fail (buffer != NULL);
	
	if (u <= 0)
		return;
	
	for (i = 1; i < u; ++i) {
		f = atan(i / (gdouble)v) * 0.636619904813;
		c = CLAMP(f, 0.0, 1.0) * 255;
		if (invert)
			c = 255 - c;
		
		PIXEL(buffer, x + v, y - i) = c;
		PIXEL(buffer, x - i, y + v) = c;
	}
}

void
shade_circle (Buffer *buffer,
			  gint x, gint y, gint radius,
			  gboolean bottom)
{
	gint	d, u, v;
	guchar	top_value, bottom_value;
	
	VALID_CIRCLE(buffer, x, y, radius);
	
	if (bottom) {
		top_value = 0x00;
		bottom_value = 0xff;
	} else {
		top_value = 0xff;
		bottom_value = 0x00;
	}
	
	d = 3 - (2 * radius);
	for (u = 0, v = radius; u <= v; ++u) {
		draw_horizontal_line (buffer, x - v, y - u, v + 1, top_value);
		draw_horizontal_line (buffer, x,     y + u, v + 1, bottom_value);
		shade_line (buffer, x, y, v + 1, u, bottom);
		
		if (d < 0) {
			d += (4*u) + 6;
		} else {
			draw_horizontal_line (buffer, x - u, y - v, u + 1, top_value);
			draw_horizontal_line (buffer, x,     y + v, u + 1, bottom_value);
			shade_line (buffer, x, y, u + 1, v, bottom);
			
			d += 4*(u - v) + 10;
			--v;
		}
	}
}

