#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2006 Jd & Hap Hazard
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#

import sys, os, select, stat
import paramiko
import getpass

import xen.xend.XendClient
import xmlrpclib
import XenServerProxy
from XenServerProxy import ServerProxy, SSHTunnelTransport

import subprocess
from phelper import PHelper


#TODO : Special handling for xenconsole (look in to interactive.py)

class CommandException(Exception):
    def __init__(self, errno, description):
        self.errno = errno
        self.description = description

    def __repr__(self):
        return "[Error %d], %s" % (self.errno, self.description)

    def __str__(self):
        return self.__repr__()




class Node(XenServerProxy.ServerProxy):
    """ The Node represents either local or remote note and provides
        -- Local/Remote Xend interface
        -- Local/Remote command execution
        -- Local/Remote file handling
    """
    DEFAULT_TCP_PORT = '8005'
    DEFAULT_PATH = '/RPC2'
    DEFAULT_USER = 'root'


    file_functions =  {"open": "open",
                       "mkdir": "os.mkdir",
                       "listdir": "os.listdir",
                       "remove": "os.remove",
                       "rmdir": "os.rmdir",
                       "rename": "os.rename",
                       "unlink": "os.unlink",
                       "chdir": "os.chdir",
                       "chmod": "os.chmod",
                       "symlink": "os.symlink"}

    
    def _init_fptrs(self):
        if self._fptrs != None:
            return self._fptrs
        
        self._fptrs = {}
        for fn in self.file_functions.keys():
            if self.isRemote:
                self._fptrs[fn] = eval("self.sftp_client." + fn)
            else:
                self._fptrs[fn] = eval(self.file_functions[fn]) # built in
                
        return self._fptrs

    def _init_sftp(self):
        if self._sftp_client != None:
            return self._sftp_client
        
        self._sftp_client = \
                         paramiko.SFTPClient.from_transport(self.ssh_transport)
        return self._sftp_client
    

    def __getattr__(self, name):
        if name == 'sftp_client': 
            return self._init_sftp()
        if name == 'fptrs': 
            return self._init_fptrs()
        if name in self.file_functions:
            return self.fptrs[name]
        return XenServerProxy.ServerProxy.__getattr__(self, name)
        

    # Node follows
    def __init__(self,
                 hostname = None,
                 ssh_port=22,
                 username=DEFAULT_USER,
                 password=None,
                 isRemote=False,
                 protocol = "ssh_tunnel",
                 tcp_port = DEFAULT_TCP_PORT):
                 

        self.isRemote = isRemote
        self.hostname = hostname
        self.ssh_port = ssh_port
        self.username = username
        self.password = password
        
        self.tcp_port = tcp_port
        self.protocol = protocol

        self.ssh_transport = None
        self._sftp_client = None
        
        # pointers to functions for doing file / directory manipulations
        self._fptrs = None

        if isRemote:
            try:
                self.ssh_transport = \
                PHelper.init_ssh_transport(self.hostname,
                                           ssh_port = self.ssh_port,
                                           username=self.username,
                                           password=self.password)
            except Exception, ex:
                print "Could not initialize ssh for %s %d" % (hostname,
                                                              ssh_port)
                raise

            if self.protocol == "tcp":
                ServerProxy.__init__(self,'http://' + hostname + ':' + 
                                     str(self.tcp_port) + Node.DEFAULT_PATH)
            
            if self.protocol == "ssh":
                ServerProxy.__init__(self,'ssh://' + self.username +'@' +
                                     hostname + Node.DEFAULT_PATH)

            if self.protocol == "ssh_tunnel":
                # create SSHTunnelTransport and pass it to ServerProxy
                self.tunnel_transport = SSHTunnelTransport(hostname, tcp_port,
                                            ssh_transport = self.ssh_transport)
                ServerProxy.__init__(self,'ssh_tunnel://' + self.username +
                                     '@' +
                                     hostname + ":" + str(self.tcp_port) +
                                     Node.DEFAULT_PATH, password=password,
                                     transport=self.tunnel_transport)    
        else:
           xen.util.xmlrpclib2.ServerProxy.__init__(self,
                                        'httpu:///var/run/xend/xmlrpc.sock')


    ## TBD : fix cleanup : IMPORTANT
    def cleanup(self):
        #if self.tunnel_transport is not None:
        #    self.tunnel_transport.cleanup()
        pass
            
    def exec_cmd(self, cmd):
        if self.isRemote:
            return self.remote_exec_cmd(cmd)
        else:
            return self.local_exec_cmd(cmd)
                  
    def local_exec_cmd(self, cmd):
        p1 = subprocess.Popen(cmd,
                              stdout=subprocess.PIPE,stderr=subprocess.STDOUT,
                              universal_newlines=True,
                              shell=True, close_fds=True)
        out = p1.communicate()[0]
        exit_code   = p1.returncode
        return out, exit_code


    def remote_exec_cmd(self, cmd):
        """
        Open channel on transport, run remote command,
        returns (stdout/stdin,process exit code)
        """

        out = ''
        exit_code = 0
        if self.ssh_transport is None:
            raise CommandException(0, "Transport not initialized")

        try:
            chan = self.ssh_transport.open_session()
            chan.set_combine_stderr(True)
            chan.setblocking(0)

            x = chan.exec_command(cmd)

            ### Read when data is available
            while select.select([chan,], [], []):
                    x = chan.recv(1024)
                    if not x: break
                    out += x
                    select.select([],[],[],.1)
            exit_code = chan.recv_exit_status()
            chan.close()
        except SSHException, ex:
            raise CommandException(0, str(ex))
        
        return out, exit_code
    

    # File operations
    # see __getattr__ and fptrs which implement pass through to the
    # sftp client object.
    # some additional ones implemented here.

    def file_exists(self, filename):
        if self.isRemote:
            try:
                file_attribs = self.sftp_client.lstat(filename)
            except IOError, err:
                if err.errno == 2:  # ENOENT
                    return False
                raise
            return True
        else:
            return os.path.exists(filename)

    def file_is_writable(self, filename):
        """ Check for write permissions on 'filename'"""
        try:
            if self.isRemote:
                mode = self.sftp_client.stat(filename).st_mode
                return bool(stat.S_IMODE(mode) & stat.S_IWRITE)
            else:
                return os.access(filename,os.W_OK)
        except IOError:
            return False        
        

    def put(self, local, remote):
        if self.isRemote:
            self.sftp_client.put(local, remote)
        else:
            raise Exception("Put allowed only in remote mode")


    def get(self, local, remote):
        if self.isRemote:
            self.sftp_client.get(local, remote)
        else:
            raise Exception("get allowed only in remote mode")
    



# main block
username = 'root'
hostname = '192.168.123.155'
hostport = 22


if __name__ == '__main__':

    # quick test for exists.
    node = Node(hostname,isRemote=True, protocol="tcp")
    # Test FTP
    node.put("/tmp/send", "/tmp/send_r")
    node.get("/tmp/send_r", "/tmp/received")

    fd = node.open('/tmp/test_writable','w')
    off = 1024L
    fd.seek(off,0)
    fd.write('\x00')
    fd.close()

    print 'exists?: ',node.file_exists('/tmp/test_writable')
    print 'isWritable?: ', node.file_is_writable('/tmp/test_writable')
    node.remove('/tmp/test_writable')
    print 'exists?: ', node.file_exists('/tmp/test_writable')
    

    sys.exit(0)
    
    print node.xend.domains(0)

    fname = "/etc/xenman.conf"
    if node.file_exists(fname):
        print fname, "Exists"
    else:
        print fname, "does not Exist"
        

    fname = "/etc/xen/junk12"
    if node.file_exists(fname):
        print fname, "Exists"
    else:
        print fname, "does not Exist"
        

    for remote in (False,True):

        node = Node(hostname,isRemote=remote, protocol="tcp")

        # test file operations
        f = node.open("/etc/xenman.conf")
        x= f.read(1024)
        print x
        f.close()

        try:
            node.mkdir("/tmp/node_test")
        except (OSError, IOError), err:
            print str(err)

        w = node.open("/tmp/node_test/test", "w")
        w.writelines(["hello this is test", "hello this is second test"])
        w.close()

        r = node.open("/tmp/node_test/test")
        x = r.readline()
        while x != None and x != "": 
            print x
            x= r.readline()
        r.close()
        
        node.remove("/tmp/node_test/test")
        node.rmdir("/tmp/node_test")

    
        
        print node.xend.domains(0)

        output,code = node.exec_cmd('ls -la /')
        print output
        print "EXIT CODE = ", code

        output,code = node.exec_cmd('find /tmp')
        print output
        print "EXIT CODE = ", code

        output,code = node.exec_cmd('junk /tmp')
        print output
        print "EXIT CODE = ", code


        output,code = node.exec_cmd('touch x')
        print output
        print "EXIT CODE = ", code

        node.cleanup()

        

