// molecule_tools.cpp - Molecule's implementation of (more) functions

#include <qobject.h>
#include <qlist.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <iostream.h>

#include "graphdialog.h"
#include "moldata.h"
#include "render2d.h"
#include "drawable.h"
#include "molecule.h"
#include "dpoint.h"
#include "defs.h"
#include "namer.h"

// Prediction stuff
#include "boshcp.h"
// SSSR (see comments in this file)
#include "molecule_sssr.h"

// note that this function builds the DPoint::neighbors list as well
void Molecule::MakeSSSR() {
  QList<DPoint> groupAtoms = AllPoints();

  for (tmp_pt = groupAtoms.first(); tmp_pt != 0; tmp_pt = groupAtoms.next() ) {
    tmp_pt->neighbors.clear();
    for(tmp_bond = bonds.first(); tmp_bond != 0; tmp_bond = bonds.next() ) {
      if (tmp_bond->Find(tmp_pt) == true) {
	tmp_pt->neighbors.append(tmp_bond->otherPoint(tmp_pt));
	tmp_pt->bondorder.at(tmp_pt->neighbors.count() - 1) = tmp_bond->Order();
      }
    }
  }
  this_sssr.BuildSSSR(groupAtoms);
  this_sssr.PrintSSSR();
  this_sssr.FindAromatic(bonds);
}

void Molecule::CalcName() {
  Namer n1;
  QString nom = n1.GetName(AllPoints(), bonds);
  cout << nom << endl;
}

void Molecule::Calc13CNMR() {
  BremserOneSphereHOSECodePredictor boshcp;
  QStringList hosecodes;
  QList<DPoint> up;
  QList<DPoint> sphere1;
  QString tmp_str, ts;
  DPoint *tmp_pt2, *tmp_pt3;
  int hs, hsmax;

  // get list of unique points
  up = AllPoints();
  // find rings (esp. find aromaticity) - do after CopyTextToDPoint()
  MakeSSSR();

  // Scan for keto groups
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next()) {
    tmp_pt->ketos = 0;
    for (tmp_bond = bonds.first(); tmp_bond != 0; tmp_bond = bonds.next()) {
      if (tmp_bond->Find(tmp_pt) == true) {
	tmp_pt2 = tmp_bond->otherPoint(tmp_pt);
	if ( (tmp_pt2->element == "O") && (tmp_bond->Order() == 2) )
	  tmp_pt->ketos += 1;
      }
    }
  }
  // Determine atoms surrounding each atom and build HOSE code list
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next()) {
    if (tmp_pt->element == "C") {  // only look at carbons
      for (tmp_bond = bonds.first(); tmp_bond != 0; tmp_bond = bonds.next()) {
	if (tmp_bond->Find(tmp_pt) == true) {
	  tmp_str = "";
	  hs = 0;
	  tmp_pt2 = tmp_bond->otherPoint(tmp_pt);
	  if (tmp_bond->Order() == 3) { tmp_str.append("%"); hs += 300; }
	  if ( (tmp_pt->aromatic == true) && (tmp_pt2->aromatic == true) ) {
	    tmp_str.append("*"); hs += 100;
	  } else {
	    if (tmp_bond->Order() == 2) {
	      if (tmp_bond->Dash() == 0) 
		{ tmp_str.append("="); hs += 200; }
	      if (tmp_bond->Dash() == 1)
		{ tmp_str.append("*"); hs += 100; }
	    }
	  }
	  if (tmp_pt2->element == "C") { tmp_str.append("C"); hs += 24; }
	  if (tmp_pt2->element == "O") { tmp_str.append("O"); hs += 22; }
	  if (tmp_pt2->element == "OH") { tmp_str.append("O"); hs += 22; }
	  if (tmp_pt2->element == "HO") { tmp_str.append("O"); hs += 22; }
	  if (tmp_pt2->element == "N") { tmp_str.append("N"); hs += 20; }
	  if (tmp_pt2->element == "NH2") { tmp_str.append("N"); hs += 20; }
	  if (tmp_pt2->element == "H2N") { tmp_str.append("N"); hs += 20; }
	  if (tmp_pt2->element == "NH") { tmp_str.append("N"); hs += 20; }
	  if (tmp_pt2->element == "HN") { tmp_str.append("N"); hs += 20; }
	  if (tmp_pt2->element == "P") { tmp_str.append("P"); hs += 18; }
	  if (tmp_pt2->element == "S") { tmp_str.append("S"); hs += 16; }
	  if (tmp_pt2->element == "Si") { tmp_str.append("Q"); hs += 14; }
	  if (tmp_pt2->element == "B") { tmp_str.append("B"); hs += 12; }
	  if (tmp_pt2->element == "F") { tmp_str.append("F"); hs += 10; }
	  if (tmp_pt2->element == "Cl") { tmp_str.append("X"); hs += 8; }
	  if (tmp_pt2->element == "Br") { tmp_str.append("Y"); hs += 6; }
	  if (tmp_pt2->element == "I") { tmp_str.append("I"); hs += 4; }
	  if (tmp_pt2->ketos == 2) { tmp_str.append("$$"); }
	  if (tmp_pt2->ketos == 1) { tmp_str.append("$"); }
	  tmp_pt2->tmphose = tmp_str;
	  tmp_pt2->hosescore = hs;
	  sphere1.append(tmp_pt2);
	}
      }
      tmp_str = "";
      do {
	hsmax = -1;
	for (tmp_pt3 = sphere1.first(); tmp_pt3 != 0; 
	     tmp_pt3 = sphere1.next()) {
	  if (tmp_pt3->hosescore > hsmax) { 
	    hsmax = tmp_pt3->hosescore; 
	    ts = tmp_pt3->tmphose;
	    tmp_pt2 = tmp_pt3;
	  }
	}
	sphere1.remove(tmp_pt2);
	tmp_str.append(ts);
      } while (sphere1.count() > 0);
      tmp_str.append("(//)");
      hosecodes.append(tmp_str);
      cout << tmp_str << endl;
    }
  }

  GraphDialog g(r, "Predicted 13C-NMR");
  // print (for now) list of NMR values
  for ( QStringList::Iterator it = hosecodes.begin(); 
	it != hosecodes.end(); ++it ) {
    //cout << boshcp.predictFull(*it) << endl;
    g.AddPeak(boshcp.predict(*it), boshcp.getMult(*it), 
	      boshcp.predictFull(*it));
  }

  QPixmap mol = r->MakeFullPixmap();
  QRect rr1 = BoundingBoxAll();
  rr1.setLeft(rr1.left() - 4);
  rr1.setTop(rr1.top() - 4);
  rr1.setBottom(rr1.bottom() + 4);
  rr1.setRight(rr1.right() + 4);
  QPixmap mol1(rr1.size());
  bitBlt(&mol1, QPoint(0,0), &mol, rr1);

  g.AddPixmap(mol1);
  g.exec();
}

Text *Molecule::CalcEmpiricalFormula(bool from_mw) {
  QList<DPoint> up;
  QString ef;
  QStringList allatoms, oddatoms;

  up = AllPoints();
  // Split all labels into allatoms (one atom per entry)
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next() ) {
    // parse this string
    QString x = tmp_pt->element;
    QString iso;  // isotope MW
    QString thiselement;  // current element
    QString repeatnum;  // number of repeats
    int ptr = 0;  // cursor position (pointer) in x
    int isoflag = false;  // isotope-override normal MW lookup if MW specified
    int repeat = 1; // number of this atom
    do {
      iso.remove(0,999);
      thiselement.remove(0,999);
      repeatnum.remove(0,999);
      // Check if token starts with a number
      if (x.at(ptr).isNumber() == true) { // read isotope value
	isoflag = true;
	iso.append(x.at(ptr));
	ptr++;
	if (x.at(ptr).isNumber() == true) {
	  iso.append(x.at(ptr));
	  ptr++;
	}
	if (x.at(ptr).isNumber() == true) {
	  iso.append(x.at(ptr));
	  ptr++;
	}
      }
      // ptr now points to first letter of element
      thiselement.append(x.at(ptr));
      ptr++;
      // if next letter is lowercase, add it too
      if (x.at(ptr).category() == QChar::Letter_Lowercase) {
	thiselement.append(x.at(ptr));
	ptr++;
      }
      // if next character is number, it's the subscript
      if (x.at(ptr).isNumber() == true) {
	repeatnum.append(x.at(ptr));
	ptr++;
	if (x.at(ptr).isNumber() == true) {
	  repeatnum.append(x.at(ptr));
	  ptr++;
	}
	if (x.at(ptr).isNumber() == true) {
	  repeatnum.append(x.at(ptr));
	  ptr++;
	}
	repeat = repeatnum.toInt();
      }
      // Move to next letter/number
      if (ptr < x.length()) {
	if (x.at(ptr).isSpace() == true) ptr++;
      }
      // add atoms to list
      for (int cc = 0; cc < repeat; cc++)
	allatoms.append(thiselement);
      isoflag = false;
      repeat = 1;
    } while (ptr < x.length());
  }
  // need to find implicit hydrogens here!
  int num_c = 0, num_h = 0, num_n = 0, num_o = 0, num_p = 0, num_s = 0;
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next() ) {
    int possible_h = 0;
    possible_h = MolData::Hydrogens(tmp_pt->element);
    for (tmp_bond = bonds.first(); tmp_bond != 0; tmp_bond = bonds.next()) {
      if (tmp_bond->Find(tmp_pt))
	possible_h -= tmp_bond->Order();
    }
    if (possible_h < 0) {
      cout << tmp_pt->element << " resulted in negative hydrogens ";
      cout << "in Molecule::CalcEmpiricalFormula()" << endl;
      possible_h = 0;
    }
    num_h += possible_h;
  }

  // convert allatoms to formula: first extract C,H,O,N,P,S
  // calculate molecular weight too
  nmw = (double)num_h * 1.00794;
  for (QStringList::Iterator it = allatoms.begin(); it != allatoms.end();
       ++it) {
    nmw += MolData::NameToMW(*it);
    if (*it == "C") { num_c++; continue; }
    if (*it == "H") { num_h++; continue; }
    if (*it == "O") { num_o++; continue; }
    if (*it == "N") { num_n++; continue; }
    if (*it == "P") { num_p++; continue; }
    if (*it == "S") { num_s++; continue; }
    oddatoms.append(*it);
  }

  cout << "Found " << oddatoms.count() << " odd atoms." << endl;
  // sort odd atom list alphabetically (if necessary)
  if (oddatoms.count() > 0) oddatoms.sort();

  QString finalef, n1;
  if (num_c > 0) { 
    n1.setNum(num_c); 
    finalef = finalef + "C";
    if (num_c > 1) finalef = finalef + n1;
  }
  if (num_h > 0) { 
    n1.setNum(num_h); 
    finalef = finalef + "H"; 
    if (num_h > 1) finalef = finalef + n1;
  }
  if (num_o > 0) { 
    n1.setNum(num_o); 
    finalef = finalef + "O"; 
    if (num_o > 1) finalef = finalef + n1;
  }
  if (num_n > 0) { 
    n1.setNum(num_n); 
    finalef = finalef + "N"; 
    if (num_n > 1) finalef = finalef + n1;
  }
  if (num_p > 0) { 
    n1.setNum(num_p); 
    finalef = finalef + "P"; 
    if (num_p > 1) finalef = finalef + n1;
  }
  if (num_s > 0) { 
    n1.setNum(num_s); 
    finalef = finalef + "S"; 
    if (num_s > 1) finalef = finalef + n1;
  }
  // add odd atoms
  if (oddatoms.count() > 0) {
    do {
      QString te = oddatoms.first();
      int tc = 0;
      for (QStringList::Iterator ir = oddatoms.begin(); ir != oddatoms.end();
	   ++ir) {
	if (*ir == te) tc++;
      }
      finalef = finalef + te;
      n1.setNum(tc);
      if (tc > 1) finalef = finalef + n1;
      oddatoms.remove(te);
    } while (oddatoms.count() > 0);
  }

  // these are used for elemental analysis, don't delete
  nc = num_c; nh = num_h; nn = num_n; no = num_o;

  double nx, ny;

  QRect nr = BoundingBoxAll();
  ny = nr.top() - 16.0;
  nx = (nr.left() + nr.right()) / 2.0;

  tmp_text = new Text(r);
  tmp_text->setJustify(JUSTIFY_TOPLEFT);
  tmp_text->setText(finalef);
  for (int c = 0; c < finalef.length(); c++) {
    if (finalef[c].isLetter()) finalef.replace(c, 1, " ");
    if (finalef[c].isNumber()) finalef.replace(c, 1, "-");
  }
  tmp_text->setTextMask(finalef);
  tmp_pt = new DPoint(nx, ny);
  tmp_text->setPoint(tmp_pt);

  //if (from_mw == false) {
  //  text_formula = tmp_text;
  //  tmp_text->setMolecule(this);
  //  tmp_text->setDataType(TEXT_DATA_FORMULA);
  //}

  return tmp_text;
}

Text *Molecule::CalcMW(bool from_change) {
  Text *tmp_txt1, *tmp_text;

  tmp_txt1 = CalcEmpiricalFormula(true);

  double nx, ny;

  QRect nr = BoundingBoxAll();
  ny = nr.bottom() + 5.0;
  nx = (nr.left() + nr.right()) / 2.0;

  tmp_text = new Text(r);
  tmp_text->setJustify(JUSTIFY_TOPLEFT);
  QString finalmw;
  finalmw.setNum(nmw);
  finalmw = QString("MW = ") + finalmw;
  tmp_text->setText(finalmw);
  finalmw.fill(' ');
  tmp_text->setTextMask(finalmw);
  tmp_pt = new DPoint(nx, ny);
  tmp_text->setPoint(tmp_pt);

  //if (from_change == false) {
  //  text_mw = tmp_text;
  //  tmp_text->setMolecule(this);
  //  tmp_text->setDataType(TEXT_DATA_MW);
  //}

  return tmp_text;
}

/*
Text *Molecule::CalcMW() {
  QList<DPoint> up;
  double sum_mw = 0.0;
  int sumh = 0, sumbonds = 0;
  QString finalmw;

  // Get all unique DPoint's
  up = AllPoints();
  // Sum up molecular weight of each DPoint and get maximum hydrogens
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next() ) {
    sum_mw += MolData::MW(tmp_pt->element);
    sumh += MolData::Hydrogens(tmp_pt->element);
  }
  // Sum up bond orders
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next() ) {
    for (tmp_bond = bonds.first(); tmp_bond != NULL; tmp_bond = bonds.next()) {
      if (tmp_bond->Find(tmp_pt))
	sumbonds += tmp_bond->Order();
    }
  }
  // Add hydrogens
  cout << "Base MW = " << sum_mw << endl;
  cout << "Max hydrogens = " << sumh << endl;
  cout << "Bond order = " << sumbonds << endl;
  sumh -= sumbonds;
  sum_mw += (sumh * 1.00794);

  cout << "MW = " << sum_mw << endl;
  finalmw.setNum(sum_mw);

  double nx, ny;

  QRect nr = BoundingBoxAll();
  ny = nr.bottom() + 5.0;
  nx = (nr.left() + nr.right()) / 2.0;

  tmp_text = new Text(r);
  tmp_text->setJustify(JUSTIFY_TOPLEFT);
  finalmw = QString("MW = ") + finalmw;
  tmp_text->setText(finalmw);
  finalmw.fill(' ');
  tmp_text->setTextMask(finalmw);
  tmp_pt = new DPoint(nx, ny);
  tmp_text->setPoint(tmp_pt);

  nmw = sum_mw;
  return tmp_text;
}
*/

void Molecule::CalcElementalAnalysis() {
  QString ea, n1;

  // calc empirical formula and total molecular weight
  tmp_text = CalcMW();
  nc *= 12.011;
  nh *= 1.00794;
  nn *= 14.0067;
  no *= 15.994;
  nc = nc * 100 / nmw;
  nh = nh * 100 / nmw;
  no = no * 100 / nmw;
  nn = nn * 100 / nmw;

  ea.append("Elemental analysis:\n");
  ea.append("C = ");
  n1.setNum(nc);
  ea.append(n1);
  ea.append("%\n");
  ea.append("H = ");
  n1.setNum(nh);
  ea.append(n1);
  ea.append("%\n");
  ea.append("O = ");
  n1.setNum(no);
  ea.append(n1);
  ea.append("%\n");
  ea.append("N = ");
  n1.setNum(nn);
  ea.append(n1);
  ea.append("%");

  QMessageBox::information(r, "Elemental analysis", ea);
}

QString Molecule::IUPAC_Name() {
  return "Spam";
}

// add hydrogens to atoms.  Only add to carbon if to_carbons == true
void Molecule::AddHydrogens(bool to_carbon) {
  up = AllPoints();
  int h, sumbonds;
  double dx;
  QString hnum;
  int least_hindered_side;
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next()) {
    sumbonds = 0; least_hindered_side = 0;
    // don't add to carbons unless specifically instructed
    if ( (tmp_pt->element == "C") && (to_carbon == false) ) continue;
    // don't do fragments with charges
    if (tmp_pt->element.contains("+") > 0) continue;
    if (tmp_pt->element.contains("-") > 0) continue;
    // N typically needs correcting...
    if (tmp_pt->element == "HN") tmp_pt->element = "N";
    if (tmp_pt->element == "NH") tmp_pt->element = "N";
    if (tmp_pt->element == "H2N") tmp_pt->element = "N";
    if (tmp_pt->element == "NH2") tmp_pt->element = "N";
    // so does O
    if (tmp_pt->element == "HO") tmp_pt->element = "O";
    if (tmp_pt->element == "OH") tmp_pt->element = "O";
    // let's do thiols too, so H ends up on proper side
    if (tmp_pt->element == "HS") tmp_pt->element = "S";
    if (tmp_pt->element == "SH") tmp_pt->element = "S";
    // don't add if hydrogen already present
    if ( tmp_pt->element.contains("H") > 0) continue;
    // find order of bonds
    for (tmp_bond = bonds.first(); tmp_bond != NULL; tmp_bond = bonds.next()) {
      if (tmp_bond->Find(tmp_pt)) {
	if (tmp_bond->Order() < 4)
	  sumbonds += tmp_bond->Order();
	else
	  sumbonds += 1; // order 5 and 7 (stereo) and order 1 bonds
	dx = tmp_pt->x - tmp_bond->otherPoint(tmp_pt)->x;
	if (dx > 0.5)
	  least_hindered_side++;
	if (dx < -0.5)
	  least_hindered_side--;
      }
    }
    h = MolData::Hydrogens(tmp_pt->element) - sumbonds;
    cout << h << endl;
    if (h > 1)
      hnum.setNum(h);
    else
      hnum = "";
    if (h > 0) {
      if (least_hindered_side < 0)
	tmp_pt->element.prepend("H" + hnum);
      else
	tmp_pt->element.append("H" + hnum);
    }
    // now find Text which reference this DPoint and copy back
    for (tmp_text = labels.first(); tmp_text != NULL; 
	 tmp_text = labels.next()) {
      if (tmp_text->Start() == tmp_pt) {
	cout << tmp_pt->element << endl;
	tmp_text->setText(tmp_pt->element);
	// *sigh* I suppose the least I can do is compute a proper textmask
	tmp_pt->element.fill(' ');
	if (h > 1) {
	  if (least_hindered_side < 0) {
	    tmp_pt->element.replace(1, 1, "-");
	  } else {
	    tmp_pt->element.replace(tmp_pt->element.length() - 1, 1, "-");
	  }
	}
	tmp_text->setTextMask(tmp_pt->element);
      }
    }
  }
}

void Molecule::Calc1HNMR() {
  QList<DPoint> up;

  // get list of unique points
  up = AllPoints();
  // find rings (esp. find aromaticity)
  MakeSSSR();
}

void Molecule::CalcIR() {
  QList<DPoint> up;
  int atom1, atom2, lorder, swp;

  // get list of unique points
  up = AllPoints();
  // find rings (esp. find aromaticity)
  MakeSSSR();

  GraphDialog g(r, "Predicted IR");
  // iterate thru all Bonds in Molecule
  QListIterator<Bond> outer(bonds), inner(bonds);
  for ( ; outer.current(); ++outer) {
    // check for 'obvious' cases
    tmp_bond = outer.current();
    // rules for atoms and groups
    lorder = tmp_bond->Order();
    if (lorder > 3) lorder = 1;  // stereo bonds (5,7) are single bonds (1)
    atom1 = tmp_bond->Start()->getAtomicNumber();
    atom2 = tmp_bond->End()->getAtomicNumber();
    if (atom1 > atom2) { swp = atom1; atom1 = atom2; atom2 = swp; }
    if (lorder == 1) {
      if ( (atom1 == 6) && (atom2 == 7) )
	g.AddPeak(1300.0, QString("C-N"), QString("~1350-1000, C-N"));
      if ( (atom1 == 6) && (atom2 == 8) )
	g.AddPeak(1300.0, QString("C-O"), QString("~1300-1000, C-O"));
    }
    if (lorder == 2) {
      if ( (atom1 == 6) && (atom2 == 6) )
	g.AddPeak(1660.0, QString("C=C"), QString("~1660-1600, C=C (cis/vinyl strong; trans weak)"));
      if ( (atom1 == 6) && (atom2 == 7) )
	g.AddPeak(1670.0, QString("C=N"), QString("~1690-1640, C=N"));
      if ( (atom1 == 6) && (atom2 == 8) )
	g.AddPeak(1700.0, QString("C=O"), QString("~1700 (narrow), C=O"));
      if ( (atom1 == 8) && (atom2 == 16) )
	g.AddPeak(1350.0, QString("S=O"), QString("~1350-1300, S=O (~1050 if R-(S=O)-R')"));
    }
    if (lorder == 3) {
      if ( (atom1 == 6) && (atom2 == 7) )
	g.AddPeak(2250.0, QString("CN"), QString("~2250 (narrow), nitrile"));
    }

    /* ####################################################################
    if (tmp_bond->Order() == 1) {
      if ( (tmp_bond->Start()->element == "C") &&
	   (tmp_bond->End()->element.contains("O") > 0) ) {
	g.AddPeak(1300.0, QString("C-O"), QString("~1300-1000, C-O"));
      }
      if ( (tmp_bond->Start()->element.contains("O") > 0) &&
	   (tmp_bond->End()->element == "C") ) {
	g.AddPeak(1300.0, QString("C-O"), QString("~1300-1000, C-O"));
      }
      if ( (tmp_bond->Start()->element == "C") &&
	   (tmp_bond->End()->element == "N") ) {
	g.AddPeak(1300.0, QString("C-N"), QString("~1350-1000, C-N"));
      }
      if ( (tmp_bond->Start()->element == "N") &&
	   (tmp_bond->End()->element == "C") ) {
	g.AddPeak(1300.0, QString("C-N"), QString("~1350-1000, C-N"));
      }      
    }
    if (tmp_bond->Order() == 2) {
      if ( (tmp_bond->Start()->element == "C") &&
	   (tmp_bond->End()->element == "O") ) {  // ketone
	g.AddPeak(1700.0, QString("C=O"), QString("~1700 (narrow), ketone"));
      }
      if ( (tmp_bond->Start()->element == "O") &&
	   (tmp_bond->End()->element == "C") ) {  // ketone
	g.AddPeak(1700.0, QString("C=O"), QString("~1700 (narrow), ketone"));
      }
      if ( (tmp_bond->Start()->element == "C") &&
	   (tmp_bond->End()->element == "C") ) {  // carbon double bond
	g.AddPeak(1700.0, QString("C=C"), QString("~1660-1600, C=C (cis/vinyl strong; trans weak)"));
      }
    }
    if (tmp_bond->Order() == 3) {
      if ( (tmp_bond->Start()->element == "C") &&
	   (tmp_bond->End()->element == "N") ) {  // nitrile
	g.AddPeak(2250.0, QString("CN"), QString("~2250 (narrow), nitrile"));
      }
      if ( (tmp_bond->Start()->element == "N") &&
	   (tmp_bond->End()->element == "C") ) {  // nitrile
	g.AddPeak(2250.0, QString("CN"), QString("~2250 (narrow), nitrile"));
      }
    }
    #########################################################################
    */
  }
  // iterate thru unique atoms, look for functional groups
  for (tmp_pt = up.first(); tmp_pt != NULL; tmp_pt = up.next()) {
    cout << "|" << tmp_pt->element << "|" << endl;
    if ( (tmp_pt->element == "C") && (tmp_pt->substituents < 4) )
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if (tmp_pt->element == "CH")
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if (tmp_pt->element == "HC")
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if (tmp_pt->element == "CH2")
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if (tmp_pt->element == "H2C")
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if (tmp_pt->element == "CH3")
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if (tmp_pt->element == "H3C")
      g.AddPeak(3000.0, QString("CH"), QString("~3000 (broad), C-H"));
    if ( (tmp_pt->element == "N") && (tmp_pt->substituents == 1) )
      g.AddPeak(3350.0, QString("NH"), QString("two peaks: ~3400, ~3300, primary N-H"));
    if ( (tmp_pt->element == "N") && (tmp_pt->substituents == 2) )
      g.AddPeak(3300.0, QString("NH"), QString("~3300 (broad), secondary N-H"));
    if (tmp_pt->element == "NH")
      g.AddPeak(3300.0, QString("NH"), QString("~3300 (broad), secondary N-H"));
    if (tmp_pt->element == "HN")
      g.AddPeak(3300.0, QString("NH"), QString("~3300 (broad), secondary N-H"));
    if (tmp_pt->element == "NH2")
      g.AddPeak(3350.0, QString("NH"), QString("two peaks: ~3400, ~3300, primary N-H"));
    if (tmp_pt->element == "H2N")
      g.AddPeak(3350.0, QString("NH"), QString("two peaks: ~3400, ~3300, primary N-H"));
    if ( (tmp_pt->element == "S") && (tmp_pt->substituents < 2) )
      g.AddPeak(2550.0, QString("SH"), QString("~2550 (broad), S-H"));
    if ( (tmp_pt->element == "SH") && (tmp_pt->substituents < 2) )
      g.AddPeak(2550.0, QString("SH"), QString("~2550 (broad), S-H"));
    if ( (tmp_pt->element == "HS") && (tmp_pt->substituents < 2) )
      g.AddPeak(2550.0, QString("SH"), QString("~2550 (broad), S-H"));
    if ( (tmp_pt->element == "O") && (tmp_pt->substituents < 2) )
      g.AddPeak(2550.0, QString("OH"), QString("~3400 (broad), O-H"));
    if (tmp_pt->element == "CN")
      g.AddPeak(2250.0, QString("CN"), QString("~2250 (narrow), nitrile"));
    if (tmp_pt->element == "NC")
      g.AddPeak(2250.0, QString("CN"), QString("~2250 (narrow), nitrile"));
    if (tmp_pt->element == "NCO")
      g.AddPeak(2270.0, QString("NCO"), QString("~2270 (narrow), -N=C=O"));
    if (tmp_pt->element == "NCS")
      g.AddPeak(2125.0, QString("NCS"), QString("~2125 (narrow), -N=C=S"));
    if (tmp_pt->element == "OH")
      g.AddPeak(3400.0, QString("OH"), QString("~3400 (broad), O-H"));
    if (tmp_pt->element == "HO")
      g.AddPeak(3400.0, QString("OH"), QString("~3400 (broad), O-H"));
    if (tmp_pt->element == "NO2")
      g.AddPeak(1525.0, QString("NO2"), QString("~1525 (narrow), -NO2"));
    if (tmp_pt->element == "O2N")
      g.AddPeak(1525.0, QString("NO2"), QString("~1525 (narrow), -NO2"));
    if (tmp_pt->aromatic == true) {
      g.AddPeak(1600.0, QString("aromatic"), QString("~1600 (narrow), aromatic ring C=C"));
      g.AddPeak(1475.0, QString("aromatic"), QString("~1475 (narrow), aromatic ring C=C"));
    }
  }

  QPixmap mol = r->MakeFullPixmap();
  QRect rr1 = BoundingBoxAll();
  rr1.setLeft(rr1.left() - 4);
  rr1.setTop(rr1.top() - 4);
  rr1.setBottom(rr1.bottom() + 4);
  rr1.setRight(rr1.right() + 4);
  QPixmap mol1(rr1.size());
  bitBlt(&mol1, QPoint(0,0), &mol, rr1);

  g.AddPixmap(mol1);
  g.exec();
}

