/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT6MApp.h"

#include <wx/config.h>
#include <wx/filename.h>
#include <wx/ffile.h>

IMPLEMENT_APP(CJT6MApp)

#include "common/SoundCard.h"
#include "common/Exception.h"
#include "common/Log.h"

#include "JT6MDefs.h"

const wxString SECTION_GENERAL  = wxT("General");
const wxString SECTION_TRANSMIT = wxT("Transmit");
const wxString SECTION_RECEIVE  = wxT("Receive");
const wxString SECTION_SOUND    = wxT("Sound");
const wxString SECTION_PTT      = wxT("PTT");
const wxString SECTION_PATHS    = wxT("Paths");
const wxString SECTION_PROTOCOL = wxT("Protocol");
const wxString SECTION_MESSAGES = wxT("Messages");

const wxString KEY_CALLSIGN     = wxT("callsign");
const wxString KEY_LOCATOR      = wxT("locator");
const wxString KEY_FIRST        = wxT("first");
const wxString KEY_MIN_LENGTH   = wxT("min_length");
const wxString KEY_TYPE         = wxT("type");
const wxString KEY_FILENAME     = wxT("filename");
const wxString KEY_DEVICE       = wxT("device");
const wxString KEY_AUDIO        = wxT("audio");
const wxString KEY_TEXT         = wxT("text");
const wxString KEY_MESSAGE      = wxT("message_");

const wxString DEFAULT_CALLSIGN = wxT("G4KLX");
const wxString DEFAULT_LOCATOR  = wxT("IO93FB");

const bool DEFAULT_TX_ENABLED = false;
const bool DEFAULT_TX_FIRST   = true;

const long DEFAULT_MIN_LENGTH     = 10L;

const wxString DEFAULT_PTT_TYPE   = wxT("none");
const wxString DEFAULT_PTT_DEVICE = wxEmptyString;

#ifdef __WINDOWS__
const wxString DEFAULT_SOUND_FILENAME = wxT("Wave Mapper");
#else
const wxString DEFAULT_SOUND_FILENAME = wxT("/dev/dsp");
#endif

const wxString DEFAULT_MESSAGE_0 = wxT("%T %M ");
const wxString DEFAULT_MESSAGE_1 = wxT("%T %M %R %R ");
const wxString DEFAULT_MESSAGE_2 = wxT("%T %M R%R R%R ");
const wxString DEFAULT_MESSAGE_3 = wxT("%M RRRRRRRR ");
const wxString DEFAULT_MESSAGE_4 = wxT("73 %M ");
const wxString DEFAULT_MESSAGE_5 = wxT("CQ %M ");


CJT6MApp::CJT6MApp() :
wxApp(),
m_frame(NULL),
m_controller(NULL),
m_messages(NULL),
m_file(NULL),
m_saving(false),
m_recording(false),
m_txEnable(DEFAULT_TX_ENABLED)
{
}

CJT6MApp::~CJT6MApp()
{
}

bool CJT6MApp::OnInit()
{
	SetVendorName(JT6M_VENDOR_NAME);
	SetAppName(JT6M_APPL_NAME);

	wxConfigBase* config = wxConfigBase::Get();
	config->SetRecordDefaults();

	wxString pathAudio, pathText;
	getPaths(pathAudio, pathText);

	wxFileName fileName(pathText, JT6M_LOGFILE_NAME, JT6M_LOGFILE_EXT);

	wxLog* log = new CLog(fileName.GetFullPath());
	wxLog::SetActiveTarget(log);

	createController();
	::wxSleep(1);

	createFileReader();

	readRegistry();

	m_frame = new CJT6MFrame();
	m_frame->Show();

	SetTopWindow(m_frame);

	return true;
}

int CJT6MApp::OnExit()
{
	wxASSERT(m_controller != NULL);
	wxASSERT(m_file != NULL);

	m_controller->Delete();
	m_file->kill();

	delete wxConfigBase::Set(NULL);

	return 0;
}

#ifdef __WXDEBUG__
void CJT6MApp::OnAssert(const wxChar* file, int line, const wxChar* cond, const wxChar* msg)
{
	::wxLogFatalError(wxT("Assertion failed on line %d in file %s: %s %s"), line, file, cond, msg);
}
#endif

void CJT6MApp::receiveMessage(CJT6MMessage* message, EWho who) const
{
	wxASSERT(m_frame != NULL);
	wxASSERT(message != NULL);

	m_frame->showMessage(message, who);
}

void CJT6MApp::logMessage(const CJT6MMessage& message) const
{
	if (m_saving) {
		wxString   id = message.getId();
		double    tim = message.getTime();
		int       len = message.getLength();
		int  strength = message.getStrength();
		int        DF = message.getDF();
		int     count = message.getCount();
		wxString text = message.getText();

		wxString pathAudio, pathText;
		getPaths(pathAudio, pathText);

		wxFileName fileName(pathText, JT6M_MESSAGES_NAME, JT6M_MESSAGES_EXT);

		wxFFile* file = new wxFFile(fileName.GetFullPath(), "a");

		if (!file->IsOpened()) {
			wxString text;
			text.Printf(wxT("Cannot open save text file: %s"), fileName.GetFullPath().c_str());
			error(text);

			delete file;
			return;
		}

		wxString line;

		if (count > 0)
			line.Printf(wxT("%s            %d  %d  %s  * %d\n"), id.c_str(), strength, DF, text.c_str(), count);
		else
			line.Printf(wxT("%s  %.1f  %d  %d  %d  %s\n"), id.c_str(), tim, len, strength, DF, text.c_str());


		file->Write(line);

		file->Close();

		delete file;
	}
}

void CJT6MApp::showLevels(CJT6MLevels* levels, EWho who) const
{
	wxASSERT(levels != NULL);
	wxASSERT(m_frame != NULL);

	m_frame->showLevels(levels, who);
}

void CJT6MApp::showAudio(double audio, EWho who) const
{
	wxASSERT(m_frame != NULL);

	m_frame->showAudio(audio, who);
}

void CJT6MApp::error(const wxString& text) const
{
	wxASSERT(m_frame != NULL);

	m_frame->error(text);
}

void CJT6MApp::getPersonal(wxString& callsign, wxString& locator) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString callsignKey = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_CALLSIGN;
	wxString locatorKey  = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_LOCATOR;

	profile->Read(callsignKey, &callsign, DEFAULT_CALLSIGN);
	profile->Read(locatorKey,  &locator,  DEFAULT_LOCATOR);
}

void CJT6MApp::setPersonal(const wxString& callsign, const wxString& locator) const
{
	CText::setLocalCallsign(callsign);
	CText::setLocalLocator(locator);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString callsignKey = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_CALLSIGN;
	wxString locatorKey  = wxT("/") + SECTION_GENERAL + wxT("/") + KEY_LOCATOR;

	profile->Write(callsignKey, callsign);
	profile->Write(locatorKey,  locator);
	profile->Flush();
}

void CJT6MApp::setRemote(const wxString& callsign, const wxString& report)
{
	if (callsign.IsEmpty())
		CText::setRemoteCallsign(wxEmptyString);
	else
		CText::setRemoteCallsign(callsign);

	if (report.IsEmpty())
		CText::setLocalReport(wxEmptyString);
	else
		CText::setLocalReport(report);
}

bool CJT6MApp::getTXFirst() const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_TRANSMIT + wxT("/") + KEY_FIRST;

	bool txFirst;
	profile->Read(key, &txFirst, DEFAULT_TX_FIRST);

	return txFirst;
}

void CJT6MApp::setTXFirst(bool txFirst) const
{
	wxASSERT(m_controller != NULL);

	m_controller->setTXFirst(txFirst);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_TRANSMIT + wxT("/") + KEY_FIRST;

	profile->Write(key, txFirst);
	profile->Flush();
}

bool CJT6MApp::getTXEnable() const
{
	return m_txEnable;
}

void CJT6MApp::setTXEnable(bool txEnable)
{
	wxASSERT(m_controller != NULL);

	m_controller->setTXEnable(txEnable);

	m_txEnable = txEnable;
}

int CJT6MApp::getMinLength() const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_RECEIVE + wxT("/") + KEY_MIN_LENGTH;

	long minLength;
	profile->Read(key, &minLength, DEFAULT_MIN_LENGTH);

	return int(minLength);
}

void CJT6MApp::setMinLength(int minLength) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString key = wxT("/") + SECTION_RECEIVE + wxT("/") + KEY_MIN_LENGTH;

	profile->Write(key, minLength);
	profile->Flush();
}

wxString CJT6MApp::getReplacedMessage(int n) const
{
	wxASSERT(n >= 0 && n < JT6M_MAX_MESSAGES);
	wxASSERT(m_messages[n] != NULL);

	return m_messages[n]->getReplacedText();
}

wxString CJT6MApp::getOriginalMessage(int n) const
{
	wxASSERT(n >= 0 && n < JT6M_MAX_MESSAGES);
	wxASSERT(m_messages[n] != NULL);

	return m_messages[n]->getOriginalText();
}

void CJT6MApp::getMessage(int n, wxString& message) const
{
	wxASSERT(n >= 0 && n < JT6M_MAX_MESSAGES);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString number;
	number.Printf(wxT("%d"), n);

	wxString key = wxT("/") + SECTION_MESSAGES + wxT("/") + KEY_MESSAGE + number;
	wxString defMessage;

	switch (n) {
		case 0:
			defMessage = DEFAULT_MESSAGE_0;
			break;
		case 1:
			defMessage = DEFAULT_MESSAGE_1;
			break;
		case 2:
			defMessage = DEFAULT_MESSAGE_2;
			break;
		case 3:
			defMessage = DEFAULT_MESSAGE_3;
			break;
		case 4:
			defMessage = DEFAULT_MESSAGE_4;
			break;
		case 5:
			defMessage = DEFAULT_MESSAGE_5;
			break;
		default:
			defMessage = wxEmptyString;
			break;
	}

	profile->Read(key, &message, defMessage);
}

void CJT6MApp::setMessage(int n, const wxString& message)
{
	wxASSERT(n >= 0 && n < JT6M_MAX_MESSAGES);

	if (m_messages[n] != NULL)
		delete m_messages[n];

	m_messages[n] = new CText(message);

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString number;
	number.Printf(wxT("%d"), n);

	wxString key = wxT("/") + SECTION_MESSAGES + wxT("/") + KEY_MESSAGE + number;

	profile->Write(key, message);
	profile->Flush();
}

void CJT6MApp::getPTT(wxString& type, wxString& device) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString typeKey   = wxT("/") + SECTION_PTT + wxT("/") + KEY_TYPE;
	wxString deviceKey = wxT("/") + SECTION_PTT + wxT("/") + KEY_DEVICE;

	profile->Read(typeKey,   &type,   DEFAULT_PTT_TYPE);
	profile->Read(deviceKey, &device, DEFAULT_PTT_DEVICE);
}

void CJT6MApp::setPTT(const wxString& type, const wxString& device) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString typeKey   = wxT("/") + SECTION_PTT + wxT("/") + KEY_TYPE;
	wxString deviceKey = wxT("/") + SECTION_PTT + wxT("/") + KEY_DEVICE;

	profile->Write(typeKey,   type);
	profile->Write(deviceKey, device);
	profile->Flush();
}

void CJT6MApp::getPaths(wxString& pathAudio, wxString& pathText) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString audioKey = wxT("/") + SECTION_PATHS + wxT("/") + KEY_AUDIO;
	wxString textKey  = wxT("/") + SECTION_PATHS + wxT("/") + KEY_TEXT;

	wxString homeDir = wxFileName::GetHomeDir();

	profile->Read(audioKey, &pathAudio, homeDir);
	profile->Read(textKey,  &pathText,  homeDir);
}

void CJT6MApp::setPaths(const wxString& pathAudio, const wxString& pathText) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString audioKey = wxT("/") + SECTION_PATHS + wxT("/") + KEY_AUDIO;
	wxString textKey  = wxT("/") + SECTION_PATHS + wxT("/") + KEY_TEXT;

	profile->Write(audioKey, pathAudio);
	profile->Write(textKey,  pathText);
	profile->Flush();
}

void CJT6MApp::getSound(wxString& fileName) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString fileNameKey = wxT("/") + SECTION_SOUND + wxT("/") + KEY_FILENAME;

	profile->Read(fileNameKey, &fileName, DEFAULT_SOUND_FILENAME);
}

void CJT6MApp::setSound(const wxString& fileName) const
{
	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	wxString fileNameKey = wxT("/") + SECTION_SOUND + wxT("/") + KEY_FILENAME;

	profile->Write(fileNameKey, fileName);
	profile->Flush();
}

void CJT6MApp::setSaving(bool saving)
{
	m_saving = saving;
}

void CJT6MApp::deleteMessages() const
{
	wxString pathAudio, pathText;
	getPaths(pathAudio, pathText);

	wxFileName fileName(pathText, JT6M_MESSAGES_NAME, JT6M_MESSAGES_EXT);

	::wxRemoveFile(fileName.GetFullPath());
}

void CJT6MApp::setRecording(bool recording)
{
	wxASSERT(m_controller != NULL);

	m_recording = recording;

	m_controller->setRXRecording(m_recording);
}

void CJT6MApp::sendMessage(const wxString& text)
{
	wxASSERT(m_controller != NULL);

	m_controller->sendMessage(text);
}

void CJT6MApp::readFile(const wxString& fileName)
{
	wxASSERT(m_file != NULL);

	if (m_file->isRunning()) {
		error(wxT("Another WAV file is currently being processed"));
		return;
	}

	try {
		m_file->setFileName(fileName);
		m_file->start();
	}
	catch (CException& ex) {
		error(ex.getMessage());
		return;
	}
}

void CJT6MApp::createController()
{
	try {
		wxString type, fileName;

		getSound(fileName);
		CSoundCard* soundDev = new CSoundCard(fileName, JT6M_SAMPLE_RATE, JT6M_SOUNDBUF_LENGTH, JT6M_SYMBOL_LENGTH);

		getPTT(type, fileName);
		CPTTPort* pttPort = CPTTPort::createPTTPort(type, fileName);

		m_controller = new CJT6MController(soundDev, pttPort);
		m_controller->Create();
		m_controller->Run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "JT6M: an error has occured when creating the sound device or the PTT port: %s\n", ex.getMessage().c_str());
		throw;
	}
}

void CJT6MApp::createFileReader()
{
	m_file = new CJT6MFileReceive(wxT("File Receive"), RX_FILE);

	m_file->Create();
	m_file->Run();
}

void CJT6MApp::readRegistry()
{
	wxASSERT(m_controller != NULL);

	wxString callsign, locator;
	getPersonal(callsign, locator);
	CText::setLocalCallsign(callsign);
	CText::setLocalLocator(locator);

	bool txEnable = getTXEnable();
	m_controller->setTXEnable(txEnable);

	bool txFirst = getTXFirst();
	m_controller->setTXFirst(txFirst);

	m_messages = new CText*[JT6M_MAX_MESSAGES];

	for (int i = 0; i < JT6M_MAX_MESSAGES; i++) {
		wxString message;
		getMessage(i, message);
		m_messages[i] = new CText(message);
	}

	wxConfigBase* profile = wxConfigBase::Get();

	wxASSERT(profile != NULL);

	profile->Flush();
}
