/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT44MessageBase.h"
#include "JT44Lookups.h"
#include "JT44Defs.h"

#include "common/Correlation.h"

#include <cmath>
using namespace std;

CJT44MessageBase::CJT44MessageBase() :
m_data(NULL),
m_sync(NULL),
m_noise(NULL)
{
	m_data  = new CAverage[135 * JT44_ALPHABET_COUNT];
	m_sync  = new CAverage[135];
	m_noise = new CAverage[135];
}

CJT44MessageBase::~CJT44MessageBase()
{
	delete[] m_data;
	delete[] m_sync;
	delete[] m_noise;
}

double CJT44MessageBase::getData(int letter, int pos) const
{
	int index = getSymbolsIndex(letter, pos);

	if (m_data[index].getCount() > 0)
		return m_data[index].getAverage();
	else
		return 0.0;
}

double CJT44MessageBase::getSync(int pos) const
{
	wxASSERT(pos >= 0 && pos < 135);

	if (m_sync[pos].getCount() > 0)
		return m_sync[pos].getAverage();
	else
		return 0.0;
}

double CJT44MessageBase::getNoise(int pos) const
{
	wxASSERT(pos >= 0 && pos < 135);

	if (m_noise[pos].getCount() > 0)
		return m_noise[pos].getAverage();
	else
		return 0.0;
}

void CJT44MessageBase::clearData()
{
	for (int i = 0; i < (135 * JT44_ALPHABET_COUNT); i++)
		m_data[i].clear();
}

void CJT44MessageBase::clearSync()
{
	for (int i = 0; i < 135; i++)
		m_sync[i].clear();
}

void CJT44MessageBase::clearNoise()
{
	for (int i = 0; i < 135; i++)
		m_noise[i].clear();
}

void CJT44MessageBase::addData(int letter, int pos, double val)
{
	int index = getSymbolsIndex(letter, pos);

	m_data[index].addValue(val);
}

void CJT44MessageBase::addSync(int pos, double val)
{
	wxASSERT(pos >= 0 && pos < 135);

	m_sync[pos].addValue(val);
}

void CJT44MessageBase::addNoise(int pos, double val)
{
	wxASSERT(pos >= 0 && pos < 135);

	m_noise[pos].addValue(val);
}

void CJT44MessageBase::subtractData(int letter, int pos, double val)
{
	int index = getSymbolsIndex(letter, pos);

	m_data[index].subtractValue(val);
}

void CJT44MessageBase::subtractSync(int pos, double val)
{
	wxASSERT(pos >= 0 && pos < 135);

	m_sync[pos].subtractValue(val);
}

void CJT44MessageBase::subtractNoise(int pos, double val)
{
	wxASSERT(pos >= 0 && pos < 135);

	m_noise[pos].subtractValue(val);
}

/*
 * Get the message text from the FFT bin data stored in the object. It is
 * done here because this class is also used for the "average" message and
 * may contain data from more than one message.
 */
double CJT44MessageBase::getStrength() const
{
	CAverage signal;
	CAverage noise;

	CJT44Lookups lookup;

	for (int pos = 0; pos < 135; pos++) {
		if (lookup.lookupSync(pos))
			signal.addValue(getSync(pos));

		noise.addValue(getNoise(pos));
	}

	double strength;
	double ratio = signal.getAverage() / noise.getAverage() - 1.0;

	if (ratio > 0.0001)
		strength = 10.0 * ::log10(ratio) - 22.7;
	else
		strength = -99.0;

	if (strength < -39.0) strength = -39.0;

	return strength;
}

int CJT44MessageBase::getQuality() const
{
	CCorrelation correlation;

	CJT44Lookups lookup;

	for (int pos = 0; pos < 135; pos++) {
		if (lookup.lookupSync(pos))
			correlation.addProduct(JT44_SYNC_POS_COEFF, getSync(pos));
		else
			correlation.addProduct(JT44_SYNC_NEG_COEFF, getSync(pos));
	}

	double corrVal;
	bool ret = correlation.getNormalisedValue(corrVal);

	if (!ret)
		return 0;

	int quality = int(corrVal * 10.0 + 0.5);

	if (quality < 0) quality = 0;

	return quality;
}

/*
 * Extracting the text from the FFT bin data was originally done by simply
 * finding the bin with the highest value, this works well but fails in the
 * presence of interference. So instead another correlator is used to try
 * and remove any birdie signals that may be present. We use the fact that the
 * message data is held in certain positions and must be mising in others, in
 * this case when we have a synchronisation tone. A "birdie" would not follow
 * these rules.
 */
wxString CJT44MessageBase::getText() const
{
	CJT44Lookups lookup;

	wxString text;

	for (int symbol = 0; symbol < JT44_MESSAGE_LENGTH; symbol++) {
		CCorrelation correlate[JT44_ALPHABET_COUNT];

		for (int pos = 0; pos < 135; pos++) {
			double corrVal = getSymbolCorrelation(symbol, pos);

			for (int i = 0; i < JT44_ALPHABET_COUNT; i++)
				correlate[i].addProduct(corrVal, getData(i, pos));
		}

		double maxValue;
		correlate[0].getRawValue(maxValue);
		int bin = 0;

		for (int i = 1; i < JT44_ALPHABET_COUNT; i++) {
			double value;
			correlate[i].getRawValue(value);

			if (value > maxValue) {
				maxValue = value;
				bin      = i;
			}
		}

		wxChar c = lookup.lookupTone(bin + 121);

		if (c != wxChar(0))
			text.Append(c);
		else
			text.Append(wxT('*'));
	}

	return text;
}

wxString CJT44MessageBase::getOddEvenText() const
{
	CJT44Lookups lookup;

	wxString text;

	for (int symbol = 0; symbol < 2; symbol++) {
		CCorrelation correlate[JT44_ALPHABET_COUNT];

		for (int pos = 0; pos < 135; pos++) {
			double corrVal = getOddEvenSymbolCorrelation(symbol, pos);

			for (int i = 0; i < JT44_ALPHABET_COUNT; i++)
				correlate[i].addProduct(corrVal, getData(i, pos));
		}

		double maxValue;
		correlate[0].getRawValue(maxValue);
		int bin = 0;

		for (int i = 1; i < JT44_ALPHABET_COUNT; i++) {
			double value;
			correlate[i].getRawValue(value);

			if (value > maxValue) {
				maxValue = value;
				bin      = i;
			}
		}

		wxChar c = lookup.lookupTone(bin + 121);

		if (c != wxChar(0))
			text.Append(c);
		else
			text.Append(wxT('*'));
	}

	return text;
}

wxString CJT44MessageBase::getLast4Text() const
{
	CJT44Lookups lookup;

	wxString text;

	CCorrelation correlate[JT44_ALPHABET_COUNT];

	for (int pos = 0; pos < 135; pos++) {
		double corrVal = getLast4SymbolCorrelation(pos);

		for (int i = 0; i < JT44_ALPHABET_COUNT; i++)
			correlate[i].addProduct(corrVal, getData(i, pos));
	}

	double maxValue;
	correlate[0].getRawValue(maxValue);
	int bin = 0;

	for (int i = 1; i < JT44_ALPHABET_COUNT; i++) {
		double value;
		correlate[i].getRawValue(value);

		if (value > maxValue) {
			maxValue = value;
			bin      = i;
		}
	}

	wxChar c = lookup.lookupTone(bin + 121);

	if (c != wxChar(0))
		text.Append(c);
	else
		text.Append(wxT('*'));

	return text;
}

int CJT44MessageBase::getSymbolsIndex(int letter, int pos) const
{
	wxASSERT(letter >= 0 && letter < JT44_ALPHABET_COUNT);
	wxASSERT(pos >= 0 && pos < 135);

	return letter * 135 + pos;
}

/*
 * Get the values for the correlator which has a suitable value for
 * the presence of a tone and a suitable negative value for the synchronisation
 * periods. The values are weighted because there are more sync periods
 * than data periods.
 */
double CJT44MessageBase::getSymbolCorrelation(int symbol, int pos) const
{
	wxASSERT(pos >= 0 && pos < 135);
	wxASSERT(symbol >= 0 && symbol < JT44_MESSAGE_LENGTH);

	CJT44Lookups lookup;

	int n = lookup.lookupPosition(pos);

	if (n == -1)
		return -3.0 / 69.0;
	else if (n == symbol)
		return 1.0;
	else
		return 0.0;
}

double CJT44MessageBase::getOddEvenSymbolCorrelation(int symbol, int pos) const
{
	wxASSERT(pos >= 0 && pos < 135);
	wxASSERT(symbol >= 0 && symbol < JT44_MESSAGE_LENGTH);

	CJT44Lookups lookup;

	int n = lookup.lookupPosition(pos);

	if (n == -1)
		return -33.0 / 69.0;
	else if ((n % 2) == symbol)
		return 1.0;
	else
		return 0.0;
}

double CJT44MessageBase::getLast4SymbolCorrelation(int pos) const
{
	wxASSERT(pos >= 0 && pos < 135);

	CJT44Lookups lookup;

	int n = lookup.lookupPosition(pos);

	if (n == -1)
		return -12.0 / 69.0;
	else if (pos >= 18)
		return 1.0;
	else
		return 0.0;
}
