/*
 *   Copyright (C) 2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "RingBuffer.h"

#include <wx/wx.h>

CRingBuffer::CRingBuffer(int size) :
m_size(size),
m_buffer(NULL),
m_inPtr(0),
m_outPtr(0)
{
	wxASSERT(m_size > 0);

	m_buffer = new double[m_size];
}

CRingBuffer::~CRingBuffer()
{
	delete[] m_buffer;
}

void CRingBuffer::empty()
{
	m_inPtr  = 0;
	m_outPtr = 0;
}

bool CRingBuffer::isEmpty() const
{
	return getDataLength() == 0;
}

int CRingBuffer::getEmptyLength() const
{
	return m_size - getDataLength();
}

int CRingBuffer::getDataLength() const
{
	int space = m_inPtr - m_outPtr;

	if (space < 0)
		space += m_size;

	return space;
}

int CRingBuffer::putData(double* data, int len)
{
	wxASSERT(data != NULL);
	wxASSERT(len > 0);

	int l = getEmptyLength();

	if (l < len)
		len = l;

	for (int n = 0; n < len; n++) {
		m_buffer[m_inPtr] = data[n];

		m_inPtr++;

		if (m_inPtr == m_size)
			m_inPtr = 0;
	}

	return len;
}

int CRingBuffer::putData(float* data, unsigned long len)
{
	wxASSERT(data != NULL);

	unsigned int l = getEmptyLength();

	if (l < len)
		len = l;

	for (unsigned int n = 0; n < len; n++) {
		m_buffer[m_inPtr] = double(data[n]);

		m_inPtr++;

		if (m_inPtr == m_size)
			m_inPtr = 0;
	}

	return len;
}

int CRingBuffer::getData(double* data, int len)
{
	wxASSERT(data != NULL);
	wxASSERT(len > 0);

	int l = getDataLength();

	if (l < len)
		len = l;

	for (int n = 0; n < len; n++) {
		data[n] = m_buffer[m_outPtr];

		m_outPtr++;

		if (m_outPtr == m_size)
			m_outPtr = 0;
	}

	return len;
}

int CRingBuffer::getData(float* data, unsigned long len)
{
	wxASSERT(data != NULL);

	unsigned int l = getDataLength();

	if (l < len)
		len = l;

	for (unsigned int n = 0; n < len; n++) {
		data[n] = float(m_buffer[m_outPtr]);

		m_outPtr++;

		if (m_outPtr == m_size)
			m_outPtr = 0;
	}

	return len;
}
