/*
 * wmnd.c
 * by Reed Lai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/stat.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>


#include "beat.h"
#include "misc.h"
#include "master.xpm"

#define WMND_VERSION "0.2.2"


#ifdef USE_KSTAT		/* Solaris */

#include <kstat.h>

#define set_bit(x,y) ((*(y))|=(1L<<(x-1)))
#define clear_bit(x,y) ((*(y))&=(~(1L<<(x-1))))
#define change_bit(x,y) ((*(y))^=(1L<<(x-1)))
#define test_bit(x,y) (((*(y))&(1<<(x-1)))?!0L:0L)

#define MAX(x,y) (((x)<(y))?(y):(x))

kstat_ctl_t *kc = NULL;
kstat_t *if_ksp = NULL;

#else				/* do not USE_KSTAT, reverting to Linux */

#include <getopt.h>
#include <asm/bitops.h>		/* for set_bit / clear_bit / friends */

#include <net/ppp_defs.h>
#include <net/if_ppp.h>

#define NET_DEV	"/proc/net/dev"
#define NET_ROUTE  "/proc/net/route"

#endif				/* USE_KSTAT */

#ifndef MAX
#define MAX(a, b) ((a)>(b))?(a):(b)
#endif

#define MAXBUF 256

#ifdef PRO
#define eprint(level, fmt, arg...) \
		switch (level) { \
		case 0: \
			break; \
		case 1: \
			fprintf(stderr, __FUNCTION__": " fmt, ##arg); \
			fprintf(stderr, "\n"); \
			break; \
		}
#else
#define eprint(level, fmt, arg...) \
		do { } while (0)
#endif

char *active_interface = NULL;

struct Devices {
    struct Devices *next;	/* ptr to next structure */
    char *name;			/* device name from /proc/net/dev */
    time_t ts;			/* add timestamp */
    unsigned long int his[59][4];
    unsigned long int ib_stat_last;
    unsigned long int ob_stat_last;
    unsigned long int ip_stat_last;
    unsigned long int op_stat_last;
    unsigned long int ib_max_his;	/* maximum input bytes in history */
    unsigned long int ob_max_his;	/* maximum output bytes in history */
    unsigned long int ip_max_his;	/* maximum input packets in history */
    unsigned long int op_max_his;	/* maximum output packets in history */
};

struct var {
    struct var *v_next;		/* ptr to next structure */
    char *v_name;		/* option name */
    char *v_value;		/* option value */
};

typedef struct {
    unsigned int nr_devices;	/* number of devices in list */
    unsigned char current[16];	/* currently showing device */
    unsigned long int flags;	/* big bit-mapped flags mask */
    time_t pppstart;		/* special case for ppp connection time */
    unsigned int wavemode;	/* type of wave graph */
    void (*draw) (struct Devices *);	/* draw stats function */
    void (*scale) (unsigned char sign, unsigned long value, char *buf);	/* scale 
									 * function 
									 */
    unsigned int refresh;	/* speed of the refresh */
    unsigned long scroll;	/* speed of the graph scrolling */
} DevTable;

typedef struct {
    Display *d;
    int xfd;
    int screen;
    Window root;
    Window win;
    Window iconwin;
    Pixmap pixmap;
    Pixmap mask;
    GC gc;
    int width;
    int height;
    int x;
    int y;
    int button;
    int update;
} Dockapp;

typedef struct {
    int enable;
    int x;
    int y;
    int width;
    int height;
} MRegion;

struct var *vars;		/* config file option structure */
struct Devices *devices;	/* devices to monitor */
DevTable wmnd;			/* interesting information about devices */
Dockapp dockapp;		/* dockapp structure */
MRegion mr[32];			/* mouse regions */

/* store all configuration in one place */
#define RUN_ONLINE		3	/* current device is online */
#define LED_POWER		4	/* next 3 are for different leds */
#define LED_RX				5
#define LED_TX				6
#define CFG_MAXSCREEN		16	/* 1: yes 0: no, max history */
#define CFG_SHORTNAME		17	/* 1: use 0: no */
#define CFG_MODE		18	/* 1: bytes 0: packets */
#define CFG_SHOWMAX		19	/* 1: yes 0: no */

/* bit ops */
#define bit_set(n) set_bit(n, &wmnd.flags)
#define bit_off(n) clear_bit(n, &wmnd.flags)
#define bit_tgl(n) change_bit(n, &wmnd.flags)
#define bit_get(n) test_bit(n, &wmnd.flags)

/* GUI */
static void redraw_window(void);
unsigned long get_color(char *name);
void new_window(char *name, int width, int height);

/* config file read/write */
void conf_read(char *filename);
void conf_write(char *filename);
void assign(char *name, char *value);
void vdel(char *name);
struct var *lookup(char *name);
char *value(char *name);
char *vcopy(char *str);

/* support */
int add_mr(int index, int x, int y, int width, int height);
int check_mr(int x, int y);
void beat_event(void);
void click_event(unsigned int region, unsigned int button);
static void led_control(const unsigned char led, const unsigned char mode);
void scale(char *rx_buf, char *tx_buf, unsigned long rx, unsigned long tx);
void metric_scale(unsigned char sign, unsigned long value, char *buf);
void binary_scale(unsigned char sign, unsigned long value, char *buf);
void draw_string(unsigned char *buf, unsigned int x, unsigned int y);

/* device statistics */
struct Devices *add_device(char *name, struct Devices **list, time_t ts);
void delete_device(char *name, struct Devices **list);
struct Devices *lookup_device(char *name, struct Devices *list);
int device_list(void);
void cycle_devices(char *def);
void draw_interface(void);
void draw_rate(unsigned long rx, unsigned long tx);
void draw_max(unsigned long rx, unsigned long tx);
void draw_error(struct Devices *ptr);
void draw_stats(struct Devices *ptr);
static int get_statistics(const char *dev, unsigned long *ip,
			  unsigned long *op, unsigned long *ib,
			  unsigned long *ob);
int still_online(char *dev);

/* useless shit */
void usage(void);
void printversion(void);

/* device structure handling */
struct Devices *add_device(char *name, struct Devices **list, time_t ts)
{
    struct Devices *new;
    struct Devices *ptr = NULL;

    if ((ptr = lookup_device(name, *list))) {
	/* this is done to "save" history data */
	/* renew timestamp so that we don't get deleted later */
	ptr->ts = ts;
	eprint(1, "device already exists");
	return ptr;
    }

    new = (struct Devices *) malloc(sizeof(struct Devices));

    if (*list == NULL || strcmp((*list)->name, name) >= 0) {
	new->next = *list;
	*list = new;
    } else {
	ptr = *list;
	while (ptr->next && strcmp(ptr->next->name, name) <= 0) {
	    ptr = ptr->next;
	}
	new->next = ptr->next;
	ptr->next = new;
    }
    eprint(1, "adding %s", name);
    new->name = strdup(name);
    new->ts = ts;
    memset(new->his, 0, sizeof(new->his));
    new->ib_stat_last = new->ob_stat_last = 0;
    new->ip_stat_last = new->op_stat_last = 0;
    new->ib_max_his = new->ob_max_his = 0;
    new->ip_max_his = new->op_max_his = 0;
    wmnd.nr_devices++;
    return new;
}

void delete_device(char *name, struct Devices **list)
{
    struct Devices *ptr;
    struct Devices *new;

    while (*list && !strcmp((*list)->name, name)) {
	ptr = (*list)->next;
	free((struct Devices *) *list);
	*list = ptr;
	eprint(1, "deleting %s", name);
    }
    if (*list)
	for (new = *list; new->next;)
	    if (!strcmp(new->next->name, name)) {
		eprint(1, "deleting %s", name);
		ptr = new->next->next;
		free(new->next);
		new->next = ptr;
	    } else
		new = new->next;

    wmnd.nr_devices--;
}

struct Devices *lookup_device(char *name, struct Devices *list)
{
    struct Devices *ptr;

    for (ptr = list; ptr; ptr = ptr->next) {
	if (!strcmp(name, ptr->name)) {
	    eprint(0, "found %s [%p]", name, ptr);
	    return ptr;
	}
    }
    return NULL;
}

#define copy_xpm_area(x, y, w, h, dx, dy)	\
{						\
    XCopyArea(dockapp.d, dockapp.pixmap, dockapp.pixmap, dockapp.gc,	\
			  x, y, w, h, dx, dy);				\
	dockapp.update = 1;			\
}

int add_mr(int index, int x, int y, int width, int height)
{
    mr[index].enable = 1;
    mr[index].x = x;
    mr[index].y = y;
    mr[index].width = width;
    mr[index].height = height;
    return 0;
}

int check_mr(int x, int y)
{
    register int i;
    register int found = 0;

    for (i = 0; i < 32 && !found; i++) {
	if (mr[i].enable && x >= mr[i].x &&
	    x <= mr[i].x + mr[i].width &&
	    y >= mr[i].y && y <= mr[i].y + mr[i].height)
	    found = 1;
    }
    if (!found)
	return -1;
    return (i - 1);

}

static void redraw_window(void)
{
    if (dockapp.update) {
	eprint(0, "redrawing window");
	XCopyArea(dockapp.d, dockapp.pixmap, dockapp.iconwin, dockapp.gc,
		  0, 0, dockapp.width, dockapp.height, 0, 0);
	XCopyArea(dockapp.d, dockapp.pixmap, dockapp.win, dockapp.gc,
		  0, 0, dockapp.width, dockapp.height, 0, 0);
	dockapp.update = 0;
    }
}

unsigned long get_color(char *name)
{
    XColor color;
    XWindowAttributes attr;

    color.pixel = 0;

    XGetWindowAttributes(dockapp.d, DefaultRootWindow(dockapp.d), &attr);
    XParseColor(dockapp.d,
		DefaultColormap(dockapp.d, DefaultScreen(dockapp.d)), name,
		&color);
    color.flags = DoRed | DoGreen | DoBlue;
    XAllocColor(dockapp.d, attr.colormap, &color);
    eprint(1, "pixel: %08lx\n", color.pixel);
    return color.pixel;

}

void new_window(char *name, int width, int height)
{
    XpmAttributes attr;
    XpmColorSymbol cols[3] = {
	{"rx_color", NULL, 0},
	{"tx_color", NULL, 0},
	{"md_color", NULL, 0}
    };
    Pixel fg, bg;
    XGCValues gcval;
    XSizeHints sizehints;
    XClassHint classhint;
    XWMHints wmhints;

    dockapp.width = width;
    dockapp.height = height;
    dockapp.screen = DefaultScreen(dockapp.d);
    dockapp.root = DefaultRootWindow(dockapp.d);

    sizehints.flags = USSize | USPosition;
    sizehints.x = dockapp.x ? dockapp.x : 0;
    sizehints.y = dockapp.y ? dockapp.y : 0;
    sizehints.width = width;
    sizehints.height = height;

    fg = BlackPixel(dockapp.d, dockapp.screen);
    bg = WhitePixel(dockapp.d, dockapp.screen);

    dockapp.win = XCreateSimpleWindow(dockapp.d, dockapp.root,
				      sizehints.x, sizehints.y,
				      sizehints.width, sizehints.height, 1,
				      fg, bg);
    dockapp.iconwin = XCreateSimpleWindow(dockapp.d, dockapp.win,
					  sizehints.x, sizehints.y,
					  sizehints.width,
					  sizehints.height, 1, fg, bg);

    XSetWMNormalHints(dockapp.d, dockapp.win, &sizehints);
    classhint.res_name = name;
    classhint.res_class = name;
    XSetClassHint(dockapp.d, dockapp.win, &classhint);

    XSelectInput(dockapp.d, dockapp.win,
		 ExposureMask | ButtonPressMask | ButtonReleaseMask |
		 StructureNotifyMask);
    XSelectInput(dockapp.d, dockapp.iconwin,
		 ExposureMask | ButtonPressMask | ButtonReleaseMask |
		 StructureNotifyMask);

    XStoreName(dockapp.d, dockapp.win, name);
    XSetIconName(dockapp.d, dockapp.win, name);

    gcval.foreground = fg;
    gcval.background = bg;
    gcval.graphics_exposures = False;

    dockapp.gc =
	XCreateGC(dockapp.d, dockapp.win,
		  GCForeground | GCBackground | GCGraphicsExposures,
		  &gcval);

    cols[0].pixel = get_color(value("rx_color"));
    cols[1].pixel = get_color(value("tx_color"));
    cols[2].pixel = get_color(value("md_color"));
    attr.exactColors = 0;
    attr.alloc_close_colors = 1;
    attr.closeness = 1L << 15;
    attr.colorsymbols = cols;
    attr.numsymbols = 3;
    attr.valuemask =
	(XpmColorSymbols | XpmExactColors | XpmAllocCloseColors |
	 XpmCloseness);
    if (XpmCreatePixmapFromData
	(dockapp.d, dockapp.win, master_xpm, &dockapp.pixmap,
	 &dockapp.mask, &attr) != XpmSuccess) {
	fprintf(stderr, "Not enough colors!\n");
	exit(-1);
    }

    XShapeCombineMask(dockapp.d, dockapp.win, ShapeBounding, 0, 0,
		      dockapp.mask, ShapeSet);
    XShapeCombineMask(dockapp.d, dockapp.iconwin, ShapeBounding, 0, 0,
		      dockapp.mask, ShapeSet);

    wmhints.initial_state = WithdrawnState;
    wmhints.flags = StateHint;
    wmhints.icon_window = dockapp.iconwin;
    wmhints.icon_x = sizehints.x;
    wmhints.icon_y = sizehints.y;
    wmhints.window_group = dockapp.win;
    wmhints.flags =
	StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;
    XSetWMHints(dockapp.d, dockapp.win, &wmhints);

    XMapWindow(dockapp.d, dockapp.win);
    dockapp.xfd = ConnectionNumber(dockapp.d);
}

/* support function - chop cr/lf off the end of a string */
void chomp(char *buffer)
{
    int i = strlen(buffer) - 1;
    while (buffer[i] == '\n' || buffer[i] == '\r') {
	buffer[i--] = '\0';
    }
}

void conf_read(char *filename)
{
    FILE *fp;
    char buf[1024];
    int pos;
    int line = 0;

    /*
     * default settings. overwritten by actual config file. assigned here
     * with sensible defaults, incase there is no config file or one of
     * these is missing from the config file 
     */
    assign("tx_color", "#00fff2");
    assign("rx_color", "#188a86");
    assign("md_color", "#71e371");
    assign("refresh", "50000");
    assign("scroll", "1");

    fp = fopen(filename, "r");

    if (fp) {
	/* actually read in the config file, skipping over short and #lines */
	while (fgets(buf, 1024, fp)) {
	    line++;
	    if (buf[0] == '#' || (strlen(buf) < 2)) {
		continue;
	    }
	    chomp(buf);
	    pos = strcspn(buf, "=");
	    if (pos < strlen(buf)) {
		buf[pos++] = '\0';
		assign(buf, buf + pos);
	    }
	    memset(buf, 0, 1024);
	}
	fclose(fp);
    } else {			/* can't open the rc file, make a new one */
	fprintf(stderr, "Can't open WMND config file %s\n", filename);
	fprintf(stderr, "Using default settings\n");
	conf_write(filename);
    }
    /* set stuff here */
    wmnd.refresh = atoi(value("refresh"));
    wmnd.scroll = atoi(value("scroll"));
}

void conf_write(char *filename)
{
    FILE *fp;
    struct var *vp;

    fp = fopen(filename, "w");
    if (!fp) {
	fprintf(stderr, "Can't open config file for writing\n");
	exit(1);
    }
    fprintf(fp, "# WMND configuration file\n\n");
    for (vp = vars; vp; vp = vp->v_next)
	fprintf(fp, "%s=%s\n", vp->v_name, vp->v_value);
    fclose(fp);
}

void assign(char *name, char *value)
{
    struct var *vp;
    struct var *new;

    new = (struct var *) malloc(sizeof(struct var));

    vdel(name);

    if (vars == NULL || strcmp(vars->v_name, name) >= 0) {
	new->v_next = vars;
	vars = new;
    } else {
	vp = vars;
	while (vp->v_next && strcmp(vp->v_next->v_name, name) <= 0) {
	    vp = vp->v_next;
	}
	new->v_next = vp->v_next;
	vp->v_next = new;
    }
    new->v_name = vcopy(name);
    new->v_value = vcopy(value);
}

void vdel(char *name)
{
    struct var *vp;
    struct var *new;

    while (vars && !strcmp(vars->v_name, name)) {
	vp = vars->v_next;
	free((struct var *) vars);
	vars = vp;
    }
    if (vars)
	for (new = vars; new->v_next;)
	    if (!strcmp(new->v_next->v_name, name)) {
		vp = new->v_next->v_next;
		free(new->v_next);
		new->v_next = vp;
	    } else
		new = new->v_next;
}

char *vcopy(char *str)
{
    char *new;
    unsigned int len;

    if (str == NULL)
	return "";
    len = strlen(str) + 1;
    new = malloc(len);
    memcpy(new, str, len);
    return new;
}

char *value(char *name)
{
    struct var *vp;

    if ((vp = lookup(name)) == NULL)
	return NULL;
    return vp->v_value;
}

struct var *lookup(char *name)
{
    struct var *vp;

    for (vp = vars; vp; vp = vp->v_next)
	if (!strcmp(name, vp->v_name))
	    return vp;
    return NULL;
}

void beat_event(void)
{
    unsigned long diff;
    unsigned int min, hr;
    char temp[16];

    eprint(0, "activated");
    if (still_online(wmnd.current)) {
	if (!bit_get(RUN_ONLINE)) {
	    bit_set(RUN_ONLINE);
	    led_control(LED_POWER, bit_get(RUN_ONLINE));
	}
    } else {
	if (bit_get(RUN_ONLINE)) {
	    bit_off(RUN_ONLINE);
	    led_control(LED_POWER, bit_get(RUN_ONLINE));
	}
    }
    diff = difftime(time(NULL), wmnd.pppstart) / 60;
    min = diff % 60;
    hr = (diff / 60) % 100;
    sprintf(temp, "%02d.%02d", hr, min);
    draw_string(temp, 71, 36);
}

/* call me after initializing X11 */
void cycle_devices(char *def)
{
    struct Devices *ptr;

    if (!dockapp.d)
	fprintf(stderr,
		"I told you, don't call me before initializing X11\n");

    device_list();

    /*
     * dirty hack for setting default device at the beginning hahahaha
     * gawd this looks like BASIC code 
     */
    if (def != NULL) {
	ptr = lookup_device(def, devices);
	if (ptr)
	    goto ugly;
    }

    /* this will return NULL during the first call cause wmnd.current = NULL */
    ptr = lookup_device(wmnd.current, devices);
    if (ptr != NULL) {
	if (ptr->next)
	    ptr = ptr->next;
	else
	    ptr = devices;
    } else
	ptr = devices;

  ugly:
    strcpy(wmnd.current, ptr->name);
    eprint(1, "Current: %s", wmnd.current);
    if (still_online(wmnd.current))
	bit_set(RUN_ONLINE);
    else
	bit_off(RUN_ONLINE);
    draw_interface();
}

void click_event(unsigned int region, unsigned int button)
{
    char *action = NULL;

    if (region == -1)		/* no region */
	return;

    eprint(1, "clicked btn %d in region %d", button, region);

    if (region == 0) {
	switch (button) {
	case Button1:
	    cycle_devices(NULL);
	    break;
	case Button3:
	    bit_tgl(CFG_SHORTNAME);
	    eprint(1, "shortname: %d", bit_get(CFG_SHORTNAME));
	    draw_interface();
	    break;
	}
    } else if (region == 1) {
	if (button == Button1) {
	    bit_tgl(CFG_MODE);
	    led_control(LED_POWER, bit_get(RUN_ONLINE));
	}
    } else if (region == 2) {
	if (button == Button1) {
	    if (wmnd.wavemode < 2)
		wmnd.wavemode++;
	    else
		wmnd.wavemode = 0;
	    eprint(1, "wavemode: %d", wmnd.wavemode);
	}
    } else if (region == 3) {
	switch (button) {
	case Button1:
	    /* switch max screen/history */
	    bit_tgl(CFG_MAXSCREEN);
	    break;
	case Button3:
	    /* toggle max display */
	    bit_tgl(CFG_SHOWMAX);
	    break;
	}
    } else if (region == 4) {
	switch (button) {
	case Button1:
	    action = value("bt1_action");
	    break;
	case Button2:
	    action = value("bt2_action");
	    break;
	case Button3:
	    action = value("bt3_action");
	    break;
	}
	if (action)
	    execCommand(action);
    }
}


int main(int argc, char **argv)
{
    char *dispname = NULL;
    char *conf_file = NULL;
    struct Devices *ptr;
    unsigned long int ib, ob, ip, op;
    int ch;
    int btn = 0, rgn = -1;
    XEvent event;
#ifdef PRO
    int counter = 10000;
#endif

    eprint(1, "wmnd start");
    wmnd.wavemode = 2;
    wmnd.scale = metric_scale;

    while ((ch = getopt(argc, argv, "bd:i:hvw")) != EOF) {
	switch (ch) {
	case 'b':
	    wmnd.scale = binary_scale;
	    break;
	case 'd':
	    if (optarg)
		dispname = strdup(optarg);
	    break;
	case 'i':
	    if (optarg)
		active_interface = strdup(optarg);
	    break;
	case 'v':
	    printversion();
	    exit(0);
	    break;
	case 'w':
	    wmnd.wavemode = 0;
	    break;
	default:
	    usage();
	    exit(0);
	    break;
	}
    }
    wmnd.nr_devices = 0;
    wmnd.flags = 0;
    if (!device_list()) {
	fprintf(stderr, "No network interfaces found, exiting\n");
	exit(1);
    }

    /* set default config options */
    bit_set(RUN_ONLINE);
    bit_set(CFG_SHORTNAME);
    bit_set(CFG_SHOWMAX);
    bit_set(CFG_MAXSCREEN);
    bit_set(CFG_MODE);

    if (!conf_file) {
	char *tmp;
	tmp = getenv("HOME");
	conf_file = calloc(1, strlen(tmp) + 9);
	strcat(conf_file, tmp);
	strcat(conf_file, "/.wmndrc");
    }
    conf_read(conf_file);

    eprint(1, "open X display");
    if (!(dockapp.d = XOpenDisplay(dispname))) {
	fprintf(stderr, "Unable to open display '%s'\n", dispname);
	exit(-1);
    }
    new_window("wmnd", 64, 64);

    add_mr(0, 3, 3, 38, 9);	/* device */
    add_mr(1, 54, 3, 7, 9);	/* up/down packet/byte mode */
    add_mr(2, 3, 22, 58, 31);	/* main display area */
    add_mr(3, 3, 13, 58, 9);	/* scale meter */
    add_mr(4, 3, 54, 58, 7);	/* user script? huh wtf does that do */

#ifndef USE_KSTAT
    beat_init(wmnd.scroll * 1000000);	/* init graph updates */
#else
    beat_init(wmnd.refresh);	/* init graph updates */
#endif

    wmnd.draw = draw_stats;

    cycle_devices(active_interface);	/* set default device name */
    eprint(1, "found %d devices [%s]", wmnd.nr_devices, wmnd.current);

    /* fill the history with stuff so that we don't get total bytes as max 
       after startup and fuckup all the scales */
    for (ptr = devices; ptr; ptr = ptr->next) {
	ptr->ib_max_his = 0;
	ptr->ob_max_his = 0;
	ptr->ip_max_his = 0;
	ptr->op_max_his = 0;
	get_statistics(ptr->name, &ip, &op, &ib, &ob);
	ptr->ib_stat_last = ib;
	ptr->ob_stat_last = ob;
	ptr->ip_stat_last = ip;
	ptr->op_stat_last = op;
    }

    eprint(1, "looping");
    XSync(dockapp.d, False);	/* kick off X11 queue */
    /* loop forever */

#ifndef PRO
#define EVER 1
#else
#define EVER (counter--)
#endif
    while (EVER) {
	char *name;
	unsigned int j;

	/* get statistics for each existing device */
	for (ptr = devices; ptr; ptr = ptr->next) {
	    name = ptr->name;

	    if (get_statistics(name, &ip, &op, &ib, &ob) == -1)
		break;

	    ptr->his[57][0] += ib - ptr->ib_stat_last;
	    ptr->his[57][1] += ob - ptr->ob_stat_last;
	    ptr->his[57][2] += ip - ptr->ip_stat_last;
	    ptr->his[57][3] += op - ptr->op_stat_last;
	    ptr->ib_max_his = MAX(ptr->ib_max_his, ptr->his[57][0]);
	    ptr->ob_max_his = MAX(ptr->ob_max_his, ptr->his[57][1]);
	    ptr->ip_max_his = MAX(ptr->ip_max_his, ptr->his[57][2]);
	    ptr->ip_max_his = MAX(ptr->ip_max_his, ptr->his[57][3]);

	    if (!strcmp(ptr->name, wmnd.current)) {	/* displayed */
		if (ptr->ib_stat_last == ib) {
		    led_control(LED_RX, 0);
		} else {
		    led_control(LED_RX, 1);
		}
		if (ptr->ob_stat_last == ob) {
		    led_control(LED_TX, 0);
		} else {
		    led_control(LED_TX, 1);
		}
	    }
	    ptr->ib_stat_last = ib;
	    ptr->ob_stat_last = ob;
	    ptr->ip_stat_last = ip;
	    ptr->op_stat_last = op;
	}
	/* end get statistics */

#ifndef USE_KSTAT
	if (beat > 0) {
#else
	if (beat > wmnd.scroll * 1000000 / wmnd.refresh) {
#endif
	    beat_roger(0);
	    beat_event();

	    /* scroll the statistics */
	    for (ptr = devices; ptr; ptr = ptr->next) {
		if (!strcmp(ptr->name, wmnd.current)) {
		    wmnd.draw(ptr);
		}
		for (j = 1; j < 58; j++) {
		    ptr->his[j - 1][0] = ptr->his[j][0];
		    ptr->his[j - 1][1] = ptr->his[j][1];
		    ptr->his[j - 1][2] = ptr->his[j][2];
		    ptr->his[j - 1][3] = ptr->his[j][3];
		}
		ptr->his[57][0] = 0;
		ptr->his[57][1] = 0;
		ptr->his[57][2] = 0;
		ptr->his[57][3] = 0;
	    }
	}
	while (XPending(dockapp.d)) {
	    eprint(1, "X11 activity");
	    XNextEvent(dockapp.d, &event);
	    switch (event.type) {
	    case Expose:
		dockapp.update = 1;
		redraw_window();
		break;
	    case DestroyNotify:
		XCloseDisplay(dockapp.d);
		exit(0);
		break;
	    case ButtonPress:
		btn = event.xbutton.button;
		rgn = check_mr(event.xbutton.x, event.xbutton.y);
		break;
	    case ButtonRelease:
		if (btn == event.xbutton.button) {
		    if (rgn == check_mr(event.xbutton.x, event.xbutton.y)) {
			click_event(rgn, btn);
		    }
		}
		break;
	    }
	}
	redraw_window();
#ifndef PRO
#ifndef USE_KSTAT
	usleep(wmnd.refresh);
#else
	pause();
#endif
#endif
    }
    exit(0);
}

#ifndef USE_KSTAT
int device_list(void)
{
    FILE *fd;
    int found = 0;		/* number of devices we'll find */
    char temp[MAXBUF];		/* string buffer */
    char *tokens = " :\t\n";	/* parse tokens */
    char *p;			/* ptr into temp */
    time_t ts;			/* add timestamp */

    fd = fopen(NET_DEV, "r");
    ts = time(NULL);
    /* Skip the first 2 lines */
    fgets(temp, MAXBUF, fd);
    fgets(temp, MAXBUF, fd);

    /* grab all active devices, adding them as we go */
    while (fgets(temp, MAXBUF, fd)) {
	p = strtok(temp, tokens);
	if (!strncmp(p, "dummy", 5) || !strncmp(p, "irda", 4))
	    continue;
	if (strcmp(p, "lo") ||
	    (active_interface && !strcmp(active_interface, "lo"))) {
	    add_device(p, &devices, ts);
	    found++;
	    wmnd.draw = draw_stats;
	}
    }
    fclose(fd);

    while (found < wmnd.nr_devices) {
	struct Devices *ptr;
	/* something happened - device is removed */
	eprint(1, "detected device removal");

	for (ptr = devices; ptr; ptr = ptr->next) {
	    if (ptr->ts < ts) {
		eprint(1, "delete candidate name: %s, ts: %ld", ptr->name,
		       ptr->ts);
		strcpy(temp, ptr->name);
	    }
	}
	delete_device(temp, &devices);
	if (!wmnd.nr_devices) {	/* we've deleted them all, so add "null0" */
	    add_device("null0", &devices, ts);
	    wmnd.draw = draw_error;
	}
    }
    return found;
}

#else				// USE_KSTAT

int device_list(void)
{
    int found = 0;		/* number of devices we'll find */

    time_t ts;			/* add timestamp */

    kstat_t *ksp;

    if (kc == NULL) {
	if ((kc = kstat_open()) == NULL) {
	    perror("wmnd [kstat]: can't open /dev/kstat\n");
	    exit(1);
	}
    }

    ts = time(NULL);

    /* grab all active devices, adding them as we go */
    for (ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
	if (strcmp(ksp->ks_class, "net") == 0) {
	    kstat_read(kc, ksp, NULL);
	    if (kstat_data_lookup(ksp, "ipackets") &&
		kstat_data_lookup(ksp, "opackets") &&
		kstat_data_lookup(ksp, "rbytes") &&
		kstat_data_lookup(ksp, "obytes")) {
		add_device(ksp->ks_name, &devices, ts);
		found++;
		wmnd.draw = draw_stats;
	    }
	}
    }


    while (found < wmnd.nr_devices) {
	struct Devices *ptr;
	char temp[MAXBUF];
	/* something happened - device is removed */
	eprint(1, "detected device removal");

	for (ptr = devices; ptr; ptr = ptr->next) {
	    if (ptr->ts < ts) {
		eprint(1, "delete candidate name: %s, ts: %ld", ptr->name,
		       ptr->ts);
		strcpy(temp, ptr->name);
	    }
	}
	delete_device(temp, &devices);
	if (!wmnd.nr_devices) {	/* we've deleted them all, so add "null0" */
	    add_device("null0", &devices, ts);
	    wmnd.draw = draw_error;
	}
    }
    return found;
}

#endif

void binary_scale(unsigned char sign, unsigned long value, char *buf)
{
    unsigned char scale;
    unsigned int i;
    char *r;

    if (value > 1073741823) {	/* scale in giga */
	value = value >> 30;
	scale = 'G';
    } else if (value > 1048575) {	/* scale in mega */
	value = value >> 20;
	scale = 'M';
    } else if (value > 1023) {	/* scale in kilo */
	value = value >> 10;
	scale = 'K';
    } else {
	scale = ' ';
    }
    sprintf(buf, "%c%lu", sign, value);	/* to string */
    r = buf;
    r++;
    for (i = 3; i > 0 && *r != '\0'; i--) {
	if (*r == '+' || *r == '-' || *r == '.') {
	    ++i;
	}
	++r;
    }
    *r++ = scale;
    *r = '\0';
}

void metric_scale(unsigned char sign, unsigned long value, char *buf)
{
    float f;
    unsigned char scale;
    unsigned int i;
    char *r;

    f = (float) value;
    if (value > 999999999) {	/* scale in giga */
	f /= 1000000000;
	scale = 'G';
    } else if (value > 999999) {	/* scale in mega */
	f /= 1000000;
	scale = 'M';
    } else if (value > 999) {	/* scale in kilo */
	f /= 1000;
	scale = 'K';
    } else {
	scale = ' ';
    }
    sprintf(buf, "%c%f", sign, f);	/* to string */
    r = buf;
    r++;
    for (i = 3; i > 0 && *r != '\0'; i--) {
	if (*r == '+' || *r == '-' || *r == '.') {
	    ++i;
	}
	++r;
    }
    *r++ = scale;
    *r = '\0';
}

void scale(char *rx_buf, char *tx_buf, unsigned long rx, unsigned long tx)
{
    char rx_sign, tx_sign;

    if (rx > tx) {
	rx_sign = '+';
	tx_sign = '#';
    } else if (rx < tx) {
	rx_sign = '-';
	tx_sign = '*';
    } else {
	rx_sign = '-';
	tx_sign = '#';
    }
    if (wmnd.scroll != 1) {
	tx /= wmnd.scroll;
	rx /= wmnd.scroll;
    }
    wmnd.scale(tx_sign, tx, tx_buf);
    wmnd.scale(rx_sign, rx, rx_buf);
}

void draw_string(unsigned char *buf, unsigned int x, unsigned int y)
{
    unsigned int w, sx = 0, sy = 0;
    unsigned int draw;
    char *r;

    w = 3;
    draw = 0;
    for (r = buf; *r != '\0'; r++) {
	if (*r >= '0' && *r <= '9') {
	    w = 5;
	    sx = 1 + (w * (*r - '0'));
	    sy = 85;
	    draw = 1;
	} else if (*r == '.') {
	    w = 2;
	    sx = 62;
	    sy = 85;
	    draw = 1;
	} else if (*r >= 'A' && *r <= 'Z') {
	    w = 5;
	    sx = 1 + (w * (*r - 'A'));
	    sy = 93;
	    draw = 1;
	} else if (*r == '+') {
	    w = 6;
	    if (bit_get(CFG_MAXSCREEN)) {
		sx = 133;
		sy = 93;
	    } else {
		sx = 66;
		sy = 46;
	    }
	    draw = 1;
	} else if (*r == '-') {
	    w = 6;
	    if (bit_get(CFG_MAXSCREEN)) {
		sx = 139;
		sy = 93;
	    } else {
		sx = 72;
		sy = 46;
	    }
	    draw = 1;
	} else if (*r == '*') {
	    w = 6;
	    if (bit_get(CFG_MAXSCREEN)) {
		sx = 145;
		sy = 93;
	    } else {
		sx = 78;
		sy = 46;
	    }
	    draw = 1;
	} else if (*r == '#') {
	    w = 6;
	    if (bit_get(CFG_MAXSCREEN)) {
		sx = 151;
		sy = 93;
	    } else {
		sx = 84;
		sy = 46;
	    }
	    draw = 1;
	}
	if (draw) {
	    copy_xpm_area(sx, sy, w, 7, x, y);
	    draw = 0;
	}
	x += w;
    }
}

void draw_rate(unsigned long rx, unsigned long tx)
{
    char rx_buf[16];
    char tx_buf[16];

    /* clear rate bar */
    copy_xpm_area(100, 85, 58, 7, 3, 54);

    /* put rx/tx numbers into strings, scaling them */
    scale(rx_buf, tx_buf, rx, tx);

    /* draw rx/tx strings */
    draw_string(rx_buf, 3, 54);
    draw_string(tx_buf, 32, 54);
}

void draw_max(unsigned long rx, unsigned long tx)
{
    char rx_buf[16];
    char tx_buf[16];

    /* clear rate bar */
    copy_xpm_area(100, 85, 58, 7, 3, 11);

    /* put rx/tx numbers into strings, scaling them */
    scale(rx_buf, tx_buf, rx, tx);

    /* draw rx/tx strings */
    draw_string(rx_buf, 3, 11);
    draw_string(tx_buf, 32, 11);
}

void draw_error(struct Devices *ptr)
{
    copy_xpm_area(25, 75, 5, 7, 12, 30);
    copy_xpm_area(103, 75, 5, 7, 18, 30);
    copy_xpm_area(103, 75, 5, 7, 24, 30);
    copy_xpm_area(85, 75, 5, 7, 30, 30);
    copy_xpm_area(103, 75, 5, 7, 36, 30);
}

void draw_stats(struct Devices *ptr)
{
    int bpp = 1;		/* bytes per pixel */
    unsigned int k;
    unsigned long int *p;
    unsigned int in, out;
    unsigned long int rx_max_his, tx_max_his;
    unsigned long long int rx_max, tx_max, max;
    unsigned int size;

    if (bit_get(CFG_SHOWMAX))
	size = 35;		/* with max bar mode */
    else
	size = 41;		/* without max bar */

    if (bit_get(CFG_MODE)) {	/* bytes mode */
	in = 0;
	out = 1;
	rx_max_his = ptr->ib_max_his;
	tx_max_his = ptr->ob_max_his;
    } else {			/* packets mode */
	in = 2;
	out = 3;
	rx_max_his = ptr->ip_max_his;
	tx_max_his = ptr->op_max_his;
    }

    /* find maximum value in screen history */
    rx_max = tx_max = 0;
    p = (unsigned long *) ptr->his;
    for (k = 0; k < 58; k++) {
	rx_max = MAX(rx_max, p[in]);
	tx_max = MAX(tx_max, p[out]);
	p += 4;
    }
    if ((max = rx_max + tx_max) > size) {
	bpp = max / size;
	if ((max % size) > 0) {
	    ++bpp;
	}
    }

    /* draw rx/tx rate */
    p -= 4;			/* sets p to his[58] or so */
    draw_rate(p[in], p[out]);

    if (bit_get(CFG_MAXSCREEN))
	draw_max(rx_max, tx_max);
    else
	draw_max(rx_max_his, tx_max_his);

    p = (unsigned long *) ptr->his;
    switch (wmnd.wavemode) {
    case 0:			/* traditional mode */
	for (k = 0; k < 58; k++) {
	    unsigned int txlev, rxlev, nolev;
	    rxlev = p[in] / bpp;
	    txlev = p[out] / bpp;
	    nolev = size - (rxlev + txlev);
	    copy_xpm_area(66, 0, 1, txlev, k + 3, 53 - (rxlev + txlev));
	    copy_xpm_area(65, 0, 1, nolev, k + 3, 53 - size);
	    copy_xpm_area(67, 0, 1, rxlev, k + 3, 53 - rxlev);
	    p += 4;
	}
	copy_xpm_area(70, 1, 58, 1, 3, 53 - size);
	break;
    case 1:			/* wave form mode */
	for (k = 0; k < 58; k++) {
	    unsigned int txlev, rxlev, center;
	    rxlev = p[in] / bpp / 2;	/* we need them like this */
	    txlev = p[out] / bpp / 2;	/* we need it like this */
	    center = 53 - size / 2;
	    copy_xpm_area(65, 0, 1, size, k + 3, 53 - size);
	    copy_xpm_area(66, 0, 1, txlev, k + 3,
			  (center - rxlev) - txlev);
	    copy_xpm_area(66, 0, 1, txlev, k + 3, (center + rxlev));
	    copy_xpm_area(67, 0, 1, rxlev * 2, k + 3, center - rxlev);
	    p += 4;
	}
	copy_xpm_area(70, 1, 58, 1, 3, 53 - size);
	copy_xpm_area(70, 0, 58, 1, 3, 53 - size / 2);
	break;
    case 2:			/* WMNET mode */
	for (k = 0; k < 58; k++) {
	    unsigned int txlev, rxlev, nolev;
	    txlev = p[out] / bpp;
	    rxlev = p[in] / bpp;
	    nolev = size - (rxlev + txlev);
	    copy_xpm_area(66, 0, 1, txlev, k + 3, 53 - size);
	    copy_xpm_area(65, 0, 1, nolev, k + 3, (53 - size) + txlev);
	    copy_xpm_area(67, 0, 1, rxlev, k + 3, 53 - rxlev);
	    p += 4;
	}
	copy_xpm_area(70, 1, 58, 1, 3, 53 - size);
	break;
    }
    /* copy PPP connection time over the graph */
    if (strstr(ptr->name, "ppp")) {
	copy_xpm_area(70, 36, 23, 7, 37, 46);
    }
}


#ifndef USE_KSTAT

static int
get_statistics(const char *devname, unsigned long *ip,
	       unsigned long *op, unsigned long *ib, unsigned long *ob)
{
    FILE *fp;
    char temp[MAXBUF];
    char *p;
    struct ifpppstatsreq req;

    static int ppp_opened = 0;
    static int ppp_h = -1;

    eprint(0, "device: %s", devname);

    /* handle special ppp case */
    if (!strncmp(devname, "ppp", 3)) {
	if (!ppp_opened) {
	    struct stat mstat;
	    char pidname[MAXBUF];
	    snprintf(pidname, strlen(devname) + 14, "/var/run/%s.pid",
		     devname);
	    stat(pidname, &mstat);
	    eprint(1, "pidfile: %s", pidname);
	    wmnd.pppstart = mstat.st_mtime;
	    /* Open the ppp device.  */
	    if ((ppp_h = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		return -1;
	    }
	    ppp_opened = 1;
	    eprint(1, "ppp monitor connection ON\n-- %s",
		   ctime(&wmnd.pppstart));
	}
	memset(&req, 0, sizeof(req));
	req.stats_ptr = (caddr_t) & req.stats;
	sprintf(req.ifr__name, devname);

	if (ioctl(ppp_h, SIOCGPPPSTATS, &req) < 0) {
	    eprint(1, "ppp monitor disconnected");
	    close(ppp_h);
	    ppp_opened = 0;
	    wmnd.pppstart = 0;
	    cycle_devices(NULL);
	    return -1;
	}
	*op = req.stats.p.ppp_opackets;
	*ip = req.stats.p.ppp_ipackets;
	*ib = req.stats.p.ppp_ibytes;
	*ob = req.stats.p.ppp_obytes;

	eprint(0, "ppp: ip:%lu op:%lu ib:%lu ob:%lu", *ip, *op, *ib, *ob);
	return 0;
    }
    /* or else, read statistics from /proc/net/dev */
    fp = fopen(NET_DEV, "r");
    fgets(temp, MAXBUF, fp);
    fgets(temp, MAXBUF, fp);

    while (fgets(temp, MAXBUF, fp)) {
	if (strstr(temp, devname)) {
	    p = strchr(temp, ':');
	    p++;
	    sscanf(p,
		   "%lu %lu %*s %*s %*s %*s %*s %*s %lu %lu", ib, ip, ob,
		   op);
	    eprint(0, "ip:%lu op:%lu ib:%lu ob:%lu", *ip, *op, *ib, *ob);
	}
    }
    fclose(fp);
    return 0;
}


int still_online(char *dev)
{
    FILE *fp;
    char temp[MAXBUF];

    fp = fopen(NET_ROUTE, "r");
    while (fgets(temp, MAXBUF, fp)) {
	if (strstr(temp, dev)) {
	    fclose(fp);
	    return 1;		/* Line is alive */
	}
    }
    fclose(fp);
    return 0;
}

#else				//USE_KSTAT


static int
get_statistics(const char *devname, unsigned long *ip,
	       unsigned long *op, unsigned long *ib, unsigned long *ob)
{
    kstat_named_t *in_pkt, *in_byte, *out_pkt, *out_byte;

    static char *old_devname = NULL;

    eprint(0, "device: %s", devname);

    if (if_ksp == NULL || strcmp(devname, old_devname)) {
	if ((if_ksp = (kstat_lookup(kc, NULL, -1, devname))) == NULL) {
	    eprint(1, "wmnd [kstat]: no network interface "
		   "%s exporting kstats found\n", devname);
	    return -1;
	}
	if (old_devname)
	    free(old_devname);
	old_devname = strdup(devname);
    }


    if (kstat_read(kc, if_ksp, NULL) == -1) {
	(void) fprintf(stderr, "wmnd [kstat]: kstat_read failed\n");
	exit(1);
    }

    if ((in_pkt = (kstat_named_t *) kstat_data_lookup(if_ksp, "ipackets"))
	== NULL) {
	(void) fprintf(stderr, "wmnd [kstat]: kstat_data_lookup "
		       "(ipackets) failed\n");
	exit(1);
    }
    *ip = (in_pkt->value.ui32);

    if ((in_byte = (kstat_named_t *) kstat_data_lookup(if_ksp, "rbytes"))
	== NULL) {
	(void) fprintf(stderr, "wmnd [kstat]: "
		       "kstat_data_lookup (rbytes) failed\n");
	exit(1);
    }
    *ib = in_byte->value.ui32;


    if ((out_pkt = (kstat_named_t *) kstat_data_lookup(if_ksp, "opackets"))
	== NULL) {
	(void) fprintf(stderr, "wmnd [kstat]: kstat_data_lookup "
		       "(opackets) failed\n");
	exit(1);
    }
    *op = (out_pkt->value.ui32);

    if ((out_byte = (kstat_named_t *) kstat_data_lookup(if_ksp, "obytes"))
	== NULL) {
	(void) fprintf(stderr, "wmnd [kstat]: "
		       "kstat_data_lookup (obytes) failed\n");
	exit(1);
    }
    *ob = out_byte->value.ui32;


    eprint(0, "ip:%lu op:%lu ib:%lu ob:%lu", *ip, *op, *ib, *ob);

    return 0;
}


int still_online(char *dev)
{
    return 1;			/* Line is alive */
}

#endif



static void led_control(const unsigned char led, const unsigned char mode)
{
    eprint(0, "led: %02x[%02x]", led, mode);
    switch (led) {
    case LED_POWER:
	switch (bit_get(CFG_MODE)) {
	case 1:		/* bytes */
	    if (mode) {		/* on-line */
		copy_xpm_area(116, 64, 5, 7, 55, 4);
	    } else {		/* off-line */
		copy_xpm_area(122, 64, 5, 7, 55, 4);
	    }
	    break;
	case 0:		/* packets */
	    if (mode) {		/* on-line */
		copy_xpm_area(128, 64, 5, 7, 55, 4);
	    } else {		/* off-line */
		copy_xpm_area(134, 64, 5, 7, 55, 4);
	    }
	    break;
	}
	break;
    case LED_RX:
	if (mode) {		/* turn on */
	    if (bit_get(LED_RX)) {
		eprint(0, "RX led already on");
		return;		/* already on */
	    }
	    copy_xpm_area(86, 69, 5, 4, 41, 4);
	    bit_set(LED_RX);
	    eprint(0, "RX led on");
	} else {		/* turn off */
	    if (!bit_get(LED_RX)) {
		eprint(0, "RX led already off");
		return;
	    }
	    copy_xpm_area(92, 69, 5, 4, 41, 4);
	    bit_off(LED_RX);
	    eprint(0, "RX led off");
	}
	break;
    case LED_TX:
	if (mode) {		/* turn on */
	    if (bit_get(LED_TX)) {
		eprint(0, "TX led already on");
		return;		/* already on */
	    }
	    copy_xpm_area(86, 64, 5, 4, 48, 4);
	    bit_set(LED_TX);
	    eprint(0, "TX led on");
	} else {		/* turn off */
	    if (!bit_get(LED_TX)) {
		eprint(0, "TX led already off");
		return;
	    }
	    copy_xpm_area(92, 64, 5, 4, 48, 4);
	    bit_off(LED_TX);
	    eprint(0, "TX led off");
	}
	break;
    }
}

void draw_interface(void)
{
    int i;
    int c;
    int k = 3;
    unsigned char temp[7];

    /* refresh */
    copy_xpm_area(65, 54, 38, 9, 3, 3);
    led_control(LED_POWER, bit_get(RUN_ONLINE));

    if (bit_get(CFG_SHORTNAME)) {
	strncpy(temp, wmnd.current, 3);
	temp[3] = wmnd.current[strlen(wmnd.current) - 1];
	temp[4] = '\0';
    } else {
	strncpy(temp, wmnd.current, 6);
	temp[7] = '\0';
    }

    for (i = 0; temp[i]; i++) {
	c = temp[i];
	if (c >= '0' && c <= '9') {
	    c -= '0';
	    copy_xpm_area(c * 6, 64, 6, 9, k, 3);
	    k += 6;
	} else {
	    if (c >= 'a' && c <= 'z')
		c = c - 'a' + 'A';
	    c -= 'A';
	    copy_xpm_area(c * 6, 74, 6, 9, k, 3);
	    k += 6;
	}
    }
}

void usage(void)
{
    fprintf(stderr,
	    "\nwmnd - WindowMaker Network Devices v%s\n"
	    "Home page: http://www.wingeer.org/wmnd/\n\n"
	    "usage:\n"
	    "  -b base 2 scale (no fractions)\n"
	    "  -d <display name>\n"
	    "  -h				   this help screen\n"
	    "  -i <interface name>  default (as it appears in /proc/net/route)\n"
	    "  -v				   print the version number\n"
	    "  -w				   traditional mode\n"
	    "\n", WMND_VERSION);
}

void printversion(void)
{
    fprintf(stderr, "%s\n", WMND_VERSION);
}
