// This interface is based on the one in the file region.h in gtk+.
// That file didn't contain license info, so is presumably under the
// same LGPL license as the rest of that library.

// Stolen from Gtk+ and C++-ized by Ron Steinke, January 2003

#ifndef _WFTK_REGION_H_
#define _WFTK_REGION_H_

#include <SDL/SDL.h>

#include <wftk/point.h>
#include <wftk/rect.h>
#include <vector>
#include <cassert>

namespace wftk { 

/// Region specifies a 'working area' of the screen - pixels
/// outside the region are always ignored
class Region
{
 public:
  /// Empty region
  Region();
  /// Copy a region
  Region(const Region&);
  /// Fill rule for polygons
  enum FillRule
  {
    EVEN_ODD_RULE, ///< .
    WINDING_RULE   ///< .
  };
  /// Region from polygon
  Region(const std::vector<Point>, FillRule);
  /// Region from rectangle
  Region(const SDL_Rect&);
  /// Region from point
  Region(const Point&);

  ~Region() {delete [] rects;}

  /// copy one region into another
  Region& operator=(const Region&);

  /// Get clipping box
  Rect getClipbox() const
	{return Rect(extents.x1, extents.y1,
		extents.x2 - extents.x1, extents.y2 - extents.y1);}

  /// Used for things like passing an array of SDL_Rect to SDL_UpdateRects()
  class RectList {
   public:
    /// create a list of uninitialized rectangles
    RectList(unsigned n) : rects_(new SDL_Rect[n]), nrects_(n) {}
    /// copy a RectList
    RectList(const RectList& rl) : rects_(0), nrects_(0) {operator=(rl);}
    ~RectList() {delete [] rects_;}

    /// copy the contents of one RectList to another
    RectList& operator=(const RectList& rl);

    /// access the rectangles contained in the list
    SDL_Rect* rects() const {return rects_;}
    /// get a count of the rectangles in the list
    unsigned nrects() const {return nrects_;}

    // The 0 index for overflow is safe, but kind of silly
    /// access a rectangle in the list
    SDL_Rect& operator[](unsigned n) {return rects_[n < nrects_ ? n : 0];}

   private:
    SDL_Rect* rects_;
    unsigned nrects_;
  };

#ifndef WFTK_DISABLE_DEPRECATED
  /// Get component rectangles
  RectList getRectangles() const;
#endif

  // a better solution than getRectangles(), with no memory allocation
  unsigned long nRects() const {return numRects;}
  Rect getRect(unsigned long i) const
	{assert(i < (unsigned long) numRects); return rects[i];}

  /// true if region empty
  bool empty() const {return numRects == 0;}
  /// compare two regions
  bool operator==(const Region&) const;
  /// compare two regions
  bool operator!=(const Region& r) const {return !operator==(r);}
  /// for std::map only
  bool operator<(const Region&) const;
  /// check if a region contains a point
  bool contains(const Point&) const;
  /// Types of overlapping between a rectangle and a region
  enum OverlapType
  {
    OVERLAP_IN,  ///< rectangle is in region
    OVERLAP_OUT, ///< rectangle is not in region
    OVERLAP_PART ///< rectangle is partially in region
  };
  /// Check if a region contains/overlaps a rectangle
  OverlapType overlap(const SDL_Rect&) const;

  /// set the region to be empty
  void clear();

  /// Pointwise or
  friend Region operator|(const Region& r, const Region& r2)
	{Region out(r); out |= r2; return out;}
  /// Pointwise and
  friend Region operator&(const Region& r, const Region& r2)
	{Region out(r); out &= r2; return out;}
  /// Pointwise subtraction
  friend Region operator-(const Region& r, const Region& r2)
	{Region out(r); out -= r2; return out;}
  /// Pointwise xor
  friend Region operator^(const Region& r, const Region& r2)
	{Region out(r); out ^= r2; return out;}

  /// Move a region
  void offset(int dx, int dy);
  /// Move a region
  void offset(const Point& p) {offset(p.x, p.y);}
  /** Shrink a region
   * Contract (dx, dy > 0) or expand (dx, dy < 0) the region
   * in the directions of the axes. If one of (dx, dy) is positive
   * and the other negative, the region contracts in one direction
   * and expands in the other.
   **/
  void shrink(int dx, int dy);

  /// Pointwise or
  Region& operator|=(const Region&);
  /// Pointwise and
  Region& operator&=(const Region&);
  /// Pointwise subtraction
  Region& operator-=(const Region&);
  /// Pointwise xor
  Region& operator^=(const Region&);

  /// Return the boundary of the region
  /**
   * inside == true => points in the region which are next to points outside the region
   * inside == false => points outside the region which are next to points in the region
   *
   * FIXME pick a default value for 'inside'
   **/
  Region boundary(bool inside) const;

 private:
  long size, numRects;
  struct RegionBox
  {
    int x1, y1, x2, y2;

    // used in Region::getRectangles()
    operator Rect() const {return Rect(x1, y1, x2-x1, y2-y1);}

  } *rects, extents;

 public:

  // const_iterator has to be declared after RegionBox

  /// Iterator over points in the region
  friend class const_iterator
  {
   public:
    const_iterator() : box_(0) {}
    const_iterator(const RegionBox* begin, const RegionBox* end);

    bool operator==(const const_iterator& I) const
	{return box_ == I.box_ && p_ == I.p_;} // need box_ equality for end() case
    bool operator!=(const const_iterator& I) const
	{return !operator==(I);}

    const_iterator& operator++();
    const_iterator operator++(int)
	{const_iterator tmp(*this); operator++(); return tmp;}

    const Point& operator*() const {return p_;}
    const Point* operator->() const {return &p_;}

   private:
    const RegionBox *box_, *final_;
    Point p_;
  };

  const_iterator begin() const {return const_iterator(rects, rects + numRects);}
  const_iterator end() const {return const_iterator(rects + numRects, rects + numRects);}

 private:

  typedef void (Region::*overlapFunc) (
			       RegionBox *r1,
			       RegionBox *r1End,
			       RegionBox *r2,
			       RegionBox *r2End,
			       int          y1,
			       int          y2);
  typedef void (Region::*nonOverlapFunc) (
				  RegionBox *r,
				  RegionBox *rEnd,
				  int          y1,
				  int          y2);
  /*
   * used to allocate buffers for points and link
   * the buffers together
   */
  struct POINTBLOCK {
    /*
     * number of points to buffer before sending them off
     * to scanlines() :  Must be an even number
     */
    static const int NUMPTSTOBUFFER = 200;
    wftk::Point pts[NUMPTSTOBUFFER];
    POINTBLOCK *next;
  };

  void miSetExtents();
  void Compress(
	 Region *s,
	 Region *t,
	 unsigned int      dx,
	 int        xdir,
	 int        grow);
  void miIntersectO (
	      RegionBox *r1,
	      RegionBox *r1End,
	      RegionBox *r2,
	      RegionBox *r2End,
	      int          y1,
	      int          y2);
  int
  miCoalesce (
	    int       prevStart,
	    int       curStart);
  void
  miRegionOp(
	   const Region *reg1,
	   const Region *reg2,
	   overlapFunc    overlapFn,
	   nonOverlapFunc nonOverlap1Fn,
	   nonOverlapFunc nonOverlap2Fn);
  void
  miUnionNonO (
	     RegionBox *r,
	     RegionBox *rEnd,
	     int          y1,
	     int          y2);
  void
  miUnionO (
	  RegionBox *r1,
	  RegionBox *r1End,
	  RegionBox *r2,
	  RegionBox *r2End,
	  int          y1,
	  int          y2);
  void
  miSubtractNonO1 (
		 RegionBox *r,
		 RegionBox *rEnd,
		 int          y1,
		 int          y2);
  void
  miSubtractO(
	     RegionBox *r1,
	     RegionBox *r1End,
	     RegionBox *r2,
	     RegionBox *r2End,
	     int          y1,
	     int          y2);
  int PtsToRegion(
    int  numFullPtBlocks, int iCurPtBlock,
    POINTBLOCK *FirstPtBlock);
};

} // namespace

#endif // _WFTK_REGION_H_

