/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _POINTER_H
#define _POINTER_H

#include <wftk/mouse.h>
#include <wftk/surface.h>
#include <wftk/timer.h>

#include <vector>

namespace wftk {

/// a pointer which uses SDL's native pointer capabilities
class HardPointer : public Pointer
{
 public:
  /// black/white/transparent pointer from a surface
  HardPointer(const Surface&, const Point& hotspot);
  ///
  ~HardPointer() {SDL_FreeCursor(ptr_);}


  /// insert a named hardware pointer into the pointer registry
  static void insert(const std::string&, const Surface&, const Point& hotspot);

 protected:
  virtual void show() {SDL_SetCursor(ptr_); SDL_ShowCursor(1);}
  virtual void hide() {SDL_ShowCursor(0);}

 private:
  // Pointer::ResInval used the default constructor
  friend struct ResInval;
  HardPointer();

  SDL_Cursor* ptr_;
};

/// a software pointer
class SoftPointer : public Pointer
{
 public:
  /// create a software mouse pointer
  SoftPointer(Surface::Resource* image, const Point& hotspot)
	: image_(image), hotspot_(hotspot) {image_->bind();}
  ///
  ~SoftPointer() {image_->free();}

  /// insert a named software pointer into the pointer registry
  static void insert(const std::string&, Surface::Resource*, const Point& hotspot);

  /// The image being used by the pointer
  const Surface& image() const {return *image_->res();}
  /// The pointer's hotspot
  const Point& hotspot() const {return hotspot_;}

 protected:
  // RootWindow takes care of the drawing, so these don't do much
  virtual void show() {SDL_ShowCursor(0);}
  virtual void hide() {}

  /// the image used by the pointer
  Surface::Resource* image_;
  /// the hotspot used by the pointer
  Point hotspot_;
};

/// an animated software pointer
class AnimatedPointer : public SoftPointer
{
 public:
  /// image and hotspot pairs
  typedef std::pair<Surface::Resource*,Point> Data;
  /// constructor takes a vector of images and update time between images
  AnimatedPointer(const std::vector<Data>&, unsigned update);
  ///
  ~AnimatedPointer();

  /// insert a named animated pointer into the pointer registry
  static void insert(const std::string&, const std::vector<Data>&, unsigned update);

 protected:
  virtual void show() {SoftPointer::show(); update_.run();}
  virtual void hide() {SoftPointer::hide(); update_.halt();}

 private:
  Timer update_;

  void switchImage();

  std::vector<Data> data_;
  unsigned image_num_;
};

} // namespace

#endif

