/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "music.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#include "mixer.h"
#include "debug.h"

#ifdef HAVE_MIXER
#include <SDL_mixer.h>
#endif

namespace wftk {

ResourceRegistry<Music*> Music::registry;

#ifdef HAVE_MIXER

Music::Music() :
  channel_(0),
  data_(NULL)
{
  Mixer::instance(); // insure init audio 
}

Music::Music(const std::string& filename) :
  channel_(0),
  data_(NULL)
{
  Mixer::instance(); // insure init audio
  load(filename);
}


Music::~Music()
{
  stop();

  if(data_)
    {
      Debug out(Debug::SOUND);
      
      out << "Music call Mix_FreeMusic() ...";
      if(Mixer::instance()->audioAvailable())
	Mix_FreeMusic((Mix_Music*) data_);
      else
	{
	  out << "deleting Music though Mixer is not available (any more?)" 
	       << Debug::endl;
	}
      out << " [OK]"<< Debug::endl;
    }
}


bool
Music::load(const std::string& filename)
{
  Debug out(Debug::SOUND);

  out << "Starting Music::load()" << Debug::endl;

  if(!Mixer::instance()->audioAvailable()) {
    out << "Music could not load: no mixer system available." << Debug::endl;
    data_ = NULL;
    return false;
  }

  data_ = Mix_LoadMUS(filename.c_str());
	      
  if(!data_)
    out << "Music: unable to load file." << Debug::endl;
  else
    out << "Loaded music using Mix_LoadMUS()" << Debug::endl;
 
  return data_ != 0; 
}


int
Music::setVolume(int vol)
{
  return Mixer::instance()->setMusicVolume(vol);
}

bool
Music::play(int loop) const
{  
  bool success = false;
  int channel = 0;
  
  if(!Mixer::instance()->audioAvailable())
    return false;

  if(data_)
    {
      channel = Mix_PlayMusic((Mix_Music*) data_,loop);
      if(channel >= 0)
	{
	  success = true;
	  Mix_VolumeMusic(Mixer::instance()->getMusicVolume());
	}
    }

  return success;
}

bool
Music::playing() const
{  
  bool playing_ = false;

  if(Mixer::instance() && Mixer::instance()->audioAvailable())  
      {
	if(data_)
	  playing_ = Mix_PlayingMusic();
      }

  return playing_;
}


void
Music::stop() const
{
  if(Mixer::instance() && Mixer::instance()->audioAvailable())
      if(data_)
	Mix_HaltMusic();
}

void
Music::pause() const
{  
  if(Mixer::instance() && Mixer::instance()->audioAvailable())
      Mix_PauseMusic();
}

void
Music::resume() const
{  
  if(Mixer::instance() && Mixer::instance()->audioAvailable())
      Mix_ResumeMusic();
}

void
Music::rewind() const
{  
  if(Mixer::instance() && Mixer::instance()->audioAvailable())
      Mix_RewindMusic();
}

bool
Music::paused() const
{
  if(Mixer::instance() && Mixer::instance()->audioAvailable())
      return Mix_PausedMusic();
  return false;
}

#else // HAVE_MIXER
//-------------------------------------------------------------
// dummy implementation when not linking against libmixer

Music::Music()
{
}

Music::Music(const std::string& filename)
{
  load(filename);
}

Music::~Music()
{
}

bool
Music::load(const std::string& )
{
  return false;
}

void
Music::pause() const
{  
}

void
Music::resume() const
{  
}

void
Music::rewind() const
{  
}


bool
Music::paused() const
{
  return false;
}

bool
Music::play(int ) const
{  
  return false;
}

int
Music::setVolume(int)
{
  return 0;
}

bool
Music::playing() const
{  
  return false;
}

void
Music::stop() const
{

}

#endif //HAVE_MIXER
} // namespace

