/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _MOUSE_H
#define _MOUSE_H

#include <wftk/point.h>
#include <wftk/resources.h>
#include <wftk/marshal.h>

#include <sigc++/marshal.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#include <sigc++/object.h>
#endif

#include <SDL/SDL_mouse.h>
#include <SDL/SDL_events.h>

namespace wftk {

class Surface;

/// virtual base pointer class
class Pointer : virtual public SigC::Object
{
  friend class Mouse;

 public:
  Pointer() {}
  virtual ~Pointer() {}

  // can't load pointers, have to create from surfaces
  struct ResLoad {
    std::pair<Pointer*,bool> operator()(const std::string&)
    {
      assert(false);
      return std::pair<Pointer*,bool>(0, false);
    }
  };
  struct ResInval {
    typedef const Pointer* OutType;
    OutType operator()(const std::string&) const;
  };
  /// A set of named pointer resources
  static ResourceRegistry<Pointer*,ResLoad,ResInval> registry;
  /// a refcounted resource
  typedef Resource<Pointer*> Resource;

 private:
  Pointer(const Pointer&);
  Pointer& operator=(const Pointer&);

 protected:
  /// show the pointer
  virtual void show() = 0;
  /// hide the pointer
  virtual void hide() = 0;
};

/**software mouse pointer.
   it allows setting special mousepointers for specified areas of
   the screen.
*/
class Mouse : virtual public SigC::Object
{
 public:
  /// Default ctor
  Mouse();
  /// Default dtor
  ~Mouse();
  /// Check if we have initialized the singleton
  static bool isInit() {return instance_ != NULL;}
  /// Singleton class - one and only one mouse pointer
  static Mouse* instance() {return isInit() ? instance_ : new Mouse();}

	/// Available masks for buttons.
  enum Button // SDL_BUTTON_FOO is button #, SDL_BUTTON() converts to bitmask
  { 
		// Rasta ASCII Pacman required so doxygen picks these up
    LEFT   = SDL_BUTTON(SDL_BUTTON_LEFT), ///< .
    RIGHT  = SDL_BUTTON(SDL_BUTTON_RIGHT), ///< .
    MIDDLE = SDL_BUTTON(SDL_BUTTON_MIDDLE), ///< .
    // these two were in uta, but don't appear to match anything in SDL
    WHEEL_UP = SDL_BUTTON(4), ///< .
    WHEEL_DOWN = SDL_BUTTON(5) ///< .
  };

  /// absolute and relative position, button state mask
  SigC::Signal3<bool,const Point&,const Point&,Button,BoolMarshal> mouseMove;
  /// button, pressed (true) or release (false), position
  SigC::Signal3<bool,Button,bool,const Point&,BoolMarshal> mouseClick;
  /// application has lost mouse focus (signalled by window manager)
  SigC::Signal0<bool,BoolMarshal> lostMouse;
  /// application has gained mouse focus (signalled by window manager)
  SigC::Signal0<bool,BoolMarshal> gotMouse;

  /// move the mouse pointer to a new screen position
  void warp(const Point& p) {SDL_WarpMouse(p.x, p.y);}
  /// get the current mouse pointer location
  const Point& position() const {return pos_;}
  /// returns a mask indicating which buttons are pressed
  Button buttons() const {return (Button) SDL_GetMouseState(0, 0);}
  /// hidden flag accessor
  bool hidden() const {return !visible_;}

  /// Only available with software pointer
  void setPointer(const Surface&, const Point& hotspot);
  /// Only available with software pointer
  void setPointer(Pointer::Resource*);
  /// Only available with software pointer
  void setPointer(const std::string& name) {setPointer(Pointer::registry.get(name));}

  /// Only available with software pointer
  const Pointer* getPointer() const {return ptr_->res();}

  /// remove pointer from screen, and update
  void hidePointer() {ptr_->res()->hide(); visible_ = false; update();}
  /// paint pointer on screen
  void showPointer() {ptr_->res()->show(); visible_ = true; update();}

  /// called by wftk mainloop
  bool handleEvent(const SDL_Event*);

 private:
  Mouse(const Mouse&);
  Mouse& operator=(const Mouse&);

  void update();

  void destroy() {delete this;} // callback handler

  ///
  Point pos_;

  ///
  bool visible_;

  ///
  Pointer::Resource* ptr_;

  ///
  static Mouse* instance_;
}; 

} // namespace

#endif

