/*******************************************************************/
/* Les fonctions suivantes sont utilisees pour gerer des solides 
 * disposes dans le monde. On n'a pas specifie ici :
 * - la gestion des sources lumineuses et des modeles lumineux
 * - la gestion des hierarchies d'objets
 * - les fonctions d'entree/sortie pour les geometries et les scenes
 * - comment gerer plusieurs mondes ou affichages
 */

#ifndef ZV_H
#define ZV_H


#include "zmath.h"

#if CYGWIN32
#define glBindTexture(target, tex) ;
#define glGenTextures(target, tex) ;
#endif


/* solid handle */

typedef struct _ZVSolid {
  u_int16	id;
  u_int16	type;
  boolean	visible;
  boolean	bbflag;
  M4		posmat;
  int32		*displaylist;
  u_int16	nbframes;
  u_int16	curframe;
  V3		bbcenter;
  V3		bbsize;
  struct WObject *object;
  struct _ZVSolid *next, *prev;
} Solid;

typedef struct _ZVContext {
  Solid		*first_solid, *last_solid;
  V3		bbox_min;
  V3		bbox_max;
  int		first_bbox;
  M4		camera_pos;
  boolean	quality;
} ZVContext;

struct _bgcolor {
  float red;
  float green;
  float blue;
  float alpha;
};

extern ZVContext zv_context;
extern struct _bgcolor bgcolor;

void zvVertex3fv(float *v);
void zvVertex3f(float x, float y, float z);

/*
 * return True if both solids 's1' et 's2' intersect themself
 * not implemented for the moment: return 'true'
 */
int solidIntersect(Solid *s1, Solid *s2);

/*
 * return a bounding box for solid 's'
 * the bounding box is aligned with the world reference
 */
void getSolidBB(Solid *s, V3 *center, V3 *size);

/*
 * update the solid position
 * the position is defined by the matrix 4x4 'mpos'
 * the position matrix gives the transformation to do on local coords
 * towards world coords
 */ 
void setSolidPosition(Solid *s, M4 *mpos);

/* change the "frame" of the solid (ie its geometry) */
void setSolidFrame(Solid *s, u_int16 frame) ;

/* give solid's position */
void getSolidPosition(Solid *s, M4 *mpos);

/*
 * set the solid 's' visible (flag=True) or invisible (flag=False)
 * by default solids are visible
 */
void setSolidVisible(Solid *s, boolean flag);
void setSolidBBFlag(Solid *s, boolean flag);

/*
 * create a new solid.
 * string 'geometry' gives the solid's geometry
 * return the solid's handle.
 */
Solid * parseGeometry(const char *geometry);

/* delete solid 's' from the scene */
void deleteSolidFromList(Solid *s);

void SolidClose(void);

float * getGLMatrix(M4 *);

/* quality = 0: normal 3D quality, quality = 1: high 3D quality */
void RenderInit(boolean);

/* display the current scene in the current buffer */
void Render(void);

void renderMaterials(void);

/* close 3d display */
void RenderClose(void);

/* matrix given the camera position in the world
 * specifie la transformation orthogonale a effectuer des coordonnes
 * mondes vers les coordonees ecran. Dans les coordonnees ecran, l'axe
 * z est normal a l'ecran et dirige vers l'observateur et l'axe x est
 * dirige de la gauche vers la droite. Le repere est direct.
 * not fully implemented
 */
void setCameraPosition(M4 *);

/*
 * projection definition
 * fovy: largeur de champ (in degree)
 * near: distance du plan de projection  l'oeil
 * far : distance  l'oeil du plan de clipping arrire
 */
void setCameraProjection(float fovy, float near, float far);
void setCameraPerspective(float fovy, float near, float far, float ratio);

/*
 * parameters given the position of the display buffer in the display window
 * (at address 'buf'). 
 * note: 'xsize' and 'x0' must be even
void ZLib_setFrameBuffer(char *buf, int x0, int y0, int xsize, int ysize);
 */

/*
 * set sizes of the display buffer
 * note: 'xsize' must be multiple of 16.
 */
void RenderSetWindow(int xsize, int ysize);

/*
 * return the solid's handle displayed in (x,y) on the screen
 * we use here coordonninates IN THE ZBUFFER and no in the display window
 */
Solid * RenderGetSelection(int x, int y);

/*
 * Pour aider dans le traitement des clicks.
 */
void clickToVector(int, int, V3 *);
void M4_to_GL(float *gl_mat, M4 *vr_mat);

extern int count_triangles;
extern int count_triangles_textured;
extern int count_pixels;


#endif  /* ZV_H */
