#ifndef VRENGD

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "list.h"
#include "grid.h"
#include "ball.h"	/* BALL_TYPE */
#include "dart.h"	/* DART_TYPE */
#include "bullet.h"	/* BULLET_TYPE */
#include "move.h"	/* GRAVITY */
#include "walls.h"	/* wallsIntersectObject */
#include "user.h"

#include "zv.h"		/* parseGeometry */
#include "gui.h"	/* GuiAddUser */
#include "payload.h"	/* seekPayload */
#include "netobj.h"	/* getMySsrc */
#include "helpers.h"

#include "col.h"	/* COL_GHOST collision behavior */


const WClass User::wclass(USER_TYPE, "User", NULL, User::replicator);


void User::allocLocal()
{
  if ((mensuration = (char *) calloc(1, USER_MENSURATIONLEN)) == NULL)
    fatal("Can't alloc mensuration");
  if ((urlfront = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc urlfront");
  if ((urlback = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc urlback");
  if ((urlleft = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc urlleft");
  if ((urlright = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc urlright");
  if ((message = (char *) calloc(1, MESS_LEN)) == NULL)
    fatal("Can't alloc message");
  if ((vre = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc vre");
  if ((web = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc web");
  if ((model = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc model");
  if ((face = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc face");
  if ((color = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc color");
  if ((baps = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc baps");
  if ((host = (char *) calloc(1, URL_LEN)) == NULL)
    fatal("Can't alloc host");
  if ((cname = (char *) calloc(1, CNAME_LEN)) == NULL)
    fatal("Can't alloc cname");
  if ((rtcpname = (char *) calloc(1, NAME_LEN)) == NULL)
    fatal("Can't alloc rtcpname");
  if ((email = (char *) calloc(1, EMAIL_LEN)) == NULL)
    fatal("Can't alloc email");
  if ((tool = (char *) calloc(1, TOOL_LEN)) == NULL)
    fatal("Can't alloc tool");
}

/* initialize localuser */
User *
User::createLocal(const char *localusername, const char *front, const char *back)
{
  return new User(localusername, front, back);
}

User::User(const char *localusername, const char *front, const char *back)
{
  trace(DBG_WMGT, "localusername: %s", localusername);
  allocLocal();
  clearKeyTab(this);

  width = USER_DEFAULTWIDTH / 2;
  depth = USER_DEFAULTDEPTH / 2;
  height = USER_DEFAULTHEIGHT / 2;

  strcpy(name.class_name, localusername);	/* a revoir: USER_NAME */
  name.instance_name = name.class_name;	/* temporally */
  name.world_name = getCurrentWorldName();
  ssrc = getMySsrc();
  strcpy(rtcpname, AvatarName());
  strcpy(email, RtcpEmail());
  strcpy(tool, ToolName());
  trace(DBG_WMGT, "createLocal: name=%s ssrc=%x rtcpname=%s email=%s", name.instance_name, ssrc, rtcpname, email);
  entry = 0;

#ifdef RANDOM_ORIGIN
  /* ramdomize x and y */
  pos.x = (float) drand48() * 2 -1;
  pos.y = (float) drand48() * 2 -1;
  pos.az = (float) drand48() * M_2PI;
#else
  pos.x = 0;
  pos.y = 0;
  pos.az = 0;
#endif
 
  /* fill mapping fields */
  if (my_widthstr != NULL)
    width = atof(my_widthstr) / 2;
  if (my_depthstr != NULL)
    depth = atof(my_depthstr) / 2;
  if (my_heightstr != NULL)
    height = atof(my_heightstr) / 2;

  if (my_mapfrontstr != NULL)
    strncpy((char *) urlfront, my_mapfrontstr, strlen(my_mapfrontstr));
  else
    strncpy((char *) urlfront, front, strlen(front));
  if (my_mapbackstr != NULL)
    strncpy((char *) urlback, my_mapbackstr, strlen(my_mapbackstr));
  else
    strncpy((char *) urlback, back, strlen(back));
  if (my_mapleftstr != NULL)
    strncpy(urlleft, my_mapleftstr, strlen(my_mapleftstr));
  if (my_maprightstr != NULL)
    strncpy(urlright, my_maprightstr, strlen(my_maprightstr));
  if (my_vrestr != NULL)
    strncpy(vre, my_vrestr, strlen(my_vrestr));
  if (my_hoststr != NULL)
    strncpy(host, my_hoststr, strlen(my_hoststr));
  if (my_webstr != NULL)
    strncpy(web, my_webstr, strlen(my_webstr));
  if (my_modelstr != NULL)
    strncpy(model, my_modelstr, strlen(my_modelstr));
  if (my_facestr != NULL)
    strncpy(face, my_facestr, strlen(my_facestr));
  if (my_colorstr != NULL)
    strncpy(color, my_colorstr, strlen(my_colorstr));
  if (my_bapsstr != NULL)
    strncpy(baps, my_bapsstr, strlen(my_bapsstr));

  char geom[BUFSIZ];
  if (model) {
    sprintf(mensuration, "bbox,size=%.2f,%.2f,%.2f", depth, width, height);
    sprintf(geom, "%s", mensuration);
  }
  else {
    sprintf(mensuration, "box,size=%.2f,%.2f,%.2f",
            /* BUG! WARNING! invert x and y: I don't know why !!! */
            depth, width, height);
    sprintf(geom, "%s,tex_xp=%s,tex_xn=%s,tex_yp=%s,tex_yn=%s",
            mensuration, urlfront, urlback, urlleft, urlright);
  }
  trace(DBG_WMGT, "geom=%s", geom);
  soh = parseGeometry(geom);

  noh.type = USER_TYPE;
  updateObjectIn3D(this);
  updateBB(this);
  insertObjectIntoGrid(this);
  mobilelist = addObjectToList(this, mobilelist);
  //DAX initializeObject(this, USER_TYPE, VR_MOBILE);
  setSolidVisible(soh, FALSE);	/* not visible by myself */
  setObjectToSolid(this);

  setCameraProjection(USER_FOVY, USER_NEAR, USER_FAR); /* view from user */

  pos.z = height + 0.15; /* jump to fall down on the floor */
  lspeed = USER_LSPEED;
  aspeed = USER_ASPEED;
  move.perm_sec = -1;

  createNetObject((NetObject *) this, VR_VOLATILE);
#if 0
  declareObjCreation(&(this->noh)); /* we don't need because delta */
#endif

  ptrgui = GuiAddUser(this);
  return;
} 

/* create an user from the network */
WObject *
User::replicator(u_int8 type_id, struct _NetObjectId noid, Payload *pp)
{
  return new User(type_id, noid, pp);
}

User::User(u_int8 type_id, struct _NetObjectId noid, Payload *pp)
{
  noh.type = type_id;
  noh.noid = noid;

  allocLocal();

  /* hack to retrieve the name and the mapping */ 
  setProperty(&(this->noh), /*  0 */ USER_PROPHNAME, pp);
  setProperty(&(this->noh), /*  1 */ USER_PROPMAPFRONT, pp);
  setProperty(&(this->noh), /*  2 */ USER_PROPMAPBACK, pp);

  int idxgeom, idxvar, idxend = 0;
  char geometry[BUFSIZ];

  idxvar = tellPayload(pp);	/* note begin of var */
  trace(DBG_WMGT, "type = %d", type_id);
  trace(DBG_WMGT, "idxvar = %d", idxvar);
  if ((idxgeom = tellStrInPayload(pp, "box,size=")) > 0) {
    /* get replicated user characteristics from the network */
    trace(DBG_WMGT, "idxgeom = %d", idxgeom);
    setProperty(&(this->noh), /* 09 */ USER_PROPMENSURATION, pp);
    setProperty(&(this->noh), /* 10 */ USER_PROPMAPLEFT, pp);
    setProperty(&(this->noh), /* 11 */ USER_PROPMAPRIGHT, pp);
    setProperty(&(this->noh), /* 12 */ USER_PROPVRE, pp);
    setProperty(&(this->noh), /* 13 */ USER_PROPWEB, pp);
    setProperty(&(this->noh), /* 14 */ USER_PROPSSRC, pp);
    setProperty(&(this->noh), /* 15 */ USER_PROPRTCPNAME, pp);
    setProperty(&(this->noh), /* 16 */ USER_PROPRTCPEMAIL, pp);
    setProperty(&(this->noh), /* 17 */ USER_PROPMODEL, pp);
    setProperty(&(this->noh), /* 18 */ USER_PROPFACE, pp);
    setProperty(&(this->noh), /* 19 */ USER_PROPCOLOR, pp);
    idxend = tellPayload(pp); /* note end of properties */

    if (model) {
      sprintf(geometry, "%s", mensuration);
    }
    else {
      sprintf(geometry, "%s,tex_xp=%s,tex_xn=%s,tex_yp=%s,tex_yn=%s",
              mensuration, urlfront, urlback, urlleft, urlright);
    }
    /* dumpPayload(stdout, pp); */
  }
  else {
    /* builtin user geometry */
    sprintf(geometry, "box,size=%.2f,%.2f,%.2f,tex_xp=%s,tex_xn=%s",
            USER_DEFAULTWIDTH/2, USER_DEFAULTDEPTH/2, USER_DEFAULTHEIGHT/2,
            urlfront, urlback);
    trace(DBG_FORCE, "Replica user: builtin geometry");
  }

  /*
   * solid creation
   */
  nature.movable = VR_ELEM_MOVE;
  nature.selectable = VR_SELECTABLE;
  nature.bbable = VR_BB;
  nature.renderable = VR_NORMAL_RENDER;

  /* get the variable properties */
  if (idxend > 0) {
    trace(DBG_WMGT, "Replica user: read var props, idxend=%d", idxend);
    seekPayload(pp, idxvar); /* begin prop var */
    for (int np = USER_PROPBEGINVAR; np <= USER_PROPENDVAR; np++)
      setProperty(&(this->noh), np, pp);
    seekPayload(pp, idxend); /* end properties */
  }
  else {
    trace(DBG_WMGT, "Replica user: read all props");
    for (int np = USER_PROPBEGINVAR; np < propertiesnumber[noh.type]; np++)
      setProperty(&(this->noh), np, pp);
  }
  trace(DBG_FORCE, "Replica: ssrc=%x rctpname=%s", ssrc, rtcpname);
  trace(DBG_WMGT, "Replica: web=%s vre=%s", web, vre);
  trace(DBG_WMGT, "Replica: model=%s", model);
  trace(DBG_WMGT, "Replica: face=%s", face);
  trace(DBG_WMGT, "Replica: color=%s", color);
#if 0
  if (getRtcpNameBySsrc(ssrc)) {
    trace(DBG_FORCE, "Replica: ssrc=%s", getRtcpNameBySsrc(ssrc));
  }
#endif

  trace(DBG_WMGT, "Replica user: %s", geometry);
  soh = parseGeometry(geometry);

  initBB(this->pos);
  updateObjectIn3D(this);
  setObjectToSolid(this);
  updateBB(this);
  mobilelist = addObjectToList(this, mobilelist);
  //PD placeObjectInGrid(this);
  insertObjectIntoGrid(this);

  clearKeyTab(this);
  lastmess = 0;
  lspeed = USER_EXTLSPEED;	/* 10 */
  aspeed = USER_EXTASPEED;	/* 5.0 */
  move.ttl = 0.0;
  move.perm_sec = -1;
  name.instance_name = name.class_name;	/* a revoir */
  name.world_name = getCurrentWorldName();
  //rtcpname = getRtcpNameBySsrc(ssrc);
  //email = getRtcpEmailBySsrc(ssrc);
  tool = getRtcpToolBySsrc(ssrc);
  if (tool)
    trace(DBG_FORCE, "Replica user: tool=%s", tool);

  ptrgui = GuiAddUser(this);

  /* network ACK = sending fictive Delta (Ph. Dax - 1998) */
  /* Prevoir implosion des reponses -> random timer */
  //if (strcmp(worlds->name, worlds->plocaluser->vre) == 0) {
  //  trace(DBG_FORCE, "netack: %s/%s", worlds->name, worlds->plocaluser->vre);
  //  declareObjDelta(&(worlds->plocaluser->noh), USER_PROPXY);
  //  declareObjDelta(&(worlds->plocaluser->noh), USER_PROPZ);
  //  declareObjDelta(&(worlds->plocaluser->noh), USER_PROPAZ);
  //}
} 

/* destroy local user */
User::~User()
{
  trace(DBG_WMGT, "User %s quits", worlds->plocaluser->name.instance_name);
  GuiRemoveUser(this);

  // MS. if this destructor is called for a remote user,
  // we should not declare the deletion: it's not our problem.
  if (this == worlds->plocaluser)
    declareObjDeletion(&noh);

  deleteNetObject(&noh);
  // MS. prevent double delete of the solid
  if (soh)
    deleteSolidFromList(soh);

  free(mensuration);
  free(urlfront);
  free(urlback);
  free(urlleft);
  free(urlright);
  free(message);
  free(vre);
  free(web);
  free(host);
  free(model);
  free(face);
  free(color);
  free(cname);
  free(rtcpname);
  free(email);
  free(tool);
}

/* update an user on the network */
boolean User::updateToNetwork(const Pos &oldpos)
{
  boolean change = FALSE;
  
  if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
    declareObjDelta(&(this->noh), USER_PROPXY); change = TRUE;
  }
  if (ABSF(pos.z - oldpos.z) > USER_DELTAZ) { /* if d < 2cm => not sent */
    declareObjDelta(&(this->noh), USER_PROPZ); change = TRUE;
  }
  if (pos.az != oldpos.az) {
    declareObjDelta(&(this->noh), USER_PROPAZ); change = TRUE;
  }
  if (pos.ay != oldpos.ay) {
    declareObjDelta(&(this->noh), USER_PROPAY); change = TRUE;
  }
  if (pos.ax != oldpos.ax) {
    declareObjDelta(&(this->noh), USER_PROPAX); change = TRUE;
  }
  return change;
}

/* update times array */
void User::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = (float) (sec - move.sec) + (float) (usec - move.usec) / 1e6;
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    move.ttl = 0;
    move.sec = 0;
    move.usec = 0;
  }
}

static boolean pause_gravity = FALSE;

/* equations system handling permanent motions */
void User::changePermanent(float lasting)
{
  if (pause_gravity == FALSE)
    pos.z -= lasting * GRAVITY;
#if 0
  if (getRtcpNameBySsrc(ssrc)) {
    trace(DBG_FORCE, "Replica: getRtcpNameBySsrc = %s",
                     getRtcpNameBySsrc(ssrc));
  }
#endif
}

/* equations system handling imposed motions */
void User::changePosition(float lasting)
{
  pos.x += lasting * move.lspeed.v[0];
  pos.y += lasting * move.lspeed.v[1];
  pos.z += lasting * move.lspeed.v[2];
  pos.az += lasting * move.aspeed.v[0];
  pos.ay += lasting * move.aspeed.v[1];
  pos.ax += lasting * move.aspeed.v[2];
}

/* condition to do position modifications */
boolean User::change()
{
  return (move.ttl > 0.0005) ? TRUE : FALSE;
}

void bulletGetHit(Bullet *pbullet, Payload *pp)
{
  putPayload(pp, "c", pbullet->hit);
#if 0
  if (pbullet->hit == 1) {
    trace(DBG_FORCE, "bulletGetHit:");
    /* declareObjDelta(&(pbullet->noh), BULLETPROPHIT); */
    putPayload(pp, "c", pbullet->hit);
    dumpPayload(stdout, pp);
  }
#endif
}

void dartGetHit(Dart *pdart, Payload *pp)
{
  putPayload(pp, "c", pdart->hit);
#if 0
  if (pdart->hit == 1) {
    trace(DBG_FORCE, "dartGetHit:");
    /* declareObjDelta(&(pdart->noh), DARTPROPHIT); */
    putPayload(pp, "c", pdart->hit);
    dumpPayload(stdout, pp);
  }
#endif
}

void User::whenIntersect(WObject *pcur, WObject *pold)
{
  notice("%s:%s intersect %s",
         pcur->name.class_name, pcur->name.instance_name, name.instance_name);

  // User has no control over ghost objects
  if (pcur->nature.collision == COL_GHOST)
    return;

  switch (pcur->noh.type) {
  case BULLET_TYPE:
  case DART_TYPE:
    /* projectile intersects a stopped user: hit */
    if (hit == 0) {
      hit = 1;
#ifndef BULLDART        // TO RESOLVE
      if (pcur->noh.type == DART_TYPE) {
        ((Dart *)pcur)->hit = 1;
        sendObjDelta(&(pcur->noh), DART_PROPHIT);
        ((Dart *)pcur)->hit = 0;
      } else if (pcur->noh.type == DART_TYPE) {
        ((Bullet *)pcur)->hit = 1;
        sendObjDelta(&(pcur->noh), BULLET_PROPHIT);
        ((Bullet *)pcur)->hit = 0;
      }
#else                   // TO RESOLVE
      Payload payload, *pp = &payload;

      notice("%s:%s hits %s",
             pcur->name.class_name, pcur->name.instance_name, name.instance_name);
      hit = 1;
      resetPayload(pp);
      if (pcur->noh.type == BULLET_TYPE) {
        pcur->hit = 1;
        bulletGetHit((Bullet *) pcur, pp);
        pcur->hit = 0;
      }
      if (pcur->noh.type == DART_TYPE) {
        pcur->hit = 1;
        dartGetHit((Dart *) pcur, pp);
        pcur->hit = 0;
      }
#endif
    }
    copyPositionAndBB(pold, pcur);
    objectToDelete(pcur);	/* delete projectile */
    break;
  case BALL_TYPE:
    pcur->pos.x += BALL_SHIFTX; /* ball shifts */
    pcur->pos.y += BALL_SHIFTY; /* ball shifts */
    updateReplica(pcur, pold->pos);
    break;
  default:
    /* user intersects an other user: slide */
    pcur->pos.x += USER_GOTHROUGH; /* step to shift */
    pcur->pos.y += USER_GOTHROUGH; /* step to shift */
    updateObjectIn3D(pcur);
    updateBB(pcur);
  }
}

void User::whenWallIntersect(WObject *puold, V3 *norm)
{
  float nx, ny;
  float ox, oy, cx, cy;

  nx = norm->v[0];
  ny = norm->v[1];
  ox = pos.x;
  oy = pos.y;
  cx = pos.x;
  cy = pos.y;

  if ((ox != cx) || (oy != cy)) {
    float nn = nx*nx + ny*ny;

    trace(DBG_WMGT, "(ox=%.2f oy=%.2f) (cx=%.2f cy=%.2f)", ox, oy, cx, cy);
    pos.y = ((nx*ny) * (ox-cx) + ny*ny*oy + nx*nx*cy) / nn;
    pos.x = ((nx*ny) * (oy-cy) + nx*nx*ox + ny*ny*cx) / nn;

    /* Changed user position => need to update BBs */
    updateObjectIn3D(this);
    updateBB(this);
  }
  if (wallsIntersectObject(&(this->pos.bbcenter), &(this->pos.bbsize), norm))
    copyPositionAndBB(puold, this);
}

static
void userCreateBullet(User *pu, void *data, time_t sec, time_t usec)
{
  if (generalMethodList[BULLET_CREAT][BULLET_TYPE].method != NULL)
    generalMethodList[BULLET_CREAT][BULLET_TYPE].method(pu, data, sec, usec);
}

static
void userCreateDart(User *pu, void *data, time_t sec, time_t usec)
{
  if (generalMethodList[DART_CREAT][DART_TYPE].method != NULL)
    generalMethodList[DART_CREAT][DART_TYPE].method(pu, data, sec, usec);
}

static float fovy = USER_FOVY;
static float near = USER_NEAR;

static
void userDefaultZoom(User *pu, void *data, time_t sec, time_t usec)
{
  fovy = USER_FOVY;
  near = USER_NEAR;
  setCameraProjection(fovy, near, USER_FAR);
}

static
void userIncreaseZoom(User *pu, void *data, time_t sec, time_t usec)
{
  fovy -= 1.0;
  if (fovy <= 1.0) fovy = 1.0;
#if 0
  near += 0.025;
  if (near >= 1.0) near = 1.0;
#endif
  setCameraProjection(fovy, near, USER_FAR);
}

static
void userDecreaseZoom(User *pu, void *data, time_t sec, time_t usec)
{
  fovy += 1.0;
  if (fovy >= 135.0) fovy = 135.0;
#if 0
  near -= 0.025;
  if (near <= 0.0) near = 0.0;
#endif
  setCameraProjection(fovy, near, USER_FAR);
}

static
void userDefaultLinearSpeed(User *pu, void *data, time_t sec, time_t usec)
{
  pu->lspeed = USER_LSPEED;
}

static
void userDecreaseLinearSpeed(User *pu, void *data, time_t sec, time_t usec)
{
  if (pu->lspeed > 1.0)
    pu->lspeed -= 1.0;
}

static
void userIncreaseLinearSpeed(User *pu, void *data, time_t sec, time_t usec)
{
  pu->lspeed += 1.0;
}

static
void userSetLspeed(User *pu, void *data, time_t sec, time_t usec)
{
  float *lspeed = (float *) data;
  pu->lspeed = *lspeed;
}

static
void userDefaultAngularSpeed(User *pu, void *data, time_t sec, time_t usec)
{
  pu->aspeed = USER_ASPEED;
}

static
void userDecreaseAngularSpeed(User *pu, void *data, time_t sec, time_t usec)
{
  if (pu->aspeed > 0.3)
    pu->aspeed -= 0.2;
}

static
void userIncreaseAngularSpeed(User *pu, void *data, time_t sec, time_t usec)
{
  pu->aspeed += 0.2;
}

static
void userSetAspeed(User *pu, void *data, time_t sec, time_t usec)
{
  float *aspeed = (float *) data;
  pu->aspeed = *aspeed;
}

static
void userPause(User *pu, void *data, time_t sec, time_t usec)
{
  pause_gravity ^= 1;
}

static
void userPauseOn(User *pu, void *data, time_t sec, time_t usec)
{
  pause_gravity = 1;
}

static
void userPauseOff(User *pu, void *data, time_t sec, time_t usec)
{
  pause_gravity = 0;
}

void u_set_xy(WObject *po, Payload *pp)
{
  //PD testsolidobject("u_set_xy begin", po);	//OK
  set_xy(po, pp);
  //PD testsolidobject("u_set_xy end", po);		//NULL

  if (po == worlds->plocaluser) {
    notice("Something's pushing me!");
    updateCameraFromObject(po);
  }
}

void u_set_z(WObject *po, Payload *pp)
{
  set_z(po, pp);
  if (po == worlds->plocaluser) {
    notice("Something's lifting me!");
    updateCameraFromObject(po);
  }
}

void u_set_ax(WObject *po, Payload *pp)
{
  set_ax(po, pp);
  if (po == worlds->plocaluser) {
    notice("Something's twisting me!");
    updateCameraFromObject(po);
  }
}

void u_set_ay(WObject *po, Payload *pp)
{
  set_ay(po, pp);
  if (po == worlds->plocaluser) {
    notice("Something's twisting me!");
    updateCameraFromObject(po);
  }
}

void u_set_az(WObject *po, Payload *pp)
{
  set_az(po, pp);
  if (po == worlds->plocaluser) {
    notice("Something's twisting me!");
    updateCameraFromObject(po);
  }
}

void set_msg(User *pu, Payload *pp)
{
  int lastmess;
  char msg[MESS_LEN];

  if (!pu) {
    trace(DBG_FORCE, "set_msg: pu NULL");
    return;
  }
  getPayload(pp, "ds", &lastmess, msg);
  if (lastmess != pu->lastmess) {
    GuiWriteMessage("chat", pu->name.instance_name, msg);
    pu->lastmess = lastmess;
  }
}

void set_infos(WObject *po, Payload *pp)
{
  if (!po) {
    trace(DBG_FORCE, "set_infos: po NULL");
    return;
  }
  getPayload(pp, "s", po->name.infos);
  trace(DBG_WMGT, "set_infos: %s", po->name.infos);
}

void set_mensuration(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_mensuration: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->mensuration);
  trace(DBG_WMGT, "set_mensuration: %s", pu->mensuration);
}

void set_mapfront(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_mapfront: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->urlfront);
  trace(DBG_WMGT, "set_urlfront: %s", pu->urlfront);
}

void set_mapback(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_mapback: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->urlback);
  trace(DBG_WMGT, "set_urlback: %s", pu->urlback);
}

void set_mapleft(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_mapleft: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->urlleft);
  trace(DBG_WMGT, "set_urlleft: %s", pu->urlleft);
}

void set_mapright(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_mapright: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->urlright);
  trace(DBG_WMGT, "set_urlright: %s", pu->urlright);
}

void set_vre(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_vre: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->vre);
  trace(DBG_WMGT, "set_vre: %s", pu->vre);
}

void set_web(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_web: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->web);
  if (pu->web)
    trace(DBG_WMGT, "set_web: %s", pu->web);
}

void set_model(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_model: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->model);
  if (pu->model)
    trace(DBG_WMGT, "set_model: %s", pu->model);
}

void set_face(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_face: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->face);
  if (pu->face)
    trace(DBG_WMGT, "set_face: %s", pu->face);
}

void set_color(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_color: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->color);
  if (pu->color)
    trace(DBG_WMGT, "set_color: %s", pu->color);
}

void set_ssrc(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_ssrc: pu NULL");
    return;
  }
  getPayload(pp, "d", &(pu->ssrc));
  trace(DBG_WMGT, "set_ssrc: %x", pu->ssrc);
}

void set_rtcpname(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_rtcpname: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->rtcpname);
  if (pu->rtcpname)
    trace(DBG_WMGT, "set_rtcpname: %s", pu->rtcpname);
}

void set_rtcpemail(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "set_rtcpemail: pu NULL");
    return;
  }
  getPayload(pp, "s", pu->email);
  if (pu->email)
    trace(DBG_WMGT, "set_rtcpemail: %s", pu->email);
}

void get_msg(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_msg: pu NULL");
    return;
  }
  putPayload(pp, "ds", pu->lastmess, pu->message);
}

void get_infos(WObject *po, Payload *pp)
{
  if (!po) {
    trace(DBG_FORCE, "infos: po NULL");
    return;
  }
  putPayload(pp, "s", po->name.infos);
  trace(DBG_WMGT, "infos: %s", po->name.infos);
}

void get_mensuration(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_mensuration: pu NULL");
    return;
  }
  putPayload(pp, "s", pu->mensuration);
  trace(DBG_WMGT, "get_mensuration: %s", pu->mensuration);
}

void get_mapfront(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_mapfront: pu NULL");
    return;
  }
  putPayload(pp, "s", pu->urlfront);
  trace(DBG_WMGT, "get_mapfront: %s", pu->urlfront);
}

void get_mapback(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_mapback: pu NULL");
    return;
  }
  putPayload(pp, "s", pu->urlback);
  trace(DBG_WMGT, "get_mapback: %s", pu->urlback);
}

void get_mapleft(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_mapleft: pu NULL");
    return;
  }
  if (pu->urlleft) {
    putPayload(pp, "s", pu->urlleft);
    trace(DBG_WMGT, "get_mapleft: %s", pu->urlleft);
  }
}

void get_mapright(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_mapright: pu NULL");
    return;
  }
  if (pu->urlright) {
    putPayload(pp, "s", pu->urlright);
    trace(DBG_WMGT, "get_mapright: %s", pu->urlright);
  }
}

void get_vre(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_vre: pu NULL");
    return;
  }
  if (pu->vre) {
    putPayload(pp, "s", pu->vre);
    trace(DBG_WMGT, "get_vre: %s", pu->vre);
  }
}

void get_web(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_web: pu NULL");
    return;
  }
  if (pu->web) {
    putPayload(pp, "s", pu->web);
    trace(DBG_WMGT, "get_web: %s", pu->web);
  }
}

void get_model(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_model: pu NULL");
    return;
  }
  if (pu->model) {
    putPayload(pp, "s", pu->model);
    trace(DBG_WMGT, "get_model: %s", pu->model);
  }
}

void get_face(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_face: pu NULL");
    return;
  }
  if (pu->face) {
    putPayload(pp, "s", pu->face);
    trace(DBG_WMGT, "get_face: %s", pu->face);
  }
}

void get_color(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_color: pu NULL");
    return;
  }
  if (pu->color) {
    putPayload(pp, "s", pu->color);
    trace(DBG_WMGT, "get_color: %s", pu->color);
  }
}

void get_ssrc(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_ssrc: pu NULL");
    return;
  }
  putPayload(pp, "d", pu->ssrc);
  trace(DBG_WMGT, "get_ssrc: %x", pu->ssrc);
}

void get_rtcpname(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_rtcpname: pu NULL");
    return;
  }
  if (pu->rtcpname) {
    putPayload(pp, "s", pu->rtcpname);
    trace(DBG_WMGT, "get_rtcpname: %s", pu->rtcpname);
  }
}

void get_rtcpemail(User *pu, Payload *pp)
{
  if (!pu) {
    trace(DBG_FORCE, "get_rtcpemail: pu NULL");
    return;
  }
  if (pu->email) {
    putPayload(pp, "s", pu->email);
    trace(DBG_WMGT, "get_rtcpemail: %s", pu->email);
  }
}

/* user functions initialization */
void userInitFuncList(void)
{
  propertiesnumber[USER_TYPE] = USER_PROPS;
  maxlasting[USER_TYPE] = USER_LASTING;	/* 15 ms */

  setFuncList[USER_PROPHNAME][USER_TYPE].pf = WO_PAYLOAD set_hname;
  setFuncList[USER_PROPMENSURATION][USER_TYPE].pf = WO_PAYLOAD set_mensuration;
  setFuncList[USER_PROPMAPFRONT][USER_TYPE].pf = WO_PAYLOAD set_mapfront;
  setFuncList[USER_PROPMAPBACK][USER_TYPE].pf = WO_PAYLOAD set_mapback;
  setFuncList[USER_PROPMAPLEFT][USER_TYPE].pf = WO_PAYLOAD set_mapleft;
  setFuncList[USER_PROPMAPRIGHT][USER_TYPE].pf = WO_PAYLOAD set_mapright;
  setFuncList[USER_PROPXY][USER_TYPE].pf = WO_PAYLOAD u_set_xy;
  setFuncList[USER_PROPZ][USER_TYPE].pf = WO_PAYLOAD u_set_z;
  setFuncList[USER_PROPAZ][USER_TYPE].pf = WO_PAYLOAD u_set_az;
  setFuncList[USER_PROPAY][USER_TYPE].pf = WO_PAYLOAD u_set_ay;
  setFuncList[USER_PROPAX][USER_TYPE].pf = WO_PAYLOAD u_set_ax;
  setFuncList[USER_PROPMSG][USER_TYPE].pf = WO_PAYLOAD set_msg;
  setFuncList[USER_PROPVRE][USER_TYPE].pf = WO_PAYLOAD set_vre;
  setFuncList[USER_PROPWEB][USER_TYPE].pf = WO_PAYLOAD set_web;
  setFuncList[USER_PROPSSRC][USER_TYPE].pf = WO_PAYLOAD set_ssrc;
  setFuncList[USER_PROPRTCPNAME][USER_TYPE].pf = WO_PAYLOAD set_rtcpname;
  setFuncList[USER_PROPRTCPEMAIL][USER_TYPE].pf = WO_PAYLOAD set_rtcpemail;
  setFuncList[USER_PROPMODEL][USER_TYPE].pf = WO_PAYLOAD set_model;
  setFuncList[USER_PROPFACE][USER_TYPE].pf = WO_PAYLOAD set_face;
  setFuncList[USER_PROPCOLOR][USER_TYPE].pf = WO_PAYLOAD set_color;

  getFuncList[USER_PROPHNAME][USER_TYPE].pf = WO_PAYLOAD get_hname;
  getFuncList[USER_PROPMENSURATION][USER_TYPE].pf = WO_PAYLOAD get_mensuration;
  getFuncList[USER_PROPMAPFRONT][USER_TYPE].pf = WO_PAYLOAD get_mapfront;
  getFuncList[USER_PROPMAPBACK][USER_TYPE].pf = WO_PAYLOAD get_mapback;
  getFuncList[USER_PROPMAPLEFT][USER_TYPE].pf = WO_PAYLOAD get_mapleft;
  getFuncList[USER_PROPMAPRIGHT][USER_TYPE].pf = WO_PAYLOAD get_mapright;
  getFuncList[USER_PROPXY][USER_TYPE].pf = WO_PAYLOAD get_xy;
  getFuncList[USER_PROPZ][USER_TYPE].pf = WO_PAYLOAD get_z;
  getFuncList[USER_PROPAZ][USER_TYPE].pf = WO_PAYLOAD get_az;
  getFuncList[USER_PROPAY][USER_TYPE].pf = WO_PAYLOAD get_ay;
  getFuncList[USER_PROPAX][USER_TYPE].pf = WO_PAYLOAD get_ax;
  getFuncList[USER_PROPMSG][USER_TYPE].pf = WO_PAYLOAD get_msg;
  getFuncList[USER_PROPVRE][USER_TYPE].pf = WO_PAYLOAD get_vre;
  getFuncList[USER_PROPWEB][USER_TYPE].pf = WO_PAYLOAD get_web;
  getFuncList[USER_PROPSSRC][USER_TYPE].pf = WO_PAYLOAD get_ssrc;
  getFuncList[USER_PROPRTCPNAME][USER_TYPE].pf = WO_PAYLOAD get_rtcpname;
  getFuncList[USER_PROPRTCPEMAIL][USER_TYPE].pf = WO_PAYLOAD get_rtcpemail;
  getFuncList[USER_PROPMODEL][USER_TYPE].pf = WO_PAYLOAD get_model;
  getFuncList[USER_PROPFACE][USER_TYPE].pf = WO_PAYLOAD get_face;
  getFuncList[USER_PROPCOLOR][USER_TYPE].pf = WO_PAYLOAD get_color;

  setMethodFunc(USER_TYPE, BULLETUSER, WO_ACTION userCreateBullet, "");
  setMethodFunc(USER_TYPE, DARTUSER, WO_ACTION userCreateDart, "");
  setMethodFunc(USER_TYPE, FOVYORIGINAL, WO_ACTION userDefaultZoom, "");
  setMethodFunc(USER_TYPE, FOVYLESS, WO_ACTION userIncreaseZoom, "");
  setMethodFunc(USER_TYPE, FOVYGREATER, WO_ACTION userDecreaseZoom, "");
  setMethodFunc(USER_TYPE, LSPEEDORIGINAL, WO_ACTION userDefaultLinearSpeed, "");
  setMethodFunc(USER_TYPE, LSPEEDLESS, WO_ACTION userDecreaseLinearSpeed, "");
  setMethodFunc(USER_TYPE, LSPEEDGREATER, WO_ACTION userIncreaseLinearSpeed, "");
  setMethodFunc(USER_TYPE, ASPEEDORIGINAL, WO_ACTION userDefaultAngularSpeed, "");
  setMethodFunc(USER_TYPE, ASPEEDLESS, WO_ACTION userDecreaseAngularSpeed, "");
  setMethodFunc(USER_TYPE, ASPEEDGREATER, WO_ACTION userIncreaseAngularSpeed, "");
  setMethodFunc(USER_TYPE, USERPAUSE, WO_ACTION userPause, "");
  setMethodFunc(USER_TYPE, USERPAUSEON, WO_ACTION userPauseOn, "");
  setMethodFunc(USER_TYPE, USERPAUSEOFF, WO_ACTION userPauseOff, "");
  setMethodFunc(USER_TYPE, USERSETLSPEED, WO_ACTION userSetLspeed, "");
  setMethodFunc(USER_TYPE, USERSETASPEED, WO_ACTION userSetAspeed, "");
}

#endif /* !VRENGD */
