#include <GL/gl.h>

#include "global.h"

// Payload/ssrc
#include "net.h"
#include "netImpl.h"
#include "netobj.h"

// User object
#include "wobject.h"
#include "wmgt.h"

// parse* functions
#include "parse.h"

// Query for users
#include "list.h"
#include "grid.h"

// Colision behavior constants
#include "col.h"

// Java vm and classpath settings
#include "env.h"

#include "user.h"
#include "ifc.h"
#include "ifcserver.h"

#include "zv.h"		// parseGeometry
#include "defaults.h"	// DEF_HTTP_SERVER



// forward prototypes
void ifcDraw2DList(Ifc *);
void ifcGetSurfVecs(Ifc *, V3 *, V3 *, V3 *);
float ifcCalcVar(V3, V3, V3, V3);
float ifcCalcDet(V3, V3, V3);
void ifcDoSetPos(WObject *po, Ifc *pifc);
void ifcDoDeltaPos(WObject *po, Ifc *pifc);
void ifcDeclarePushDeltas(WObject *po, Ifc *pifc);

const WClass Ifc::wclass(IFC_TYPE, "Ifc", Ifc::creator);


// *******************************************************************
//                    VRENG functions
// *******************************************************************

/*
 * Ifc parser.
 * Configuration line structure:
 * - (name)
 * - position (X Y Z AZ AX)
 * - geometry (see zv/solid.c)
 * - incrx and incry of the main surface (coordinates passed to the client
 *   will be in the range ([-incrx;incrx], [-incry;incry])
 * - x and y aspect ratio (client incrx/object width)
 * - java class for the client (must implement fr.enst.vreng.<VrengClient>)
 * - codebase URL for that class (optional)
 */
static
void ifcParser(char *l, Ifc *po)
{
  boolean gotURL = FALSE;

  l = parseName(l, po);
  l = parsePosition(l, po);
  po->soh = parseGeometry(l);
  l = strtok(NULL, SEP);

  po->incrx = (int) atoi(l); l = strtok(NULL, SEP);
  po->incry = (int) atoi(l); l = strtok(NULL, SEP);
  po->fx = (float) atof(l); l = strtok(NULL, SEP);
  po->fy = (float) atof(l); l = strtok(NULL, SEP);

  strcpy(po->app, l); l = strtok(NULL, SEP);

  // Try to fin out the codebase URL for this thing
  if (l != NULL) { // url in the .vre file
    strcpy(po->url, l); l = strtok(NULL, SEP);
    gotURL = TRUE;
  }
  else { // There's none in the .vre file.
    char l2[URL_LEN];
    char *last = NULL;

    // if the world's url is like [...]/vre/world.vre
    // try to build an url like [...]/jar/vrengapp.jar
    strncpy(l2, getCurrentWorld()->url, strlen(getCurrentWorld()->url));
    last = strrchr(l2, '/');
    if (last != NULL) {
      *last = '\0'; 
    }
    last = strrchr(l2, '/');
    if (last != NULL) {
      if (strncmp(last, "/vre", 4) == 0) {
        *last = '\0';
        strncpy(po->url, l2, strlen(l2));
        strncat(po->url, "/jar/vrengapp.jar", 18);
        trace(DBG_IFC, "Ifc: cumputed codbase=%s", po->url);
        gotURL = TRUE;
      }
    }
  }
  if (!gotURL) { // Neither of the above methods worked. Just put a default url
     sprintf(po->url, "http://%s%sjar/vrengapp.jar", DEF_HTTP_SERVER, DEF_URL_PREFIX);
     trace(DBG_IFC, "Ifc: default codbase=%s", po->url);
  }

  // Defaults for the other values
  po->wantDelta = FALSE;
  po->o2 = NULL;
  po->needRedraw = FALSE;

  // Ifc objects can't currently collide with anything - they move through them
  po->nature.collision = COL_GHOST;
}

/* Create a ifc from a fileline */
void Ifc::creator(char *l)
{
  new Ifc(l);
}

Ifc::Ifc(char *l)
{
  ifcParser(l, this);

  initializeObject(this, IFC_TYPE, VR_MOBILE);
//  initializeNetObject(this, 0, IFC_PROPS, VR_VOLATILE);
  IfcServer::startApp(this);
} 

/*
 * Send a notification to the child saying a click occured.
 * The coordinates sent are relative to the object's main surface
 * (which would be described by the vectors (1,0,0), (0,1,0),
 * and a normal of (0,0,1)). This only works well if the object
 * is very thin. Clicks hitting the other surface (norm=(0,0,-1))
 * are ignored.
 * Packet contents:
 * - signed 32bit x
 * - signed 32bit y
 */
static
void ifcSendClick(Ifc *po, float x, float y)
{
  IfcMessage *msg = new IfcMessage(po, IFC_MSGT_CLICK, IFC_MSGV_CLICK);
  msg->put32((int32) x);
  msg->put32((int32) y);
  IfcServer::sendData(msg);
  delete msg;
}

/*
 * Send some object's position to the child.
 * If the position is that of the current object,
 * the type is set to 0, otherwise it's set to
 * the object's real type.
 * Packet contents:
 * - signed 8bit type
 * - signed 32bit net object id
 * - signed 16bit port id and object id
 * - signed 32bit x, y, z
 * - signed 32bit az, ax, 0
 * All position/angle variables are multiplied by 1000
 * (sending floats is not currently possible)
 */
static
void ifcSendPos(Ifc *po, WObject *who)
{
  IfcMessage *msg = new IfcMessage(po, IFC_MSGT_POS, IFC_MSGV_SET);
  msg->putOID(who);
  msg->putPos(who);
  IfcServer::sendData(msg);
  delete msg;
}

/*
 * Same as ifcSendPos, except that no position is sent.
 * This is to indicate to the controler application that
 * the object it's working on no longer exists.
 */
static
void ifcSendPosError(Ifc *po, int type, int ssrc, int port, int id)
{
  IfcMessage *msg = new IfcMessage(po, IFC_MSGT_POS, IFC_MSGV_ERROR);
  msg->put8( type);
  msg->put32(ssrc);
  msg->put16(port);
  msg->put16(id);
  IfcServer::sendData(msg);
  delete msg;
}

/*
 * Send the client a notification that a thing just entered/exited
 * its bounding box
 */
static
void ifcSendIntersect(Ifc *pifc, WObject *who, WObject *old, int inOrOut)
{
  IfcMessage *msg = new IfcMessage(pifc, IFC_MSGT_ISEC, inOrOut);
  msg->putOID(who);
  msg->putPos(who);
  msg->putPos(old);
  IfcServer::sendData(msg);
  delete msg;
}

/*
 * Answer a child's query.
 * Right now, the only query available is a query
 * to the list of mobile objects, asking for a given type
 * of WObject. A list of all the object ids is returned
 * to the child.
 */
static
void ifcAnswerTypeQuery(Ifc *po, int type)
{
  ObjectList *list;
  ObjectList *cursor;
  int count = 0;
  IfcMessage *msg = new IfcMessage(po, IFC_MSGT_QUERY, IFC_MSGV_QANS);

  if (type > 0) {
    list = getObjectsWithType(type);
    cursor = list;
    while (cursor != NULL) {
      count++;
      cursor = cursor->next;
    }
    msg->put32(count);
    cursor = list;
    while (cursor != NULL) {
      msg->putOID(cursor->pobject);
      cursor = cursor->next;
    }
    freeObjectList(list); list = NULL;
    trace(DBG_IFC, "Sending %d replies for %d", count, type);
  } else {
    msg->put32(1);
    msg->putOID(worlds->plocaluser);
  }
  IfcServer::sendData(msg);
  delete msg;
}

/* Read a header struct for the raw packet data */
void ifcDumpHeader(ifcHeader hdr)
{
  trace(DBG_IFC, "%x %d %d %d %d %d",
  hdr.proto,
  hdr.version,
  hdr.app_ssrc,
  hdr.msg_type,
  hdr.msg_val,
  hdr.data_len);
}

/*
 * Deal with input from the client app.
 * This function is called at every render phase.
 * The client socket is probed for incoming packets,
 * with an empty timeout. If no packets were queued,
 * the function returns immediatly.
 * No more than one packet is dealt with in one go.
 * Messages dealt with:
 * - MSGT_ADD:
 *     adds 2D objects to this object's rendering.
 * - MSGT_DEL:
 *     removes one or all of the existing 2d objects.
 * - MSGT_POS:
 *     MSGV_ASK: sends the queried object's position back to the client
 *     MSGV_SET: sets this object's position to the parameters passed in
 *     MSGV_UPD: considers the params passed as deltas on the object's position
 * - MSGT_QUERY:
 *     Answers to type queries (returns a list of mobile objects with that type)
 */
static
void ifcProcessClient(Ifc *po)
{
  IfcMessage *msg = NULL;
  Object2D *o2;
  V3 color;
  ifcHeader header;
  int processed = 0;
  int tag;

  // Check if the client has sent anything
  if ((msg = IfcServer::getData(po)) == NULL)
    return;
  header = msg->getHeader();

  color.v[0] = 0.0;
  color.v[1] = 0.0;
  color.v[2] = 0.0;

  // Process the client's data
  switch (header.msg_type) {

  case IFC_MSGT_ADD:
    tag = msg->read16();
    color = msg->readPoint3D();

    if ( (header.msg_val == IFC_MSGV_LINE)
    ||   (header.msg_val == IFC_MSGV_LOOP)
    ||   (header.msg_val == IFC_MSGV_FILL)) {
      // Create a new line
      o2 = ifcNewObject2D(po->o2, header.msg_val, tag, color);
      po->o2 = o2;
      while (msg->hasData(2*IFC_INT32)) {
        ifcAddPoint(o2, msg->readPoint2D());
       }
       processed = 1;
    } else if (header.msg_val == IFC_MSGV_CIRCLE) {
      // Create a new circle
      o2 = ifcNewObject2D(po->o2, header.msg_val, tag, color);
      po->o2 = o2;
      ifcAddPoint(o2, msg->readPoint2D());
      ifcAddPoint(o2, msg->readPoint2D());
      processed = 1;
    }
    po->needRedraw = TRUE;
    break;

  case IFC_MSGT_DEL:
    if (header.msg_val == IFC_MSGV_DELALL) {
      ifcFreeObject2DList(po->o2);
      po->o2 = NULL;
    }
    else {
      tag = msg->read16();
      po->o2 = ifcRemoveObject2D(po->o2, header.msg_val, tag);
    }
    po->needRedraw = TRUE;
    break;

  case IFC_MSGT_POS: {
    // Get the objet ID
    u_int8 type = msg->read8();
    u_int32 src  = msg->read32();
    u_int16 port = msg->read16();
    u_int16 id   = msg->read16();

    // Locate the object
    WObject *who = ((type == 0) ? po : findObjectInMobile(type, src, port, id));
    if (who == NULL) {
      // We didn't find anything that matched
      ifcSendPosError(po, type, src, port, id);
    }
    else if (header.msg_val == IFC_MSGV_ASK) {
      // Message was a query - send the data back
      ifcSendPos(po, who);
      processed = 1;
    }
    else if ((header.msg_val == IFC_MSGV_SET)
      || (header.msg_val == IFC_MSGV_UPD)) {
      // Message was a move
      // Get the movement information
      ifcCopyV3(&(po->posDelta), msg->readPoint3D());
      ifcCopyV3(&(po->angDelta), msg->readPoint3D());
      if (who == po) { // Move myself
        po->wantDelta = header.msg_val;
      }
      else { // Push someone else
        WObject oldobj;
        //PD Pos oldpos = po->pos;
        ObjectList *vicinity;

        // Make the changes
        copyPositionAndBB(who, &oldobj);
        //PD copyPosAndBB(who, oldpos);
        ifcDoDeltaPos(who, po);
        // Update the object in the Vreng 3D thingies
        //PD updateObjectIntoGrid(who, oldpos);
        updateObjectIntoGrid(who, oldobj.pos);
        updateObjectIn3D(who);
        updateBB(who);
        // Check if we need to move the camera
        if (who == worlds->plocaluser) {
          updateCameraFromObject(who);
        }
        // Propagate the changes
        //PD vicinity = getVicinityObjectList(who, &oldobj);
        vicinity = getVicinityObjectList(who, oldobj.pos);
        generalIntersect(who, &oldobj, vicinity);
        freeObjectList(vicinity);
      }
      processed = 1;
    }
    break;
  }

  case IFC_MSGT_QUERY:
    if (header.msg_val == IFC_MSGV_QTYPE) {
      int type = msg->read32();
      ifcAnswerTypeQuery(po, type);
      processed = 1;
    }
    break;

  default:
    break;
  }

  if (!processed)
    trace(DBG_IFC, "Message ignore %d %d\n", header.msg_type, header.msg_val);
  delete msg;
}

/* React to a user click on our surface */
void Ifc::click(V3 dir)
{
  V3 u, c, v, w, norm;
  float det, varx, vary, alpha, beta, ps;
  int x, y;

  // Calculate the intersection between the click vector
  // and our primary surface

  // Get two principal vectors and a normal one
  ifcGetSurfVecs(this, &v, &w, &norm);

  // Check if the click comes from the right side of the surface
  ps = norm.v[0] * dir.v[0] + norm.v[1] * dir.v[1] + norm.v[2] * dir.v[2];
  if (ps < 0) {
    warning("Ifc::click: bad side!");
    return;
  }
  
  // Eye position
  u = V3_New(worlds->plocaluser->pos.x,
             worlds->plocaluser->pos.y,
             worlds->plocaluser->pos.z + 0.5 * USER_DEFAULTHEIGHT);

  // Object's center coordinates
  c = V3_New(pos.x, pos.y, pos.z);

  // Determine X coord. relativly to our surface
  det  = ifcCalcDet(dir, v, w);
  varx  = ifcCalcVar(c, u, w, dir);
  alpha = varx/det;

  // Determine Y coord. relativly to our surface
  vary = ifcCalcVar(c, u, dir, v);
  beta = vary/det;

  // Scale these values
  x = (int) (alpha * (float)incrx / (float)fx);
  y = (int) (beta  * (float)incry / (float)fy);

  // Notify the child
  trace(DBG_FORCE, "Ifc::click: x=%d y=%d", x, y);
  ifcSendClick(this, x, y);
}

/* Dummy: always say we need to move */
void Ifc::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = 1.;
}

/* Propagate the last deltas to the object's position */
void ifcDoChange(Ifc *pifc)
{
  if (pifc->wantDelta == IFC_MSGV_SET)
    ifcDoSetPos(pifc, pifc);
  else if (pifc->wantDelta == IFC_MSGV_UPD)
    ifcDoDeltaPos(pifc, pifc);
  pifc->wantDelta = 0;
}

void ifcDoSetPos(WObject *po, Ifc *pifc)
{
  po->pos.x = pifc->posDelta.v[0];
  po->pos.y = pifc->posDelta.v[1];
  po->pos.z = pifc->posDelta.v[2];
  po->pos.az = pifc->angDelta.v[0];
  po->pos.ay = 0;
  po->pos.ax = pifc->angDelta.v[1];
}

void ifcDoDeltaPos(WObject *po, Ifc *pifc)
{
  po->pos.x += pifc->posDelta.v[0];
  po->pos.y += pifc->posDelta.v[1];
  po->pos.z += pifc->posDelta.v[2];
  po->pos.az += pifc->angDelta.v[0];
  po->pos.ay += 0;
  po->pos.ax += pifc->angDelta.v[1];
}

void Ifc::changePosition(float lasting)
{
  ifcDoChange(this);
}

/*
 * Notify the controler that an ingoing intersection occured.
 */
void Ifc::whenIntersect(WObject *po, WObject *pold)
{
  ifcSendIntersect(this, po, pold, IFC_MSGV_ISECIN);
}

/*
 * Notify the controler that an outgoing intersection occured.
 */
boolean Ifc::whenIntersectOut(WObject *po, WObject *pold)
{
  ifcSendIntersect(this, po, pold, IFC_MSGV_ISECOUT);
  return TRUE;
}

/* Return yes if the child has sent a delta request */
int Ifc::change()
{
  ifcProcessClient(this);
  return (wantDelta > 0 ? TRUE : FALSE);
}

/* Make sure our asynch sockets are open if they need to be.
 * If they are, see if the client said something between now
 * and the last render call.
 * Finally, draw this Ifc's 2D object list.
 */
void Ifc::render()
{
  ifcDraw2DList(this);
}

/* Turn the child on (start it up) */
void Ifc::quit()
{
  IfcServer::stopApp(this);
}

void ifcInitFuncList(void) { }

// *******************************************************************
//                    2D Object manipulation
// *******************************************************************

// 2D Creation functions
Object2D *ifcNewObject2D(Object2D *list, int type, int tag, V3 color)
{
  Object2D *o2 = (Object2D *) malloc(sizeof(Object2D));
  if (o2 == NULL)
    fatal("Couldn't allocate new 2d object");
  o2->type = type;
  o2->tag = tag;
  ifcCopyV3(&(o2->color), color);
  o2->points = NULL;
  o2->next = list;
  return o2;
}

// Remove one object from a list
Object2D *ifcRemoveObject2D(Object2D *list, int type, int tag)
{
  Object2D *ret = list;
  Object2D *prev = NULL;
  Object2D *cursor = list;

  while (cursor != NULL) {
    if ((cursor->type == type) && (cursor->tag == tag)) {
      if (prev == NULL)
        ret = cursor->next;
      else {
        prev->next = cursor->next;
        ret = list;
      }
      ifcFreeObject2D(cursor);
        return ret;
    }
    prev = cursor;
    cursor = cursor->next;
  }
  warning("Attempt to remove a non-existing 2d object (%d %d)", type, tag);
  return list;
}

// Free an object2d list
void ifcFreeObject2DList(Object2D *list)
{
  Object2D *cursor = list;
  Object2D *toDel = list;

  while (cursor != NULL) {
    toDel = cursor;
    cursor = cursor->next;
    ifcFreeObject2D(toDel);
  }
}

// Free an object2D and its point list
void ifcFreeObject2D(Object2D *o2)
{
  if (!o2)
    warning("Attempt to free a null 2d object");
  else {
    ifcFreePointList(o2->points);
    free(o2);
  }
}

// Free an object's point list
void ifcFreePointList(PointList *pl)
{
  PointList *temp;

  while (pl != NULL) {
    temp = pl->next;
    free(pl);
    pl = temp;
  }
}

// Add a point to the point list
void ifcAddPoint(Object2D *o2, V3 point)
{
  PointList *pl = (PointList *) malloc(sizeof(PointList));
  if (pl == NULL)
    fatal("Couldn't add point to 2d object");

  ifcCopyV3(&(pl->point), point);
  pl->next = o2->points;
  o2->points = pl;
}

// Render a list of object
void ifcDraw2DList(Ifc *po)
{
  Object2D *o2 = po->o2;
  float gl_mat[16];
  float fx, fy;

  // Have things changed since last time we build the dlist ?
  if (po->needRedraw) {
    po->needRedraw = FALSE;

    po->displaylist = glGenLists(1);
    glNewList(po->displaylist, GL_COMPILE);

    fx = po->incrx / po->fx;
    fy = po->incry / po->fy;
 
    glTranslatef(0, 0, 0.02); // Extra z so that the figures are drawn slightly
  			  // above the surface

    while (o2 != NULL) {
      float mat[4];

      mat[0] = 0;
      mat[1] = 1;
      mat[2] = 0;
      mat[3] = 1;

      switch (o2->type) {

      case IFC_MSGV_LINE:
      case IFC_MSGV_LOOP:
      case IFC_MSGV_FILL: {
        PointList *pts = o2->points;
        if (pts == NULL)
          return;

        //glColor3f(o2->color.v[0], o2->color.v[1], o2->color.v[2]);
        glColor3f(0.0, 0.0, 1.0);

        if (o2->type == IFC_MSGV_LINE)
          glBegin(GL_LINE_STRIP);
        else if (o2->type == IFC_MSGV_LOOP)
          glBegin(GL_LINE_LOOP);
        else 
          glBegin(GL_POLYGON);

        glColor3f(0.0, 0.0, 1.0);

        while (pts != NULL) {
          glVertex2f(pts->point.v[0]/fx, pts->point.v[1]/fy);
          pts = pts->next;
        }
        glEnd();
        }
        break;

      case IFC_MSGV_CIRCLE: {
        V3 diam = o2->points->point;
        V3 center = o2->points->next->point;
        float arc = M_2PI / 20;
        int i;

        glBegin(GL_LINE_LOOP);
        glColor3f(o2->color.v[0], o2->color.v[1], o2->color.v[2]);
        for (i=19; i>=0; i--) { 
          glVertex2f(center.v[0]/fx+diam.v[0]*cos(i*arc)/fx,
                   center.v[1]/fy+diam.v[1]*sin(i*arc)/fx);
        }
        glEnd();
        }
        break;

      default:
        break;
      }
      o2 = o2->next;
    }
    glEndList();
  }

  // Do the actual drawing
  glLoadName(po->displaylist);
  glPushMatrix();
  M4_to_GL(gl_mat, &po->soh->posmat);
  glMultMatrixf(gl_mat);
  glCallList(po->displaylist);
  glPopMatrix();
}

// *******************************************************************
//                    Line/Surface intersection utilities
// *******************************************************************

/* Returns two vectors that describe the object's surface
 * and a normal vector to that surface */
void ifcGetSurfVecs(Ifc *po, V3 *V, V3 *W, V3 *norm)
{
  M4 rot = MulM4(RotateM4(po->pos.az, UZ), RotateM4(po->pos.ax, UX));
  V3 vec = V3_New(1, 0, 0);
  MulM4V3(V, &rot, &vec);
  vec = V3_New(0, 1, 0);
  MulM4V3(W, &rot, &vec);
  vec = V3_New(0, 0, 1);
  MulM4V3(norm, &rot, &vec);
}

/* Compute a determinant for a 3x3 matrix that's useful from time to time */
float ifcCalcDet(V3 d, V3 v, V3 w)
{
  return (  d.v[0] * ( v.v[2] * w.v[1] - v.v[1] * w.v[2] )
          + d.v[1] * ( v.v[0] * w.v[2] - v.v[2] * w.v[0] )
          + d.v[2] * ( v.v[1] * w.v[0] - v.v[0] * w.v[1] ) );
}

/* Compute an quantity that's used in the line/surface intersection calculation.
 * It's actually just the same thing as above, with a little twist. */
float ifcCalcVar(V3 c, V3 u, V3 w, V3 v)
{
  return (  (c.v[0] - u.v[0]) * ( v.v[2] * w.v[1] - v.v[1] * w.v[2] )
          + (c.v[1] - u.v[1]) * ( v.v[0] * w.v[2] - v.v[2] * w.v[0] )
          + (c.v[2] - u.v[2]) * ( v.v[1] * w.v[0] - v.v[0] * w.v[1] ) );
}
