#ifndef VRENGD

#include <GL/gl.h>
#include <GL/glu.h>
#if HAVE_GLUT
#include <GL/glut.h>
#endif

#include "global.h"
#include "net.h"
#include "wobject.h"
#include "wmgt.h"
#include "parse.h"
#include "list.h"
#include "col.h"	/* COL_NEVER */
#include "defaults.h"
#include "android.h"

#include "zv.h"		/* parseGeometry */
#include "draw.h"
#include "channel.h"	/* getvrgroup */
#include "stat.h"	/* getRate */


const WClass Android::wclass(ANDROID_TYPE, "Android", Android::creator);

void manInit(Android *po);


#if 1
#define MAN_HEIGHT	69
#define MAN_HALF_HEIGHT	34.5
#define MAN_SCALE 	0.0247
#define MAN_MULT	40.5
#else
#define MAN_HEIGHT	1.70
#define MAN_HALF_HEIGHT	0.85
#define MAN_SCALE 	1
#define MAN_MULT	1
#endif

#if 0
#define PHALANX2_LEN	0.5
#define PHALANX_LEN	0.5
#define FINGER_LEN	1.5
#define HAND_LEN	4.0
#define FOREARM_LEN	7.0
#define ELBOW_RAD	0.1
#define ARM_LEN		6.7
#define SHOULDER_RAD	0.1
#define HEAD_RAD	0.3
#define NECK_LEN	2.0
#define FOOT_LEN	5.0
#define CALF_LEN	10.5
#define LEG_LEN		10.0
#define BUST_LEN	12.0
#else
#define PHALANX2_LEN	0.012
#define PHALANX_LEN	0.012
#define FINGER_LEN	0.0375
#define HAND_LEN	0.10
#define FOREARM_LEN	0.178
#define ELBOW_RAD	0.025
#define ARM_LEN		0.165
#define SHOULDER_RAD	0.0025
#define HEAD_RAD	0.0075
#define NECK_LEN	0.05
#define FOOT_LEN	0.12
#define CALF_LEN	0.265
#define LEG_LEN		0.25
#define BUST_LEN	0.30
#endif

// Phalanx
class Phalanx2 {
  GLfloat length;
public:
  Phalanx2() {length = PHALANX2_LEN;}
  int drawPhalanx2();
};

int Phalanx2::drawPhalanx2()
{
  return 1;
}

class Phalanx {
  GLfloat length;
  GLfloat angle_bend_phalanx;
public:
  class Phalanx2 *pphalanx2;
  Phalanx(class Phalanx2 *);
  int drawPhalanx();
  void bend_phalanx(GLfloat angle) {angle_bend_phalanx = angle;}
};

Phalanx::Phalanx(class Phalanx2 *phalanx2)
{
  angle_bend_phalanx = 0;
  length = PHALANX_LEN;
  pphalanx2 = phalanx2;
}

int Phalanx::drawPhalanx()
{
  return pphalanx2->drawPhalanx2();
}

// Finger
class Finger {
  GLfloat length;
  GLfloat angle_bend_finger;
  GLfloat angle_tilt_finger;
public:
  class Phalanx *pphalanx;
  Finger(class Phalanx *);
  int drawFinger(class Android *);
  void bend_finger(GLfloat angle) {angle_bend_finger = angle;}
  void tilt_finger(GLfloat angle) {angle_tilt_finger = angle;}
};

Finger::Finger(class Phalanx *phalanx)
{
  angle_bend_finger = angle_tilt_finger = 0;
  length = FINGER_LEN;
  pphalanx = phalanx;
}

int Finger::drawFinger(Android *po)
{
  return pphalanx->drawPhalanx();
}

// Hand
class Hand {
  GLfloat length;
  GLfloat angle_bend_hand;
  GLfloat angle_tilt_hand;
  GLfloat angle_turn_hand;
public:
  class Finger *pfingers[5];
  Hand(class Finger **);
  int drawHand(class Android *);
  void bend_hand(GLfloat angle) {angle_bend_hand = angle;}
  void tilt_hand(GLfloat angle) {angle_tilt_hand = angle;}
  void turn_hand(GLfloat angle) {angle_turn_hand = angle;}
};

Hand::Hand(class Finger *fingers[5])
{
  angle_bend_hand = angle_tilt_hand = angle_turn_hand = 0;
  length = HAND_LEN;
  //pfingers[0] = fingers[0];
  pfingers[1] = fingers[1];
  pfingers[2] = fingers[2];
  pfingers[3] = fingers[3];
  pfingers[4] = fingers[4];
}

int Hand::drawHand(Android *po)
{
  return pfingers[4]->drawFinger(po);
}

// Forearm
class Forearm {
  GLfloat length;
  GLfloat angle_bend_elbow;
  GLfloat angle_turn_elbow;
public:
  class Hand *phand;
  Forearm(class Hand *);
  int drawForearm(class Android *);
  void bend_elbow(GLfloat angle) {angle_bend_elbow = angle;}
  void turn_elbow(GLfloat angle) {angle_turn_elbow = angle;}
};

Forearm::Forearm(class Hand *hand)
{
  angle_bend_elbow = angle_turn_elbow = 0;
  length = FOREARM_LEN;
  phand = hand;
}

int Forearm::drawForearm(Android *po)
{
  return phand->drawHand(po);
}

// Arm
class Arm {
  GLfloat length;
  GLfloat angle_raise_arm;
  GLfloat angle_rote_arm;
  GLfloat angle_turn_arm;
public:
  class Forearm *pforearm;
  Arm(class Forearm *);
  int drawArm(class Android *);
  void raise_arm (GLfloat angle) {angle_raise_arm = angle-90;}
  void rote_arm(GLfloat angle) {angle_rote_arm = -angle;}
  void turn_arm(GLfloat angle) {angle_turn_arm = angle+90;}
};

Arm::Arm(class Forearm *forearm)
{
  angle_raise_arm = -90;
  angle_rote_arm = 0;
  angle_turn_arm = 90;
  length = ARM_LEN;
}

int Arm::drawArm(Android *po)
{
  return pforearm->drawForearm(po);
}

// Shoulder
class Shoulder {
  GLfloat radius;
  GLfloat distance_avance_shoulder;
  GLfloat distance_raise_shoulder;
public:
  class Arm *parm;
  Shoulder(class Arm *);
  int drawSoulder(class Android *);
  void avance_shoulder(GLfloat distance) {distance_avance_shoulder = distance;}
  void raise_shoulder(GLfloat distance) {distance_raise_shoulder = distance;}
};

Shoulder::Shoulder(class Arm* arm)
{
  distance_avance_shoulder = distance_raise_shoulder = 0;
  radius = SHOULDER_RAD;
  parm = arm;
}

int Shoulder::drawSoulder(Android *po)
{
  return parm->drawArm(po);
}

// Head
class Head {
  GLfloat radius;
  GLfloat angle_bend_head;
  GLfloat angle_tilt_head;
  GLfloat angle_turn_head;
  GLfloat head_diffuse[4];
public:
  Head();
  int drawHead();
  void bend_head(GLfloat angle) {angle_bend_head = angle;}
  void tilt_head(GLfloat angle) {angle_tilt_head = angle;}
  void turn_head(GLfloat angle){angle_turn_head = angle;}
};

Head::Head()
{
  angle_bend_head = angle_tilt_head = angle_turn_head = 0;
  radius = HEAD_RAD;
}

int Head::drawHead()
{
  return 1;
}

// Neck
class Neck {
  GLfloat length;
  GLfloat angle_bend_neck;
  GLfloat angle_tilt_neck;
  GLfloat angle_turn_neck;
public:
  class Head *phead;
  Neck(class Head *);
  int drawNeck();
  void bend_neck(GLfloat angle) {angle_bend_neck = angle;}
  void tilt_neck(GLfloat angle) {angle_tilt_neck = angle;}
  void turn_neck(GLfloat angle) {angle_turn_neck = angle;}
};

Neck::Neck(class Head *head)
{
  angle_bend_neck = angle_tilt_neck = angle_turn_neck = 0;
  length = NECK_LEN;
  phead = head;
}

int Neck::drawNeck()
{
  return phead->drawHead();
}

// Foot
class Foot {
  GLfloat length;
  GLfloat angle_bend_foot;
  GLfloat angle_tilt_foot;
  GLfloat angle_turn_foot;
  GLfloat foot_diffuse[4];
public:
  Foot() {
    foot_diffuse[0] = 0.96;
    foot_diffuse[1] = 0.5;
    foot_diffuse[2] = 0.95;
    foot_diffuse[3] = 1.0;
    length = FOOT_LEN;
    angle_bend_foot = angle_tilt_foot = angle_turn_foot = 0;
  }
  int drawFoot(class Android *);
  void bend_foot(GLfloat angle) {angle_bend_foot = -angle;}
  void tilt_foot(GLfloat angle) {angle_tilt_foot = -angle;}
  void turn_foot(GLfloat angle) {angle_turn_foot = -angle;}
};

int Foot::drawFoot(Android *po)
{
  return 1;
}

// Calf
class Calf {
  GLfloat length;
  GLfloat angle_bend_knee;
  GLfloat angle_turn_knee;
public:
  class Foot *pfoot;
  Calf(class Foot *);
  int drawCalf(class Android *);
  void bend_knee(GLfloat angle) {angle_bend_knee = -angle;}
  void turn_knee(GLfloat angle) {angle_turn_knee = -angle;}
};

Calf::Calf(class Foot *foot)
{
  angle_bend_knee = 0;
  length = CALF_LEN;
  pfoot = foot;
}

int Calf::drawCalf(Android *po)
{
  return pfoot->drawFoot(po);
}

// Leg
class Leg {
  GLfloat length;
  GLfloat angle_raise_leg;
  GLfloat angle_rote_leg;
  GLfloat angle_turn_leg;
public:
  class Calf *pcalf;
  Leg(class Calf *calf);
  int drawLeg(class Android *);
  void raise_leg(GLfloat angle) {angle_raise_leg = angle;}
  void rote_leg(GLfloat angle) {angle_rote_leg = angle;}
  void turn_leg(GLfloat angle) {angle_turn_leg = angle;}
};

Leg::Leg(class Calf *calf)
{
  angle_raise_leg = angle_rote_leg = angle_turn_leg = 0;
  length = LEG_LEN;
  pcalf = calf;
}

int Leg::drawLeg(Android *po)
{
  return pcalf->drawCalf(po);
}

// Bust
class Bust {
  GLfloat length;
  GLfloat angle_roll_bust;
  GLfloat angle_tilt_bust;
  GLfloat angle_turn_bust;
public:
  class Leg *plegs[2];
  class Shoulder *pshoulders[2];
  class Neck *pneck;
  Bust(class Leg **, class Shoulder **, class Neck *, class Android *);
  int drawBust(class Android *);
  void roll_bust(GLfloat angle) {angle_roll_bust = angle;}
  void tilt_bust( GLfloat angle) {angle_tilt_bust = angle;}
  void turn_bust(GLfloat angle) {angle_turn_bust = angle;}
};

Bust::Bust (class Leg *legs[2], class Shoulder *pshoulders[2], class Neck *pneck,class Android *po)
{
}

int Bust::drawBust(Android *po)
{
  return po->shoulders[1]->drawSoulder(po);
}

static
void androidInit(Android *po)
{
  // Create human body
  for (int k=0;k<10;k++)
    po->phalanx2[k] = new Phalanx2();
  for (int k=0;k<10;k++)
    po->phalanx[k] = new Phalanx(po->phalanx2[k]);
  for (int k=0;k<5;k++)
    po->fingers_r[k] = new Finger(po->phalanx[k]);
  for (int k=0;k<5;k++)
    po->fingers_l[k] = new Finger(po->phalanx[k+5]);

  po->hand_r = new Hand(po->fingers_r);
  po->hand_l = new Hand(po->fingers_l);
  po->forearm_r = new Forearm(po->hand_r);
  po->forearm_l = new Forearm(po->hand_l);
  po->arm_r = new Arm(po->forearm_r);
  po->arm_l = new Arm(po->forearm_l);
  po->shoulders[0] = new Shoulder(po->arm_r);
  po->shoulders[1] = new Shoulder(po->arm_l);
  po->head = new Head();
  po->neck =  new Neck(po->head);
  po->foot_r = new Foot();
  po->foot_l = new Foot();
  po->calf_r = new Calf(po->foot_r);
  po->calf_l = new Calf(po->foot_l);
  po->legs[0] = new Leg(po->calf_r);
  po->legs[1] = new Leg(po->calf_l);
}

static
void humanInit(Android *po)
{
  po->tx=0; po->ty=0; po->tz=0;
  po->rx=0; po->ry=0; po->rz=0;

  androidInit(po);
  po->bust = new Bust(po->legs, po->shoulders, po->neck, po);
  manInit(po);

  for (int i=0; i <= NUM_BAPS_V2; i++) { //set all values to 0
    po->bap[i]=0;
    po->mask[i]=0;
  }
  for (int i=0; i <= NUM_FAPS; i++) //set all values to 0
    po->fap[i]=0;
  po->status = ANDROID_INACTIVE;
}

///////////////
// MAN_MODEL
///////////////

static
void parseInt(char *s, int *a, int *b, int *c)
{
  int i, j;
  int x, y, z;

  x = 0; y = -1; z = -1;
  j = (int)strlen(s);
  for (i=0; i<j; i++) {
    if ((s[i] == ' ') || (s[i] == '\n')) {
      s[i] = 0;
      if (y == -1)
        y = i+1;
      else if (z == -1)
        z = i+1;
    }
  }
  *a = atoi(&s[x]);
  *b = atoi(&s[y]);
  *c = atoi(&s[z]);
}

static
void parseFloat(char *s, float *a, float *b, float *c)
{
  int i, j;
  int x, y, z;

  x = 0; y = -1; z = -1;
  j = (int)strlen(s);
  for (i=0; i<j; i++) {
    if ((s[i] == ' ') || (s[i] == '\n')) {
      s[i] = 0;
      if (y == -1)
        y = i+1;
      else if (z == -1)
        z = i+1;
    }
  }
#define NEWSCALE
#ifdef NEWSCALE
  *a = (float)atof(&s[x]) * MAN_SCALE;
  *b = (float)atof(&s[y]) * MAN_SCALE;
  *c = (float)atof(&s[z]) * MAN_SCALE;
#else
  *a = (float)atof(&s[x]);
  *b = (float)atof(&s[y]);
  *c = (float)atof(&s[z]);
#endif
}

boolean isEmpty(char *l)
{
  if (*l == '#')	// commented line
    return 1;
  if (strlen(l) == 0) 	// empty line
    return 1;
  if (isprint(*l))
    return 0;
  return 1;
}

static
void drawPart(struct model *m)
{
  glPushMatrix(); // TODO : REMOVE THOSE ?
  glBegin(GL_TRIANGLES);
  for (int i=0; i < m->pn; i++) {
    int v1, v2, v3;
    int n1, n2, n3;

    v1 = m->p[i*6];
    v2 = m->p[i*6+1];
    v3 = m->p[i*6+2];
    n1 = m->p[i*6+3];
    n2 = m->p[i*6+4];
    n3 = m->p[i*6+5];

    glNormal3f(m->n[n1*3], m->n[n1*3+1], m->n[n1*3+2]);
    glVertex3f(m->v[v1*3], m->v[v1*3+1], m->v[v1*3+2]);
    glNormal3f(m->n[n2*3], m->n[n2*3+1], m->n[n2*3+2]);
    glVertex3f(m->v[v2*3], m->v[v2*3+1], m->v[v2*3+2]);
    glNormal3f(m->n[n3*3], m->n[n3*3+1], m->n[n3*3+2]);
    glVertex3f(m->v[v3*3], m->v[v3*3+1], m->v[v3*3+2]);
  }
  glEnd();
  glPopMatrix(); // TODO : REMOVE THOSE ?
}

static
void LoadMFile(void *am, void *handle)
{
  if (handle == NULL)
    return;

  struct model *mod = (struct model *) am;
  char line[128];

  /* Get info */
  httpClearBuf();
  do {
    httpGetLine(handle, line);
  } while (isEmpty(line));
  parseInt(line, &mod->vn, &mod->nn, &mod->pn);

  /* Get vertices */
  mod->v = (float *) malloc(sizeof(float)*3*mod->vn);
  for (int i=0; i<mod->vn; i++) {
    do {
      httpGetLine(handle, line);
    } while (isEmpty(line));
    parseFloat(line, &(mod->v[i*3]), &(mod->v[i*3+1]), &(mod->v[i*3+2]));
  }

  /* Get normals */
  mod->n = (float *) malloc(sizeof(float)*3*mod->nn);
  for (int i=0; i<mod->nn; i++) {
    do {
      httpGetLine(handle, line);
    } while (isEmpty(line));
    parseFloat(line, &(mod->n[i*3]), &(mod->n[i*3+1]), &(mod->n[i*3+2]));
  }

  /* Get polygons */
  mod->p = (int *) malloc(sizeof(int)*6*mod->pn);
  for (int i=0; i<mod->pn; i++) {
    do {
      httpGetLine(handle, line);
    } while (isEmpty(line));
    for (int j=0; j<(int)strlen(line); j++) {
      int a;
      if (line[j] == 'v') a = j+2;
      if (line[j] == 'n') {
        line[j-1] = 0;
        parseInt(&line[a], &(mod->p[i*6]), &(mod->p[i*6+1]), &(mod->p[i*6+2]));
        parseInt(&line[j+2],&(mod->p[i*6+3]),&(mod->p[i*6+4]),&(mod->p[i*6+5]));
        break;
      }
    }
  }
}

static
void LoadUrlModels(Android *po, int numjoints, void *handle)
{
  char line[BUFSIZ];
  struct {
    char keystr[10];
    struct model *m;
    char url[URL_LEN];
  } jointpoints[] = {
      { "mod_head", &(po->head1) },
      { "mod_neck", &(po->neck1) },
      { "mod_bust", &(po->body_up) },
      { "mod_tumm", &(po->body_bot) },
      { "mod_la",   &(po->la_up) },
      { "mod_lm",   &(po->la_mid) },
      { "mod_lh",   &(po->la_hand) },
      { "mod_ra",   &(po->ra_up) },
      { "mod_rm",   &(po->ra_mid) },
      { "mod_rh",   &(po->ra_hand) },
      { "mod_ll",   &(po->ll_up) },
      { "mod_lm",   &(po->ll_mid) },
      { "mod_lf",   &(po->ll_foot) },
      { "mod_rl",   &(po->rl_up) },
      { "mod_rm",   &(po->rl_mid) },
      { "mod_rf",   &(po->rl_foot) },
      { "",         NULL }
      };

  for (int i=0; i < numjoints; ) {
    char *keystr, *url;

    if (httpGetLine(handle, line) < 0)
      break;
    if (line[0] == '#')
      continue;
    trace(DBG_MAN, "LoadUrlModels: %s", line);

    keystr = strtok(line, "=");
    url = strtok(NULL, " ");
    if (strcasecmp(jointpoints[i].keystr, keystr) == 0) {
      trace(DBG_MAN, "LoadUrlModels: i=%d key=%s url=%s", i, keystr, url);
      strcpy(jointpoints[i].url, url);
    }
    i++;
  }

  // get faces.index
  while (httpGetLine(handle, line) >= 0) {
    char *keystr, *url;

    if (line[0] == '#')
      continue;
    keystr = strtok(line, "=");
    url = strtok(NULL, " ");
    if (strcasecmp("faces", keystr) == 0) {
      strcpy(po->faces_url, url);
      trace(DBG_MAN, "LoadUrlModels: faces_url=%s", po->faces_url);
    }
  }

  // now we can download models
  for (int i=0; i < numjoints; i++) {
    trace(DBG_MAN, "LoadUrlModels: i=%d url=%s", i, jointpoints[i].url);
    httpOpen(jointpoints[i].url, LoadMFile, jointpoints[i].m, THREAD_NO_BLOCK);
  }
}

static
void LoadJoints(void *_po, void *handle)
{
  if (handle == NULL)
    return;

  int numjoints = 0;
  Android *po = (Android *) _po;
  char line[BUFSIZ];

  httpClearBuf();
  do {
    if (httpGetLine(handle, line) < 0)
      return;
    trace(DBG_MAN, "LoadJoints: %s", line);
  } while (line[0] == '#') ;
  numjoints = atoi(line);
  trace(DBG_MAN, "LoadJoints: numjoints=%d", numjoints);

  for (int i=0; i < numjoints;) {
    if (httpGetLine(handle, line) < 0)
      break;
    if (line[0] == '#')
      continue;
    trace(DBG_MAN, "LoadJoints: %s", line);

    char *l;
    int number;

    l = strtok(line, " \t");
    number = (int)atoi(l);
    l = strtok(NULL, " \t");
#ifdef NEWSCALE
    po->jp.x[number] = (float)atof(l) * MAN_SCALE; l = strtok(NULL, " \t");
    po->jp.y[number] = (float)atof(l) * MAN_SCALE; l = strtok(NULL, " \t");
    po->jp.z[number] = (float)atof(l) * MAN_SCALE;
#else
    po->jp.x[number] = (float)atof(l); l = strtok(NULL, " \t");
    po->jp.y[number] = (float)atof(l); l = strtok(NULL, " \t");
    po->jp.z[number] = (float)atof(l);
#endif
    i++;
  }
  LoadUrlModels(po, numjoints, handle);
}

void manInit(Android *po)
{
  httpOpen(po->name.url, LoadJoints, po, THREAD_NO_BLOCK);

  glShadeModel(GL_SMOOTH);

  GLfloat light_ambient[] = {0.0, 0.0, 0.0, 1.0};
  GLfloat light_diffuse[] = {1.0, 1.0, 1.0, 1.0};
  GLfloat light_specular[] = {1.0, 1.0, 1.0, 1.0};
  GLfloat light_position[] = {10.0f, -10.0f, -1.0f, 0.0f};
  
  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);
  
  glEnable(GL_LIGHT0);
  glDepthFunc(GL_LESS);	// effect ?
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_LIGHTING);

  po->manlist = glGenLists(16);

  glNewList(po->manlist,GL_COMPILE); drawPart(&(po->head1)); glEndList();
  glNewList(po->manlist+1,GL_COMPILE); drawPart(&(po->body_up)); glEndList();
  glNewList(po->manlist+2,GL_COMPILE); drawPart(&(po->body_bot)); glEndList();
  glNewList(po->manlist+3,GL_COMPILE); drawPart(&(po->ra_up)); glEndList();
  glNewList(po->manlist+4,GL_COMPILE); drawPart(&(po->ra_mid)); glEndList();
  glNewList(po->manlist+5,GL_COMPILE); drawPart(&(po->ra_hand)); glEndList();
  glNewList(po->manlist+6,GL_COMPILE); drawPart(&(po->la_up)); glEndList();
  glNewList(po->manlist+7,GL_COMPILE); drawPart(&(po->la_mid)); glEndList();
  glNewList(po->manlist+8,GL_COMPILE); drawPart(&(po->la_hand)); glEndList();
  glNewList(po->manlist+9,GL_COMPILE); drawPart(&(po->rl_up)); glEndList();
  glNewList(po->manlist+10,GL_COMPILE); drawPart(&(po->rl_mid)); glEndList();
  glNewList(po->manlist+11,GL_COMPILE); drawPart(&(po->rl_foot)); glEndList();
  glNewList(po->manlist+12,GL_COMPILE); drawPart(&(po->ll_up)); glEndList();
  glNewList(po->manlist+13,GL_COMPILE); drawPart(&(po->ll_mid)); glEndList();
  glNewList(po->manlist+14,GL_COMPILE); drawPart(&(po->ll_foot)); glEndList();
  glNewList(po->manlist+15,GL_COMPILE); drawPart(&(po->neck1)); glEndList();
}

//PDstatic GLfloat skindef[] = {0.847,0.6117,0.5137,1};
//PD static GLfloat skindef[] = {1.0,0.8,0.8,1};
static GLfloat skindef[] = {1.00,0.75,0.70,1};

static
void manRender(Android *po)
{
#define MANLIGHT
#ifdef MANLIGHT
  GLfloat light_ambient[] = {0.4, 0.4, 0.4, 1};
  GLfloat light_diffuse[] = {1.0, 1.0, 1.0, 1};
  GLfloat light_specular[] = {0.1, 0.1, 0.1, 1};
  //PD GLfloat light_position [] = {500,500,200,1};
  //PD GLfloat light_position [] = {-5,-5,-2,1};
  GLfloat light_position [] = {-0,-0, 2,1};

  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);

  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHTING);
#endif

  GLfloat specular[] = {0.1,0.1,0.1,1};
  GLfloat shininess[] = {100};

  glMaterialfv(GL_FRONT, GL_SPECULAR,  specular);
  glMaterialfv(GL_FRONT, GL_SHININESS, shininess);
  // glColorMaterial(GL_FRONT, GL_DIFFUSE);	// AJ

  glEnable(GL_DEPTH);
  glDepthFunc(GL_LEQUAL);

  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(po->skin);
  glPushMatrix();	// pelvic
  //======================== Pelvic (Body Bottom)
  glTranslatef(po->jp.x[PELVIC], po->jp.y[PELVIC], po->jp.z[PELVIC]);
  glRotatef(po->bap[PELVIC_TILT],1,0,0);
  glRotatef(po->bap[PELVIC_TORSION],0,0,1);
  glRotatef(po->bap[PELVIC_ROLL],0,1,0);
  glTranslatef(-po->jp.x[PELVIC], -po->jp.y[PELVIC], -po->jp.z[PELVIC]);
  glCallList(po->manlist+2);
  glPushMatrix();	// spinal
  //======================== Spinal (Body Up)
  glTranslatef(po->jp.x[SPINAL], po->jp.y[SPINAL], po->jp.z[SPINAL]);
  glRotatef(po->bap[T5ROLL],0,1,0);
  glRotatef(po->bap[T5TORSION],0,0,1);
  glRotatef(po->bap[T5TILT],1,0,0);
  glTranslatef(-po->jp.x[SPINAL], -po->jp.y[SPINAL], -po->jp.z[SPINAL]);
  glCallList(po->manlist+1);
  glPushMatrix();	// neck
  //======================== Lower Neck
  glTranslatef(po->jp.x[LOWER_NECK],po->jp.y[LOWER_NECK],po->jp.z[LOWER_NECK]);
  glRotatef(po->bap[C4ROLL],0,1,0);
  glRotatef(po->bap[C4TORSION],0,0,1);
  glRotatef(po->bap[C4TILT],1,0,0);
  glTranslatef(-po->jp.x[LOWER_NECK],-po->jp.y[LOWER_NECK],-po->jp.z[LOWER_NECK]);
  glCallList(po->manlist+15);

  glEnable(GL_COLOR_MATERIAL);
  glColor4fv(po->skin);
  glPushMatrix();	// head
  //======================== Upper Neck -> Head
  glTranslatef(po->jp.x[UPPER_NECK],po->jp.y[UPPER_NECK],po->jp.z[UPPER_NECK]);
  glRotatef(po->bap[C1ROLL],0,1,0);
  glRotatef(po->bap[C1TORSION],0,0,1);
  glRotatef(po->bap[C1TILT],1,0,0);

#define FACERENDER
#ifdef FACERENDER
#ifdef NEWSCALE
  glScalef(FACE_SCALE, FACE_SCALE, FACE_SCALE);     // PD
#else
  glScalef(3.3,3.3,3.3);	// YR
#endif
  glTranslatef(0, 0.5, -0.7);	// PD
  glRotatef(90,1,0,0);		// YR

  po->faceObject->render();	// YR
#else
  glCallList(po->manlist);
#endif // FACERENDER
  glTranslatef(-po->jp.x[UPPER_NECK],-po->jp.y[UPPER_NECK],-po->jp.z[UPPER_NECK]);
  glPopMatrix(); 	// head

  glPopMatrix(); 	// neck

  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(skindef);
  glPushMatrix();	// l_shoulder
  //========================  Left Shoulder (Arm Up)
  glTranslatef(po->jp.x[L_SHOULDER], po->jp.y[L_SHOULDER], po->jp.z[L_SHOULDER]);
  glRotatef(-90,0,0,1);
  glRotatef(-po->bap[L_SHOULDER_FLEXION],0,1,0);
  glRotatef(po->bap[L_SHOULDER_ABDUCT],1,0,0);
  glRotatef(-po->bap[L_SHOULDER_TWIST],0,0,1);
  glTranslatef(-po->jp.x[L_SHOULDER], -po->jp.y[L_SHOULDER], -po->jp.z[L_SHOULDER]);
  glCallList(po->manlist+6);
  //========================  Left Elbow (Arm Mid)
  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(skindef);
  glTranslatef(po->jp.x[L_ELBOW], po->jp.y[L_ELBOW], po->jp.z[L_ELBOW]);
  glRotatef(-po->bap[L_ELBOW_FLEXION],0,1,0);
  glRotatef(-po->bap[L_ELBOW_TWIST],0,0,1);
  glTranslatef(-po->jp.x[L_ELBOW], -po->jp.y[L_ELBOW], -po->jp.z[L_ELBOW]);
  glCallList(po->manlist+7);
  //======================== Left Wrist (Arm Hand)
  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(skindef);
  glTranslatef(po->jp.x[L_WRIST], po->jp.y[L_WRIST], po->jp.z[L_WRIST]);
  glRotatef(po->bap[L_WRIST_FLEXION],1,0,0);
  glRotatef(po->bap[L_WRIST_PIVOT],0,1,0);
  glRotatef(po->bap[L_WRIST_TWIST],0,0,1);
  glTranslatef(-po->jp.x[L_WRIST], -po->jp.y[L_WRIST], -po->jp.z[L_WRIST]);
  glCallList(po->manlist+8);
  glPopMatrix();	// l_shoulder
  //======================== Right Shoulder (Arm Up)
  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(skindef);
  glPushMatrix();	// r_shoulder
  glTranslatef(po->jp.x[R_SHOULDER],po->jp.y[R_SHOULDER],po->jp.z[R_SHOULDER]);
  glRotatef(90,0,0,1);
  glRotatef(po->bap[R_SHOULDER_FLEXION],0,1,0);
  glRotatef(-po->bap[R_SHOULDER_ABDUCT],1,0,0);
  glRotatef(-po->bap[R_SHOULDER_TWIST],0,0,1);
  glTranslatef(-po->jp.x[R_SHOULDER],-po->jp.y[R_SHOULDER],-po->jp.z[R_SHOULDER]);
  glCallList(po->manlist+3);
  //======================== Right Elbow (Arm Mid)
  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(skindef);
  glTranslatef(po->jp.x[R_ELBOW], po->jp.y[R_ELBOW], po->jp.z[R_ELBOW]);
  //glRotatef(90,0,0,1);
  glRotatef(po->bap[R_ELBOW_FLEXION],0,1,0);
  glRotatef(-po->bap[R_ELBOW_TWIST],0,0,1);
  glTranslatef(-po->jp.x[R_ELBOW], -po->jp.y[R_ELBOW], -po->jp.z[R_ELBOW]);
  glCallList(po->manlist+4);
  //======================== Right Wrist (Arm Hand)
  glEnable(GL_COLOR_MATERIAL);
  glColor4fv(skindef);
  glTranslatef(po->jp.x[R_WRIST], po->jp.y[R_WRIST], po->jp.z[R_WRIST]);
  glRotatef(po->bap[R_WRIST_FLEXION],1,0,0);
  glRotatef(po->bap[R_WRIST_PIVOT],0,1,0);
  glRotatef(po->bap[R_WRIST_TWIST],0,0,1);
  glTranslatef(-po->jp.x[R_WRIST], -po->jp.y[R_WRIST], -po->jp.z[R_WRIST]);
  glCallList(po->manlist+5);
  glPopMatrix();	// r_shoulder
  glPopMatrix();	// spinal
  glPopMatrix();	// pelvic

  glEnable(GL_COLOR_MATERIAL);	//PD
  glColor4fv(po->skin);
  glPushMatrix();	// l_leg
  //======================== Left Hip (Leg Up)
  glTranslatef(po->jp.x[L_HIP], po->jp.y[L_HIP], po->jp.z[L_HIP]);
  glRotatef(po->bap[L_HIP_FLEXION],1,0,0);
  glRotatef(po->bap[L_HIP_ABDUCT],0,1,0);
  glRotatef(po->bap[L_HIP_TWIST],0,0,1);
  glTranslatef(-po->jp.x[L_HIP], -po->jp.y[L_HIP], -po->jp.z[L_HIP]);
  glCallList(po->manlist+12);
  //======================== Left Knee (Leg Mid)
  glTranslatef(po->jp.x[L_KNEE], po->jp.y[L_KNEE], po->jp.z[L_KNEE]);
  glRotatef(po->bap[L_KNEE_FLEXION],1,0,0);
  glRotatef(po->bap[L_KNEE_TWIST],0,0,1);
  glTranslatef(-po->jp.x[L_KNEE], -po->jp.y[L_KNEE], -po->jp.z[L_KNEE]);
  glCallList(po->manlist+13);
  //======================== Left Ankle (Leg Foot)
  glTranslatef(po->jp.x[L_ANKLE], po->jp.y[L_ANKLE], po->jp.z[L_ANKLE]);
  glRotatef(po->bap[L_ANKLE_FLEXION],1,0,0);
  glRotatef(po->bap[L_ANKLE_TWIST],0,0,1);
  glTranslatef(-po->jp.x[L_ANKLE], -po->jp.y[L_ANKLE], -po->jp.z[L_ANKLE]);
  glCallList(po->manlist+14);
  glPopMatrix();	// l_leg
  //======================== Right Hip (Leg Up)
  glPushMatrix();	// r_leg
  glTranslatef(po->jp.x[R_HIP], po->jp.y[R_HIP], po->jp.z[R_HIP]);
  glRotatef(po->bap[R_HIP_FLEXION],1,0,0);
  glRotatef(po->bap[R_HIP_ABDUCT],0,1,0);
  glRotatef(po->bap[R_HIP_TWIST],0,0,1);
  glTranslatef(-po->jp.x[R_HIP], -po->jp.y[R_HIP], -po->jp.z[R_HIP]);
  glCallList(po->manlist+9);
  //======================== Right Knee (Leg Mid)
  glTranslatef(po->jp.x[R_KNEE], po->jp.y[R_KNEE], po->jp.z[R_KNEE]);
  glRotatef(po->bap[R_KNEE_FLEXION],1,0,0);
  glRotatef(po->bap[R_KNEE_TWIST],0,0,1);
  glTranslatef(-po->jp.x[R_KNEE], -po->jp.y[R_KNEE], -po->jp.z[R_KNEE]);
  glCallList(po->manlist+10);
  //======================== Right Ankle (Leg Foot)
  glTranslatef(po->jp.x[R_ANKLE], po->jp.y[R_ANKLE], po->jp.z[R_ANKLE]);
  glRotatef(po->bap[R_ANKLE_FLEXION],1,0,0);
  glRotatef(po->bap[R_ANKLE_TWIST],0,0,1);
  glTranslatef(-po->jp.x[R_ANKLE], -po->jp.y[R_ANKLE], -po->jp.z[R_ANKLE]);
  glCallList(po->manlist+11);
  glPopMatrix();	// r_leg

  glDisable(GL_COLOR_MATERIAL);	//PD
}

void Android::render()
{
#undef ANDROIDLIGHT
#ifdef ANDROIDLIGHT
  const GLfloat blank[] = {1,1,1,1};
  const GLfloat grik[] = {0.7,0.7,0.7,1};

  glEnable(GL_LIGHT0);
  glLightfv(GL_LIGHT0,GL_SPECULAR, blank);
  glLightfv(GL_LIGHT0,GL_DIFFUSE, grik);
  const GLfloat pos0[] = {-3, 0, 0, 1};
  glLightfv(GL_LIGHT0,GL_POSITION, pos0);

  glEnable(GL_LIGHT1);
  glLightfv(GL_LIGHT1,GL_SPECULAR, blank);
  glLightfv(GL_LIGHT1,GL_DIFFUSE, grik);
  const GLfloat pos1[] = {0, 0, 2, 1};
  glLightfv(GL_LIGHT1,GL_POSITION, pos1);

  glEnable(GL_LIGHTING);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
#endif

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);	//AJ effect ?
  glPushMatrix();

  // XYZ POSITION: x=10 y={0,2,-2} z=0 az={90,180,0}
#ifdef NEWSCALE
  glTranslatef(pos.x + tx, pos.y + ty, pos.z +MAN_HALF_HEIGHT*MAN_SCALE + tz);
#else
  glScalef(MAN_SCALE, MAN_SCALE, MAN_SCALE);	// 1/40 = 68m/1.70m
  glTranslatef(41.5*pos.x + tx, 40*pos.y + ty, pos.z+MAN_HALF_HEIGHT + tz);
#endif

  glRotatef(-90 + ry, 0,1,0);	//PD8  horiz -> vertic axe vre Y
  glRotatef(-90 + rz, 0,0,1);	//PD8  axe vre X
  glRotatef(-90 + rx, 1,0,0);	//PD8  axe vre Y
  glRotatef(rotz - 90, 0,0,1);	//PD6

  manRender(this);
  glPopMatrix();
  glPopMatrix();
}

//===================================== Network

static
int initUdpReceiver(Android *po)
{
  if ((po->sdudp = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    trace(DBG_FORCE, "initUdpReceiver socket");
    return 0;
  }

  /* bind local port */
  po->udpsa.sin_family = AF_INET;
  po->udpsa.sin_port = htons(po->bap_port);
  if (po->ipmode == ANDROID_MULTICAST)
    inet_pton(AF_INET, getvrgroup(getCurrentChannelName()), &(po->udpsa.sin_addr.s_addr));
  else
    po->udpsa.sin_addr.s_addr = htonl(INADDR_ANY);

  int one = 1;
  if (setsockopt(po->sdudp, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one))<0)
    perror("reuseaddr");
  if (bind(po->sdudp, (struct sockaddr *) &(po->udpsa), sizeof(po->udpsa)) <0) {
    trace(DBG_FORCE, "initUdpReceiver: can't bind port=%d", po->bap_port);
    return 0;
  }
  if (po->ipmode == ANDROID_MULTICAST) {	// Multicast
    inet_pton(AF_INET, getvrgroup(getCurrentChannelName()),
              &(po->mreq.imr_multiaddr.s_addr));
    po->mreq.imr_interface.s_addr = htonl(INADDR_ANY);
    if (setsockopt(po->sdudp, IPPROTO_IP, IP_ADD_MEMBERSHIP,
                   &(po->mreq), sizeof(po->mreq)) < 0) {
      perror("IP_ADD_MEMBERSHIP");
      trace(DBG_FORCE, "initUdpReceiver: cannot join multicast group %s",
                        getvrgroup(getCurrentChannelName()));
      return -1;
    }
    trace(DBG_FORCE, "initUdpReceiver: waiting for Multicast on %x/%d/%d", po->udpsa.sin_addr.s_addr, po->udpsa.sin_port, getvrttl(getCurrentChannelName()));
  }
  else {	// Unicast
    trace(DBG_FORCE, "initUdpReceiver: waiting for Unicast on port %d", po->bap_port);
  }
  return 1;
}

int connectToBapServer(Android *po, int ipmode)
{
  if ((po->sdtcp = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("socket bap/fap failed");
    return 0;
  }

  struct hostent *ph;
  struct in_addr *pa;

  if ((ph = my_gethostbyname(po->baps)) == NULL)
    return 0;
  if ((pa = (struct in_addr*) (ph->h_addr)) == NULL) {
    my_free_netdb(ph);
    return 0;
  }
  my_free_netdb(ph);
  po->tcpsa.sin_family = AF_INET;
  po->tcpsa.sin_port = htons(BAPFAP_PORT);
  po->tcpsa.sin_addr.s_addr = pa->s_addr;	// no htonl !!!

  char dst[64];

  if (connect(po->sdtcp, (struct sockaddr *) &(po->tcpsa), sizeof(po->tcpsa)) < 0) {
    perror("connect Bap server");
    trace(DBG_FORCE, "Connection failed with the bap/fap server: %s (%s)",
          po->baps,
          inet_ntop(AF_INET, &po->tcpsa.sin_addr, dst, sizeof(dst)));
    return 0;
  }
  trace(DBG_MAN, "Connection established with the bap/fap server: %s (%s)",
        po->baps,
        inet_ntop(AF_INET, &po->tcpsa.sin_addr, dst, sizeof(dst)));

  char command[128];

  po->ipmode = ipmode;
  if (po->ipmode == ANDROID_MULTICAST) {
    sprintf(command, "setup a=%s p=%d t=%d r=%.2f ",
            getvrgroup(getCurrentChannelName()),
            po->bap_port,
            getvrttl(getCurrentChannelName()),
            getRate());
  }
  else {
    sprintf(command, "setup p=%d r=%.2f ",
            po->bap_port,
            getRate());
  }
  trace(DBG_MAN, "connectToBapServer: command=%s", command);
  write(po->sdtcp, command, strlen(command));	// setup packet
  return 1;
}

static
void disconnectFromBapServer(Android *po)
{
  if (po->sdtcp > 0) {
    if (po->ipmode == ANDROID_MULTICAST) {
      if (setsockopt(po->sdudp, IPPROTO_IP, IP_DROP_MEMBERSHIP, &(po->mreq), sizeof(po->mreq)) < 0)
        perror("IP_DROP_MEMBERSHIP");
    }
    if (po->sdudp > 0) {
      close(po->sdudp);
    }
    write(po->sdtcp, "stop", 4);
    close(po->sdtcp);
  }
  po->sdtcp = po->sdudp = 0;
}

void AndroidListen(Android *po, void *data, time_t sec, time_t usec)
{
#if IPMC_ENABLE
  if (po->status == ANDROID_INACTIVE) {
    if (!connectToBapServer(po, ANDROID_MULTICAST))
      return;
    if (!initUdpReceiver(po))
      return;
    po->status = ANDROID_LISTENING;
  }
  else
#endif
    notice("Listen without effect");
}

static
int readBapFrame(Android *po)
{
  fd_set set;
  struct timeval tv;
  socklen_t slen = sizeof(struct sockaddr_in);
  int len;
  float bap_rate, vreng_rate, ratio_rate;

  if (po->sdudp <= 0)
    return 0;	// not connected

  FD_ZERO(&set);
  FD_SET(po->sdudp, &set);
  tv.tv_sec = tv.tv_usec = 0;	// select passing
  if (select(FD_SETSIZE, &set, NULL, NULL, &tv) == 0)
    return 0;	// nothing to read

  memset(po->line, 0, BAPFAP_BUFSIZ);

  /* receive Bap line */
  if ((len = recvfrom(po->sdudp, po->line, BAPFAP_BUFSIZ, 0,
                      (struct sockaddr *) &(po->udpsa), &slen)) < 0) {
    trace(DBG_FORCE, "readBapFrame: err");
    return 0;
  }
  if (len == 0) { // eof
    trace(DBG_MAN, "readBapFrame: eof");
    //TODO DON'T DISCONNECT NEXT LINE
    //PD disconnectFromBapServer(po);	// send stop command
    //PD po->status = ANDROID_INACTIVE;
    return 0;
  }

  trace(DBG_MAN, "%s", po->line);
  if (strcmp(po->line, "") == 0)
    return 0;
  else if (po->line[0] == '#')
    return 0;
  else { // Header or Data is there
    char *l;

    l = strtok(po->line, " ");
    if (strncmp(l, "3.1", 3) == 0) {	// Header
      po->num_baps = NUM_BAPS;
      l = strtok(NULL, " ");
      l = strtok(NULL, " ");
      bap_rate = (float) atof(l);
      vreng_rate = getRate();
      ratio_rate = vreng_rate / bap_rate;
      po->bfflag = TYPE_BAP_V1;
      trace(DBG_MAN, "readBapHeader: num_baps=%d", po->num_baps);
    }
    else if (strncmp(l, "3.2", 3) == 0) {	// Header
      po->num_baps = NUM_BAPS_V2;
      l = strtok(NULL, " ");
      l = strtok(NULL, " ");
      bap_rate = (float) atof(l);
      vreng_rate = getRate();
      ratio_rate = vreng_rate / bap_rate;
      po->bfflag = TYPE_BAP_V2;
      trace(DBG_MAN, "readBapHeader: num_baps=%d", po->num_baps);
    }
    else if (strncmp(l, "FAP", 3) == 0) {	// Fap header
      po->num_baps = NUM_FAPS;
      l = strtok(NULL, " ");
      l = strtok(NULL, " ");
      bap_rate = (float) atof(l);
      vreng_rate = getRate();
      ratio_rate = vreng_rate / bap_rate;
      po->bfflag = TYPE_FAP;
      trace(DBG_MAN, "readBapHeader: num_baps=%d", po->num_baps);
    }
    else {	// Data
      if (po->bfflag == TYPE_BAP_V1 || po->bfflag == TYPE_BAP_V2) {
        if (po->num_baps == 0)
          po->num_baps = NUM_BAPS;

        for (int i=1; i <= po->num_baps; i++) {
          po->mask[i] = atoi(l);	//extract all the mask values
          l = strtok(NULL, " ");
        }
        trace(DBG_MAN, "readBapHeader: num_frame=%s", l);
        for (int i=1; i <= po->num_baps; i++) {
          if (po->mask[i] == 0)
	    continue;
	  if ((l = strtok(NULL, " ")) == NULL)
            break;
  
          if (i >= TR_VERTICAL && i <= TR_FRONTAL) {	// translations
            //PD po->bap[i] = atoi(l) / 1000;	// millimeters
            po->bap[i] = atoi(l);	// millimeters
          }
          else {	// rotations
            if (po->num_baps == NUM_BAPS_V2)
              po->bap[i] = atoi(l) / 555; //magic formula (555)	//GB
            else
              po->bap[i] = atoi(l) / 1745; //magic formula (1745)
             //po->bap[i] = (atoi(l)*180)/(int)(M_PI*1e5);
          }
          trace(DBG_MAN, "readBapHeader: l=%s bap[%d]=%d", l, i, po->bap[i]);
        }
      }
      else if (po->bfflag == TYPE_FAP) {
        if (po->num_baps == 0)
          po->num_baps = NUM_FAPS;

        for (int i=1; i <= po->num_baps; i++) {
          po->mask[i] = atoi(l);	//extract all the mask values
          l = strtok(NULL, " ");
        }
        trace(DBG_MAN, "readBapHeader: num_frame=%s", l);
        for (int i=1; i <= po->num_baps; i++) {
          if (po->mask[i] == 0)
	    continue;
	  if ((l = strtok(NULL, " ")) == NULL)
            break;
  
          po->fap[i] = atoi(l) / 5; //unknown formula
          trace(DBG_MAN, "readBapHeader: l=%s fap[%d]=%d", l, i, po->fap[i]);
        }
      }
      else {
        return 0;
      }
    }
    return 1;
  }
}

/* system of equations handling permanent motion */
void Android::changePermanent(float lasting)
{
  if (readBapFrame(this)) {
    if (bfflag == TYPE_BAP_V1 || bfflag == TYPE_BAP_V2) {
      if (mask[PELVIC_TILT]) bust->tilt_bust(bap[PELVIC_TILT]); //X
      if (mask[PELVIC_TORSION]) bust->turn_bust(bap[PELVIC_TORSION]); //Y
      if (mask[PELVIC_ROLL]) bust->roll_bust(bap[PELVIC_ROLL]); //Z
      if (mask[L_HIP_FLEXION]) legs[1]->rote_leg(bap[L_HIP_FLEXION]); //X
      if (mask[R_HIP_FLEXION]) legs[0]->rote_leg(bap[R_HIP_FLEXION]);
      if (mask[L_HIP_ABDUCT]) legs[1]->raise_leg(bap[L_HIP_ABDUCT]); //Z
      if (mask[R_HIP_ABDUCT]) legs[0]->raise_leg(bap[R_HIP_ABDUCT]);
      if (mask[L_HIP_TWIST]) legs[1]->turn_leg(bap[L_HIP_TWIST]); //Y
      if (mask[R_HIP_TWIST]) legs[0]->turn_leg(bap[R_HIP_TWIST]);
      if (mask[L_KNEE_FLEXION]) calf_l->bend_knee(bap[L_KNEE_FLEXION]); //X
      if (mask[R_KNEE_FLEXION]) calf_r->bend_knee(bap[R_KNEE_FLEXION]);
      if (mask[L_KNEE_TWIST]) calf_l->turn_knee(bap[L_KNEE_TWIST]); //Y
      if (mask[R_KNEE_TWIST]) calf_r->turn_knee(bap[R_KNEE_TWIST]);
      if (mask[L_ANKLE_FLEXION]) foot_l->bend_foot(bap[L_ANKLE_FLEXION]); //X
      if (mask[R_ANKLE_FLEXION]) foot_r->bend_foot(bap[R_ANKLE_FLEXION]);
      if (mask[L_ANKLE_TWIST]) foot_l->turn_foot(bap[L_ANKLE_TWIST]); //Y
      if (mask[R_ANKLE_TWIST]) foot_r->turn_foot(bap[R_ANKLE_TWIST]);
      if (mask[L_SHOULDER_FLEXION]) arm_l->rote_arm(bap[L_SHOULDER_FLEXION]);//X
      if (mask[R_SHOULDER_FLEXION]) arm_r->rote_arm(bap[R_SHOULDER_FLEXION]);
      if (mask[L_SHOULDER_ABDUCT]) arm_l->raise_arm(bap[L_SHOULDER_ABDUCT]); //Z
      if (mask[R_SHOULDER_ABDUCT]) arm_r->raise_arm(bap[R_SHOULDER_ABDUCT]);
      if (mask[L_SHOULDER_TWIST]) arm_l->turn_arm(bap[L_SHOULDER_TWIST]); //Y
      if (mask[R_SHOULDER_TWIST]) arm_r->turn_arm(bap[R_SHOULDER_TWIST]);
      if (mask[L_ELBOW_FLEXION]) forearm_l->bend_elbow(bap[L_ELBOW_FLEXION]);//X
      if (mask[R_ELBOW_FLEXION]) forearm_r->bend_elbow(bap[R_ELBOW_FLEXION]);
      if (mask[L_WRIST_FLEXION]) hand_l->tilt_hand(bap[L_WRIST_FLEXION]); //Z
      if (mask[R_WRIST_FLEXION]) hand_r->tilt_hand(bap[R_WRIST_FLEXION]);
      if (mask[L_WRIST_PIVOT]) hand_l->bend_hand(bap[L_WRIST_PIVOT]); //X
      if (mask[R_WRIST_PIVOT]) hand_r->bend_hand(bap[R_WRIST_PIVOT]);
      if (mask[L_WRIST_TWIST]) hand_l->turn_hand(bap[L_WRIST_TWIST]); //Y
      if (mask[R_WRIST_TWIST]) hand_r->turn_hand(bap[R_WRIST_TWIST]);
      if (mask[C1ROLL]) head->bend_head(bap[C1ROLL]); //X
      if (mask[C1TORSION]) head->turn_head(bap[C1TORSION]); //Y
      if (mask[C1TILT]) head->tilt_head(bap[C1TILT]); //Z
      if (mask[C4ROLL]) neck->bend_neck(bap[C4ROLL]); //X
      if (mask[C4TORSION]) neck->turn_neck(bap[C4TORSION]); //Y
      if (mask[C4TILT]) neck->tilt_neck(bap[C4TILT]); //Z
#if 0
      if (mask[69]) head->tilt_head(bap[69]);
      if (mask[70]) head->turn_head(bap[70]);
      if (mask[71]) head->bend_head(bap[71]);

      if (mask[69]) neck->tilt_neck(bap[69]);
      if (mask[70]) neck->turn_neck(bap[70]);
      if (mask[71]) neck->bend_neck(bap[71]);
#endif
      //PD if (mask[TR_VERTICAL]) tx = (GLfloat) bap[TR_VERTICAL] / 300.;
      if (mask[TR_VERTICAL]) tz = (GLfloat) bap[TR_VERTICAL] / 300.;
      if (mask[TR_LATERAL]) ty = (GLfloat) bap[TR_LATERAL] / 300.;
      //PD if (mask[TR_FRONTAL]) tz = (GLfloat) bap[TR_FRONTAL] / 300.;
      if (mask[TR_FRONTAL]) tx = (GLfloat) bap[TR_FRONTAL] / 300.;

      if (mask[RT_BODY_TURN]) rx = bap[RT_BODY_TURN];
      if (mask[RT_BODY_ROLL]) ry = bap[RT_BODY_ROLL];
      if (mask[RT_BODY_TILT]) rz = bap[RT_BODY_TILT];
    }
    else if (bfflag == TYPE_FAP) {
      if (mask[CLOSE_T_L_EYELID])
        faceObject->animate(CLOSE_T_L_EYELID, fap[CLOSE_T_L_EYELID]);
      if (mask[CLOSE_T_R_EYELID])
        faceObject->animate(CLOSE_T_R_EYELID, fap[CLOSE_T_R_EYELID]);
      if (mask[CLOSE_B_L_EYELID])
        faceObject->animate(CLOSE_B_L_EYELID, fap[CLOSE_B_L_EYELID]);
      if (mask[CLOSE_B_R_EYELID])
        faceObject->animate(CLOSE_B_R_EYELID, fap[CLOSE_B_R_EYELID]);
      if (mask[YAW_L_EYEBALL])
        faceObject->animate(YAW_L_EYEBALL, fap[YAW_L_EYEBALL]);
      if (mask[YAW_R_EYEBALL])
        faceObject->animate(YAW_R_EYEBALL, fap[YAW_R_EYEBALL]);
      if (mask[PITCH_L_EYEBALL])
        faceObject->animate(PITCH_L_EYEBALL, fap[PITCH_L_EYEBALL]);
      if (mask[PITCH_R_EYEBALL])
        faceObject->animate(PITCH_R_EYEBALL, fap[PITCH_R_EYEBALL]);
    }
  }
  else {
    //TODO: predictive interpollation
    ;
  }
  faceObject->animate();	//PD8
}

static u_int16 bap_offset_port = BAP_OFFSET_PORT;

/* create from a fileline */
void Android::creator(char *l)
{
  new Android(l);
}

Android::Android(char *l)
{
  char face_url[URL_LEN];

  l = parseName(l, this);
  l = parsePosition(l, this);
  memset(faces_url, 0, sizeof(faces_url));
  memset(face_url, 0, sizeof(face_url));
  for (int i=0; i<4; i++)
    skin[i] = skindef[i];
  l = parseURL(l, this);
  strcpy(baps, BAPFAP_HOST);	// default

  while (l) {
    // parse face
    if (strncmp(l, "face", 4) == 0) {
      l = strchr(l, '=');
      strcpy(face_url, ++l);
      l = strtok(NULL, SEP);
      continue;
    }
    // parse skin color
    if (strncmp(l, "color", 5) == 0) {
      l = strchr(l, '=');
      skin[0] = (float) atof(++l);
      l = strchr(l, ',');
      skin[1] = (float) atof(++l);
      l = strchr(l, ',');
      skin[2] = (float) atof(++l);
      l = strtok(NULL, SEP);
      continue;
    }
    // parse bap_server
    if (strncmp(l, "baps", 4) == 0) {
      l = strchr(l, '=');
      strcpy(baps, ++l);
      l = strtok(NULL, SEP);
      continue;
    }
  }

  char geom[80];
  sprintf(geom, "bbox,size=%.2f,%.2f,%.2f",0.12,0.07,0.85);
  if ((soh = parseGeometry(geom)) == NULL) {
    trace(DBG_FORCE, "android: can't create solid");
    return;
  }
  nature.movable = VR_NO_ELEM_MOVE;
  nature.renderable = VR_SPECIAL_RENDER;
  nature.collision = COL_NEVER;
  initializeObject(this, ANDROID_TYPE, VR_MOBILE);

  rotz = (float) RADIAN2DEGREE(pos.az);
  rotx = (float) RADIAN2DEGREE(pos.ax);
  roty = 90;
  move.perm_sec = 1;
  bap_offset_port += 2;
  bap_port = getvrport(getCurrentChannelName()) + bap_offset_port;
  bfflag = 0;
  sdudp = 0;

  humanInit(this);

  if (*faces_url) {
    faceObject = new FACE(faces_url);	// PD
  }
  //PD else {
  //PD   faceObject = new FACE();	// YR
  //PD }
  if (*face_url) {
    trace(DBG_MAN, "android creator: face_url=%s", face_url);
    this->faceObject->loadFace(face_url);
  }

#ifndef STANDALONE
  AndroidListen(this, NULL, 0, 0);
#endif
}

static
void sendPlayToBapServer(Android *po, const char *bap_file)
{
  if (po->status == ANDROID_INACTIVE) {
    if (!connectToBapServer(po, ANDROID_UNICAST)) {
      return;
    }
  }
  if (po->sdtcp <= 0) {
    trace(DBG_FORCE, "sendPlayToBapServer: not connected");
    return;
  }

  char command[80];
  sprintf(command, "play f=%s ", bap_file);
  write(po->sdtcp, command, strlen(command));	// play packet
  if (po->status == ANDROID_INACTIVE) {
    if (!initUdpReceiver(po)) {
      perror("initUdpReceiver");
      return;
    }
    po->status = ANDROID_PLAYING;
  }
}

void Android::quit()
{
  disconnectFromBapServer(this);
  bap_offset_port = BAP_OFFSET_PORT;

  delete faceObject;	// YR
}

void AndroidHi(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "hi.bap"); }

void AndroidBye(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "bye.bap"); }

void AndroidAsk(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "rightarm.bap"); }

void AndroidDown(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "down.bap"); }

void AndroidUp(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "up.bap"); }

void AndroidNack(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "comprendpas.bap"); }

void AndroidClap(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "clap.bap"); }

void AndroidTest(Android *po, void *data, time_t sec, time_t usec)
{ sendPlayToBapServer(po, "test.bap"); }

void AndroidStop(Android *po, void *data, time_t sec, time_t usec)
{
  if (po->status == ANDROID_PLAYING || po->status == ANDROID_LISTENING) {
    po->status = ANDROID_INACTIVE;
    disconnectFromBapServer(po);
  }
}

/* android functions initialization */
void androidInitFuncList(void)
{
  setMethodFunc(ANDROID_TYPE, 0, WO_ACTION AndroidHi, "Hi");
  setMethodFunc(ANDROID_TYPE, 1, WO_ACTION AndroidBye, "Bye");
  setMethodFunc(ANDROID_TYPE, 2, WO_ACTION AndroidAsk, "Ask");
  setMethodFunc(ANDROID_TYPE, 3, WO_ACTION AndroidDown, "Down");
  setMethodFunc(ANDROID_TYPE, 4, WO_ACTION AndroidUp, "Up");
  setMethodFunc(ANDROID_TYPE, 5, WO_ACTION AndroidNack, "Nack");
  setMethodFunc(ANDROID_TYPE, 6, WO_ACTION AndroidClap, "Clap");
  setMethodFunc(ANDROID_TYPE, 7, WO_ACTION AndroidTest, "Test");
  setMethodFunc(ANDROID_TYPE, 8, WO_ACTION changeFace, "New");
  setMethodFunc(ANDROID_TYPE, 9, WO_ACTION changeMoveYes, "Yes");
  setMethodFunc(ANDROID_TYPE, 10, WO_ACTION changeMoveNo, "No");
  setMethodFunc(ANDROID_TYPE, 11, WO_ACTION changeMoveMouth, "Mouth");
  setMethodFunc(ANDROID_TYPE, 12, WO_ACTION changeMoveSmile, "Smile");
  setMethodFunc(ANDROID_TYPE, 13, WO_ACTION changeMoveSulk, "Sulk");
  setMethodFunc(ANDROID_TYPE, 14, WO_ACTION changeMoveEyeR, "Right");
  setMethodFunc(ANDROID_TYPE, 15, WO_ACTION changeMoveEyeL, "Left");
  setMethodFunc(ANDROID_TYPE, 16, WO_ACTION changeMoveNose, "Nose");
  setMethodFunc(ANDROID_TYPE, 17, WO_ACTION AndroidStop, "Stop");
  setMethodFunc(ANDROID_TYPE, 18, WO_ACTION AndroidListen, "Listen");
}

#endif // !VRENGD
