#include "global.h"

#include "net.h"
#include "netdefs.h"		/* SD_R_RTP */
#include "netImpl.h"
#include "netobj.h"		/* my_host_id */
#include "channel.h"
#include "payload.h"		/* isValidPayload */
#include "stat.h"		/* pkts_sent */
#include "rtpsess.h"		/* RtpDeleteSource */
#include "pkt.h"


extern int sendSRSDES(Channel *, struct sockaddr_in *);


/* Send a packet
 * Return result of sendto(): -1 if error, else >=0
 */
int sendPacket(int fd, u_int8 *pkt, int pkt_len, struct sockaddr_in *to)
{
  int r;

  if (fd < 0)
    return -1;

  /*
   * Send the packet
   */
  if ((r = sendto(fd, pkt, pkt_len, 0, (struct sockaddr*) to, sizeof(*to))) < 0) {
#if IPMC_ENABLE
    statSendPacket(pkt_len);
#endif
    return -1;
  }
  statSendPacket(pkt_len);
  return r;
}

/* Send a payload
   * TODO: send queue to group messages into only one packet
   * called by sendObj* (proto.c)
 */
int sendPayload(struct sockaddr_in *to, const Payload *ppload)
{
  u_int8 pkt[PKTSIZE];
  u_int8 *ppl;
  int pkt_len, fd_send_rtp;
  rtp_hdr_t *rtp_hdr = (rtp_hdr_t *) pkt;
  Channel *pchan;

  /*
   * initial checks
   */
  if (!isValidPayload(ppload)) {
    trace(DBG_FORCE, "sendPayload: invalid payload");
    return -1;
  }
  if (ppload->len == 0) {
    trace(DBG_NET, "sendPayload: payload empty");
    return -1;
  }
  if (to == NULL) {
    trace(DBG_FORCE, "sendPayload: socket NULL");
    return -1;
  }

  /*
   * Build the RTP Header
   */
  if ((pchan = getChannelbysa(to)) == NULL) {
    pchan = getCurrentChannel(); /* hack !!! */
    trace(DBG_FORCE, "sendPayload: pchan NULL, to=%x", to);
  }
  RtpCreateHeader(pchan->session, rtp_hdr, pchan->ssrc);

  /*
   * Build the Payload Header
   */
  ppl = pkt + RTP_HDR_SIZE;

  ppl[PAYLOAD_HDR_VERSION] = VREP_VERSION;
  ppl[PAYLOAD_HDR_FLAGS] = 1; /* payload header length in words */
#if WANT_IPV6
  ppl[PAYLOAD_HDR_FAMILY] = AF_INET6;
#else /* IPV4 */
  ppl[PAYLOAD_HDR_FAMILY] = AF_INET;
#endif /* WANT_IPV6 */

  ppl[PAYLOAD_HDR_LEN] = (ppload->len + 3 + PAYLOAD_HDR_SIZE) >> 2; /* words */
  pkt_len = RTP_HDR_SIZE + (ppl[PAYLOAD_HDR_LEN] <<2); /* bytes */

  memcpy(ppl+PAYLOAD_HDR_SIZE, ppload->data, ppload->len);
  ppl[PAYLOAD_HDR_SIZE + ppload->len] = 0;	/* end of payload chunk */

  trace(DBG_NET, "S: %02x%02x%02x%02x/%02x",
                     ppl[0], ppl[1], ppl[2], ppl[3], ppl[4]);

  /*
   * Find the file descriptor
   */
  if ((fd_send_rtp = getFdSendRTP(to)) <= 0)
    fd_send_rtp = pchan->sd[SD_W_RTP]; /* hack !!! */

  /*
   * Send the packet
   */
  sendPacket(fd_send_rtp, pkt, pkt_len, to);
  statSendRTP(pkt_len);

  /* Update SR, TODO: compute RCTP interval */
  pchan->session->sr.psent++;
  pchan->session->sr.osent += pkt_len;

  if (pkts_sent && ((pkts_sent % 100) == 0)) {
    /*
     * Send RTCP compound (SR + SDES)
     */
    RtpRefreshSDES(pchan->session);
    sendSRSDES(pchan, to);
  }

  return pkt_len;
}

/* Set common part of rtcp packet */
static
void setRTCPcommon(rtcp_common_t *prtcp_hdr, u_int8 pt)
{
  memset(prtcp_hdr, 0, sizeof(rtcp_common_t));	/* clear P:1, count:5 */
  prtcp_hdr->version = RTP_VERSION;
  prtcp_hdr->pt = pt;
}

/* Set a SR packet */
static
int setSR(rtcp_common_t *prtcp_hdr, u_int8* pkt, Channel *pchan)
{
  rtcp_sr_t sr = pchan->session->sr;
  struct timeval ts;
  int len = 0;

  setRTCPcommon(prtcp_hdr, RTCP_SR);
  prtcp_hdr->count = 0;	/* only one SSRC */
  sr.ssrc = htonl(pchan->ssrc);
  gettimeofday(&ts, NULL);
  sr.rtp_ts = htonl(ts.tv_sec*1000 + ts.tv_sec/1000);
  prtcp_hdr->length = htons(sizeof(rtcp_sr_t) >>2);
  memcpy(pkt, prtcp_hdr, sizeof(rtcp_common_t));
  len += sizeof(rtcp_common_t);
  memcpy(pkt+len, &sr, sizeof(rtcp_sr_t));
  len += sizeof(rtcp_sr_t);

  trace(DBG_RTP, "setSR: ssrc=%x len=%d", sr.ssrc, len);
  return len;
}

/* Set a SDES packet, fill rtcp buffer and return length */
static
int setSDES(rtcp_common_t *prtcp_hdr, u_int8* pkt, Channel *pchan)
{
  u_int32 ssrc = htonl(pchan->ssrc);
  u_int8 xitem = 0;
  u_int8 items[PKTSIZE-sizeof(rtcp_sr_t)-sizeof(rtcp_common_t)-sizeof(ssrc)];
  sdes_item *pchunk;
  int len = 0;

  setRTCPcommon(prtcp_hdr, RTCP_SDES);
  memset(items, 0, sizeof(items));
  prtcp_hdr->count = 1; /* Only one SDES with multiple chunks */

  /* fill chunks */
  for (pchunk = pchan->session->sdes; pchunk; pchunk = pchunk->next) {
    items[xitem++] = pchunk->type;
    items[xitem++] = pchunk->len;
    memcpy(items+xitem, pchunk->str, pchunk->len);
    xitem += pchunk->len;
  }
  items[xitem++] = RTCP_SDES_END; /* End of chunks is indicated by a zero */

  while (xitem % 4) /* Pad to 4 bytes word boundary and store zeros there */
    items[xitem++] = 0;

  /* SDES header */
  prtcp_hdr->length = htons(1 + (xitem >> 2));
  memcpy(pkt, prtcp_hdr, sizeof(rtcp_common_t));
  len += sizeof(rtcp_common_t);
  memcpy(pkt+len, &ssrc, sizeof(ssrc));
  len += sizeof(ssrc);
  memcpy(pkt+len, items, xitem);
  len += xitem;

  trace(DBG_RTP, "setSDES: ssrc=%x len=%d xitem=%d", pchan->ssrc, len, xitem);
  return len;
}

/* Set a BYE packet */
static
int setBYE(rtcp_common_t *prtcp_hdr, u_int8* pkt, Channel *pchan)
{
  u_int32 ssrc = htonl(pchan->ssrc);
  int len = 0;

  setRTCPcommon(prtcp_hdr, RTCP_BYE);
  prtcp_hdr->count = 1; /* me only says bye */
  prtcp_hdr->length = htons(1);
  memcpy(pkt, prtcp_hdr, sizeof(rtcp_common_t));
  len = sizeof(rtcp_common_t);
  memcpy(pkt+len, &ssrc, sizeof(ssrc));
  len += sizeof(ssrc);

  trace(DBG_RTP, "setBYE: ssrc=%x len=%d", ssrc, len);
  return len;
}

/*
 * Send a RTCP packet
 */
int sendRTCPPacket(Channel *pchannel, struct sockaddr_in *to, u_int8 pt)
{
  int pkt_len = 0, r, fd_send_rtcp;
  rtcp_common_t rtcp_hdr;
  struct sockaddr_in *sin_rtcp;
  u_int8 pkt[PKTSIZE];

  switch (pt) {
  case RTCP_SR: /* send a SR */
    pkt_len = setSR(&rtcp_hdr, pkt, pchannel);
    break;
  case RTCP_SDES: /* send a SDES */
    pkt_len = setSDES(&rtcp_hdr, pkt, pchannel);
    break;
  case RTCP_BYE: /* send a BYE */
    pkt_len = setBYE(&rtcp_hdr, pkt, pchannel);
    break;
  }
  if ((sin_rtcp = getSaRTCP(to)) == NULL) {
    trace(DBG_FORCE, "sendRTCPPacket: sin_rtcp NULL");
    return -1;
  }
  if ((fd_send_rtcp = getFdSendRTCP(sin_rtcp)) < 0) {
    trace(DBG_FORCE, "sendRTCPPacket: fd_send_rtcp <0");
    return -1;
  }

  r = sendPacket(fd_send_rtcp, pkt, pkt_len, sin_rtcp);
  statSendRTCP(pkt_len);
  return r; 
}

/*
 * Send a RTCP compound packet
 */
int sendSRSDES(Channel *pchannel, struct sockaddr_in *to)
{
  int pkt_len = 0, len = 0, r, fd_send_rtcp;
  rtcp_common_t rtcp_hdr;
  struct sockaddr_in *sin_rtcp;
  u_int8 pkt[PKTSIZE];

  /* send (SR + SDES) */
  len = setSR(&rtcp_hdr, pkt, pchannel);
  pkt_len += len;
  len = setSDES(&rtcp_hdr, pkt+len, pchannel);
  pkt_len += len;

  if ((sin_rtcp = getSaRTCP(to)) == NULL) {
    trace(DBG_FORCE, "sendSRSDES: sin_rtcp NULL");
    return -1;
  }
  if ((fd_send_rtcp = getFdSendRTCP(sin_rtcp)) < 0) {
    trace(DBG_FORCE, "sendSRSDES: fd_send_rtcp <0");
    return -1;
  }

  trace(DBG_RTP, "sendSRSDES: pkt_len=%d", pkt_len);
  r = sendPacket(fd_send_rtcp, pkt, pkt_len, sin_rtcp);
  statSendRTCP(pkt_len);
  return r; 
}

void sendBYE(Channel *pchannel)
{
  sendRTCPPacket(pchannel, pchannel->sa[SA_RTCP], RTCP_SR);
  sendRTCPPacket(pchannel, pchannel->sa[SA_RTCP], RTCP_BYE);
  trace(DBG_RTP, "sendBYE: on channel=%x", pchannel);
}


/*
 * Receive a packet
 *
 * put datas into the Payload and sender into sender
 * -1 if errorr, else >=0
   * TODO:
   * decoding multi-messages packets
 */
int recvPacket(int fd, struct sockaddr_in *sender, Payload *ppload)
{
  u_int8 rtp_hdr_size;
  u_int8 pkt[PKTSIZE];
  u_int8 *ppl;
  u_int8 payload_hdr_size;
  u_int8 payload_version;
  u_int8 len;
  int pkt_len;
  u_int16 payload_len;
  socklen_t l = sizeof(struct sockaddr_in);
  rtp_hdr_t *rtp_hdr = (rtp_hdr_t *) pkt;
  rtcp_common_t *rtcp_hdr = (rtcp_common_t *) pkt;

  assert(sender);
  memset(sender, 0, sizeof(*sender));
  if ((pkt_len = recvfrom(fd, pkt, sizeof(pkt), 0, (struct sockaddr *)sender, &l)) <0) {
    trace(DBG_FORCE, "recvfrom: %s on %d", strerror(errno), fd);
    return pkt_len;	/* here pkt_len < 0 -> error */
  }
  statReceivePacket(pkt_len);

  /*
   * Test if the packet was sent by myself
   */
#if NEEDLOOPBACK
  if (my_host_id == sender->sin_addr.s_addr) {
    // If two apps are running on the same machine,
    // you need to sort out the packets on something
    // else than just the host ID.
    if (ntohl(rtp_hdr->ssrc) == my_ssrc_id) {
      return 0; /* Loopback from same app */
    }
  }
#else
  // TODO: This is probably broken anyway, loopback or not
  if (my_host_id == ntohl(sender->sin_addr.s_addr)) {
     return 0;  /* Loopback: ignore it */
  }
#endif

  /*
   * Test if it is a valid RTP header
   */
  if (rtp_hdr->version == RTP_VERSION &&
      (rtp_hdr->pt == PAYLOAD_TYPE || rtp_hdr->pt == PAYLOAD_TYPE_V1)) {
    /*
     * it's a RTP packet
     */
    u_int16 seq;	/* RTP seq */
    source_info *psi;

    rtp_hdr_size = RTP_HDR_SIZE;
    seq = ntohs(rtp_hdr->seq);
    if ((psi = RtpGetSource(ntohl(rtp_hdr->ssrc))) != NULL) {
      source *s;

      psi->rr.lost += seq - psi->rr.last_seq - 1;
      psi->rr.last_seq = seq;
      s = &(psi->s);
      RtpUpdateSeq(s, seq);
      psi->extended_max = s->cycles + s->max_seq;
      psi->expected = psi->extended_max - s->base_seq + 1;
      psi->lost = psi->expected - s->received;	/* number of packets lost */
    }
  }
  else if (rtcp_hdr->version == RTP_VERSION) {
    /*
     * it's a RTCP packet
     */
    switch (rtcp_hdr->pt) {
      case RTCP_SR:
      case RTCP_SDES:
      case RTCP_BYE:
      case RTCP_APP:
        return recvRTCPPacket(sender, pkt, pkt_len);
      case RTCP_RR:	/* why send a RR ? we only send SR */
        return 0;	/* like action done */
      default:
        return -3;	/* unknown RTCP packet type */
    }
  }
  else {
   /*
    * it's an UDP packet, old version of VREng <= 1.5.8
    */
    rtp_hdr_size = 0;
  }

  ppl = pkt + RTP_HDR_SIZE;
  trace(DBG_NET, "R: %02x%02x%02x%02x/%02x",ppl[0],ppl[1],ppl[2],ppl[3],ppl[4]);

  /*
   * Compatibility with older VREP Protocol
   */
  payload_version = VREP_VERSION;
  payload_hdr_size = PAYLOAD_HDR_SIZE;
  len = PAYLOAD_HDR_LEN;
  sender->sin_family = ppl[PAYLOAD_HDR_FAMILY];
  if (ppl[PAYLOAD_HDR_FLAGS] == 1)
    payload_len = ppl[len] << 2; /* length in words */
  else
    payload_len = ppl[len];	/* length in bytes <= 1.5.8 */

  if (ppl[PAYLOAD_HDR_VERSION] != VREP_VERSION &&
      ppl[PAYLOAD_HDR_FAMILY]  != AF_INET) {
    /* old version <= 1.5.8 */
    payload_version = 1;
    payload_hdr_size = PAYLOAD_HDR_SIZE_V1;
    len = PAYLOAD_HDR_LEN_V1;
    sender->sin_family = AF_INET;
    memcpy(&sender->sin_addr.s_addr, ppl+PAYLOAD_HDR_SRCID_V1, 4);
    memcpy(&sender->sin_port, ppl+PAYLOAD_HDR_PORTID_V1, 2);
  }

  /*
   * Check if valid payload header
   */
  if (pkt_len != rtp_hdr_size + payload_len || payload_len < payload_hdr_size) {
    warning("recvPacket: wrong size, pkt_len=%d pl_len=%d rtp_hdr_size=%d",
             pkt_len, payload_len, rtp_hdr_size);
    trace(DBG_FORCE, "R: %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
                         pkt[0], pkt[1], pkt[2], pkt[3],
                         pkt[4], pkt[5], pkt[6], pkt[7],
                         pkt[8], pkt[9], pkt[10], pkt[11]);
    trace(DBG_FORCE, "R: %02x%02x%02x%02x/%02x",
                         ppl[0], ppl[1], ppl[2], ppl[3], ppl[4]);
    trace(DBG_NET, "wrong packet size, pkt_len=%d pl_len=%d",
                   pkt_len, payload_len);
    return -2;	/* unknown packet type */
  }

  /*
   * Set Payload
   */
  resetPayload(ppload);
  ppload->len = payload_len - payload_hdr_size;
  memcpy(ppload->data, ppl+payload_hdr_size, ppload->len);

  return pkt_len;
}

int recvRTCPPacket(struct sockaddr_in *sender, u_int8 *pkt, int pkt_len)
{
  rtcp_common_t rtcp_hdr;
  rtcp_t *end;
#ifdef CHECK_RTCP_VALIDITY
  rtcp_t *r;
#endif
  u_int16 length = 0;
  u_int32 ssrc = 0;
  int len = 0;
  Channel *pchan = getCurrentChannel();	/* maybe indeterminated */

  /* we are supposed to receive compounds RTCP packets */
  while (len < pkt_len) {
    memcpy(&rtcp_hdr, pkt+len, sizeof(rtcp_common_t));
    length = ntohs(rtcp_hdr.length);
    //DAX end = (rtcp_t *) ((u_int32 *) &rtcp_hdr + (length<<2) + sizeof(rtcp_common_t));
    len += sizeof(rtcp_common_t);

#ifdef CHECK_RTCP_VALIDITY
    /* Check RTCP validity */
    r = (rtcp_t *) &rtcp_hdr;
    do
      r = (rtcp_t *) ((u_int32 *) r + ntohs(r->common.length + sizeof(rtcp_common_t)));
    while (r < end && r->common.version == RTP_VERSION);
    if (r != end) {
      trace(DBG_FORCE, "RTCP wrong size: r=%x end=%x length=%d pkt_len=%d sizeof(rtcp_common_t)=%d", r, end, length<<2, pkt_len, sizeof(rtcp_common_t));
      len += (length << 2);
      continue;
    }
#endif

    switch (rtcp_hdr.pt) {

     /* Receive a SR */
      case RTCP_SR: {
        rtcp_sr_t sr;
        source_info *psi;

        trace(DBG_RTP, "Got SR: length=%d len=%d", length<<2, len);
        memcpy(&sr, pkt+len, sizeof(rtcp_sr_t));
        if ((psi = RtpGetSource(ntohl(sr.ssrc))) != NULL)
          psi->sr = sr;
        len += (length << 2);
      }
      break;

      /* Receive a SDES */
      case RTCP_SDES: {
        sdes_item *sitem;
        source_info *psi;
        u_int8 *p = pkt + len;

        end = (rtcp_t *) ((u_int32 *) p + (length<<2) + sizeof(rtcp_common_t));
        memcpy(&ssrc, p, sizeof(ssrc));
        p += sizeof(ssrc);
        trace(DBG_RTP, "Got SDES: ssrc=%x pkt_len=%d length=%d len=%d p=%x",
              ntohl(ssrc), pkt_len, length<<2, len, p);
        if ((psi = RtpGetSource(ntohl(ssrc))) != NULL) { /* source found */
          sitem = &(psi->sdes);	/* first sitem which ever exists */ 

          /* Parse SDES chunks */
          while (p < (u_int8 *) end) {
            trace(DBG_RTP, "SDES: sitem=%x p=%x end=%x", sitem, p, end);
            sitem->type = *p++;
            sitem->len = *p++;
            if (sitem->str == NULL) {	/* free asumption */
              /* alloc string space */
              if ((sitem->str = (u_int8 *) calloc(1, sitem->len + 1)) == NULL) {
                trace(DBG_RTP, "SDES: can't alloc sitem->str");
                return -1;
              }
            }
            memcpy(sitem->str, p, sitem->len);
            sitem->str[sitem->len] = 0; /* now string null terminated */
            trace(DBG_RTP, "SDES: %s [%d]", sitem->str, sitem->len);
            p += sitem->len;	/* next item */
            if (*p == RTCP_SDES_END) {
              trace(DBG_RTP, "end SDES: p=%x end=%x", p, end);
              break;	/* end of SDES packet */
            }

            if (sitem->next == NULL) {
              /* alloc next item */
              if ((sitem->next = (sdes_item *) calloc(1, sizeof(sdes_item))) == NULL) {
                trace(DBG_FORCE, "SDES: can't alloc next sitem");
                return -1;
              }
            }
            sitem = sitem->next;
          }
        }
        len += (length << 2);
      }
      break;

     /* Receive a BYE */
      case RTCP_BYE: {
        memcpy(&ssrc, pkt+len, sizeof(ssrc));
        trace(DBG_RTP, "Got BYE: ssrc=%x", ntohl(ssrc));
#ifdef DELETESESSION
        RtpDeleteSession(ntohl(ssrc));
#else
        RtpDeleteSource(pchan->session, ntohl(ssrc));
#endif
        len += (length << 2);
      }
      break;

     /* Receive a APP */
      case RTCP_APP:
        trace(DBG_RTP, "Got APP");

     /* Receive something unknown */
      default:
        len += (length << 2);
        trace(DBG_FORCE, "Got RTCP unknown type: pt=%u, pkt_len=%d len=%d",
                         rtcp_hdr.pt, pkt_len, len);
        trace(DBG_FORCE, "%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
                         pkt[0], pkt[1], pkt[2], pkt[3],
                         pkt[4], pkt[5], pkt[6], pkt[7],
                         pkt[8], pkt[9], pkt[10], pkt[11]);
        return -3;
    }
  }
  return 0;
}
