/* ==================================================== ======== ======= *
 *
 *  xwin.cc : X-Window initialization and management for the VREng GUI
 *  NOTE: this file should be common to all X-Window GUI variants
 *
 *  VREng Project [Elc::2000]
 *  Author: Eric Lecolinet
 *
 *  (C) 1999-2000 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 */

#ifndef VRENGD

#include "global.h"
#include "zv.h"		/* RenderIniet */
#include "net.h"
#include "stat.h"
#include "world.h"

#include "guiclass.h"
#include "gui.h"
#include "guiImpl.hh"
#include "widgets.hh"


// cheks regularly for various updates
static void netTimeOut(XtPointer pgui, XtIntervalId *iid);

// called once after the mainloop is started for init. World mgt and rendering
static Boolean initWorldAndRenderingWorkProc(XtPointer);

// calculates the New World Order and renders it
static Boolean renderingWorkProc(XtPointer pgui);

// detects when the toplevel is resized, moved, etc.
static void configureEH(Widget, XtPointer pgui, XEvent *ev, Boolean*);

// inits. the GL rendering window (and the GL Visual and Colormap)
static void initGLZone(GUI *g);

// Xlib error handler
static int X11ErrorHandler(Display *d, XErrorEvent *ev);
// Xlib error status: if != 0 -> X fatal error
static int x11error;
// this variable will point to the previous (= standard) error handler
static int (*std_x11error_handler)(Display *, XErrorEvent *);


/* ==================================================== ======== ======= */


void GUI::createToplevel(int argc, char *argv[], char **fallback_resources) {

  toplevel = XtVaAppInitialize(&appContext, "Vreng",
		               NULL, 0, &argc, argv, fallback_resources, NULL);
  if (!toplevel)
    fatal("GUI::createToplevel: Toplevel could not be created");

  display = XtDisplay(toplevel);
}

/* ==================================================== ======== ======= */


void GUI::initX() {
  // note that toplevel, display and appContext 
  // have been  previously initialized
  Dimension ww, hh;

  /* handling errors */
  x11error = 0;
  std_x11error_handler = XSetErrorHandler(X11ErrorHandler);

  // makes the toplevel window visible
  XtRealizeWidget(toplevel);

  // get the toplevel size (which is stored for managing resizes)
  XtVaGetValues(toplevel, XtNwidth, &ww, XtNheight, &hh, NULL);
  appWidth = ww; appHeight = hh;

  // initialize OpenGL, its visual and the rendering window
  // !CAUTION: must be called after XtRealizeWidget
  initGLZone(this);
}


/* ==================================================== ======== ======= */


void GUI::mainLoop() {

  //!!!!!!
  if (vrengInitCB) vrengInitCB();

  // this fct. is called once for init. World Management and Rendering
  // Note: as this init. may be quite long, it is postponed so that
  // we can start the main loop immediately and make the "Please wait"
  // message appear
  XtAppAddWorkProc(appContext, initWorldAndRenderingWorkProc, (XtPointer)this);

  // This handler is called each time the window is resized, moved, etc.
  // Note: we don't use ResizeRedirectMask that is intended for Window Managers
  XtAddEventHandler(toplevel, StructureNotifyMask, False, 
		    configureEH, (XtPointer)this);

  // fprintf(stderr, ">START MainLoop!!\n");
  XtAppMainLoop(appContext);
}


/* ==================================================== ======== ======= */
// called once after the mainloop is started for init. World mgt and rendering


static Boolean initWorldAndRenderingWorkProc(XtPointer pgui) {
  GUI *gui = (GUI*)pgui;

  // fonction d'initalisation longue qui est lancee apres l'ouverture de la fenetre
  // en pratique c'est WmgtInit qui DOIT de plus etre lancee APRES initGLZone

  // fprintf(stderr, "\n!Initializing the World! Please wait ...\n\n");
  // if (gui->vrengInitCB) gui->vrengInitCB();

  // makes the GL window appear on the top of the toplevel
  XMapWindow(gui->display, gui->glwin);

  // closes the "Please Wait" alert box
  gui->guiWidgets->showAlertBox(false);

  stopTime(&ptime_init);

  // renderingWorkProc is called regularly by the main loop when it is 
  // idle (ie. when current event processing is over and timeOut fcts.
  // havee been processed)
  XtAppAddWorkProc(gui->appContext, renderingWorkProc, (XtPointer)gui);

  // netTimeOut is called regularly during the session
  XtAppAddTimeOut(gui->appContext, NET_TIMEOUT, netTimeOut, (XtPointer)gui);

  // the GUI is ready for rendering
  gui->readyAndVisible = TRUE;

  return True;  // dont' call it again
}

/* ==================================================== ======== ======= */
// calculates the New World Order and renders it


static Boolean renderingWorkProc(XtPointer pgui) {
  GUI *gui = (GUI*)pgui;

  if (gui->pendingPostponedKRs())  // at least one postponed Key Release event
    gui->flushPostponedKRs();

  // the GUI is ready for rendering and is visible
  // (no rendering is performed while iconified)
  if (gui->readyAndVisible) {      

    // Performs virtual world calculation (cf. wmgt.c)
    startTime(&ptime_world);
    doWorldCalculation(ptime_world.start.tv_sec, ptime_world.start.tv_usec);
    stopTime(&ptime_world);
     
    // compute the 3D
    startTime(&ptime_render);
    Render();
    stopTime(&ptime_render);
     
    // displays on the GL rendering window
    startTime(&ptime_buffer);
    glXSwapBuffers(gui->display, gui->glwin);
    // useless: XFlush(gui->display);
    stopTime(&ptime_buffer);
    
    // count cycles
    gui->cycles++;
  }
  return False;  // call this function again
}


/* ==================================================== ======== ======= */
// Handlers and TimeOuts


// This timeOut cheks regularly if various updates are needed

static void netTimeOut(XtPointer pgui, XtIntervalId *iid) {
  GUI *gui = (GUI*)pgui;

  // networkTimeout() cheks if various updates are needed
  u_int32 next_timeout = networkTimeout();

  // TimeOuts are only called ONCE by Xt ==> register this callback again
  // with the same call_data
  XtAppAddTimeOut(gui->appContext, next_timeout,
		  netTimeOut, (XtPointer)gui);
}


/* ==================================================== ======== ======= */
// detects when the toplevel is resized, moved, iconified, etc.

static void configureEH(Widget, XtPointer pgui, XEvent *ev, Boolean*) {
  GUI *gui = (GUI*)pgui;

  if (ev->type == MapNotify) {
    // visible again: restart rendering
    gui->readyAndVisible = TRUE;
  }
  else if (ev->type == UnmapNotify) {
    // iconified => stop rendering
     gui->readyAndVisible = FALSE;
    // a quoi ca pourrait bien servir de reactualiser le Monde qd l'appli
    // est iconifiee ?
    //XtAppAddTimeOut(appContext, WMGT_TIMEOUT, TimeOut, (XtPointer)WMGTTO);
  }
  else if (ev->type == DestroyNotify) {
    fprintf(stderr, ">DestroyNotify!!\n");
  }

  // !Caution: we don't know if the window was moved or resized
  // => test if the size has changed
  //
  else if (ev->type == ConfigureNotify
	   && (gui->appWidth != ev->xconfigure.width
	       || gui->appHeight != ev->xconfigure.height)
	   ) {
    //printf("conf w=%d, h=%d\n", ev->xconfigure.width,ev->xconfigure.height);
    resources.width3D  += ev->xconfigure.width - gui->appWidth;
    resources.height3D += ev->xconfigure.height - gui->appHeight;
    gui->appWidth  = ev->xconfigure.width;
    gui->appHeight = ev->xconfigure.height;

    // change sizes of:
    // - the GL rendering X-Window
    XResizeWindow(gui->display, gui->glwin, 
		  resources.width3D, resources.height3D);
    // - the corresponding GLZone widget of the GUI
    gui->guiWidgets->resizeGLZone(resources.width3D, resources.height3D);

    // - the OpenGL Viewport
    RenderSetWindow(resources.width3D, resources.height3D);
  }
}

/* ==================================================== ======== ======= */
// called when something occurs on a file-descriptor

static void FDEvent(XtPointer, int *fd, XtInputId *iid) {
  incoming(*fd);
}

void GUI::addInputTable(int cnt, int *table, int table_no) {
  //printf("addInputTable: table_no=%d\n", table_no);
  inputTable[table_no] = (XtInputId*) malloc(cnt * sizeof(XtInputId));
  for (int k=0; k < cnt; k++)
    inputTable[table_no][k] = XtAppAddInput(appContext, table[k],
					    (XtPointer) XtInputReadMask, 
					    FDEvent, (XtPointer)NULL);
}

void GUI::removeInputTable(int cnt, int table_no) {
  //printf("removeInputTable: table_no=%d\n", table_no);
  for (int k=0; k < cnt; k++)
    XtRemoveInput(inputTable[table_no][k]);

  free(inputTable[table_no]);
  inputTable[table_no] = NULL;
}

/* ==================================================== ======== ======= */
// for Handling X errors

static int X11ErrorHandler(Display *d, XErrorEvent *ev) {
  /* errors for shared-mem */
  if (ev->request_code == 129) {
    fprintf(stderr, "Shared memory unavailable\n");
    x11error = 1;
  }
  else {
    QuitVreng(0);
    std_x11error_handler(d, ev);
    exit(-1);
  }
  return 0;
}


static void initGLZone(GUI *g) {
  Colormap cmap = None;
  Screen *scr = XtScreen(g->toplevel);
  int scr_no = XScreenNumberOfScreen(scr);
  int dummy, toplevel_depth;
  int glxmajor = 0, glxminor = 0;
  Window wtmp;
  

  /* Check if GLX is supported */
  if (!glXQueryExtension(g->display, &dummy, &dummy))
    warning("X server has no OpenGL GLX extension");

#if defined(WITH_TINYGL)
  glxmajor = 0, glxminor = 0;
#else
  /* Get GLX version */
  glXQueryVersion(g->display, &glxmajor, &glxminor);
  notice("glXQueryVersion: GLX%d.%d", glxmajor, glxminor);
#endif

  /* TODO: test if GLX1.3 to do glxCreateWindow */

  /* ?? POURQUOI 16 ?? */
  static int gl_options[] = {
    GLX_RGBA, GLX_DEPTH_SIZE, 16, GLX_DOUBLEBUFFER,
    None
  };
  
  // find a Visual that match requested depth and OpenGL options
  if (! (g->glvisual = glXChooseVisual(g->display, scr_no, gl_options)))
    fatal("could not get appropriate OpenGL visual");

  // messages
  XtVaGetValues(g->toplevel, XtNdepth, &toplevel_depth, NULL);
  trace(DBG_GUI, "initGLZone: toplevel_depth: %d, rendering zone depth: %d", 
	toplevel_depth, g->glvisual->depth);
  trace(DBG_GUI, "toplevel_depth=%d, visual.depth=%d, visualid=%x, class=%d", 
	toplevel_depth, g->glvisual->depth, g->glvisual->visualid, getVisualClass(g->glvisual));


  //  if (toplevel_depth == 8 || toplevel_depth == 16) {
  if (toplevel_depth > 16)
    cmap = DefaultColormapOfScreen(scr); // hum hum ????
  else {
    trace(DBG_GUI, "XCreateColormap");
    cmap = XCreateColormap(g->display, RootWindow(g->display, scr_no),
			   g->glvisual->visual, AllocNone);
    if (cmap == None) 
      warning("Could not create private Colormap");

    trace(DBG_GUI, "getVisualClass = %d [PseudoColor=%d, TrueColor=%d]", 
	  getVisualClass(g->glvisual), PseudoColor, TrueColor);

    if (getVisualClass(g->glvisual) == PseudoColor) {
      // astuce: recopier les premieres couleurs de la Colormap standard
      // dans la Colormap privee pour eviter que toutes les couleurs ne
      // changent (y compris le noir et le blanc!) qunad la souris change
      // de fenetre
      XColor colors[256];
      u_long i;

      // recuperer SHARED_COLOR_COUNT premieres couleurs de la shared Colormap
      trace(DBG_GUI, "DefaultColormapOfScreen");
      Colormap defcmap = DefaultColormapOfScreen(scr);
      for (i = 0; i < SHARED_COLOR_COUNT; i++)
        colors[i].pixel = i;

      trace(DBG_GUI, "XQueryColors");
      XQueryColors(g->display, defcmap, colors, SHARED_COLOR_COUNT);
    }
  } 

  /* Create the OpenGL rendering window
   * IMPORTANT NOTES:
   * -1- XtWindow(g->toplevel) must be != None ==> the toplevel must already
   *     be REALIZED (= XtRealizeWidget must have been called previously)
   * -2- it is mandatory to specify the background color and a reasonnable size
   *     (otherwise, the X lib, a very intelligent system, will crash!)
   * -3- this window has special characteristics that may DIFFER from the toplevel
   *     (and other) window(s) because it is initialized from 'g->glvisual' 
   *     and the 'cmap' Colormap (which may differ from those of the toplevel).
   * NOTICEABLY:
   *     the depth of this window may be different from the toplevel depth !!!    
   *     (typically, the toplevel depth will be 8 on a Sparc (the default size)
   *     while the gl window depth may possibly be 8, 16 or 24)
   */
  {
    Window windows[2];
    XSetWindowAttributes wattr;
    wattr.colormap = cmap;
    wattr.background_pixel = wattr.border_pixel = BlackPixelOfScreen(scr);

    trace(DBG_GUI, "XCreateWindow");
    g->glwin = XCreateWindow(g->display, XtWindow(g->toplevel), 
			     // offsets of this win from toplevel origin
			     // (set by function GuiCreateWidgets)
			     g->guiWidgets->glwin_x,
			     g->guiWidgets->glwin_y,
			     resources.width3D, resources.height3D, 0, 
			     g->glvisual->depth,  //!!same depth as glvisual
			     (unsigned int)InputOutput, 
			     g->glvisual->visual,
			     CWColormap|CWBackPixel|CWBorderPixel, &wattr); 

    // fait en sorte que la Colormap change qunad la souris change de fenetre
    // ATT: si depth de toplevel est different de depth de cmap ne rien faire
    // sinon BadMatch X (et d'ailleurs, s'ils sont differents, c'est probablement
    // que g->glvisual->depth est en 24 bits et qu'on n'a pas besoin de changer
    // de colormap ...???)

    if (toplevel_depth == g->glvisual->depth
	&& toplevel_depth <= 16    /// hum hum ????
	) {
      trace(DBG_GUI, "XSetWindowColormap");
      XSetWindowColormap(g->display, XtWindow(g->toplevel), cmap);
      //    XSetWindowColormap(g->display, g->glwin, cmap);
    }

    // ne semble pas faire grand chose ?
    trace(DBG_GUI, "XInstallColormap");
    XInstallColormap(g->display, cmap);

    // set the Colormap property on the toplevel Shell and the 3d window
    windows[0] = XtWindow(g->toplevel);
    windows[1] = g->glwin;
    trace(DBG_GUI, "XtWindow");
    wtmp = XtWindow(g->toplevel);
    trace(DBG_GUI, "XSetWMColormapWindows");
    XSetWMColormapWindows(g->display, XtWindow(g->toplevel), windows, 2);
 
    // CRASH sometimes : avec OpenGL si ne peut pas allouer un Visual
    // raisonnable (il faut 24 bits (ou 16 ?) pour que ca marche)
    // NOTE:
    // une difference notable entre OpenGL et MesaGL dans le cas d'un
    // ecran 8 bits est que :
    // -- OpenGL alloue tout de meme un visual TrueColor. 
    //    Ca signifie que chaque composante RGB est codee sur 2 ou 3 bits !
    // -- MesaGL alloue par contre un visual PseudoColor ce qui permet 
    //    d'y associer une Colormap de 256 valeurs
    // C'est visiblement de la que vient le probleme ...

    // NB: the window is made visible after VrengWmgt init so that we can
    // show a "PLease Wait" msg first
  }

  // Create an OpenGL rendering context
  trace(DBG_GUI, "glXCreateContext");
  if (! (g->glxc = glXCreateContext(g->display, g->glvisual, None, True)))
    fatal("could not create OpenGL rendering context");

  /* here, we call ZLib initialization */
  trace(DBG_GUI, "glXMakeCurrent");
  glXMakeCurrent(g->display, g->glwin, g->glxc);
  trace(DBG_GUI, "RenderInit");
  RenderInit(resources.quality);
  trace(DBG_GUI, "RenderSetWindow");
  RenderSetWindow(resources.width3D, resources.height3D);
  trace(DBG_INIT, "GUI X-Window initialized");
}



#endif /* !VRENGD */
