/* Volq.cc. This program interacts with `vold', the volume daemon. 
   (c) 1997 David A. van Leeuwen

   $Id: volq.cc,v 1.2 1997/12/26 15:51:57 david Exp $
   It can be called being named `volq', `eject' or `unmount'.
   If used as `volq', it will report the status of the volumes.
 */

#include "vold.h"
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <fstream.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/cdrom.h>
#include <signal.h>

#define REVISION "$Revision: 1.2 $"
#define VOLQ_FMT "%-32s\t%-20s\n"

void perror(char * call, int fatal=0) 
{
     cerr << "volq: " << call << ": " << strerror(errno) << "\n";
     if (fatal) exit(errno);
}

void error(char * message, int fatal=0)
{
     cerr << message << "\n";
     if (fatal) exit(-fatal);
}

mel * start = NULL;

int read_mtab(void) {
     int nvol=0;
     struct mntent * mep;
     mel * tmp;
     FILE* mt;
     if ((mt = setmntent(MOUNTED, "r")) == NULL) perror("setmntent",1);
     while((mep = getmntent(mt))!=NULL) {
	  tmp = new(mel); 
	  tmp->next = start; tmp->mounted=not_mounted;
	  start = tmp; tmp->mep = new Mntent(mep);
	  start->dir = tmp->mep->dir.before("/",-1);
	  nvol++;
     }
     endmntent(mt);
     return nvol;
}  

mel * find_entry(const String& dir) // finds mount point or /etc/voltab entry
{
     mel * m;
     for(m=start; m; m=m->next) 
	  if (m->dir == dir || m->mep->dir == dir) break;
     return m;
}

main(int ac, char ** av) 
{
     // check if vold is running
     ifstream f(VOLPID, ios::in);
     int pid=-1;
     if (f && f >> pid) f.close();
     else error("Can't read from " VOLPID ", is vold running?");

  // parse the command line
     int arg=0, quit=0;
     String cmd=av[arg++], volume=""; // what is the program name?
     int eject = cmd.contains(Regex("eject$"));
     int unmount = cmd.contains(Regex("un?mount$"));
     int remount = cmd.contains(Regex("remount$"));
     int volq = cmd.contains(Regex("volq$"));
     if (volq && arg<ac) {
	  cmd=av[arg++];
	  eject = (cmd == "eject");
	  unmount = (cmd.matches(Regex("un?mount")));
	  remount = (cmd == "remount");
	  quit = (cmd == "quit");
	  volq = !(eject || unmount || remount || quit);
     }
     if ((eject || unmount || remount) && arg<ac) 
	  volume = av[arg++];

  // read mtab
     int nmounted = read_mtab();

     // execute according to requests
     if (volq) {				// inquiry
	  if (nmounted) printf(VOLQ_FMT, "Volume", "Mounted on");
	  else printf ("No volumes mounted by vold\n");
	  for (mel * i = start; i; i = i->next) {
		    if (hasmntopt(*i->mep, "vold")) {
			 String volume = i->mep->dir.after("/", -1);
			 printf(VOLQ_FMT, (const char *) volume, 
				(const char *) i->dir);
		    }
	  }
     } else if (eject || unmount || remount || quit) {
	  // check if the command pipe exists
	  struct stat s;
	  if (stat(VOLPIPE, &s)) perror ("can't stat " VOLPIPE, 1);
	  if (!S_ISFIFO(s.st_mode)) error(VOLPIPE " not a pipe", 1);
	  if (volume == "") volume = "/cdrom";
	  else volume.prepend("/cdrom/");
	  ofstream pipe(VOLPIPE, ios::out | ios::nocreate);
	  if (!pipe) perror("opening pipe", 1);
	  if (quit) {
	       uid_t uid = getuid();
	       if (uid) {
		    cerr << "Only root may do that\n";
		    exit (-2);
	       }
	       pipe << "quit";
	  } else {
	       cerr << "Trying to " << cmd << " " << volume << "\n";
	       if (eject) pipe << "eject";
	       else if (unmount) pipe << "umount";
	       else if (remount) pipe << "mount";
	       pipe << " " << volume;
	  }  
	  pipe << "\n";
	  pipe.close();
	  if (pid>0) kill(pid, SIGUSR1);
     }
	  
}

//
// Local variables:
// compile-command: "make volq"
// c-file-style: "K&R"
// End:
//
