(**
  A special group object for toolbar buttons.
**)

MODULE VOToolbar;

(*
    A layout group, that layouts a number of buttons horizontaly.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D := VODisplay,
       F := VOFrame,
       G := VOGUIObject;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button row is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 frame* : LONGINT;
               END;

  Toolbar*     = POINTER TO ToolbarDesc;
  ToolbarDesc* = RECORD(G.GroupDesc)
                     prefs     : Prefs;
                     frame     : F.Frame;
                     mWidth    : LONGINT;
                     equalSize : BOOLEAN;
                   END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.single3DOut;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(t : Toolbar);

  BEGIN
    t.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      t.SetBackgroundObject(p.background.Copy());
      t.backgroundObject.source:=t;
    END;
  END SetPrefs;


  PROCEDURE (t : Toolbar) Init*;

  BEGIN
    t.Init^;

    prefs.SetPrefs(t);

    t.mWidth:=0;
    t.equalSize:=FALSE;

    NEW(t.frame);     (* We create a frame object to frame our toolbar *)
    t.frame.Init;     (* We must of cause call Frame.Init of the frame *)
    (* The frame must be resizeable in all directions *)
    t.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (t : Toolbar) SetEqualSize*(equal : BOOLEAN);

  BEGIN
    t.equalSize:=equal;
  END SetEqualSize;

  PROCEDURE (t : Toolbar) CalcSize*(display : D.Display);

  VAR
    object : G.Object;

  BEGIN
    t.width:=0;
    t.height:=0;

    t.frame.SetInternalFrame(t.prefs.frame);
    t.frame.CalcSize(display);

    t.mWidth:=0;

    object:=t.list;
    WHILE object#NIL DO
      object.CalcSize(display);
      t.height:=G.MaxLong(t.height,object.oHeight);
      t.mWidth:=G.MaxLong(t.mWidth,object.oWidth);
      INC(t.width,object.oWidth);

      object:=object.next;
    END;

    IF t.equalSize THEN
      t.width:=t.count*t.mWidth;
    END;

    INC(t.width,t.frame.leftBorder+t.frame.rightBorder);
    INC(t.height,t.frame.topBorder+t.frame.bottomBorder);

    t.minWidth:=t.width;
    t.minHeight:=t.height;

    t.CalcSize^(display);
  END CalcSize;

  PROCEDURE (t : Toolbar) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    object : G.Object;
    pos    : LONGINT;

  BEGIN
    t.Draw^(x,y,draw);

    t.DrawBackground(t.x,t.y,t.width,t.height);

    t.frame.Resize(t.width,t.height);
    t.frame.Draw(t.x,t.y,draw);

    pos:=t.x+t.frame.leftBorder;
    object:=t.list;
    WHILE object#NIL DO
      IF t.equalSize THEN
        object.Resize(t.mWidth,t.height-t.frame.topBorder-t.frame.bottomBorder);
      ELSE
        object.Resize(-1,t.height-t.frame.topBorder-t.frame.bottomBorder);
      END;
      object.Draw(pos,t.y + (t.height-object.oHeight) DIV 2,draw);
      INC(pos,object.width);
      object:=object.next;
    END;
  END Draw;

  PROCEDURE (t : Toolbar) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    IF t.visible & t.Intersect(x,y,w,h) THEN
      t.Refresh^(x,y,w,h);
      t.frame.Refresh(x,y,w,h);
    END;
  END Refresh;

  PROCEDURE (t : Toolbar) Hide*;

  VAR
    help : G.Object;

  BEGIN
    IF t.visible THEN
      t.frame.Hide;
      help:=t.list;
      WHILE help#NIL DO
        help.Hide;
        help:=help.next;
      END;
    END;
    t.Hide^;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;

END VOToolbar.