(**
   Implements a textinput field. The string gadget is drag source and drop
   destination.

  TODO

  * Optimize redrawing to avoid flicker.

**)

MODULE VOString;

(*
    Implements a string gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

(*
  Contributions:

  * introducing centered and right alignment. Beat Christen, 08-Dec-97

*)


IMPORT D   := VODisplay,
       DD  := VODragDrop,
       E   := VOEvent,
       F   := VOFrame,
       G   := VOGUIObject,
       O   := VOObject,
       P   := VOPrefs,
       U   := VOUtil,
       V   := VOValue,

       CC  := CharClass,
       str := Strings;

CONST
  enteredMsg * = 0;
  escapeMsg  * = 1;

  leftAligned   * = 0;
  rightAligned  * = 1;
  centerAligned * = 2;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 frame*       : LONGINT; (* the frame to use for the button *)
                 gridDisable* : BOOLEAN;
               END;

  Alignment* = LONGINT;

  String*     = POINTER TO StringDesc;
  StringDesc* = RECORD (G.GadgetDesc)
                  prefs       : Prefs;
                  frame       : F.Frame;
            		  textAlign   : Alignment;
                  textWidth,
                  visWidth,
                  offset,
                  cursor,
                  markA,markB,
                  textPos     : LONGINT;
                  font        : LONGINT;
                  string-     : V.ValueModel;
                  selected    : BOOLEAN;
                END;

  EnteredMsg*     = POINTER TO EnteredMsgDesc;
  EnteredMsgDesc* = RECORD (O.MessageDesc)
                    END;

  EscapeMsg*      = POINTER TO EscapeMsgDesc;
  EscapeMsgDesc*  = RECORD (O.MessageDesc)
                    END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
    p.gridDisable:=TRUE;
  END Init;

  PROCEDURE (s : String) Init*;

  BEGIN
    s.Init^;

    s.prefs:=prefs;

    INCL(s.flags,G.canFocus);
    EXCL(s.flags,G.stdFocus);

    s.textWidth:=256;
    s.visWidth:=20;
    s.offset:=0;
    s.cursor:=0;
    s.markA:=-1;
    s.markB:=-1;
    s.selected:=FALSE;
    s.string:=NIL;
    s.textAlign:=leftAligned;
    s.font:=D.normalFont;

    NEW(s.frame);
    s.frame.Init;
    s.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  (**
    Set a new font to be used by the string gadget.
  **)

  PROCEDURE (s : String) SetFont*(font : LONGINT);

  BEGIN
    s.font:=font;
  END SetFont;

  (**
    Assign a string model to string gadget.
  **)

  PROCEDURE (s : String) SetModel*(model : V.ValueModel);

  BEGIN
    IF s.string#NIL THEN
      s.UnattachModel(s.string);
    END;
    s.string:=model;
    s.AttachModel(model);
  END SetModel;

  (**
    Set the maximum text with of string gadget and the visible with
    of the string gadget in letters.

    NOTE
    The with in letters will be estimated and thus the real width
    may differ from the given one.
  **)

  PROCEDURE (s : String) SetStringWidth*(width,visWidth : LONGINT);

  VAR
    help,help2 : U.Text;

  BEGIN
    IF (s.string#NIL) & ~s.string.IsNull() THEN
      NEW(help,width);
      help2:=s.string.GetText();
      COPY(help2^,help^);
      s.string.SetText(help);
    END;
    s.textWidth:=width;
    s.visWidth:=visWidth;
  END SetStringWidth;

  (**
    Set the aligment for the text within the string object.
  **)

  PROCEDURE (s: String) SetStringAlignment*(a:Alignment);
  BEGIN
    s.textAlign := a;
  END SetStringAlignment;

  PROCEDURE (s : String) CalcSize*(d : D.Display);

  VAR
    font : D.Font;

  BEGIN
    s.frame.SetInternalFrame(s.prefs.frame);
    s.frame.CalcSize(d);

    font:=d.GetFont(s.font);

    s.width:=s.frame.leftBorder+s.visWidth*d.spaceWidth+s.frame.rightBorder;
    s.height:=s.frame.topBorder+font.height+s.frame.bottomBorder;

    s.minWidth:=s.width;
    s.minHeight:=s.height;

    s.CalcSize^(d);
  END CalcSize;

  (**
    Sets the cursor under the mouse.
  **)

  PROCEDURE (s : String) GetCursorPos(x : LONGINT):LONGINT;

  VAR
    y,
    widthA,
    widthB  : LONGINT;
    help    : U.Text;
    extent  : D.FontExtentDesc;
    found   : BOOLEAN;
    font    : D.Font;

  BEGIN
    font:=s.draw.display.GetFont(s.font);

    IF x<s.textPos THEN
      RETURN 0;
    END;
    DEC(x,s.textPos);

    help:=s.string.GetTextCopy();

    found:=FALSE;
    y:=0;
    WHILE ~found & (y<=s.string.GetTextLength()) DO (* TODO: Optimize *)
      font.TextExtent(help^,y,{},extent);
      widthA:=extent.width;
      font.TextExtent(help^,y+1,{},extent);
      widthB:=extent.width;
      IF (widthA<=x) & (x<=widthB) THEN
        IF (x-widthA)>(widthB-widthA) DIV 2 THEN
          INC(y);
        END;
        found:=TRUE;
      ELSE
        INC(y);
      END;
    END;

    RETURN y;
  END GetCursorPos;

  PROCEDURE (s : String) Selected():BOOLEAN;

  BEGIN
    RETURN s.markA>=0;
  END Selected;

  PROCEDURE (s : String) SetSelection(a,b : LONGINT);

  BEGIN
    IF s.display.RegisterSelection(s,s.draw.vWindow) THEN
      s.markA:=a;
      s.markB:=b;
    END;
  END SetSelection;

  PROCEDURE (s : String) ClearSelection;

  BEGIN
    s.markA:=-1;
    s.markB:=-1;
    s.display.CancelSelection;
  END ClearSelection;

  PROCEDURE (s : String) DeleteSelection;

  VAR
    a,b : LONGINT;

  BEGIN
    IF s.Selected() THEN
      a:=s.markA;
      b:=s.markB;
      s.ClearSelection;
      s.string.Delete(a,b-a);
    END;
  END DeleteSelection;

  (* ---- Drag and drop stuff *)

  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (s : String) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  BEGIN
    IF s.visible & s.PointIsIn(x,y) & (s.string#NIL) & ~s.string.IsNull()
    & (drag OR ~s.disabled) THEN
      (* we can drag and drop *)
      RETURN s;
    ELSE
      RETURN NIL;
    END;
  END GetDnDObject;

  (**
    Return a list of supported datatypes.
  **)

  PROCEDURE (s : String) GetDragInfo*(VAR dragInfo : DD.DnDDataInfo);

  BEGIN
    dragInfo.AddDataType(DD.text,DD.none,{DD.copy,DD.move,DD.insert},DD.copy);
  END GetDragInfo;

  (**
    The object gets a list of supported datatypes of the drag object and
    has to choose one of them. If there is no fitting datatype it must return
    FALSE.
  **)

  PROCEDURE (s : String) GetDropDataType*(VAR dragInfo : DD.DnDDataInfo;
                                          VAR group, type, action : LONGINT):BOOLEAN;

  BEGIN
    group:=DD.text;
    type:=DD.joker;
    RETURN dragInfo.FindDataType(group,type,action);
  END GetDropDataType;

  (**
    Called if we want to get the drag & drop data from the object.
    data will we garanted to be initialized.
  **)

  PROCEDURE (s : String) GetDragData*(group, type, action : LONGINT):DD.DnDData;

  VAR
    data : DD.DnDStringData;
    text : U.Text;

  BEGIN
    IF (group=DD.text) & (s.string#NIL) & ~s.string.IsNull() THEN
      NEW(data);
      IF s.Selected() THEN
        NEW(data.string,s.markB-s.markA+2);
        text:=s.string.GetText();
        str.Extract(text^,SHORT(s.markA),SHORT(s.markB-s.markA),data.string^);
        data.string[s.markB-s.markA]:=0X;

        IF action=DD.move THEN
          s.string.Delete(SHORT(s.markA),SHORT(s.markB-s.markA));
        END;
      ELSE
        data.string:=s.string.GetTextCopy();
        IF action=DD.move THEN
          s.string.SetString("");
        END;
      END;
      RETURN data;
    ELSE
      RETURN NIL;
    END;
  END GetDragData;

  PROCEDURE (s : String) HandleDrop*(data : DD.DnDData; action : LONGINT):BOOLEAN;

  VAR
    x : LONGINT;

  BEGIN
    WITH data : DD.DnDStringData DO

      x:=0;
      WHILE data.string[x]#0X DO
        IF CC.IsControl(data.string[x]) THEN
          RETURN FALSE;
        END;
        INC(x);
      END;

      (*
        TODO: Exchange selection if selection exists
      *)

      IF s.string.GetTextLength()+LEN(data.string^)-1>s.textWidth THEN
        RETURN FALSE;
      END;

      CASE action OF
        DD.insert:
          INC(s.cursor,LEN(data.string^)-1);
          s.string.Insert(data.string^,s.cursor-LEN(data.string^)+1);
      | DD.copy,
        DD.move:
          s.string.SetString(data.string^);
      ELSE
        RETURN FALSE;
      END;
      RETURN TRUE;
    ELSE
      RETURN FALSE;
    END;
  END HandleDrop;

  PROCEDURE (s : String) DrawText;

  VAR
    x,y,
    width,
    height,
    cursorPos,
    markA,
    markB     : LONGINT;
    help      : U.Text;
    help2     : ARRAY 2 OF CHAR;
    extent1,
    extent2   : D.FontExtentDesc;
    string    : U.Text;
    font      : D.Font;

  BEGIN
    font:=s.display.GetFont(s.font);

    (* x = left starting position of string, y = top position *)
    x:=s.x+s.frame.leftBorder+s.display.spaceWidth DIV 2;
    y:=s.y+s.frame.topBorder;
    width:=s.width-s.display.spaceWidth-s.frame.leftBorder-s.frame.rightBorder;
    height:=s.height-s.frame.topBorder-s.frame.bottomBorder;

    IF s.disabled & ~s.prefs.gridDisable THEN
      s.draw.PushForeground(D.backgroundColor);
    ELSE
      s.draw.PushForeground(D.textBackgroundColor);
    END;

    s.draw.FillRectangle(s.x+s.frame.leftBorder,y,
                         s.width-s.frame.leftBorder-s.frame.rightBorder,
                         height);
    s.draw.PopForeground;

    IF s.string#NIL THEN
      IF s.string.IsNull() THEN
        NEW(string,1);
        string[0]:=0X;
      ELSE
        string:=s.string.GetText();
      END;
    ELSE
      string:=NIL;
    END;

    IF string#NIL THEN
      s.draw.InstallClip;
      s.draw.AddRegion(s.x+s.frame.leftBorder,s.y+s.frame.topBorder,
                       s.width-s.frame.leftBorder-s.frame.rightBorder,
                       s.height-s.frame.topBorder-s.frame.bottomBorder);


      IF s.cursor>str.Length(string^) THEN
        s.cursor:=str.Length(string^);
      END;

      IF s.textAlign = leftAligned THEN
        s.textPos:=x;
      ELSIF s.textAlign = rightAligned THEN
        s.textPos:=x+width -font.TextWidth(string^,str.Length(string^), {});
      ELSE (* center *)
        s.textPos:=x+(width -font.TextWidth(string^, str.Length(string^), {})) DIV 2;
      END;

      (* Copy string from 0 before cursor to help *)
      NEW(help,s.cursor+1);
      str.Extract(string^,0,SHORT(s.cursor),help^);

      (* calculate bounds of help^ *)
      font.TextExtent(help^,s.cursor,{},extent1);

      (* Calculate bound of first letter of help^ *)
      help2[0]:=string[0];
      help2[1]:=0X;
      font.TextExtent(help2,1,{},extent2);
      (*
        correct starting position of string by left hand
        space of starting character
       *)
      DEC(s.textPos,extent2.lbearing);

      (* Calculate cursor position *)
      cursorPos:=s.textPos+extent1.width;

      (* correct cursor position by first character of 2nd string *)
      IF s.cursor<str.Length(string^) THEN
        help2[0]:=string[s.cursor];
        font.TextExtent(help2,1,{},extent2);
        INC(cursorPos,extent2.lbearing);
      END;

      (* Make cursor visible in gadget by correcting starting offset *)
      IF cursorPos-s.offset>x+width-2 THEN
        INC(s.offset,cursorPos-s.offset-(x+width-2));
      ELSIF cursorPos-s.offset<x THEN
        DEC(s.offset,x-cursorPos+s.offset);
      END;

      (* correct textstart and cursorpos, too *)
      DEC(s.textPos,s.offset);
      DEC(cursorPos,s.offset+1);

      IF s.Selected() THEN
        (* Copy string from 0 before cursor to help *)
        NEW(help,s.markA+1);
        str.Extract(string^,0,SHORT(s.markA),help^);

        (* calculate bounds of help^ *)
        font.TextExtent(help^,s.markA,{},extent1);

        (* Calculate cursor position *)
        markA:=s.textPos+extent1.width;

        (* correct cursor position by first character of 2nd string *)
        IF s.markA<str.Length(string^) THEN
          help2[0]:=string[s.markA];
          font.TextExtent(help2,1,{},extent2);
          INC(markA,extent2.lbearing);
        END;


        (* Copy string from 0 before cursor to help *)
        NEW(help,s.markB+1);
        str.Extract(string^,0,SHORT(s.markB),help^);

        (* calculate bounds of help^ *)
        font.TextExtent(help^,s.markB,{},extent1);

        (* Calculate cursor position *)
        markB:=s.textPos+extent1.width;

        (* correct cursor position by first character of 2nd string *)
        IF s.markB<str.Length(string^) THEN
          help2[0]:=string[s.markB];
          font.TextExtent(help2,1,{},extent2);
          INC(markB,extent2.lbearing);
        END;

        DEC(markA,s.offset+1);
        DEC(markB,s.offset+1);

        s.draw.PushForeground(D.fillColor);
        s.draw.FillRectangle(markA,y,markB-markA+1,height);
        s.draw.PopForeground;
      END;

      (* Draw the string *)
      s.draw.PushFont(s.font,{});
      IF s.disabled & ~s.prefs.gridDisable THEN
        s.draw.PushForeground(D.disabledColor);
      ELSE
        s.draw.PushForeground(D.textColor);
      END;

      s.draw.DrawString(s.textPos,y+font.ascent,string^,str.Length(string^));

      s.draw.PopForeground;
      s.draw.PopFont;

      (* Drawing the cursor *)
      IF s.selected THEN
        s.draw.PushDrawMode(D.invert);
        s.draw.PushForeground(D.textColor);
        s.draw.DrawLine(cursorPos,y,cursorPos,y+font.height-1);
        s.draw.PopForeground;
        s.draw.PopDrawMode;
      END;

      s.draw.FreeLastClip;
    END;
  END DrawText;

  (**
    Clear the current selection.
  **)

  PROCEDURE (s : String) Deselect*;

  BEGIN
    s.ClearSelection;
    IF s.visible THEN
      s.DrawText;
    END;
  END Deselect;

  PROCEDURE (s : String) GetFocus*(event : E.Event):G.Object;

  BEGIN
    IF ~s.visible OR s.disabled OR (s.string=NIL) THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & s.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        s.selected:=TRUE;
        s.cursor:=s.GetCursorPos(event.x);
        s.ClearSelection;
        s.DrawText;
        RETURN s;
      ELSIF (event.type=E.mouseUp) & (event.button=E.dragDropButton) THEN
        s.selected:=TRUE;
        s.cursor:=s.GetCursorPos(event.x);
        s.ClearSelection;
        s.DrawText;
        IF ~s.display.QuerySelection(s.draw.vWindow,s,D.text) THEN END;
        RETURN s;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  PROCEDURE (s : String) HandleKeys(event :  E.KeyEvent):BOOLEAN;

  VAR
    count   : LONGINT;
    buffer  : ARRAY 256 OF CHAR;
    keysym  : LONGINT;
    entered : EnteredMsg;
    escape  : EscapeMsg;

  BEGIN
    keysym:=event.GetKey();
    CASE keysym OF
      E.return:
        IF ~s.HasFocus() THEN
          s.selected:=FALSE;
          s.DrawText;
        ELSE
          s.LeaveFocus;
        END;

        NEW(entered);
        s.Send(entered,enteredMsg);
    | E.escape:
        IF ~s.HasFocus() THEN
          s.selected:=FALSE;
          s.DrawText;
        END;

        NEW(escape);
        s.Send(escape,escapeMsg);
    | E.left:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor>0 THEN
            IF ~s.Selected() THEN
              DEC(s.cursor);
              s.SetSelection(s.cursor,s.cursor+1);
              s.DrawText;
            ELSIF (s.markA+1=s.markB) & (s.markB=s.cursor) THEN
              s.ClearSelection;
              DEC(s.cursor);
              s.DrawText;
            ELSIF s.cursor=s.markA THEN
              DEC(s.cursor);
              s.SetSelection(s.cursor,s.markB);
              s.DrawText;
            ELSIF s.cursor=s.markB THEN
              DEC(s.cursor);
              s.SetSelection(s.markA,s.cursor);
              s.DrawText;
            END;
          END;
        ELSE
          s.ClearSelection;
          IF s.cursor>0 THEN
            DEC(s.cursor);
          END;
          s.DrawText;
        END;
    | E.right:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor<s.string.GetTextLength() THEN
            IF ~s.Selected() THEN
              INC(s.cursor);
              s.SetSelection(s.cursor-1,s.cursor);
              s.DrawText;
            ELSIF (s.markA+1=s.markB) & (s.markA=s.cursor) THEN
              s.ClearSelection;
              INC(s.cursor);
              s.DrawText;
            ELSIF s.cursor=s.markB THEN
              INC(s.cursor);
              s.SetSelection(s.markA,s.cursor);
              s.DrawText;
            ELSIF s.cursor=s.markA THEN
              INC(s.cursor);
              s.SetSelection(s.cursor,s.markB);
              s.DrawText;
            END;
          END;
        ELSE
          s.ClearSelection;
          IF s.cursor<s.string.GetTextLength() THEN
            INC(s.cursor);
          END;
          s.DrawText;
        END;
    | E.home:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor#0 THEN
            s.SetSelection(0,s.cursor);
          END;
          s.cursor:=0;
          s.DrawText;
        ELSE
          s.ClearSelection;
          s.cursor:=0;
          s.DrawText;
        END;
    | E.end:
        IF event.qualifier=E.shiftMask THEN
          IF s.cursor#s.string.GetTextLength() THEN
            s.SetSelection(s.cursor,s.string.GetTextLength());
          END;
          s.cursor:=s.string.GetTextLength();
          s.DrawText;
        ELSE
          s.ClearSelection;
          s.cursor:=s.string.GetTextLength();
          s.DrawText;
        END;
    | E.backspace:
        IF s.Selected() THEN
          s.cursor:=s.markA;
          s.DeleteSelection;
        ELSIF s.cursor>0 THEN
          s.ClearSelection;
          DEC(s.cursor);
          s.string.Delete(s.cursor,1);
        END;
    | E.delete:
        IF s.Selected() THEN
          s.cursor:=s.markA;
          s.DeleteSelection;
        ELSIF s.cursor<s.string.GetTextLength() THEN
          s.ClearSelection;
          s.string.Delete(s.cursor,1);
        END;
    | E.insert:
        IF s.Selected() THEN
          s.DeleteSelection;
        END;
        IF ~s.display.QuerySelection(s.draw.vWindow,s,D.text) THEN
        END;
    ELSE
      IF ~s.disabled THEN
        count:=event.GetText(buffer);
        IF (count>0) & ~CC.IsControl(buffer[0]) THEN
          IF s.string.GetTextLength()+count<=s.textWidth THEN
            s.DeleteSelection;
            INC(s.cursor,count);
            s.string.Insert(buffer,s.cursor-count);
          ELSE
            RETURN FALSE;
            s.draw.Beep;
          END;
        ELSE
          IF buffer[0]#0X THEN
            (*Err.String(buffer); Err.Ln;
            Err.Hex(keysym,8); Err.Ln;*)
            s.draw.Beep;
            RETURN FALSE;
          END;
        END;
      END;
    END;
    RETURN TRUE;
  END HandleKeys;

  PROCEDURE (s : String) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    help : LONGINT;

  BEGIN
    WITH
      event : E.MouseEvent DO
        IF (event.type=E.mouseDown) THEN
          IF ~s.PointIsIn(event.x,event.y) THEN
            (* Action: "Entered", when text changed *)
            event.reUse:=TRUE;
            IF ~s.HasFocus() THEN
              s.selected:=FALSE;
              s.DrawText;
            END;
            RETURN TRUE;
          ELSE
            s.cursor:=s.GetCursorPos(event.x);
            s.ClearSelection;
            s.DrawText;
          END;
        ELSIF (event.button=E.button1) & (event.type=E.mouseUp) THEN
          help:=s.GetCursorPos(event.x);
          IF help>s.cursor THEN
            s.SetSelection(s.cursor,help);
            s.cursor:=help;
            s.DrawText;
          ELSIF help<s.cursor THEN
            s.SetSelection(help,s.cursor);
            s.DrawText;
          END;
        ELSIF (event.type=E.mouseUp) & (event.button=E.dragDropButton) THEN
          IF ~s.display.QuerySelection(s.draw.vWindow,s,D.text) THEN END;
        END;
    | event : E.MotionEvent DO
        IF event.qualifier={E.button1} THEN
          help:=s.GetCursorPos(event.x);
          IF help>s.cursor THEN
            s.SetSelection(s.cursor,help);
          ELSIF help<s.cursor THEN
            s.SetSelection(help,s.cursor);
          ELSE
            s.ClearSelection;
          END;
          s.DrawText;
        END;
    | event : E.KeyEvent DO
        IF (event.type=E.keyDown) & (s.string#NIL) THEN
          RETURN s.HandleKeys(event);
        END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  (**
    Called, when you got the keyboard focus.
  **)

  PROCEDURE (s : String) DrawFocus*;

  BEGIN
(*    s.CatchedFocus^;*)
    s.selected:=TRUE;

    IF s.string#NIL THEN
    (*
      s.markA:=0;
      s.markB:=s.string.length;
      s.cursor:=s.string.length;
    *)
    END;

    s.DrawText;
  END DrawFocus;

  (**
    Call, when the keyboard focus has been taken away from you.
  **)

  PROCEDURE (s : String) HideFocus*;

  BEGIN
(*    s.LostFocus^;*)
    s.selected:=FALSE;

    (*
      Hmmm, no good: LostFocus gets called when Quickhelp occures,
      window gets deactive, etc...
    *)
(*    s.markA:=-1;
    s.markB:=-1;*)

    s.DrawText;
  END HideFocus;

  PROCEDURE (s : String) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    IF event.type=E.keyDown THEN
      RETURN s.HandleKeys(event);
    ELSE
      RETURN FALSE;
    END;
  END HandleFocusEvent;

  PROCEDURE (s : String) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    s.Draw^(x,y,draw);

    s.frame.Resize(s.width,s.height);
    s.frame.Draw(s.x,s.y,draw);

    s.DrawText;

    IF s.disabled & s.prefs.gridDisable THEN
      s.DrawDisabled;
    END;
  END Draw;

  PROCEDURE (s : String) Hide*;

  BEGIN
    IF s.visible THEN
      s.frame.Hide;
      s.DrawHide;
      s.ClearSelection;
      s.Hide^;
    END;
  END Hide;

  PROCEDURE (s : String) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF s.visible THEN
      s.DrawText;
    END;
  END Resync;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOString.