(**
   A layouting class for grouping objects horizontaly or verticaly.
   The size of the objects can be changed by some sliders.
**)

MODULE VOPane;

(*
    Implements a layouting group.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       E := VOEvent,
       G := VOGUIObject,
       P := VOPrefs,
       V := VOVecImage;

CONST
  horizontal* = 0;
  vertical*   = 1;

  (* Differen modes for DrawSelection *)

  start       = 0;
  refresh     = 1;
  end         = 2;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 hMover,
                 vMover : LONGINT;
               END;

  Pane*     = POINTER TO PaneDesc;
  PaneDesc* = RECORD (G.GroupDesc)
                prefs  : Prefs;
                type   : LONGINT;
                oldPos : LONGINT;
                o1,o2,
                mover  : G.Object; (* The objects before and after the mover and the mover itself *)
              END;

VAR
  prefs : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hMover:=V.w95HMover(*V.hMover*);
    p.vMover:=V.w95VMover(*V.vMover*);
  END Init;


  PROCEDURE (p : Pane) Init*;

  BEGIN
    p.Init^;

    p.prefs:=prefs;

    p.type:=vertical;
  END Init;

  (**
    Tell if the objects should be arranged horizontally or vertically.
  **)

  PROCEDURE (p : Pane) Set*(type : LONGINT);

  BEGIN
    p.type:=type;
  END Set;

  (**
    Add an object to the group. Pane will add mover objects automatically
    when appropiate.
  **)

  PROCEDURE (p : Pane) Add*(object : G.Object);

  VAR
    image : V.VecImage;

  BEGIN
    IF p.list=NIL THEN
      p.list:=object;
    ELSE
      NEW(image);
      image.Init;
      image.parent:=p;
      IF p.type=vertical THEN
        image.Set(p.prefs.vMover);
        image.SetFlags({G.horizontalFlex});
      ELSE
        image.Set(p.prefs.hMover);
        image.SetFlags({G.verticalFlex});
      END;
      INC(p.count);

      p.last.next:=image;
      image.next:=object;
    END;
    p.last:=object;
    object.parent:=p;
    INC(p.count);
  END Add;

  PROCEDURE (p : Pane) CalcSize*(display : D.Display);

  VAR
    object : G.Object;

  BEGIN
    p.width:=0;
    p.height:=0;
    p.minWidth:=0;
    p.minHeight:=0;

    object:=p.list;
    WHILE object#NIL DO

      object.CalcSize(display);

      IF p.type=vertical THEN
        p.width:=G.MaxLong(p.width,object.oWidth);
        INC(p.height,object.oHeight);

        p.minWidth:=G.MaxLong(p.minWidth,object.oMinWidth);
        INC(p.minHeight,object.oMinHeight);
      ELSE
        INC(p.width,object.oWidth);
        p.height:=G.MaxLong(p.height,object.oHeight);

        INC(p.minWidth,object.oMinWidth);
        p.minHeight:=G.MaxLong(p.minHeight,object.oMinHeight);
      END;

      object:=object.next;
    END;

    p.CalcSize^(display);
  END CalcSize;

  PROCEDURE (p : Pane) MoveObjects(o1,o2,mover : G.Object; pos : LONGINT);

  VAR
    old : LONGINT;

  BEGIN
    o1.Hide;
    o2.Hide;
    mover.Hide;

    IF p.type=horizontal THEN
      pos:=G.MinLong(p.x+p.width-1-o2.oMinWidth-mover.oWidth,pos);
      pos:=G.MaxLong(p.x+o1.oMinWidth-1,pos);

      old:=p.width-mover.oWidth;

      o1.Resize(pos-o1.x,-1);
      o2.Resize(old-o1.oWidth,-1);
      o1.Draw(o1.x,o1.y,p.draw);
      mover.Draw(o1.x+o1.oWidth,mover.y,p.draw);
      o2.Draw(o1.x+o1.oWidth+mover.oWidth,o2.y,p.draw);
    ELSE
      pos:=G.MinLong(p.y+p.height-1-o2.oMinHeight-mover.oHeight,pos);
      pos:=G.MaxLong(p.y+o1.oMinHeight-1,pos);

      old:=p.height-mover.oHeight;

      o1.Resize(-1,pos-o1.y);
      o2.Resize(-1,old-o1.oHeight);
      o1.Draw(o1.x,o1.y,p.draw);
      mover.Draw(mover.x,o1.y+o1.oHeight,p.draw);
      o2.Draw(o2.x,o1.y+o1.oHeight+mover.oHeight,p.draw);
    END;
  END MoveObjects;

  PROCEDURE (p : Pane) DrawSelection(mode : INTEGER; x,y : LONGINT);

  BEGIN
    p.draw.PushDrawMode(D.invert);
    p.draw.PushForeground(D.blackColor);
    IF p.type=horizontal THEN
      IF (mode=refresh) OR (mode=end) THEN (* clear old *)
        p.draw.DrawLine(p.oldPos,p.y,p.oldPos,p.y+p.height-1);
      END;
      IF (mode=refresh) OR (mode=start) THEN (* draw new *)
        p.draw.DrawLine(x,p.y,x,p.y+p.height-1);
        p.oldPos:=x;
      END;
    ELSE
      IF (mode=refresh) OR (mode=end) THEN (* clear old *)
        p.draw.DrawLine(p.x,p.oldPos,p.x+p.width-1,p.oldPos);
      END;
      IF (mode=refresh) OR (mode=start) THEN (* draw new *)
        p.draw.DrawLine(p.x,y,p.x+p.width-1,y);
        p.oldPos:=y;
      END;
    END;
    p.draw.PopDrawMode;
    p.draw.PopForeground;

(*
    IF p.type=horizontal THEN
      p.MoveObjects(p.o1,p.o2,p.mover,x);
    ELSE
      p.MoveObjects(p.o1,p.o2,p.mover,y);
    END;*)
  END DrawSelection;

  PROCEDURE (p : Pane) GetFocus*(event : E.Event):G.Object;

  VAR
    object,
    last   : G.Object;

  BEGIN
    (*
      Going trough the list of objects and investigate every second object
      which should be a mover image.
     *)

    object:=p.list;
    IF object#NIL THEN
      last:=object;
      object:=object.next;
    END;

    WHILE object#NIL DO
      WITH event : E.MouseEvent DO
        IF object.PointIsIn(event.x,event.y)
        & (event.type=E.mouseDown) & (event.button=E.button1) THEN
          p.o1:=last;
          p.mover:=object;
          p.o2:=object.next;
          p.DrawSelection(start,event.x,event.y);
          RETURN p;
        END;
      ELSE
      END;
      object:=object.next;
      IF object#NIL THEN
        last:=object;
        object:=object.next;
      END;
    END;

    (* No mover image selected, we delegate event handling to our baseclass. *)

    RETURN p.GetFocus^(event);
  END GetFocus;

  PROCEDURE (p : Pane) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        p.DrawSelection(end,event.x,event.y);
        IF p.type=horizontal THEN
          p.MoveObjects(p.o1,p.o2,p.mover,event.x);
        ELSE
          p.MoveObjects(p.o1,p.o2,p.mover,event.y);
        END;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      p.DrawSelection(refresh,event.x,event.y);
    ELSE
    END;
    RETURN FALSE;
  END HandleEvent;


  PROCEDURE (p : Pane) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    pos  : LONGINT;
    help : G.Object;

  BEGIN
    p.Draw^(x,y,draw);

    IF p.type=vertical THEN

      (* Liste abarbeiten *)

      help:=p.list;
      pos:=p.y;
      WHILE help#NIL DO
        help.Resize(p.width,-1);
        IF help.next=NIL THEN
          help.Resize(-1,p.y+p.height-pos+1);
        END;
        help.Draw(p.x + (p.width-help.oWidth) DIV 2,pos,draw);
        INC(pos,help.oHeight);
        help:=help.next;
      END;

    ELSE

      (* Liste abarbeiten *)

      help:=p.list;
      pos:=p.x;
      WHILE help#NIL DO
        help.Resize(-1,p.height);
        IF help.next=NIL THEN
          help.Resize(p.x+p.width-pos+1,-1);
        END;
        help.Draw(pos,p.y + (p.height-help.oHeight) DIV 2,draw);
        INC(pos,help.oWidth);
        help:=help.next;
      END;

    END;
  END Draw;

  PROCEDURE (p : Pane) Refresh*(x,y,w,h : LONGINT);

  VAR
    help : G.Object;

  BEGIN
    IF p.visible & p.Intersect(x,y,w,h) THEN
      help:=p.list;
      WHILE help#NIL DO
        help.Refresh(x,y,w,h);
        help:=help.next;
      END;
    END;
  END Refresh;


  PROCEDURE (p : Pane) Hide*;

  VAR
    help : G.Object;

  BEGIN
    IF p.visible THEN
      help:=p.list;
      WHILE help#NIL DO
        help.Hide;
        help:=help.next;
      END;
    END;
    p.Hide^;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOPane.